/*
 * BeschriftungApplyer.java
 * eu.gronos.kostenrechner.controller (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller;

import java.awt.Component;
import java.awt.Window;
import java.util.List;

import javax.accessibility.AccessibleContext;
import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.border.TitledBorder;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.interfaces.Bebildert;
import eu.gronos.kostenrechner.interfaces.Vorsorgend;
import eu.gronos.kostenrechner.model.beschriftungen.Beschriftung;
import eu.gronos.kostenrechner.model.beschriftungen.BildOhneTitel;
import eu.gronos.kostenrechner.model.beschriftungen.LangBeschriftung;
import eu.gronos.kostenrechner.model.beschriftungen.LangVorsorgeBeschriftung;
import eu.gronos.kostenrechner.model.beschriftungen.NameContainer;
import eu.gronos.kostenrechner.model.beschriftungen.UnBeschriftung;
import eu.gronos.kostenrechner.model.beschriftungen.VorsorgendeBeschriftung;
import eu.gronos.kostenrechner.view.RechnerhammerIcons;

/**
 * Neue Klasse, die Anwendungen für {@link NameContainer} bereithält. Mit
 * {@link #beschrifte(Component, NameContainer)} kann man einen
 * {@link NameContainer} auf eine {@link Component} anwenden. Mit
 * {@link #changeTo(Vorsorgend, int)} kann man bei einer
 * {@link Vorsorgend}-Klasse Title und ShortDescription wechseln.
 *
 * @author Peter Schuster (setrok)
 * @date 01.02.2020
 *
 */
public class ComponentBeschrifter {

	public static int MAX_ZEICHEN_PRO_ZEILE = 40;

	/**
	 * Neue Methode setzt in der Basisklasse {@link NameContainer} nur
	 * comp.setName(), bei einer {@link Beschriftung} weiter den Text/Title/Label
	 * und die shortDescription/toolTip.
	 * 
	 * holt sich bei {@link UnBeschriftung} aus {@link UnBeschriftung#getRef()} den
	 * Title und die shortDescription und setzt die auch.
	 * 
	 * @param das   die {@link Component}, die die Methode beschriften soll
	 * @param damit der {@link NameContainer}
	 * @throws IllegalArgumentException wenn {@link JComponent} null ist
	 */
	public Component beschrifte(Component das, NameContainer damit) throws IllegalArgumentException {
		if (das == null)
			throw new IllegalArgumentException("Component darf nicht null sein!");
		// System.out.println("Beschreibe " + damit.getId());
		das.setName(damit.getId());
		if (damit instanceof Beschriftung) {
			deployTitle(das, (Beschriftung) damit);
			deployShortDescription(das, (Beschriftung) damit);
			deployMnemonic(das, (Beschriftung) damit);
			if (damit instanceof LangBeschriftung) {
				deployActionCommandKey(das, (LangBeschriftung) damit);
				deployImage(das, (LangBeschriftung) damit);
				deployAcceleratorKey(das, (LangBeschriftung) damit);
				deployLongDescription(das, (LangBeschriftung) damit);
			}
		} else if (damit instanceof UnBeschriftung) {
			// getRef().applyTo(comp) geht nicht, sonst würde auch ID geändert.
			deployAccessibleName(das, createCompleteTitle((UnBeschriftung) damit));
			// deployAccessibleDescription
			deployShortDescription(das, ((UnBeschriftung) damit).getRef());
			if (damit instanceof BildOhneTitel)
				deployImage(das, (Bebildert) damit);
		}
		return das;
	}

	/**
	 * Die Methode setzt die {@link Action#getValue(String)}s der {@link Action}
	 * 
	 * @param action {@link Action}
	 * @param damit  die {@link Beschriftung}
	 */
	public Action beschrifte(Action action, LangBeschriftung damit) {
		// action.putValue(Action.NAME, damit.getTitle());
		deployTitle(action, damit);
		deployShortDescription(action, damit);
		deployLongDescription(action, damit);
		deployActionCommandKey(action, damit);
		deployImage(action, damit);
		deployMnemonic(action, damit);
		deployAcceleratorKey(action, damit);
		return action;
	}

	/**
	 * Die Methode setzt einen {@link Beschriftung#getTitle()} aus den
	 * {@link Vorsorgend#getAlternativeTitles()}, sofern er enthalten ist, und die
	 * {@link Beschriftung#getShortDescription()} aus den
	 * {@link Vorsorgend#getAlternativeDescriptions()}.
	 * 
	 * Danach muss {@link #beschrifte(Component, NameContainer)} aufgerufen werden.
	 * 
	 * @param vorsorgend ein {@link Vorsorgend}, also eine
	 *                   {@link VorsorgendeBeschriftung} oder eine
	 *                   {@link LangVorsorgeBeschriftung}
	 * @param index      der Index des neuen Titles
	 * 
	 * @return die {@link VorsorgendeBeschriftung} mit dem neuen
	 *         {@link Beschriftung#getTitle()}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.Vorsorgend
	 */
	public Vorsorgend<?> changeTo(Vorsorgend<?> vorsorgend, int index) {
		// changeTitle(index);
		changeTitle(vorsorgend, index);
		changeDescription(vorsorgend, index);
		return vorsorgend;
	}

	/**
	 * Die Methode setzt über {@link #changeTo(Vorsorgend, int)} den
	 * {@link Beschriftung#getTitle()} und den
	 * {@link Beschriftung#getShortDescription()}. Dann formatiert sie den
	 * {@link Beschriftung#getTitle()} mit Hilfe der <code>args</code> über
	 * {@link String#format(String, Object...)}, sofern
	 * {@link Vorsorgend#isFormatted()} gesetzt ist.
	 * 
	 * Danach muss {@link #beschrifte(Component, NameContainer)} aufgerufen werden.
	 * 
	 * @param vorsorgend ein {@link Vorsorgend}, also eine
	 *                   {@link VorsorgendeBeschriftung} oder eine
	 *                   {@link LangVorsorgeBeschriftung}
	 * @param index      der Index des neuen Titles
	 * @param args       die weiteren Argumente für
	 *                   {@link String#format(String, Object...)}
	 * @return die {@link VorsorgendeBeschriftung} mit dem neuen
	 *         {@link Beschriftung#getTitle()}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.Vorsorgend
	 * @see java.lang.String#format(String, Object...)
	 */
	public Vorsorgend<?> changeTo(Vorsorgend<?> vorsorgend, int index, Object... args) {
		changeTo(vorsorgend, index);
		if (vorsorgend != null && vorsorgend.isFormatted()) {
			Beschriftung beschriftung = (Beschriftung) vorsorgend;
			beschriftung.setTitle(String.format(beschriftung.getTitle(), args));
		}
		return vorsorgend;
	}

	/**
	 * Die Methode setzt einen {@link Beschriftung#getTitle()} aus den
	 * {@link Vorsorgend#getAlternativeTitles()}, sofern der Text aus
	 * <code>title</code> in einer von ihnen enthalten ist, und die
	 * {@link Beschriftung#getShortDescription()} aus den
	 * {@link Vorsorgend#getAlternativeDescriptions()}.
	 * 
	 * Danach muss {@link #beschrifte(Component, NameContainer)} aufgerufen werden.
	 *
	 * @param vorsorgend ein {@link Vorsorgend}, also eine
	 *                   {@link VorsorgendeBeschriftung} oder eine
	 *                   {@link LangVorsorgeBeschriftung}
	 * @param title      den {@link Beschriftung#getTitle()}, der herausgesucht
	 *                   werden soll
	 * @return <code>true</code>, wenn er in {@link #getAlternativeTitles()}
	 *         gefunden wird
	 * @see eu.gronos.kostenrechner.interfaces.Vorsorgend
	 */
	public boolean changeTo(Vorsorgend<?> vorsorgend, String title) {
		int index = indexOf(vorsorgend.getAlternativeTitles(), title);
		if (index > -1) {
			changeTo(vorsorgend, index);
			return true;
		}
		Kostenrechner.getLogger().info(String.format("Beschriftung nicht gefunden: '%s'", title));
		return false;
	}

	/**
	 * Die Methode prüft, ob ein {@link String} in den <code>altenativen</code> ist.
	 * 
	 * @param alternativen die {@link List} aus {@link String}s mit
	 *                     {@link Vorsorgend#getAlternativeTitles()} oder
	 *                     {@link Vorsorgend#getAlternativeDescriptions()}
	 * @param neu          der zu prüfende Text als {@link String}
	 * @return den index von <code>neu</code> in <code>alternativen</code>
	 */
	public int indexOf(List<String> alternativen, String neu) {
		for (int index = 0; index < alternativen.size(); index++) {
			if (istGleicherText(alternativen.get(index), neu)) {
				return index;
			}
		}
		return -1;
	}

	/**
	 * Die Methode prüft, ob zwei {@link String}s unabhängig von
	 * Groß-/Kleinschreibung und führenden Leerzeichen gleich sind.
	 * 
	 * @param erste  {@link String}
	 * @param zweite {@link String}
	 * @return <code>true</code>, wenn beide gleich sind
	 */
	public boolean istGleicherText(String erste, String zweite) {
		//return erste.trim().toLowerCase().equals(zweite.trim().toLowerCase());
		return deWhiteSpace(erste).toLowerCase().equals(deWhiteSpace(zweite).toLowerCase());
	}

	/**
	 * Die Methode in dem übergebenen {@link String} nach einer bestimmten Anzahl
	 * von Zeichen ({@link #MAX_ZEICHEN_PRO_ZEILE}) einen Zeilenumbruch einzufügen,
	 * aber nicht mitten im Wort.
	 * 
	 * @param string der zu verarbeitende {@link String}
	 * @return der Ausgabe-{@link String} mit den eingefügten Zeilenumbrüchen.
	 */
	public String brecheZeilen(String string) {
		StringBuffer gesamt = new StringBuffer();
		StringBuffer zeile = new StringBuffer();
		String[] zeilen = string.trim().split("\\s+");
		for (String wort : zeilen) {
			if (zeile.length() + wort.length() > MAX_ZEICHEN_PRO_ZEILE) {
				gesamt.append(zeile);
				gesamt.append("\n");
				zeile = new StringBuffer();
			}
			zeile.append(wort);
			zeile.append(" ");

		}
		gesamt.append(zeile);
		gesamt.append("\n");
		return gesamt.toString();
	}

	/**
	 * Die Methode entfernt überflüssigen Whitespace aus einem String, der beim
	 * XML-Import entsteht.
	 * 
	 * @param string der {@link String}
	 * @return den um zuviel Whitespace bereinigten {@link String}
	 */
	public String deWhiteSpace(String string) {
		return string.replace('\t', ' ').replace('\n', ' ').replace('\r', ' ').replaceAll("\\s", " ")
				.replaceAll("\\s+", " ").trim();
	}

	/**
	 * Die Methode setzt den {@link Beschriftung#getTitle()} einer
	 * {@link Beschriftung} auf eine Vielzahl von {@link Component}s oder auf eine
	 * {@link Action}.
	 * 
	 * @param das          eine {@link Component} oder {@link Action}
	 * @param beschriftung die {@link Beschriftung}
	 */
	private void deployTitle(Object das, Beschriftung beschriftung) {
		String title = beschriftung.getTitle();
		if (das instanceof Action) {
			((Action) das).putValue(Action.NAME, title);
		} else if (das instanceof JComponent) {
			if (das instanceof JLabel)
				((JLabel) das).setText(title);
			else if (das instanceof AbstractButton) {
				if (((AbstractButton) das).getAction() != null)
					deployTitle(((AbstractButton) das).getAction(), beschriftung);
				else
					((AbstractButton) das).setText(title);
			} else if ((das instanceof JPanel || das instanceof JScrollPane)
					&& (((JComponent) das).getBorder() instanceof TitledBorder))
				((TitledBorder) ((JComponent) das).getBorder()).setTitle(title);
		} else if (das instanceof Window) {
			if (das instanceof JDialog)
				((JDialog) das).setTitle(title);
			else if (das instanceof JFrame)
				((JFrame) das).setTitle(title);
		} else if (das instanceof Component)
			deployAccessibleName((Component) das, title);
	}

	/**
	 * Die Methode setzt den {@link AccessibleContext#getAccessibleName()} einer
	 * {@link Component}
	 * 
	 * @param das   {@link Component}
	 * @param title der String, der als
	 *              {@link AccessibleContext#getAccessibleName()} gesetzt werden
	 *              soll.
	 */
	private void deployAccessibleName(Component das, String title) {
		if (das instanceof JScrollPane && ((JScrollPane) das).getVerticalScrollBar() != null
				&& ((JScrollPane) das).getVerticalScrollBar().getAccessibleContext() != null) {
			((JScrollPane) das).getVerticalScrollBar().getAccessibleContext().setAccessibleName(title);
		} else if (das.getAccessibleContext() != null) {
			das.getAccessibleContext().setAccessibleName(title);
		} else
			System.err.println("AccessibleContext null bei " + das.getClass().toString());
	}

	/**
	 * Die Methode verknüpft {@link Beschriftung#getTitle()},
	 * {@link UnBeschriftung#getPrefix()} und {@link UnBeschriftung#getSuffix()}
	 * 
	 * @param un die {@link UnBeschriftung}
	 * 
	 * @return den String
	 */
	private String createCompleteTitle(UnBeschriftung un) {
		String title = un.getRef().getTitle();
		if (title == null || title.trim().isEmpty())
			return "";
		title = deWhiteSpace(title);
		final StringBuilder sb = new StringBuilder(title);
		if (un.getPrefix() != null && !un.getPrefix().isEmpty())
			sb.insert(0, un.getPrefix() + " ");
		if (un.getSuffix() != null && !un.getSuffix().isEmpty())
			sb.append(" " + un.getSuffix());
		return sb.toString();
	}

	/**
	 * Die Methode setzt die {@link Beschriftung#getShortDescription()} einer
	 * {@link Beschriftung} auf eine Vielzahl von {@link Component}s oder auf eine
	 * {@link Action}. Sie setzt außerdem die
	 * {@link AccessibleContext#getAccessibleDescription()}.
	 * 
	 * @param das          eine {@link Component} oder {@link Action}
	 * @param beschriftung die {@link Beschriftung}
	 */
	private void deployShortDescription(Object das, Beschriftung beschriftung) {
		String shortDescription = beschriftung.getShortDescription();
		if (shortDescription != null && !shortDescription.isEmpty()) {
			shortDescription = deWhiteSpace(shortDescription);
			shortDescription = brecheZeilen(shortDescription);
		}
		if (das instanceof Action)
			((Action) das).putValue(Action.SHORT_DESCRIPTION, shortDescription);
		else if (das instanceof AbstractButton && ((AbstractButton) das).getAction() != null)
			deployShortDescription(((AbstractButton) das).getAction(), beschriftung);
		else if (das instanceof JComponent)
			((JComponent) das).setToolTipText(shortDescription);
		if (das instanceof Component && ((Component) das).getAccessibleContext() != null)
			((Component) das).getAccessibleContext().setAccessibleDescription(shortDescription);
	}

	/**
	 * Die Methode setzt das {@link Beschriftung#getMnemonic()} einer
	 * {@link Beschriftung} auf eine Vielzahl von {@link Component}s oder auf eine
	 * {@link Action}.
	 * 
	 * @param das          eine {@link Component} oder {@link Action}
	 * @param beschriftung die {@link Beschriftung}
	 */
	private void deployMnemonic(Object das, Beschriftung beschriftung) {
		if (beschriftung.hasMnemonic()) {// getMnemonic() != null
			if (das instanceof Action)
				((Action) das).putValue(Action.MNEMONIC_KEY, Integer.valueOf(beschriftung.mnemonicIntValue()));
			else if (das instanceof JComponent) {
				if (das instanceof JLabel)
					((JLabel) das).setDisplayedMnemonic(beschriftung.mnemonicIntValue());
				else if (das instanceof AbstractButton) {
					if (((AbstractButton) das).getAction() != null)
						deployMnemonic(((AbstractButton) das).getAction(), beschriftung);
					else
						((AbstractButton) das).setMnemonic(beschriftung.mnemonicIntValue());
				}
			}
		}
	}

	/**
	 * Die Methode setzt den {@link LangBeschriftung#getActionCommandKey()} einer
	 * {@link LangBeschriftung} auf einen {@link AbstractButton} oder auf eine
	 * {@link Action}.
	 * 
	 * @param das          ein {@link AbstractButton} oder eine {@link Action}
	 * @param beschriftung die {@link LangBeschriftung}
	 */
	private void deployActionCommandKey(Object das, LangBeschriftung beschriftung) {
		if (beschriftung.getActionCommandKey() != null) {
			if (das instanceof Action)
				((Action) das).putValue(Action.ACTION_COMMAND_KEY, beschriftung.getActionCommandKey());
			else if (das instanceof AbstractButton) {
				if (((AbstractButton) das).getAction() != null)
					deployActionCommandKey(((AbstractButton) das).getAction(), beschriftung);
				else
					((AbstractButton) das).setActionCommand(beschriftung.getActionCommandKey());
			}
		}
	}

	/**
	 * Die Methode setzt die {@link LangBeschriftung#getLongDescription()} einer
	 * {@link LangBeschriftung} auf einen {@link AbstractButton} oder auf eine
	 * {@link Action}.
	 * 
	 * @param das          ein {@link AbstractButton} oder eine {@link Action}
	 * @param beschriftung die {@link LangBeschriftung}
	 */
	private void deployLongDescription(Object das, LangBeschriftung damit) {
		if (damit.getLongDescription() != null)
			if (das instanceof Action)
				((Action) das).putValue(Action.LONG_DESCRIPTION, damit.getLongDescription());
			else if (das instanceof AbstractButton && ((AbstractButton) das).getAction() != null)
				deployLongDescription(((AbstractButton) das).getAction(), damit);
	}

	/**
	 * Die Methode setzt den {@link LangBeschriftung#getAcceleratorKey()} einer
	 * {@link LangBeschriftung} eine {@link Action}. Auf einen
	 * {@link AbstractButton} nur, wenn er eine {@link AbstractButton#getAction()}
	 * vorhanden ist oder er ein {@link JMenuItem} ist.
	 * 
	 * @param das          ein {@link AbstractButton} oder eine {@link Action}
	 * @param beschriftung die {@link LangBeschriftung}
	 */
	private void deployAcceleratorKey(Object das, LangBeschriftung damit) {
		if (damit.getAcceleratorKey() != null) {
			if (das instanceof Action)
				((Action) das).putValue(Action.ACCELERATOR_KEY, damit.getAcceleratorKey().toKeyStroke());
			else if (das instanceof AbstractButton) {
				if (((AbstractButton) das).getAction() != null)
					deployAcceleratorKey(((AbstractButton) das).getAction(), damit);
				else if (das instanceof JMenuItem)
					((JMenuItem) das).setAccelerator(damit.getAcceleratorKey().toKeyStroke());
			}
		}
	}

	/**
	 * Die Methode setzt das {@link Bebildert#getImage()} einer
	 * {@link LangBeschriftung} oder eines {@link BildOhneTitel}s auf ein
	 * {@link JLabel}, einen {@link AbstractButton} oder auf eine {@link Action}.
	 * 
	 * @param das   ein {@link JLabel}, einen {@link AbstractButton} oder auf eine
	 *              {@link Action}
	 * @param damit eine {@link LangBeschriftung} oder ein {@link BildOhneTitel}
	 */
	private void deployImage(Object das, Bebildert damit) {
		if (damit.getImage() != null) {
			ImageIcon ii = RechnerhammerIcons.loadIcon(damit.getImage());
			ii.setDescription(damit.getImage());
			if (das instanceof Action)
				((Action) das).putValue(Action.LARGE_ICON_KEY, ii);
			else if (das instanceof JLabel)
				((JLabel) das).setIcon(RechnerhammerIcons.loadIcon(damit.getImage()));
			else if (das instanceof AbstractButton) {
				if (((AbstractButton) das).getAction() != null)
					deployImage(((AbstractButton) das).getAction(), damit);
				else
					((AbstractButton) das).setIcon(RechnerhammerIcons.loadIcon(damit.getImage()));
			}
		}
	}

	/**
	 * Die Methode setzt einen Title aus den vorhandenen
	 * {@link #getAlternativeTitles()}.
	 * 
	 * @param vorsorgend eine {@link VorsorgendeBeschriftung} oder
	 *                   {@link LangVorsorgeBeschriftung}
	 * @param index      der Index des neuen Titles
	 */
	private void changeTitle(Vorsorgend<?> vorsorgend, int index) {
		if (index > -1 && index < vorsorgend.getAlternativeTitles().size()) {
			((Beschriftung) vorsorgend).setTitle(deWhiteSpace(vorsorgend.getAlternativeTitles().get(index)));
		} else {
			Kostenrechner.getLogger().info(String.format("Titel Nr. %d aus Beschriftung %d nicht vorhanden: !%n", index,
					((NameContainer) vorsorgend).getKey()));
		}
	}

	/**
	 * Die Methode setzt eine {@link #getShortDescription()} aus den
	 * {@link #getAlternativeDescriptions()}.
	 * 
	 * @param vorsorgend eine {@link VorsorgendeBeschriftung} oder
	 *                   {@link LangVorsorgeBeschriftung}
	 * @param index      der Index der neuen {@link #getShortDescription()}
	 */
	private void changeDescription(Vorsorgend<?> vorsorgend, int index) {
		if (index > -1 && vorsorgend.getAlternativeDescriptions() != null
				&& index < vorsorgend.getAlternativeDescriptions().size()) {
			((Beschriftung) vorsorgend)
					.setShortDescription(deWhiteSpace(vorsorgend.getAlternativeDescriptions().get(index)));
		} else {
			Kostenrechner.getLogger()
					.info(String.format("Kurzbeschreibung Nr. %d aus Beschriftung %d nicht vorhanden!%n", index,
							((NameContainer) vorsorgend).getKey()));
		}
	}

	// for (int index = 0; index < vorsorgend.getAlternativeTitles().size();
	// index++) {
	// if (vorsorgend.getAlternativeTitles().get(index).contains(title)) {
	// }
	// if (alternativen.get(index).contains(title))

}

/*
 * Die Methode setzt einen Title aus den vorhandenen {@link
 * #getAlternativeTitles()}.
 * 
 * @param index der Index des neuen Titles private void changeTitle(int index) {
 * if (index > -1 && index < getAlternativeTitles().size()) {
 * super.setTitle(BeschriftungApplyer.deWhiteSpace(getAlternativeTitles().get(
 * index))); } else { Kostenrechner.getLogger().info(String.
 * format("Beschriftung nicht vorhanden: %d!%n", index)); } }
 */
/*
 * Konstruktor: (kommentieren)
 * 
 * public ComponentBeschrifter() { // Auto-generated constructor stub }
 */
/*
 * 
 * public Component beschreibe(Component comp, NameContainer damit) throws
 * IllegalArgumentException { if (comp == null) throw new
 * IllegalArgumentException("Component darf nicht null sein!");
 * System.out.println("Beschreibe " + damit.getId());
 * comp.setName(damit.getId()); if (damit instanceof Beschriftung) {
 * Beschriftung beschriftung = (Beschriftung) damit; if (comp instanceof JLabel)
 * { ((JLabel) comp).setText(beschriftung.getTitle()); if
 * (beschriftung.hasMnemonic()) ((JLabel)
 * comp).setDisplayedMnemonic(beschriftung.mnemonicIntValue()); } else if (comp
 * instanceof JCheckBoxMenuItem || comp instanceof JMenuItem || comp instanceof
 * JCheckBox) { ((AbstractButton) comp).setText(beschriftung.getTitle()); if
 * (beschriftung.hasMnemonic()) ((AbstractButton)
 * comp).setMnemonic(beschriftung.mnemonicIntValue()); } else if (comp
 * instanceof JPanel || comp instanceof JScrollPane) { if (((JComponent)
 * comp).getBorder() instanceof TitledBorder) { ((TitledBorder) ((JComponent)
 * comp).getBorder()).setTitle(beschriftung.getTitle()); } } else if (comp
 * instanceof JDialog) { ((JDialog) comp).setTitle(beschriftung.getTitle()); }
 * else if (comp instanceof JFrame) { ((JFrame)
 * comp).setTitle(beschriftung.getTitle()); } else { if
 * (comp.getAccessibleContext() != null) {
 * comp.getAccessibleContext().setAccessibleName(beschriftung.getTitle()); } }
 * if (comp instanceof JComponent) ((JComponent)
 * comp).setToolTipText(beschriftung.getShortDescription()); // HTML-Code in der
 * AccessibleDescription verwirrt assistive Technik
 * comp.getAccessibleContext().setAccessibleDescription(beschriftung.
 * getShortDescription()); if (beschriftung instanceof LangBeschriftung) {
 * LangBeschriftung lulatsch = (LangBeschriftung) beschriftung; if (comp
 * instanceof AbstractButton) { final AbstractButton button = (AbstractButton)
 * comp; if (button.getAction() != null) { beschreibe(button.getAction(),
 * lulatsch); } else { button.setText(lulatsch.getTitle()); if
 * (lulatsch.hasMnemonic()) ((AbstractButton)
 * comp).setMnemonic(lulatsch.mnemonicIntValue());
 * button.setActionCommand(lulatsch.getActionCommandKey()); if
 * (lulatsch.getImage() != null)
 * button.setIcon(RechnerhammerIcons.loadIcon(lulatsch.getImage())); if (button
 * instanceof JMenuItem && lulatsch.getAcceleratorKey() != null) { ((JMenuItem)
 * button).setAccelerator(lulatsch.getAcceleratorKey().toKeyStroke()); } } } } }
 * if (damit instanceof UnBeschriftung) { UnBeschriftung un = (UnBeschriftung)
 * damit; // getRef().applyTo(comp) geht nicht, sonst würde auch ID geändert. if
 * (comp instanceof JScrollPane && ((JScrollPane) comp).getVerticalScrollBar()
 * != null && ((JScrollPane) comp).getVerticalScrollBar().getAccessibleContext()
 * != null) { ((JScrollPane) comp).getVerticalScrollBar().getAccessibleContext()
 * .setAccessibleName(createCompleteTitle(un)); } else if
 * (comp.getAccessibleContext() != null) {
 * comp.getAccessibleContext().setAccessibleName(createCompleteTitle(un)); }
 * else System.err.println("AccessibleContext null bei " +
 * comp.getClass().toString()); if (comp instanceof JComponent) ((JComponent)
 * comp).setToolTipText(un.getRef().getShortDescription());
 * comp.getAccessibleContext().setAccessibleDescription(un.getRef().
 * getShortDescription()); if (un instanceof BildOhneTitel) { BildOhneTitel bild
 * = (BildOhneTitel) un; if (comp instanceof JLabel) { if (bild.getImage() !=
 * null) { JLabel label = (JLabel) comp;
 * label.setIcon(RechnerhammerIcons.loadIcon(bild.getImage())); } } } } return
 * comp; }
 */
/*
 * 
 * action.putValue(Action.NAME, damit.getTitle());
 * action.putValue(Action.SHORT_DESCRIPTION, damit.getShortDescription()); if
 * (damit.getLongDescription() != null) action.putValue(Action.LONG_DESCRIPTION,
 * damit.getLongDescription()); if (damit.getActionCommandKey() != null)
 * action.putValue(Action.ACTION_COMMAND_KEY, damit.getActionCommandKey()); if
 * (damit.getImage() != null) { ImageIcon ii =
 * RechnerhammerIcons.loadIcon(damit.getImage());
 * ii.setDescription(damit.getImage()); action.putValue(Action.LARGE_ICON_KEY,
 * ii); } if (damit.hasMnemonic())// getMnemonic() != null
 * action.putValue(Action.MNEMONIC_KEY,
 * Integer.valueOf(damit.mnemonicIntValue())); if (damit.getAcceleratorKey() !=
 * null) action.putValue(Action.ACCELERATOR_KEY,
 * damit.getAcceleratorKey().toKeyStroke()); return action;
 */

// new ImageIcon(getClass().getClassLoader().getResource(this.getImage()));

// new ImageIcon(getClass().getClassLoader().getResource(this.getImage()));

// new ImageIcon(getClass().getClassLoader().getResource(this.getImage()));
// new ImageIcon(getClass().getClassLoader().getResource(this.getImage()));
// Beschriftung beschriftung = (Beschriftung) damit;
// if (das instanceof JLabel) {
// ((JLabel) comp).setText(beschriftung.getTitle());
// if (beschriftung.hasMnemonic())
// ((JLabel) das).setDisplayedMnemonic(beschriftung.mnemonicIntValue());
// } else if (das instanceof JCheckBoxMenuItem || das instanceof JMenuItem ||
// das instanceof JCheckBox) {
// ((AbstractButton) das).setText(beschriftung.getTitle());
// if (beschriftung.hasMnemonic())
// ((AbstractButton) das).setMnemonic(beschriftung.mnemonicIntValue());
// } else if (das instanceof JPanel || das instanceof JScrollPane) {
// if (((JComponent) das).getBorder() instanceof TitledBorder) {
// ((TitledBorder) ((JComponent)
// das).getBorder()).setTitle(beschriftung.getTitle());
// }
// } else if (das instanceof JDialog) {
// ((JDialog) das).setTitle(beschriftung.getTitle());
// } else if (das instanceof JFrame) {
// ((JFrame) das).setTitle(beschriftung.getTitle());
// } else {
// if (das.getAccessibleContext() != null) {
// das.getAccessibleContext().setAccessibleName(beschriftung.getTitle());
// }
// }
// if (das instanceof JComponent)
// ((JComponent) das).setToolTipText(beschriftung.getShortDescription());
// HTML-Code in der AccessibleDescription verwirrt assistive Technik
// das.getAccessibleContext().setAccessibleDescription(beschriftung.getShortDescription());
// LangBeschriftung lulatsch = (LangBeschriftung) beschriftung;
// if (das instanceof AbstractButton) {
// final AbstractButton button = (AbstractButton) das;
// if (button.getAction() != null) {
// beschreibe(button.getAction(), (LangBeschriftung) damit);
// } else {
// button.setText(lulatsch.getTitle());
// if (lulatsch.hasMnemonic())
// ((AbstractButton) das).setMnemonic(lulatsch.mnemonicIntValue());
// button.setActionCommand(lulatsch.getActionCommandKey());
// if (lulatsch.getImage() != null)
// button.setIcon(RechnerhammerIcons.loadIcon(lulatsch.getImage()));
// if (button instanceof JMenuItem && lulatsch.getAcceleratorKey() != null) {
// ((JMenuItem)
// button).setAccelerator(lulatsch.getAcceleratorKey().toKeyStroke());
// }
// }
// }
//{
//BildOhneTitel bild = (BildOhneTitel) un;
//if (das instanceof JLabel) {
//if (bild.getImage() != null) {
//JLabel label = (JLabel) das;
//label.setIcon(RechnerhammerIcons.loadIcon(bild.getImage()));
//}
//}
//}
// UnBeschriftung un = (UnBeschriftung) damit;
// {
// if (((Component) das).getAccessibleContext() != null) {
// ((Component) das).getAccessibleContext().setAccessibleName(title);
// }
// }
// JCheckBoxMenuItem || das instanceof JMenuItem || das instanceof JCheckBox)

/*
 * Die Methode einer {@link Component}, die sie aus der {@link Beschriftung}
 * nimmt.
 * 
 * @param das die {@link Component}
 * 
 * @param beschriftung die {@link Beschriftung}
 */
// private void deployAccessibleDescription(Component das, Beschriftung
// beschriftung) {
// if (das instanceof JComponent && ((JComponent) das).getToolTipText() == null)
// ((JComponent) das).setToolTipText(beschriftung.getShortDescription());
//// HTML-Code in der AccessibleDescription verwirrt assistive Technik
// das.getAccessibleContext().setAccessibleDescription(beschriftung.getShortDescription());
// }