/*
 * NumberExtractor.java
 * eu.gronos.kostenrechner.controller (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller;

import java.text.ParseException;
import java.util.Formatter;

import javax.swing.JFormattedTextField;
import javax.swing.JLabel;

/**
 * Die Klasse zieht den Zahlwert aus dem übergebenen
 * {@link JFormattedTextField}. Der Konstruktor
 * {@link NumberExtractor#NumberExtractor(JFormattedTextField, String)} benötigt
 * das {@link JFormattedTextField} und die beschriftung. Die Methode
 * {@link #extractDouble()} zieht den double-Wert aus dem {@link Formatter} und
 * die Methoden {@link #extractInteger()} und {@link #extractLong()} die
 * Ganzzahl-Werte.
 *
 * @author Peter Schuster (setrok)
 * @date 13 Apr 2019
 *
 */
public class NumberExtractor {
	public static final String FEHLER_KEINE_ZAHL = "Es konnte keine Zahl im Feld '%s' erkannt werden: '%s'!";
	private final JFormattedTextField formattedTextField;
//	private final JLabel label;
	private String beschriftung;

	/**
	 * Konstruktor:
	 * 
	 * @param formattedTextField das {@link JFormattedTextField}, das ausgelesen
	 *                           werden soll.
	 * @param beschriftung       die Beschriftung des dazugehörigen {@link JLabel}s
	 */
	public NumberExtractor(JFormattedTextField formattedTextField, String beschriftung) {// JLabel label
		this.formattedTextField = formattedTextField;
		this.beschriftung = beschriftung;
	}

	/**
	 * Die Methode zieht den Zahlwert aus dem übergebenen
	 * {@link JFormattedTextField}
	 * 
	 * @return den ausgelesenen Wert als double
	 */
	public double extractDouble() {
		Number number = extractNumber();
		if (number instanceof Double)
			return ((Double) number).doubleValue();
		else
			return new Double((number).doubleValue());
	}

	/**
	 * Die Methode zieht den Zahlwert aus dem übergebenen
	 * {@link JFormattedTextField}
	 * 
	 * @return den ausgelesenen Wert als int
	 */
	public int extractInteger() {
		Number number = extractNumber();
		if (number instanceof Integer)
			return ((Integer) number).intValue();
		else
			return new Integer(number.intValue());
	}

	/**
	 * Die Methode zieht den Zahlwert aus dem übergebenen
	 * {@link JFormattedTextField}
	 * 
	 * @return den ausgelesenen Wert als long
	 */
	public long extractLong() {
		Number number = extractNumber();
		if (number instanceof Long)
			return ((Long) number).longValue();
		else
			return new Long(number.longValue());
	}

	/**
	 * Die Methode ist der Kern: Sie sagt dem {@link JFormattedTextField}, dass man
	 * zuende bearbeitet hat und holt sich den Wert, prüft, ob es eine Zahl ist und
	 * gibt eine {@link Number} zurück.
	 * 
	 * @return eine {@link Number}
	 * 
	 * @throws NumberFormatException wenn es keine Zahl ist
	 */
	private Number extractNumber() {
		String text = formattedTextField.getText();
		try {
			// Nötig, damit getValue nicht null ist
			formattedTextField.commitEdit();
			Object value = formattedTextField.getValue();
			if (value != null && value instanceof Number)
				return (Number) value;
			else
				throw new NumberFormatException(String.format(FEHLER_KEINE_ZAHL, beschriftung, text));
		} catch (ParseException e) {
			NumberFormatException nfe = new NumberFormatException(String.format(FEHLER_KEINE_ZAHL, beschriftung, text));
			nfe.addSuppressed(e);
			throw nfe;
		}
	}

}
