/*
 * SprungmarkenHorcher.java
 * eu.gronos.kostenrechner.view (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller;

import java.awt.Desktop;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;

import javax.swing.JEditorPane;
import javax.swing.event.HyperlinkEvent;
import javax.swing.event.HyperlinkListener;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;

/**
 * /** HYPERLINK_LISTENER ist ein HyperlinkListener, der auf Klicks reagiert und
 * entweder interne Links im selben Dokument anspringt oder für externe Links
 * den Standardbrowser öffnet.
 * 
 * @url "http://www.dpunkt.de/java/Programmieren_mit_Java/Oberflaechenprogrammierung/59.html"
 * @url "http://stackoverflow.com/questions/10967451/open-a-link-in-browser-with-java-button"
 * @url "http://stackoverflow.com/questions/14054601/jumping-to-internal-anchors-in-jeditorpane"
 *
 * @author Peter Schuster (setrok)
 * @date 5 Mar 2019
 *
 */
public class SprungmarkenHorcher implements HyperlinkListener {
	final Desktop desktop = Desktop.isDesktopSupported() ? Desktop.getDesktop() : null;

	@Override
	public void hyperlinkUpdate(HyperlinkEvent hle) {
		if (hle.getEventType() == HyperlinkEvent.EventType.ACTIVATED) {
			if (!isNull(hle) && !interneSprungmarke(hle) && hle.getURL() != null)
				launchBrowser(hle.getURL());

		}
	}

	/**
	 * Die Methode dient dazu, alle denkbaren {@link NullPointerException}s
	 * auszuschließen
	 * 
	 * @param hle ein {@link HyperlinkEvent}
	 * @return <code>true</code>, wenn {@link Desktop#isDesktopSupported()}
	 *         <code>false</code> ist, {@link HyperlinkEvent#getURL()} oder
	 *         {@link HyperlinkEvent#getDescription()} <code>null</code> liefern.
	 */
	private boolean isNull(HyperlinkEvent hle) {
		if (hle.getDescription() == null) {
			// System.out.println
			Kostenrechner.getLogger().info("Keine Linkbeschreibung");
			return true;
		}
		return desktop == null;
	}

	/**
	 * Die Methode dient dazu, interne Sprungmarken anzusteuern
	 * 
	 * @param hle das {@link HyperlinkEvent}
	 * @return <code>true</code>, wenn es sich um eine Sprungmarke handelte.
	 */
	private boolean interneSprungmarke(HyperlinkEvent hle) {
		String desc = hle.getDescription();
		if (!desc.startsWith("#")) {
			/* Ohne Raute keine Sprünge */
			// System.out.println(
			Kostenrechner.getLogger().info(String.format("Externer Link: %s", desc));
			return false;
		} else {
			/* Raute (#) entfernen */
			// System.out.println(
			Kostenrechner.getLogger().info(String.format("Interner Link: %s", desc));
			desc = desc.substring(1);
		}
		if (hle.getSource() instanceof JEditorPane) {
			JEditorPane ep = (JEditorPane) hle.getSource();
			ep.scrollToReference(desc);
		} else
			// System.out.println
			Kostenrechner.getLogger().info("Kein Editorpane");
		return true;
	}

	/**
	 * Die Methode übernimmt das Konvertieren in eine {@link URI} und ruft über
	 * {@link #launchBrowser(URI)} den Standardbrowser des Anwenders auf.
	 * 
	 * @param url die anzusteuernde {@link URL}
	 */
	private void launchBrowser(URL url) {
		try {
			launchBrowser(url.toURI());
			// System.out.println(
			Kostenrechner.getLogger().info(String.format("URL: %s", url.toString()));
		} catch (URISyntaxException use) {
			FehlerHelper.zeigeFehler("Der ausgewählte Link enthält einen Syntaxfehler!", use);
			use.printStackTrace();
		}
	}

	/**
	 * Die Methode ruft den Standardbrowser des Anwenders auf.
	 * 
	 * @param uri die anzuspringende {@link URI}
	 * 
	 * @see Desktop#browse(URI)
	 */
	private void launchBrowser(URI uri) {
		if (desktop != null && desktop.isSupported(Desktop.Action.BROWSE)) {
			try {
				desktop.browse(uri);
			} catch (IllegalArgumentException | NullPointerException iae) {
				FehlerHelper.zeigeFehler("URI konnte nicht geöffnet werden oder Link ist NULL", iae);
			} catch (SecurityException se) {
				FehlerHelper.zeigeFehler("Sicherheitsverletzung beim Start des Browsers.", se);
			} catch (IOException ioe) {
				FehlerHelper.zeigeFehler(
						"Browser konnte nicht gestartet werden. Kein Standardbrowser definiert, definierter Standardbrowser nicht gefunden oder das System konnte ihn nicht starten.",
						ioe);
			} catch (Exception e) {
				FehlerHelper.zeigeFehler("Sonstiger Fehler beim Starten des Browsers", e);
			}
		}
	}
}