/**
 * TabulierendBaukasten.java
 * eu.gronos.kostenrechner.controller (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.swing.JTable;

import eu.gronos.kostenrechner.interfaces.Tabulierend;
import eu.gronos.kostenrechner.interfaces.TenorVorbereitend;
import eu.gronos.kostenrechner.model.tenordaten.BegruendungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.CaretRange;
import eu.gronos.kostenrechner.model.tenordaten.DoubleDataRows;
import eu.gronos.kostenrechner.model.tenordaten.TenorDatenContainer;
import eu.gronos.kostenrechner.view.result.TenorDialog;

/**
 * Die Klasse speichert die BegründungsTabelle eines {@link Tabulierend} und
 * hält Methoden zum Umgang damit bereit: So erstellt
 * {@link #toTableCellValues()} die eigentlichen Tabellendaten für
 * {@link Tabulierend#getTableCellValues()} und {@link #toColumnHeaders()}
 * erstellt die Kopfdaten für die Tabelle.
 *
 * @author Peter Schuster (setrok)
 * @date 17.08.2018
 *
 */
public class TabulierendZeilen {
	// final int[] range;
	private final CaretRange range;
	private final List<DoubleDataRows> doubleValues;
	private final List<String> columnHeaders;
	private final List<String> rowHeaders;

	/**
	 * Der Standardkonstruktor legt alle Listen selbst an
	 * 
	 */
	public TabulierendZeilen() {
		this.range = new CaretRange();// = new int[2];
		this.doubleValues = new ArrayList<DoubleDataRows>();
		this.columnHeaders = new ArrayList<String>();
		this.rowHeaders = new ArrayList<String>();
	}

	/**
	 * Der Konstruktor für den, der schon alles hat (sprich: alle Tabellendaten),
	 * braucht diese in drei Listen
	 * 
	 * @param doubleValues  {@link #doubleValues}
	 * @param columnHeaders {@link #columnHeaders}
	 * @param rowHeaders    {@link #rowHeaders}
	 * @param range         {@link #range}
	 */
	public TabulierendZeilen(List<DoubleDataRows> doubleValues, List<String> columnHeaders, List<String> rowHeaders,
			CaretRange range) {// int[]
		this.range = range;
		this.doubleValues = new ArrayList<DoubleDataRows>(doubleValues);/* ArrayList<Double> */
		this.columnHeaders = new ArrayList<String>(columnHeaders);
		this.rowHeaders = new ArrayList<String>(rowHeaders);
	}

	/**
	 * Die Methode baut Tabellendaten aus {@link #rowHeaders} und
	 * {@link #doubleValues}. Die Spaltenüberschriften gibt sie nicht zurück,
	 * sondern die Methode {@link #toColumnHeaders()}.
	 * 
	 * @return ein Array Object[][]
	 */
	public Object[][] toTableCellValues() {
		if ((doubleValues == null || columnHeaders == null || rowHeaders == null)
				|| (doubleValues.size() < 1 || columnHeaders.size() < 1 || rowHeaders.size() < 1)) {
			return null;
		}
		if (doubleValues.size() != rowHeaders.size()) {
			throw new IllegalArgumentException("doubleValues und rowHeaders müssen gleich viele Zeilen haben!");
		}
		/*
		 * Die Tabelle hat so viele Zeilen wie tableCellValues groß ist und so viele
		 * Spalten wie columnHeaders groß ist. Denn: Erste Zeile sind NICHT die
		 * Spaltenüberschriften (die werden gesondert ausgegeben)
		 */
		Object[][] tableCellValues = new Object[doubleValues.size()][columnHeaders.size()];
		/*
		 * Jede Zeile besteht aus einem Stück rowHeaders und einer Zeile
		 * tableCellValues. ACHTUNG: Dabei hat doubleValues eine Spalte weniger als
		 * columnHeaders!
		 */
		for (int row = 0; row < doubleValues.size(); row++) {
			tableCellValues[row][0] = rowHeaders.get(row);
			for (int column = 1; column < columnHeaders.size(); column++) {
				tableCellValues[row][column] = doubleValues.get(row).get(column - 1);
			}
		}
		return tableCellValues;
	}

	/**
	 * Die Methode konvertiert die {@link ArrayList}&lt;{@link String}>
	 * {@link #columnHeaders} zu einem Array String[] mit den
	 * Kopfdaten/Spaltenüberschriften für {@link Tabulierend#getColumnHeaders()}.
	 * 
	 * @return ein Array String[] mit den Kopfdaten/Spaltenüberschriften
	 */
	public String[] toColumnHeaders() {
		if (columnHeaders == null || columnHeaders.isEmpty()) {
			return null;
		}
		String[] array = new String[columnHeaders.size()];
		return columnHeaders.toArray(array);
	}

	/**
	 * Die Methode gibt die {@link ArrayList}&lt;{@link String}> {@link #rowHeaders}
	 * als {@link Collections#unmodifiableList(List)} mit den
	 * Kopfdaten/Spaltenüberschriften für
	 * {@link TenorDatenContainer}/{@link BegruendungsElemente#columnHeaders}
	 * zurück.
	 * 
	 * columnHeaders enthält die Kopfdaten/Spaltenüberschriften als List&lt;String>.
	 * Diese werden hinterher zur ersten Zeile/Überschriftenzeile der Tabelle
	 * 
	 * @return eine {@link List}&lt;{@link String}> mit den
	 *         Kopfdaten/Spaltenüberschriften
	 */
	public List<String> getColumnHeaders() {
		if (columnHeaders == null || columnHeaders.isEmpty()) {
			return null;
		}
		return Collections.unmodifiableList(columnHeaders);
	}

	/**
	 * Die Methode gibt die {@link ArrayList}&lt;{@link String}> {@link #rowHeaders}
	 * als {@link Collections#unmodifiableList(List)} mit den Zeilenüberschriften
	 * für {@link TenorDatenContainer}/{@link BegruendungsElemente#rowHeaders}
	 * zurück.
	 * 
	 * rowHeaders enthält die Zeilenüberschriften als List&lt;String>. Diese werden
	 * zur ersten Spalte der Tabelle.
	 * 
	 * @return eine {@link List}&lt;{@link String}> mit den Zeilenüberschriften
	 */
	public List<String> getRowHeaders() {
		if (rowHeaders == null || rowHeaders.isEmpty()) {
			return null;
		}
		return Collections.unmodifiableList(rowHeaders);
	}

	/**
	 * Die Methode gibt die {@link ArrayList}&lt;{@link Double}>
	 * {@link #doubleValues} als {@link Collections#unmodifiableList(List)} Double[]
	 * mit den Rohdaten für
	 * {@link TenorDatenContainer}/{@link BegruendungsElemente#doubleData}.
	 * 
	 * doubleValues enthält die eigentlichen Tabellendaten (die Double-Werte ohne
	 * Zeilen- oder Spalten-Überschriften) als List&lt;Double[]>.
	 * 
	 * @return eine {@link List}&lt;{@link Double}[]> mit den Rohdaten
	 */
	public List<DoubleDataRows> getDoubleData() {
		if (doubleValues == null || doubleValues.isEmpty()) {
			return null;
		}
		return doubleValues;
//		List<DoubleDataRows> list = new ArrayList<DoubleDataRows>();
//		for (ArrayList<Double> dv : doubleValues) {
//			list.add(new DoubleDataRows(dv));
//		}
//		return list;
		// return (ArrayList<ArrayList<Double>>)
		// Collections.unmodifiableList(doubleValues);
//		return /*doubleValues*/ new DoubleDataRows(doubleValues);
	}

	/**
	 * Die Methode sagt, welche Datentypen von {@link #toTableCellValues()}
	 * zurückgegeben werden.
	 * 
	 * @return ein Array Class&lt;?>[], das so groß ist wie {@link #columnHeaders}
	 *         und deren erstes Objekt {@link String#getClass()} ist und alle
	 *         übrigen {@link Double#getClass()}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.Tabulierend#getColumnClasses()
	 */
	public Class<?>[] toColumnClasses() {
		Class<?>[] columnClasses;
		if (columnHeaders == null || columnHeaders.size() < 1)
			return null;
		columnClasses = new Class<?>[columnHeaders.size()];
		columnClasses[0] = String.class;
		for (int i = 1; i < columnHeaders.size(); i++)
			columnClasses[i] = Double.class;
		return columnClasses;
	}

	/**
	 * Die Methode dient dazu, aus der {@link #zeilenListe} einen Text mit
	 * Tabulatoren zu basteln, der von der {@link JTable} im {@link TenorDialog}
	 * überlagert werden kann. Gleichzeitig wird auch
	 * {@link eu.gronos.kostenrechner.interfaces.Tabulierend#getRange() range}
	 * gesetzt.
	 * 
	 * @param gruende ein {@link StringBuilder} mit den Gründen, der auch verändert
	 *                wird
	 * @return gibt die veränderten gruende auch wieder zurück
	 */
	public StringBuilder toStringBuilder(StringBuilder gruende) {
		// range[0]
		range.beginn = gruende.length() - 1;
		// Erst die Überschriften
		for (int index = 0; index < columnHeaders.size(); index++) {
//			text.append(String.format("%,.2f", row.get(cell)));
			gruende.append(columnHeaders.get(index));
			// Tabulator oder Zeilenumbruch.
			if (index == columnHeaders.size() - 1) {
				gruende.append("\n");
			} else {
				gruende.append("\t");
			}
		}
		// dann die Double-Werte
		for (int index = 0; index < doubleValues.size(); index++) {
			gruende.append(rowToText(index));
		}
		// range[1]
		range.ende = gruende.length() - 1;
		return gruende;
	}

	/**
	 * Die Methode fügt der Tabelle die Spaltenüberschriften hinzu
	 * 
	 * @param values eine List&lt;String> mit Spaltenüberschriften. Muss eine Spalte
	 *               enthalten, als die Doublewerte haben.
	 * @return true, wenn es geklappt hat
	 */
	public boolean add(List<String> values) {
		return columnHeaders.addAll(values);
	}

	/**
	 * Die Methode fügt der Tabelle eine Zeile hinzu, bestehend aus einer
	 * Zeilenüberschrift und einem Array mit Double-Werten
	 * 
	 * @param header Zeilenüberschrift als String
	 * @param values Tabellenwerte als List&lt;Double>
	 * @return true, wenn es geklappt hat
	 * 
	 * @see List#add(Object)
	 */
	public boolean add(String header, DoubleDataRows values) {
		if (header == null || values == null) {
			throw new NullPointerException();
		}
		return rowHeaders.add(header) && doubleValues.add(new DoubleDataRows(values));
	}

	/**
	 * range speichert Beginn und Ende der Begründungstabelle im Begründungstext als
	 * int[].
	 * 
	 * @return
	 * 
	 * @return gibt {@link #range} als int [] zurück.
	 * 
	 * @see Tabulierend#getRange()
	 */
	// int[]
	public CaretRange getRange() {
		return range;
	}

	/**
	 * Die Methode wandelt die über den index angegebene Tabellenzeile, bestehend
	 * aus einem String aus {@link #rowHeaders} und einer Zeile aus
	 * {@link #doubleValues} in eine Tab-separierte Textzeile als {@link String} um
	 * 
	 * @param index die Zeile aus {@link #doubleValues}
	 * @return eine Tab-separierte Textzeile (String)
	 */
	private String rowToText(int index) {
		if (doubleValues == null || rowHeaders == null || doubleValues.size() <= index || rowHeaders.size() <= index) {
			return "";
		}
		StringBuffer text = new StringBuffer();
		text.append(rowHeaders.get(index));
		text.append("\t");
		List<Double> row = doubleValues.get(index);
		for (int cell = 0; cell < row.size(); cell++) {
			text.append(String.format("%,.2f", row.get(cell)));
			// Tabulator oder Zeilenumbruch.
			if (cell == row.size() - 1) {
				text.append("\n");
			} else {
				text.append("\t");
			}
		}
		return text.toString();
	}

	/**
	 * Die Methode hilft {@link TenorVorbereitend}-Klassen, ein
	 * {@link TenorDatenContainer} zu befüllen. Dazu liefert es die nötigen Daten
	 * für ein {@link BegruendungsElemente} zurück.
	 * 
	 * @param gruende Die Gründe selbst kann {@link TabulierendZeilen} nicht
	 *                erzeugen, deshalb muss die aufrufende Methode sie übergeben.
	 * @return das erstellte {@link BegruendungsElemente}
	 */
	public BegruendungsElemente toBegruendungsElemente(String gruende) {
		BegruendungsElemente begruendung = new BegruendungsElemente();
		begruendung.text = gruende;
		begruendung.range = getRange();
		begruendung.columnHeaders = getColumnHeaders();
		begruendung.rowHeaders = getRowHeaders();
		begruendung.doubleData = getDoubleData();
		return begruendung;
	}
}
