/*
 * TenorAction.java
 * eu.gronos.kostenrechner.controller.gebuehren (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller;

import java.awt.event.ActionEvent;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.files.XmlTransferHandler;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.interfaces.DialogLieferant;
import eu.gronos.kostenrechner.interfaces.TenorVorbereitend;
import eu.gronos.kostenrechner.model.beschriftungen.LangBeschriftung;
import eu.gronos.kostenrechner.model.tenordaten.TenorDatenContainer;
import eu.gronos.kostenrechner.model.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.view.result.TenorDialog;

/**
 * Abstrakte Oberklasse für alle {@link BeschriebeneAktion}en, die als
 * {@link DialogLieferant} einen {@link TenorDialog} aufrufen. Die Klasse
 * kümmert sich um den {@link DialogLieferant#getDialog()} und um
 * {@link XmlTransferHandler#leseAlleWertefuerContainer()}.
 * 
 * Die abgeleitete Klasse muss in einer Methode {@link #allesOk()} prüfen, ob
 * die nötigen Oberflächenelemente non-null sind. Sie muss weiter in einer
 * Methode {@link #tenorDaten(VerfahrensDatenContainer)} über die
 * fachspezifische {@link TenorVorbereitend}-Klasse einen
 * {@link TenorDatenContainer} erzeugen.
 *
 * @author Peter Schuster (setrok)
 * @date 15.02.2020
 *
 */
public abstract class TenorAction extends BeschriebeneAktion implements DialogLieferant {

	private static final long serialVersionUID = 5202806129481793793L;
	private TenorDialog dialog = null;

	/**
	 * 
	 * @param beschriftung eine {@link LangBeschriftung}, der auch
	 *                     ({@link javax.swing.Action#NAME}),
	 *                     ({@link javax.swing.Action#SHORT_DESCRIPTION}),
	 *                     {@link javax.swing.Action#MNEMONIC_KEY},
	 *                     {@link javax.swing.Action#ACCELERATOR_KEY} und
	 *                     ({@link javax.swing.Action#ACTION_COMMAND_KEY}) entnommen
	 *                     werden.
	 * 
	 * @see eu.gronos.kostenrechner.controller.BeschriebeneAktion#BeschriebeneAktion(LangBeschriftung)
	 */
	public TenorAction(LangBeschriftung beschriftung) {
		super(beschriftung);
	}

	/**
	 * Die Methode ruft für die abgeleiteten Klassen
	 * {@link XmlTransferHandler#leseAlleWertefuerContainer()} auf. Sie ruft dann
	 * den {@link TenorDialog} auf und kümmert sich um {@link #getDialog()}.
	 * 
	 * @param ae ein {@link ActionEvent}
	 * 
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 * @see eu.gronos.kostenrechner.controller.files.XmlTransferHandler#leseAlleWertefuerContainer()
	 */
	@Override
	public void actionPerformed(ActionEvent ae) {
		try {
			allesOk();
			final VerfahrensDatenContainer container = new XmlTransferHandler().leseAlleWertefuerContainer();
			final TenorDatenContainer tenorDaten = tenorDaten(container);
			setDialog(new TenorDialog(Kostenrechner.getInstance(), tenorDaten)).showDialog();
			setDialog(null);
		} catch (NumberFormatException e) {
			// TODO Überflüssig? Werte prüfen! Werden woanders geprüft?
			FehlerHelper.zeigeFehler(e.getLocalizedMessage(), e);
		} catch (NullPointerException | IllegalArgumentException e) {
			FehlerHelper.zeigeFehler(e.getLocalizedMessage(), e);
		}
	}

	/**
	 * @return gibt {@link #dialog} als {@link TenorDialog} zurück.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.DialogLieferant#getDialog()
	 */
	@Override
	public TenorDialog getDialog() {
		return dialog;
	}

	/**
	 * Die Methode prüft, ob alle nötigen Oberflächenelemente non-null sind
	 * 
	 * @return <code>true</code>, wenn {@link #alleWerte()} gefahrlos laufen kann.
	 * @throws IllegalArgumentException wird ansonsten geworfen (<code>false</code>
	 *                                  gibt es nicht), damit die {@link Exception}
	 *                                  direkt behandelt werden kann.
	 */
	protected abstract boolean allesOk() throws IllegalArgumentException;

	/**
	 * Die Methode ruft den einschlägigen
	 * {@link TenorVorbereitend#erzeugeContainer()} auf, um den
	 * {@link TenorDatenContainer} zu erzeugen.
	 * 
	 * @param container einen {@link VerfahrensDatenContainer}, den
	 *                  {@link TenorAction} über
	 *                  {@link XmlTransferHandler#leseAlleWertefuerContainer()}
	 *                  zusammenstellt
	 * @return den {@link TenorDatenContainer}
	 * @throws IllegalArgumentException wenn was schief läuft
	 */
	protected abstract TenorDatenContainer tenorDaten(VerfahrensDatenContainer container)
			throws IllegalArgumentException;

	/**
	 * @param dialog d. {@link #dialog}, d. gesetzt werden soll als
	 *               {@link TenorDialog}.
	 * @return den {@link TenorDialog} zum Weiterarbeiten
	 */
	private TenorDialog setDialog(TenorDialog dialog) {
		this.dialog = dialog;
		return dialog;
	}

}

/*
 * @param tenorDaten der {@link TenorDatenContainer}
 */
// private void showDialog(final TenorDatenContainer tenorDaten) {
// }

// private VerfahrensDatenContainer alleWerte() {
// return new XmlTransferHandler().leseAlleWertefuerContainer();
// }
// showDialog(tenorDaten);// alleWerte()