/*
 * BeteiligtenPruefer.java
 * eu.gronos.kostenrechner.controller.baumbach (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.baumbach;

import java.awt.Component;

import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JFormattedTextField;

import eu.gronos.kostenrechner.controller.NumberExtractor;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.interfaces.ParsendUndBauend;
import eu.gronos.kostenrechner.model.baumbach.BaumbachBeteiligter;
import eu.gronos.kostenrechner.model.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.view.HinzufuegenDialog;

/**
 * Der Pruefer kontrolliert alle Werte zum Bau eines {@link Beteiligter}n
 *
 * @author Peter Schuster (setrok)
 * @date 16.07.2019
 *
 */
public class BeteiligtenPruefer implements ParsendUndBauend<BaumbachBeteiligter> {

	private BaumbachBeteiligter rueckgabewert;
	private JComboBox<Beteiligter> cmbTypGenusNumerus;
	private JCheckBox cbAnWiderklageBeteiligt;
	private JFormattedTextField ftfUnterliegen;
	private JFormattedTextField ftfStreitwert;

	/**
	 */
	public BeteiligtenPruefer() {
		super();
	}

	/**
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#leseComponents(Component[])
	 */
	@SuppressWarnings("unchecked")
	@Override
	public ParsendUndBauend<BaumbachBeteiligter> leseComponents(Component... components) {
		if (components != null) {
			for (Component comp : components) {
				JComponent component = (JComponent) comp;
				if ("35005".equals(component.getName())) {// cmbTypGenusNumerus
					cmbTypGenusNumerus = (JComboBox<Beteiligter>) component;
				} else if ("35011".equals(component.getName())) {// cbAnWiderklageBeteiligt
					cbAnWiderklageBeteiligt = (JCheckBox) component;
				} else if ("35035".equals(component.getName())) {// ftfUnterliegen
					ftfUnterliegen = (JFormattedTextField) component;
				} else if ("35025".equals(component.getName())) {// ftfStreitwert
					ftfStreitwert = (JFormattedTextField) component;
				} else {
					System.out.println("Komisch: " + component.getName());
				}
			}
		}
		return this;
	}

	/**
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#parseEingabe()
	 */
	@Override
	public boolean parseEingabe() {
		try {
			rueckgabewert = leseFelder();
			return parseEingabe(rueckgabewert);
		} catch (IllegalArgumentException | NullPointerException ne) {
			FehlerHelper.zeigeFehler(ne.getLocalizedMessage(), ne);
			return false;
		}
	}

	/**
	 * Methode zum Plausis checken, wird vom OK-Button aufgerufen.
	 * 
	 * @return true, wenn alle Plausis eingehalten werden
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#parseEingabe(java.lang.Object)
	 */
	@Override
	public boolean parseEingabe(BaumbachBeteiligter beteiligter) throws IllegalArgumentException, NullPointerException {
		this.rueckgabewert = beteiligter;
		if (rueckgabewert.getStreitwert() < rueckgabewert.getUnterliegen())
			throw new NumberFormatException(HinzufuegenDialog.FEHLER_BT_UNTERLIEGEN_GT_STREITWERT);
		if (rueckgabewert.getStreitwert() <= 0.0
				&& (Beteiligter.KLAEGER != rueckgabewert.getTyp() || rueckgabewert.isAnWiderklageBeteiligt()))
			throw new IllegalArgumentException(HinzufuegenDialog.FEHLER_BK_STREITWERT);
		if (Beteiligter.DRITTWIDERBEKLAGTE == rueckgabewert.getTyp() && !rueckgabewert.isAnWiderklageBeteiligt())
			throw new IllegalArgumentException("Ein Drittwiderbeklagter muss widerbeklagt sein!");
		// Ein nicht widerbeklagter Kläger mit Streitwert/Unterliegen > 0 führt zu Fehlern
		if (Beteiligter.KLAEGER == rueckgabewert.getTyp() && !rueckgabewert.isAnWiderklageBeteiligt()
				&& (rueckgabewert.getStreitwert() > 0.0 || rueckgabewert.getUnterliegen() > 0.0))
			throw new IllegalArgumentException(
					"Ein nicht widerbeklagter Kläger darf nicht in Anspruch genommen oder verurteilt werden (Streitwert oder Verurteilung dürfen nicht größer als 0 sein)!");
		return true;
	}

	/**
	 * Die Methode gibt einen Rückgabewert zurück, der aus den Eingabefeldern des
	 * Dialogs zusammengebaut wurde, der dann von {@link #showDialogAndAsk()} an die
	 * aufrufende Maske zurückgegeben werden kann.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#getRueckgabewert()
	 */
	@Override
	public BaumbachBeteiligter getRueckgabewert() {
		return rueckgabewert;
	}

	private BaumbachBeteiligter leseFelder() throws IllegalArgumentException, NullPointerException {
		/*
		 * Ob überhaupt Widerkläger/beklagte angezeigt werden, erkennt der Prüfer an
		 * cbAnWiderBeteiligt#isVisible
		 */
		final boolean anWiderklageBeteiligt = (cbAnWiderklageBeteiligt.isVisible()
				&& cbAnWiderklageBeteiligt.isSelected());
		final double unterliegen = new NumberExtractor(ftfUnterliegen,
				ftfUnterliegen.getAccessibleContext().getAccessibleName()).extractDouble();
		final double streitwert = new NumberExtractor(ftfStreitwert,
				ftfStreitwert.getAccessibleContext().getAccessibleName()).extractDouble();
		rueckgabewert = new BaumbachBeteiligter(getSelectedItem(), streitwert, unterliegen, anWiderklageBeteiligt);
		return rueckgabewert;
	}

	private Beteiligter getSelectedItem() {
		return cmbTypGenusNumerus.getModel().getElementAt(cmbTypGenusNumerus.getSelectedIndex());
	}

//	double streitwert = 0.0;
//	double unterliegen = 0.0;
//	try {
//		streitwert = doubleAusFormatter(ftfStreitwert, lblStreitwert.getText());// Double.parseDouble(ftfStreitwert.getText());
//		unterliegen = doubleAusFormatter(ftfUnterliegen, lblUnterliegen.getText());// Double.parseDouble(ftfUnterliegen.getText());
//		if (streitwert < unterliegen)
//			throw new NumberFormatException(HinzufuegenDialog.FEHLER_BT_UNTERLIEGEN_GT_STREITWERT);
//	} catch (NumberFormatException nfe) {
//		FehlerDialog.zeigeFehler(FehlerDialog.FEHLER_BLANKO, nfe.getLocalizedMessage(), nfe);
//		// nfe.printStackTrace();
//		return false;
//	}
//	return true;

//	return new BaumbachBeteiligter(beteiligtenTyp, cmbTypGenusNumerus.getSelectedIndex(),
//			doubleAusFormatter(ftfStreitwert, lblStreitwert.getText()),
//			doubleAusFormatter(ftfUnterliegen, lblUnterliegen.getText()),
//			widerKlageAnzeigen ? cbAnWiderklageBeteiligt.isSelected() : false);
	// BeteiligtenHinzufuegenDialog.TEXT_VERURTEILUNG
	// BeteiligtenHinzufuegenDialog.TEXT_STREITWERT

}
