/*
 * GesamtschuldPruefer.java
 * eu.gronos.kostenrechner.controller.baumbach (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.baumbach;

import java.awt.Component;

import javax.swing.JComponent;
import javax.swing.JFormattedTextField;
import javax.swing.JList;

import eu.gronos.kostenrechner.controller.NumberExtractor;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.interfaces.ParsendUndBauend;
import eu.gronos.kostenrechner.model.baumbach.BaumbachBeteiligtenListe;
import eu.gronos.kostenrechner.model.baumbach.BaumbachBeteiligter;
import eu.gronos.kostenrechner.model.baumbach.BaumbachGesamtschuldnerschaft;
import eu.gronos.kostenrechner.model.baumbach.BeteiligtenListModel;
import eu.gronos.kostenrechner.model.tenordaten.Beteiligter;

/**
 * Der Pruefer kontrolliert alle Werte zum Bau einer
 * {@link BaumbachGesamtschuldnerschaft}
 *
 * @author Peter Schuster (setrok)
 * @date 15.07.2019
 *
 */
public class GesamtschuldPruefer implements ParsendUndBauend<BaumbachGesamtschuldnerschaft> {

	private static final String FEHLER_UNTERLIEGEN_GROESSER_STREITWERT = "Die gesamtschuldnerische Verurteilung (%,.2f EUR) darf zusammen mit der Verurteilung %s %ndarüber hinaus (%,.2f EUR) den Streitwert (%,.2f EUR) nicht überschreiten!";
	private static final String FEHLER_NIX_GENUG_BETEILIGTE = "Nicht genug Beteiligte ausgewählt. Wählen Sie mindestens zwei Beteiligte aus!";
	private BaumbachGesamtschuldnerschaft rueckgabewert;
	private JList<BaumbachBeteiligter> beteiligtenAuswahlListe;
	private JFormattedTextField ftfUnterliegen;
	private BaumbachBeteiligtenListe baumbachBeteiligtenListe;

	/**
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#leseComponents(Component[])
	 */
	@SuppressWarnings("unchecked")
	@Override
	public ParsendUndBauend<BaumbachGesamtschuldnerschaft> leseComponents(Component... components) {
		if (components != null) {
			for (Component comp : components) {
				JComponent component = (JComponent) comp;
				if ("36011".equals(component.getName())) {
					beteiligtenAuswahlListe = (JList<BaumbachBeteiligter>) component;
					baumbachBeteiligtenListe = ((BeteiligtenListModel) beteiligtenAuswahlListe.getModel()).getValues();
				} else if ("36025".equals(component.getName())) {
					ftfUnterliegen = (JFormattedTextField) component;
				}
			}
		}
		return this;
	}

	/**
	 * Die Methode wird aufgerufen, um zu prüfen, ob alle {@link JComponent}s zur
	 * Eingabe mit gültigen Werten gefüllt sind und diese zueinander nicht im
	 * Widerspruch stehen.
	 * 
	 * Hier wirft die Methode Fehler, wenn weniger als zwei Beteiligte ausgewählt
	 * sind oder wenn das Gesamtunterliegen bei einem Beteiligten größer als der
	 * Streitwert ist.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#parseEingabe()
	 */
	@Override
	public boolean parseEingabe() {
		try {
			rueckgabewert = leseFelder();
			return parseEingabe(rueckgabewert);
		} catch (IllegalArgumentException | NullPointerException ne) {
			FehlerHelper.zeigeFehler(ne.getLocalizedMessage(), ne);
			return false;
		}
	}

	/**
	 * Die Methode wird aufgerufen, um zu prüfen, ob der Rückgabewert gültige
	 * Teilwerte hat, die zueinander nicht im Widerspruch stehen.
	 * 
	 * Hier wirft die Methode Fehler, wenn weniger als zwei Beteiligte ausgewählt
	 * sind oder wenn das Gesamtunterliegen bei einem Beteiligten größer als der
	 * Streitwert ist.
	 * 
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#parseEingabe(java.lang.Object)
	 */
	@Override
	public boolean parseEingabe(BaumbachGesamtschuldnerschaft gesamtschuld)
			throws IllegalArgumentException, NullPointerException {
		this.rueckgabewert = gesamtschuld;

		if (rueckgabewert.getAufzaehlung() == null || rueckgabewert.getAufzaehlung().length < 2) {
			throw new NullPointerException(FEHLER_NIX_GENUG_BETEILIGTE);
		}
		for (int index : rueckgabewert.getAufzaehlung()) {
			// Gesamtschuldnerschaften dürfen nicht ausgewählt sein
			if (rueckgabewert.getBaumbachBeteiligtenListe().get(index) instanceof BaumbachGesamtschuldnerschaft) {
				throw new IllegalArgumentException("Sie dürfen keine Gesamtschuldnerschaften auswählen!");
			}
			// Das Unterliegen darf zusammengenommen nicht den Streitwert überschreiten.
			if ((rueckgabewert.getUnterliegen()
					+ rueckgabewert.getBaumbachBeteiligtenListe().get(index).getUnterliegen()) > rueckgabewert
							.getBaumbachBeteiligtenListe().get(index).getStreitwert()) {
				throw new IllegalArgumentException(
						String.format(FEHLER_UNTERLIEGEN_GROESSER_STREITWERT, rueckgabewert.getUnterliegen(),
								rueckgabewert.getBaumbachBeteiligtenListe().parteiBezeichner(Beteiligter.GENITIV,
										index),
								rueckgabewert.getBaumbachBeteiligtenListe().get(index).getUnterliegen(),
								rueckgabewert.getBaumbachBeteiligtenListe().get(index).getStreitwert()));
			}
		}
		return true;
	}

	/**
	 * Die Methode soll aus den Eingabefeldern des Dialogs einen Rückgabewert
	 * zusammenzubauen, der dann von {@link #showDialogAndAsk()} an die aufrufende Maske
	 * zurückgegeben werden kann.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#getRueckgabewert()
	 */
	@Override
	public BaumbachGesamtschuldnerschaft getRueckgabewert() {
		return rueckgabewert;
	}

	private BaumbachGesamtschuldnerschaft leseFelder() throws IllegalArgumentException, NullPointerException {
		NumberExtractor extractor = new NumberExtractor(ftfUnterliegen,
				ftfUnterliegen.getAccessibleContext().getAccessibleName());

		int typ = getLetztenBeteiligten().getTyp();
		if (typ == Beteiligter.KLAEGER)
			typ = Beteiligter.DRITTWIDERBEKLAGTE;

		int[] indices = beteiligtenAuswahlListe.getSelectedIndices();
		int genusNumerus = getLetztenBeteiligten().getGenusNumerus();
		if (genusNumerus < Beteiligter.PLURAL)
			genusNumerus += Beteiligter.PLURAL;

		/*
		 * anWiderklageBeteiligt darf beim gesamtschuldnerischen Beklagten // nicht
		 * gesetzt werden (eine Widerklage führt man nicht // gesamtschuldnerisch)
		 */
		final boolean anWiderklageBeteiligt = baumbachBeteiligtenListe.enthaeltAnWiderklageBeteiligte()
				&& typ != Beteiligter.BEKLAGTE;
		double unterliegen = extractor.extractDouble();
		return new BaumbachGesamtschuldnerschaft(typ, genusNumerus, unterliegen, anWiderklageBeteiligt, indices,
				baumbachBeteiligtenListe);
	}

	private Beteiligter getLetztenBeteiligten() {
		return this.beteiligtenAuswahlListe.getModel().getElementAt(beteiligtenAuswahlListe.getModel().getSize() - 1);
	}

}
// GesamtschuldnerDialog.TEXT_UNTERLIEGEN
// beteiligtenAuswahlListe
// ftfUnterliegen
//	int typ = getLetztenBeteiligten().getTyp();
//	if (typ == Beteiligter.KLAEGER)
//		typ = Beteiligter.DRITTWIDERBEKLAGTE;
//	int[] indices = beteiligtenAuswahlListe.getSelectedIndices();
//	int genusNumerus = getLetztenBeteiligten().getGenusNumerus();
//	if (genusNumerus < Beteiligter.PLURAL)
//		genusNumerus += Beteiligter.PLURAL;
//	/*
//	 * anWiderklageBeteiligt darf beim gesamtschuldnerischen Beklagten // nicht
//	 * gesetzt werden (eine Widerklage führt man nicht // gesamtschuldnerisch)
//	 * 
//	 * 
//	 */
//	final boolean anWiderklageBeteiligt = baumbachBeteiligtenListe.enthaeltAnWiderklageBeteiligte()
//			&& typ != Beteiligter.BEKLAGTE;
//	double unterliegen = 0.0;
//	unterliegen = Double.parseDouble(ftfUnterliegen.getText());
//	return new BaumbachGesamtschuldnerschaft(typ, genusNumerus, unterliegen, anWiderklageBeteiligt, indices,
//			baumbachBeteiligtenListe);
//
//	try {
//		Double.parseDouble(ftfUnterliegen.getText());
//		int[] indices = beteiligtenAuswahlListe.getSelectedIndices();
//		if (indices == null || indices.length < 2)
//			throw new NullPointerException(
//					"Nicht genug Beteiligte ausgewählt. Wählen Sie mindestens zwei Beteiligte aus!");
//		// Gesamtschuldnerschaften dürfen nicht ausgewählt sein
//	} catch (NumberFormatException | NullPointerException e) {
//		FehlerDialog.zeigeFehler(FehlerDialog.FEHLER_BLANKO, e.getLocalizedMessage(), e);
//		e.printStackTrace();
//		return false;
//	}
//	return true;
