/*
 * StreitgenossenAssistentPruefer.java
 * eu.gronos.kostenrechner.controller.baumbach (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.baumbach;

import java.awt.Component;
import java.awt.Container;
import java.util.ArrayList;

import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JFormattedTextField;
import javax.swing.JList;
import javax.swing.JSpinner;
import javax.swing.JTable;
import javax.swing.SpinnerNumberModel;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.NumberExtractor;
import eu.gronos.kostenrechner.controller.files.XmlTransferHandler;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.interfaces.ParsendUndBauend;
import eu.gronos.kostenrechner.interfaces.UnterContainerKlasse;
import eu.gronos.kostenrechner.model.baumbach.BaumbachBeteiligtenListe;
import eu.gronos.kostenrechner.model.baumbach.BaumbachBeteiligter;
import eu.gronos.kostenrechner.model.baumbach.BaumbachGesamtschuldnerschaft;
import eu.gronos.kostenrechner.model.baumbach.ProtoBeteiligtenWerte;
import eu.gronos.kostenrechner.model.baumbach.ScalableBeteiligtenTableModel;
import eu.gronos.kostenrechner.model.baumbach.Streitgenossen;
import eu.gronos.kostenrechner.model.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.model.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.view.HinzufuegenDialog;
import eu.gronos.kostenrechner.view.baumbach.BaumbachAssistentDialog;
import eu.gronos.kostenrechner.view.baumbach.SimpleBeteiligtenTable;

/**
 * Ein {@link ParsendUndBauend} für die {@link UnterContainerKlasse}
 * {@link Streitgenossen}. Wegen der Komplexität des
 * {@link BaumbachAssistentDialog}s kann er auch etwas mehr:
 * {@link #leseFelder()} liest die Felder (und baut den
 * {@link #getRueckgabewert()}), {@link #parseEingabe(int)} prüft die
 * Stimmigkeit und {@link #setzeWerte(int)} schreibt die Werte ins nächste Feld.
 * 
 *
 *
 * @author Peter Schuster (setrok)
 * @date 06.09.2019
 *
 */
public class StreitgenossenAssistentPruefer extends StreitgenossenPruefer {

	private ProtoBeteiligtenWerte primitives;// = new ProtoBeteiligtenWerte();
	private int zustand;
	public static final int HOECHSTER_ZUSTAND_OHNE_WIDERKLAGE = 6;
	public static final int HOECHSTER_ZUSTAND = 11;

	private Streitgenossen rueckgabewert;
	private JSpinner spiKlaegerZahl;
	private JSpinner spiBeklagtenZahl;
	private JTable tabKlaegerGenus;
	private JTable tabBeklagtenGenus;
	private JFormattedTextField ftfBkStreitwert;
	private JTable tabBeklagtenSw;
	private JCheckBox chbBkSwEinheitlich;
	private JList<Beteiligter> liBkGesamtschuldner;
	private JFormattedTextField ftfBkGesamtUnterliegen;
	private SimpleBeteiligtenTable tbBkUnt;
	private JCheckBox chBkGesamtschuldner;
	private JList<Beteiligter> liBkWk;
	private JSpinner spiDwbkAnz;
	private JCheckBox chbKlWk;
	private SimpleBeteiligtenTable tbDwbkGenus;
	private SimpleBeteiligtenTable tabDwbkSw;
	private SimpleBeteiligtenTable tabKlSw;
	private JCheckBox chbWkEinheitlich;
	private JFormattedTextField ftfWkSw;
	private JList<Beteiligter> liWkGesamtschuldner;
	private JCheckBox chbWkGesamtschuld;
	private JFormattedTextField ftfWkUnterliegen;
	private SimpleBeteiligtenTable tbKlDarueber;
	private SimpleBeteiligtenTable tbDwbkDarueber;
	private ScalableBeteiligtenTableModel btmKl;
	private ScalableBeteiligtenTableModel btmBk;
	private ScalableBeteiligtenTableModel btmDwbk;

	public Component focusComponent;

	public StreitgenossenAssistentPruefer() {
		super(true);
	}

	/**
	 * @param components die benötigten {@link JComponent}s, namentlich mehr oder
	 *                   weniger alle.
	 * @return diesen {@link StreitgenossenPruefer}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#leseComponents(Component[])
	 */
	@SuppressWarnings("unchecked")
	@Override
	public ParsendUndBauend<Streitgenossen> leseComponents(Component... components) {
		if (components != null) {
			for (Component comp : components) {
				String compName;
				if (comp.getName() == null)
					compName = "";
				else
					compName = comp.getName();
				if ("38025".equals(compName)) {
					spiKlaegerZahl = (JSpinner) comp;
				} else if ("38035".equals(compName)) {
					spiBeklagtenZahl = (JSpinner) comp;
				} else if ("38111".equals(compName)) {
					tabKlaegerGenus = (JTable) comp;
				} else if ("38131".equals(compName)) {
					tabBeklagtenGenus = (JTable) comp;
				} else if ("38215".equals(compName)) {
					ftfBkStreitwert = (JFormattedTextField) comp;
				} else if ("38221".equals(compName)) {
					tabBeklagtenSw = (JTable) comp;
				} else if ("38201".equals(compName)) {
					chbBkSwEinheitlich = (JCheckBox) comp;
				} else if ("38321".equals(compName)) {
					liBkGesamtschuldner = (JList<Beteiligter>) comp;
				} else if ("38335".equals(compName)) {
					ftfBkGesamtUnterliegen = (JFormattedTextField) comp;
				} else if ("38411".equals(compName)) {
					tbBkUnt = (SimpleBeteiligtenTable) comp;
				} else if ("38301".equals(compName)) {
					chBkGesamtschuldner = (JCheckBox) comp;
				} else if ("38511".equals(compName)) {
					liBkWk = (JList<Beteiligter>) comp;
				} else if ("38525".equals(compName)) {
					spiDwbkAnz = (JSpinner) comp;
				} else if ("38531".equals(compName)) {
					chbKlWk = (JCheckBox) comp;
				} else if ("38611".equals(compName)) {
					tbDwbkGenus = (SimpleBeteiligtenTable) comp;
				} else if ("38751".equals(compName)) {
					tabDwbkSw = (SimpleBeteiligtenTable) comp;
				} else if ("38731".equals(compName)) {
					tabKlSw = (SimpleBeteiligtenTable) comp;
				} else if ("38701".equals(compName)) {
					chbWkEinheitlich = (JCheckBox) comp;
				} else if ("38715".equals(compName)) {
					ftfWkSw = (JFormattedTextField) comp;
				} else if ("38821".equals(compName)) {
					liWkGesamtschuldner = (JList<Beteiligter>) comp;
				} else if ("38801".equals(compName)) {
					chbWkGesamtschuld = (JCheckBox) comp;
				} else if ("38835".equals(compName)) {
					ftfWkUnterliegen = (JFormattedTextField) comp;
				} else if ("38911".equals(compName)) {
					tbKlDarueber = (SimpleBeteiligtenTable) comp;
				} else if ("38931".equals(compName)) {
					tbDwbkDarueber = (SimpleBeteiligtenTable) comp;
				} else if (comp instanceof JComponent && ((Container) comp).getComponents() != null
						&& ((Container) comp).getComponents().length > 0) {
					leseComponents(((Container) comp).getComponents());
				}
			}
		}
		return this;
	}

	/**
	 * @param primitives d. {@link #primitives}, d. gesetzt werden soll als
	 *                   {@link ProtoBeteiligtenWerte}.
	 */
	public void setPrimitives(ProtoBeteiligtenWerte primitives) {
		this.primitives = primitives;
		Kostenrechner.getLogger().info(primitives.toString());
	}

	/**
	 * Die Methode überprüft den {@link #getRueckgabewert()}, nachdem er Felder
	 * ausgelesen hat.
	 * 
	 * @return <code>true</code>, wenn die Plausibilitäten zufrieden sind
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#parseEingabe()
	 */
	@Override
	public boolean parseEingabe() throws IllegalArgumentException, NumberFormatException, NullPointerException {
		boolean boole = false;
		try {
			rueckgabewert = leseFelder();
			if (primitives.mitWiderklage) {
				boole = getZustand() == HOECHSTER_ZUSTAND - 1 && parseEingabe(10);
			} else {
				boole = getZustand() == HOECHSTER_ZUSTAND_OHNE_WIDERKLAGE - 1 && parseEingabe(5);
			}
			// nur strenge Variante ruft parseEingabe(rueckgabewert) auf
			if (boole)
				boole = parseEingabe(rueckgabewert);
		} catch (NumberFormatException | NullPointerException ne) {
			FehlerHelper.zeigeFehler(ne.getLocalizedMessage(), ne);
			return false;
		}
		return boole;
		// return true;
	}

	/**
	 * Überprüft die Eingabe und übergibt sie an das nächste Tab
	 * 
	 * Zu jedem Panel überlegen, was ausgelesen und was in das nächste Panel
	 * geschrieben werden muss. Mögliche Exceptions und RuntimeExceptions müssen
	 * abgefangen werden
	 * 
	 * @param tab die Nummer des Tabulators
	 * @return true, wenn das Überprüfen und Übergeben klappt
	 */
	public boolean parseEingabe(int tab) throws IllegalArgumentException, NumberFormatException, NullPointerException {
		boolean boole = false;
		rueckgabewert = leseFelder();
		// boolean klWk;
		switch (tab) {
		case 0:
			// Willkommen: nix zu prüfen
			boole = true;
			break;
		case 1:
			// Anzahl Kläger und Beklagte
			if (primitives.klaegerZahl == 1 && primitives.beklagtenZahl > 0) {
				boole = true;
			} else {
				boole = false;
			}
			break;
		case 2:
			/* Bezeichnung von Kläger und Beklagten (nix zu beachten) */
			boole = true;
			break;
		case 3:
			/* Hier prüfen, ob der Streitwert über 0 ist. */
			if (primitives.bkStreitwert > 0.0) {
				boole = true;
			} else {
				boole = false;
				throw new IllegalArgumentException(HinzufuegenDialog.FEHLER_BK_STREITWERT);
			}
			break;
		case 4:
			/*
			 * Prüfen: Häkchen bei gesamtschuldnerischer Verurteilung gesetzt? Wenn ja, dann
			 * muss das Unterliegen auch über 0 sein, sonst ist die Eingabe kappes.
			 */
			if (primitives.bkGesamt) {
				if (primitives.bkGesamtIndices != null && primitives.bkGesamtIndices.length > 1
						&& primitives.bkGesamtUnterliegen >= 0
						&& primitives.bkGesamtUnterliegen <= primitives.bkStreitwert) {
					boole = true;
				} else {
					throw new IllegalArgumentException(HinzufuegenDialog.FEHLER_BK_GESAMTSCHULDNERSCHAFT);
				}
			} else {
				// sonst: alles egal, keine Plausi zu prüfen, nichts zu übertragen
				boole = true;
			}
			break;
		case 5:
			/*
			 * Wurden einige Beklagte darüber hinaus verurteilt? Welche, zu wie viel?
			 * 
			 * Prüfen, ob einer der Beteiligten ein höheres Unterliegen als der streitwert
			 * hat (ggfs. zusammen mit der gesamtschuldnerischen Verurteilung)
			 */
			boole = true;
			for (int row = 0; row < primitives.beklagtenZahl; row++) {
				// TODO Plausis?
				double d = rueckgabewert.beklagte.get(row).getUnterliegen();
				if (d > primitives.bkStreitwert)
					boole = false;
				if (chBkGesamtschuldner.isSelected() && contains(primitives.bkGesamtIndices, row)
						&& (d + primitives.bkGesamtUnterliegen) > primitives.bkStreitwert)
					boole = false;
			}
			if (!boole) {
				throw new IllegalArgumentException(HinzufuegenDialog.FEHLER_UNTERLIEGEN_GT_STREITWERT);
			}
			break;
		case 6:
			/*
			 * Mindestens ein Bk muss ausgewählt sein, dwbkZahl darf nicht negativ sein und
			 * wenn schon mitWiderklage, dann muss der Kläger widerbeklagt sein oder
			 * wenigstens ein Dwbk vorhanden sein
			 */
			if (primitives.bkWiderklaegerIndex > -1 && primitives.dwbkZahl > -1
					&& (primitives.klWk || primitives.dwbkZahl > 0)) {
				boole = true;
			} else {
				boole = false;
				throw new IllegalArgumentException(HinzufuegenDialog.FEHLER_NIX_WIDERKLAEGER_WIDERBEKLAGTER);
			}
			break;
		case 7:
			/*
			 * Bezeichnung der Drittwiderbeklagten (nix zu prüfen)
			 * 
			 * Denkt auch daran, dass man Dwbk auch weglassen kann
			 */
			boole = true;
			break;
		case 8:
			ftfWkSw.setText(Double.toString(primitives.wkStreitwert));
			if (primitives.wkStreitwert <= 0) {
				boole = false;
				throw new IllegalArgumentException(HinzufuegenDialog.FEHLER_WK_STREITWERT);
			} else {
				boole = true;
			}
			// Sonst geht das nirgends sinnvoll...
			ftfWkUnterliegen.setText(Double.toString(primitives.wkGesamtUnterliegen));
			break;
		case 9:
			/*
			 * Prüfen: Häkchen bei gesamtschuldnerischer Verurteilung gesetzt? Wenn ja, dann
			 * muss das Unterliegen auch über 0 sein, sonst ist die Eingabe kappes.
			 */
			if (primitives.wkGesamt) {
				if (primitives.wkGesamtIndices != null && primitives.wkGesamtIndices.length > 1
						&& primitives.wkGesamtUnterliegen > 0
						&& primitives.wkGesamtUnterliegen <= primitives.wkStreitwert)
					boole = true;
				else {
					boole = false;
					throw new IllegalArgumentException(HinzufuegenDialog.FEHLER_WK_GESAMTSCHULDNERSCHAFT);
				}
				/*
				 * Wenn der Kläger nicht widerbeklagt ist, darf er nicht ausgewählt werden
				 */
				if (!primitives.klWk && contains(primitives.wkGesamtIndices, 0)) {
					boole = false;
					throw new IllegalArgumentException(HinzufuegenDialog.FEHLER_KLAEGER_NIX_WBK);
				}
			} else {
				/*
				 * Wenn keine Gesamtschuldnerschaft angehakt, keine Plausis zu prüfen.
				 */
				boole = true;
			}
			/* Panel 10 ist schon unter case 8 gebaut worden. */
			break;
		case 10:
			/*
			 * Prüfen, ob einer der Beteiligten ein höheres Unterliegen als der streitwert
			 * hat (ggfs. zusammen mit der gesamtschuldnerischen Verurteilung)
			 */
			boole = true;
			/*
			 * Wenn Kl nicht wbk, dann darf man für ihn auch nichts eingeben, sonst prüfen,
			 * ob Unterliegen den Streitwert überschreitet
			 */
			if (primitives.klWk) {
				for (int row = 0; row < tbKlDarueber.getModel().getRowCount(); row++) {
					double d = ((Double) tbKlDarueber.getModel().getValueAt(row, 3)).doubleValue();
					if (d > primitives.wkStreitwert) {
						boole = false;
					} else if (chbWkGesamtschuld.isSelected() && contains(primitives.wkGesamtIndices, row)
							&& (d + primitives.wkGesamtUnterliegen) > primitives.wkStreitwert) {
						boole = false;
					}
				}
			}
			/* Dran denken, dass man Dwbk auch weglassen kann */
			if (primitives.dwbkZahl > 0) {
				for (int row = 0; row < tbDwbkDarueber.getModel().getRowCount(); row++) {
					int absRow = row + primitives.klaegerZahl;
					double d = ((Double) tbDwbkDarueber.getModel().getValueAt(row, 3)).doubleValue();
					if (d > primitives.wkStreitwert)
						boole = false;
					else if (chbWkGesamtschuld.isSelected() && contains(primitives.wkGesamtIndices, absRow)
							&& (d + primitives.wkGesamtUnterliegen) > primitives.wkStreitwert)
						boole = false;
				}
			}
			if (!boole) {
				throw new IllegalArgumentException(HinzufuegenDialog.FEHLER_UNTERLIEGEN_GT_STREITWERT);
			}
			// Wenn Widerklage erhoben: hier fertig!
			break;
		default:
			boole = false;
		}
		return boole;
	}

	/**
	 * Die Methode setzt Daten aus dem {@link UnterContainerKlasse}
	 * {@link Streitgenossen} und den {@link ProtoBeteiligtenWerte}n in das
	 * jeweilige Tab.
	 * 
	 * Wird derzeit von
	 * {@link AssistentWeiterAction#actionPerformed(java.awt.event.ActionEvent)}
	 * aufgerufen, nachdem (!) der Zustand heraufgesetzt wurde.
	 * 
	 * @param tab die Nummer des Tabs, in dessen Felder die Methode Werte schreiben
	 *            soll.
	 */
	public void setzeWerte(int tab) {
		switch (tab) {
		case 0:
			// TODO ggfs: Aktenzeichen in HTML schreiben, id heißt "az"!
			focusComponent = null;
			break;
		case 1:
			spiKlaegerZahl.setModel(new SpinnerNumberModel(primitives.klaegerZahl, 1, 1, 1));
			spiBeklagtenZahl.setModel(new SpinnerNumberModel(primitives.beklagtenZahl, 0, 100, 1));
			focusComponent = spiBeklagtenZahl;
			break;
		case 2:
			btmKl.setColumnCount(2);
			tabKlaegerGenus.setModel(btmKl);
			btmBk.setColumnCount(2);
			tabBeklagtenGenus.setModel(btmBk);
			focusComponent = null;
			break;
		case 3:
			ftfBkStreitwert.setText(Double.toString(primitives.bkStreitwert));
			btmBk.setColumnCount(3);
			btmBk.setEachStreitwert(primitives.bkStreitwert);
			tabBeklagtenSw.setModel(btmBk);
			// Dem Change-Listener auf die Sprünge helfen...
			chbBkSwEinheitlich.setSelected(false);
			chbBkSwEinheitlich.setSelected(true);
			chbBkSwEinheitlich.setSelected(primitives.bkSwEinheitlich);
			/*
			 * Wenn der Assistent RK4 überspringt, weil es nur einen Beklagten gibt, kann
			 * der Streitwert nur einheitlich sein: Das sollte man dann auch nicht ändern
			 * können, außer der Sw ist aus irgendwelchen Gründen als uneinheitlich
			 * angegeben.
			 */
			chbBkSwEinheitlich.setEnabled(!primitives.tab4ueberspringen || !primitives.bkSwEinheitlich);
			// chbBkSwEinheitlich.requestFocusInWindow();
			focusComponent = ftfBkStreitwert;// .requestFocusInWindow();
			break;
		case 4:
			/* Panel 4 vorbereiten: Array Beteiligter[], nur erste Spalte */
			chBkGesamtschuldner.setSelected(primitives.bkGesamt);
			ftfBkGesamtUnterliegen.setText(Double.toString(primitives.bkGesamtUnterliegen));
			liBkGesamtschuldner.setListData(btmBk.getBeteiligterColumn());
			focusComponent = ftfBkGesamtUnterliegen;
			break;
		case 5:
			/* Panel 5 vorbereiten */
			btmBk.setColumnCount(4);
			btmBk.setEachUnterliegen(primitives.bkUnterliegen);
			tbBkUnt.setModel(btmBk);
			focusComponent = null;
			break;
		case 6:
			/*
			 * Für die Rückgabe muss anWiderklageBeteiligt erst einmal auf false gesetzt
			 * werden, wo es nicht gesetzt wurde
			 */
			chbKlWk.setSelected(primitives.klWk);
			spiDwbkAnz.setModel(new SpinnerNumberModel(primitives.dwbkZahl, 0, 100, 1));
			/* Array Beteiligter[] wird auch für Panel 6 gebraucht */
			liBkWk.setListData(btmBk.getBeteiligterColumn());
			liBkWk.setSelectedIndex(primitives.bkWiderklaegerIndex);
			focusComponent = liBkWk;
			break;
		case 7:
			// Auf Panel 7 setzen
			if (primitives.dwbkZahl > 0) {
				btmDwbk.setColumnCount(2);
				tbDwbkGenus.setModel(btmDwbk);
				tbDwbkGenus.setEnabled(true);
				focusComponent = tbDwbkGenus;
			} else {
				tbDwbkGenus.setEnabled(false);
				focusComponent = null;
			}
			break;
		case 8:
			/*
			 * Für Panel 8 werden dann wieder die Klägerdaten gebraucht: angezeigt werden
			 * sollen sie immer. Der ChangeListener der checkbox sorgt dafür, dass die
			 * Kläger nur bearbeitet werden können, wenn der Kläger was mit der Wk am Hut
			 * hat.
			 */
			if (primitives.klWk) {
				btmKl.setColumnCount(3);
				btmKl.setEachStreitwert(primitives.wkStreitwert);
				tabKlSw.setModel(btmKl);
			} else
				tabKlSw.setEnabled(false);
			if (primitives.dwbkZahl > 0) {
				btmDwbk.setColumnCount(3);
				btmDwbk.setEachStreitwert(primitives.wkStreitwert);
				tabDwbkSw.setModel(btmDwbk);
			} else
				tabDwbkSw.setEnabled(false);
			ftfWkSw.setText(Double.toString(primitives.wkStreitwert));
			/*
			 * Einmal den ChangeListener aufmerksam machen, der schreibt auch den
			 * wkStreitwert ins Textfeld.
			 */
			chbWkEinheitlich.setSelected(false);
			chbWkEinheitlich.setSelected(true);
			chbWkEinheitlich.setSelected(primitives.wkSwEinheitlich);
			/*
			 * Wenn der Assistent RK9 überspringt, weil es nur einen Widerbeklagten gibt,
			 * kann der Streitwert nur einheitlich sein: Das sollte man dann auch nicht
			 * ändern können, außer der Sw ist aus irgendwelchen Gründen als uneinheitlich
			 * angegeben.
			 */
			chbWkEinheitlich.setEnabled(!primitives.tab9ueberspringen || !primitives.wkSwEinheitlich);
			// chbWkEinheitlich.requestFocusInWindow();
			focusComponent = ftfWkSw;// .requestFocusInWindow();
			break;
		case 9:
			/*
			 * Panel 9 setzen: Immer eine Liste aus Kl (auch wenn nicht wbk) und Dwbk bauen,
			 * damit die Indices beim Bauen einer BaumbachGesamtschuldnerschaft stimmen.
			 */
			ArrayList<Beteiligter> tempWiderbeklagte = new ArrayList<Beteiligter>();
			tempWiderbeklagte.addAll(rueckgabewert.klaeger);
			if (primitives.dwbkZahl > 0) {
				tempWiderbeklagte.addAll(rueckgabewert.drittwiderbeklagte);
			}
			liWkGesamtschuldner.setListData(tempWiderbeklagte.toArray(new Beteiligter[tempWiderbeklagte.size()]));
			chbWkGesamtschuld.setSelected(false);
			chbWkGesamtschuld.setSelected(true);
			chbWkGesamtschuld.setSelected(primitives.wkGesamt && tempWiderbeklagte.size() > 1);
			chbWkGesamtschuld.setEnabled(tempWiderbeklagte.size() > 1);
			ftfWkUnterliegen.setText(Double.toString(primitives.wkGesamtUnterliegen));
			focusComponent = liWkGesamtschuldner;
			break;
		case 10:
			/*
			 * Panel 10 setzen: Wenn Kl nicht wbk, dann darf man für ihn auch nichts
			 * eingeben. Genauso, wenn es keine Drittwiderbeklagten gibbet.
			 */
			if (primitives.klWk) {
				btmKl.setColumnCount(4);
				btmKl.setEachUnterliegen(primitives.wkUnterliegen);
				tbKlDarueber.setModel(btmKl);
				tbKlDarueber.setEnabled(true);
			} else {
				tbKlDarueber.setEnabled(false);
			}
			if (primitives.dwbkZahl > 0) {
				btmDwbk.setColumnCount(4);
				btmDwbk.setEachUnterliegen(primitives.wkUnterliegen);
				tbDwbkDarueber.setModel(btmDwbk);
				tbDwbkDarueber.setEnabled(true);
			} else {
				tbDwbkDarueber.setEnabled(false);
			}
			focusComponent = null;
			break;
		default:
			break;
		}
	}

	/**
	 * Bestimmte Zustände können übersprungen werden, wenn Xjustiz-Daten importiert
	 * wurden.
	 * 
	 * @param tab der Index des zu überprüfenden Zustands.
	 * @return Ob der Zustand übersprungen werden kann
	 */
	public boolean isUeberspringbar(int tab) {
		if (tab == 1 && primitives.tab1ueberspringen)
			return true;
		if (tab == 2 && primitives.tab2ueberspringen)
			return true;
		if (tab == 4 && primitives.tab4ueberspringen)
			return true;
		if (tab == 6 && primitives.tab6ueberspringen)
			return true;
		if (tab == 7 && primitives.tab7ueberspringen)
			return true;
		if (tab == 9 && primitives.tab9ueberspringen)
			return true;
		return false;
	}

	/**
	 * Mit dieser Methode kann {@link BaumbachAssistentDialog} die Werte aus dem
	 * {@link VerfahrensDatenContainer} übergeben - oder einen leeren
	 * {@link UnterContainerKlasse} {@link Streitgenossen}.
	 * 
	 * @param rueckgabewert d. {@link #rueckgabewert}, d. gesetzt werden soll als
	 *                      {@link Streitgenossen}.
	 */
	public void setRueckgabewert(Streitgenossen rueckgabewert) {
		this.rueckgabewert = rueckgabewert;
		btmKl = new ScalableBeteiligtenTableModel(Beteiligter.KLAEGER, rueckgabewert.klaeger);
		btmBk = new ScalableBeteiligtenTableModel(Beteiligter.BEKLAGTE, rueckgabewert.beklagte);
		btmDwbk = new ScalableBeteiligtenTableModel(Beteiligter.DRITTWIDERBEKLAGTE, rueckgabewert.drittwiderbeklagte);
	}

	/**
	 * Die Methode dient dazu, die Verfahrensdaten vom XJustiz-Import zu prüfen und
	 * dann ein paar Felder schon einmal vorzubelegen
	 * 
	 * @param container  {@link VerfahrensDatenContainer}
	 * @param primitives die {@link ProtoBeteiligtenWerte}, die vorbelegt werden
	 *                   sollen
	 * @return <code>true</code>, wenn alles stimmt
	 * @throws IllegalArgumentException wenn die Plausis nicht stimmen!
	 */
	public ProtoBeteiligtenWerte pruefeVerfahrensDatenContainer(VerfahrensDatenContainer container,
			ProtoBeteiligtenWerte primitives) throws IllegalArgumentException {
		if (container == null)
			throw new IllegalArgumentException("Leere Datei oder Datei konnte nicht gelesen werden!");
		else
			primitives.ohneXjustiz = container == null;
		primitives.tab1ueberspringen = true;
		primitives.tab2ueberspringen = true;
		primitives.tab6ueberspringen = true;
		primitives.tab7ueberspringen = true;
		primitives.mitWiderklage = container.streitgenossen.widerklage;
		//VerfahrensDatenContainerUtil.
		XmlTransferHandler xmlHelfer = new XmlTransferHandler();
		if (!xmlHelfer.gibtsKlaeger(container) || container.streitgenossen.klaeger.size() != 1)
			throw new IllegalArgumentException("Nur ein Kläger erlaubt!");
		primitives.klaegerZahl = container.streitgenossen.klaeger.size();
		primitives.klWk = (primitives.mitWiderklage
				&& container.streitgenossen.klaeger.enthaeltAnWiderklageBeteiligte());
		/* Gibt's überhaupt Beklagte und wenn ja, maximal einen Widerkläger? */
		Kostenrechner.getLogger().info(String.format("Beklagtenzahl: %d.%n", container.streitgenossen.beklagte.size()));
		if (container.streitgenossen.beklagte.size() < 1) {
			/*
			 * Wenn keine Beklagten vorhanden sind, darf man die Tabs 1+2 (ggfs. auch 6)
			 * nicht ausblenden...
			 */
			primitives.tab1ueberspringen = false;
			primitives.tab2ueberspringen = false;
			primitives.tab6ueberspringen = false;
		} else if (container.streitgenossen.beklagte.zaehleAnWiderklageBeteiligte() > 1)
			throw new IllegalArgumentException("Nur ein Widerkläger erlaubt!");
		primitives.beklagtenZahl = container.streitgenossen.beklagte.size();
		/*
		 * Wenn es weniger als einen Beklagten gibt, RK4 überspringen
		 */
		primitives.tab4ueberspringen = (primitives.beklagtenZahl < 2);
		/*
		 * Wenn RK4 übersprungen wird, darf Bk-Gesamtschuld nicht vorbelegt sein, sonst
		 * muss sie vorbelegt sein.
		 */
		primitives.bkGesamt = !primitives.tab4ueberspringen;
		if (primitives.mitWiderklage) {
			for (int i = 0; i < container.streitgenossen.beklagte.size(); i++)
				if (container.streitgenossen.beklagte.get(i).isAnWiderklageBeteiligt())
					primitives.bkWiderklaegerIndex = i;
		}
		//VerfahrensDatenContainerUtil
		if (xmlHelfer.gibtsDrittwiderbeklagte(container.streitgenossen))
			primitives.dwbkZahl = container.streitgenossen.drittwiderbeklagte.size();
		else
			primitives.dwbkZahl = 0;
		/*
		 * Gibt es Widerkläger ohne Widerbeklagte oder Widerbeklagte ohne Widerkläger?
		 */
		if (primitives.mitWiderklage) {
			if (container.streitgenossen.beklagte.zaehleAnWiderklageBeteiligte() > 0
					&& container.streitgenossen.klaeger.zaehleAnWiderklageBeteiligte() < 1 && primitives.dwbkZahl < 1) {
				/*
				 * Wenn Widerkläger ohne Widerbeklagte, darf man nur die Tabs 6+7 nicht
				 * ausblenden, dann kann man den Kläger als Widerbeklagter markieren oder
				 * Drittwiderbeklagte hinzufügen
				 */
				primitives.tab6ueberspringen = false;
				primitives.tab7ueberspringen = false;
			} else if ((container.streitgenossen.klaeger.zaehleAnWiderklageBeteiligte() > 0 || primitives.dwbkZahl > 0)
					&& container.streitgenossen.beklagte.zaehleAnWiderklageBeteiligte() < 1) {
				/*
				 * Wenn Widerbeklagte ohne Widerkläger, darf man nur das Tab 6 nicht ausblenden,
				 * dort kann der Widerkläger markiert werden... Das hat zur Folge, dass der
				 * Nutzer die Anzahl der Drittwiderbeklagten ändern kann, deshalb darf dann auch
				 * Tab 7 nicht übersprungen werden.
				 */
				primitives.tab6ueberspringen = false;
				primitives.tab7ueberspringen = false;
			}
			/*
			 * Prüfen, ob es mehr als einen Widerbeklagten gibt. Wenn nicht, kann sich der
			 * Assistent die Frage nach Widerklage-Gesamtschuld auf RK9 sparen!
			 */
			if (primitives.klWk)
				primitives.wbkZahl = 1;
			else
				primitives.wbkZahl = 0;
			primitives.wbkZahl += primitives.dwbkZahl;
			primitives.tab9ueberspringen = (primitives.wbkZahl < 2);
			// wenn der Assistent RK9 _nicht_ überspringt, Gesamtschuld vorbelegen.
			primitives.wkGesamt = !primitives.tab9ueberspringen;
		}
		// hier einen Default-Wert für den Streitwert nehmen (z.B. Xjustiz)
		primitives.bkStreitwert = container.allgemein.streitwert;
		primitives.wkStreitwert = container.allgemein.streitwert;
		// return true;
		return primitives;
	}

	/**
	 * @return gibt {@link #zustand} als int zurück.
	 */
	public int getZustand() {
		return zustand;
	}

	/**
	 * @param zustand d. {@link #zustand}, d. gesetzt werden soll als int
	 */
	public void setZustand(int zustand) {
		this.zustand = zustand;
	}

	/**
	 * Gibt abhängig von Widerklage oder nicht den tatsächlichen höchstmöglichen
	 * Zustand aus.
	 * 
	 * @return {@link StreitgenossenAssistentPruefer#HOECHSTER_ZUSTAND} oder
	 *         {@link StreitgenossenAssistentPruefer#HOECHSTER_ZUSTAND_OHNE_WIDERKLAGE}
	 */
	public int getHoechsterZustand() {
		if (primitives.mitWiderklage) {
			return HOECHSTER_ZUSTAND;
		} else {
			return HOECHSTER_ZUSTAND_OHNE_WIDERKLAGE;
		}
	}

	/**
	 * Die Methode packt alle Programmzustände in einen {@link UnterContainerKlasse}
	 * {@link Streitgenossen}.
	 * 
	 * @return den Untercontainer von
	 *         {@link VerfahrensDatenContainer#streitgenossen}
	 */
	private Streitgenossen leseFelder() throws NumberFormatException, NullPointerException {
		if (rueckgabewert == null) {
			rueckgabewert = new Streitgenossen();
		}
		switch (getZustand()) {
		default:
		case 0:
			break;
		case 1:
			// Überspringbar!
			// Anzahl Kläger und Beklagte
			primitives.klaegerZahl = ((SpinnerNumberModel) spiKlaegerZahl.getModel()).getNumber().intValue();
			/*
			 * bbl Kläger anlegen: x BaumbachBeteiligte hinzufügen. Dabei aufpassen, falls
			 * über Xjustiz schon etwas übergeben wurde, deshalb bei der Zählung nicht
			 * zwingend mit 0 beginnen
			 */
			for (int lfd = rueckgabewert.klaeger.size(); rueckgabewert.klaeger.size() < primitives.klaegerZahl; lfd++) {
				BaumbachBeteiligter klaeger = new BaumbachBeteiligter();
				klaeger.setLfdNr(lfd);
				klaeger.setTyp(Beteiligter.KLAEGER);
				rueckgabewert.klaeger.add(klaeger);
				Kostenrechner.getLogger().info(String.format("Kläger Nr. %d hinzugefügt.%n", lfd + 1));
			}
			primitives.beklagtenZahl = ((SpinnerNumberModel) spiBeklagtenZahl.getModel()).getNumber().intValue();
			/*
			 * Wenn es weniger als einen Beklagten gibt, muss der Assistent die RK4
			 * überspringen
			 */
			primitives.tab4ueberspringen = (primitives.beklagtenZahl < 2);
			/*
			 * Wenn RK4 übersprungen wird, darf Bk-Gesamtschuld nicht vorbelegt sein, sonst
			 * muss sie vorbelegt sein.
			 */
			primitives.bkGesamt = !primitives.tab4ueberspringen;
			/*
			 * bbl Beklagte anlegen: x BaumbachBeteiligte hinzufügen. Dabei aufpassen, falls
			 * über Xjustiz schon etwas übergeben wurde, deshalb bei der Zählung nicht
			 * zwingend mit 0 beginnen
			 */
			for (int lfd = rueckgabewert.beklagte.size(); rueckgabewert.beklagte
					.size() < primitives.beklagtenZahl; lfd++) {
				BaumbachBeteiligter beklagter = new BaumbachBeteiligter();
				beklagter.setLfdNr(lfd);
				beklagter.setTyp(Beteiligter.BEKLAGTE);
				rueckgabewert.beklagte.add(beklagter);
				Kostenrechner.getLogger().info(String.format("Beklagten Nr. %d hinzugefügt.%n", lfd + 1));
			}
			/*
			 * Wenn die Widerklage aktiviert ist und es nur einen Beklagten gibt, muss er
			 * der Widerkläger sein
			 */
			if (primitives.mitWiderklage && primitives.beklagtenZahl == 1)
				primitives.bkWiderklaegerIndex = 0;
			break;
		case 2:
			// Überspringbar!
			// genus bei Kläger steht schon in Tabelle, deshalb auch in Werten
			// genus bei Beklagten steht schon in Tabelle, deshalb auch in Werten
			break;
		case 3:
			// Streitwert Bk: einheitlich oder einzeln
			primitives.bkSwEinheitlich = chbBkSwEinheitlich.isSelected();
			if (primitives.bkSwEinheitlich) {
				primitives.bkStreitwert = new NumberExtractor(ftfBkStreitwert,
						ftfBkStreitwert.getAccessibleContext().getAccessibleName()).extractDouble();
				// bk Streitwert setzen
				for (int row = 0; row < primitives.beklagtenZahl; row++)
					rueckgabewert.beklagte.get(row).setStreitwert(primitives.bkStreitwert);
			} else {
				primitives.bkStreitwert = rueckgabewert.beklagte.getHoechstenStreitwert(Beteiligter.BEKLAGTE);
				// einzeln in bbl der bk übertragen unnötig, weil in Tabelle eingetragen
			}
			break;
		case 4:
			primitives.bkGesamt = chBkGesamtschuldner.isSelected();
			if (primitives.bkGesamt) {
				primitives.bkGesamtIndices = liBkGesamtschuldner.getSelectedIndices();
				primitives.bkGesamtUnterliegen = new NumberExtractor(ftfBkGesamtUnterliegen,
						ftfBkGesamtUnterliegen.getAccessibleContext().getAccessibleName()).extractDouble();
				// Bk Ges.sch. bauen,
				BaumbachGesamtschuldnerschaft gessch = new BaumbachGesamtschuldnerschaft();
				gessch.setTyp(Beteiligter.BEKLAGTE);
				gessch.setGenusNumerus(rueckgabewert.beklagte.get(primitives.beklagtenZahl - 1).getGenusNumerus()
						+ Beteiligter.PLURAL);
				gessch.setUnterliegen(primitives.bkGesamtUnterliegen);
				// anWiderklageBeteiligt wird bei Beklagten nicht gesetzt
				gessch.setAufzaehlung(primitives.bkGesamtIndices);
				gessch.setBaumbachBeteiligtenListe(rueckgabewert.beklagte);
				// Bk Ges.sch. der bbl Bk hinzfügen
				rueckgabewert.beklagte.add(gessch);
			}
			break;
		case 5:
			// Bk unterliegen (einzeln), steht schon in bbl
			break;
		case 6:
			// Überspringbar!
			// Welcher Beklagte klagt? Wie viele Drittwiderbeklagten?
			rueckgabewert.widerklage = primitives.mitWiderklage;
			primitives.dwbkZahl = ((SpinnerNumberModel) spiDwbkAnz.getModel()).getNumber().intValue();
			// Wenn es keine Drittwiderbeklagten gibt, kann Tab 7 übersprungen werden!
			if (primitives.dwbkZahl < 1) {
				primitives.tab7ueberspringen = true;
			} else {
				primitives.tab7ueberspringen = false;
			}
			/*
			 * bbl Drittwiderbeklagte anlegen: x BaumbachBeteiligte hinzufügen. Dabei
			 * aufpassen, falls über Xjustiz schon etwas übergeben wurde, deshalb bei der
			 * Zählung nicht zwingend mit 0 beginnen.
			 */
			for (int lfd = rueckgabewert.drittwiderbeklagte.size(); rueckgabewert.drittwiderbeklagte
					.size() < primitives.dwbkZahl; lfd++) {
				BaumbachBeteiligter drittwiderbeklagter = new BaumbachBeteiligter();
				drittwiderbeklagter.setLfdNr(lfd);
				drittwiderbeklagter.setTyp(Beteiligter.DRITTWIDERBEKLAGTE);
				// Drittwiderbeklagte sind immer an der Wk beteiligt
				drittwiderbeklagter.setAnWiderklageBeteiligt(true);
				rueckgabewert.drittwiderbeklagte.add(drittwiderbeklagter);
				Kostenrechner.getLogger().info(String.format("Drittwiderbeklagten Nr. %d hinzugefügt.%n", lfd + 1));
			}
			/*
			 * Schreiben, ob Wk auch gegen den Kläger geht, als Widerbeklagten kennzeichnen
			 * – oder auch nicht.
			 */
			primitives.klWk = chbKlWk.isSelected();
			rueckgabewert.klaeger.get(0).setAnWiderklageBeteiligt(primitives.klWk);
			/*
			 * Prüfen, ob es mehr als einen Widerbeklagten gibt. Wenn nicht, kann sich der
			 * Assistent die Frage nach Widerklage-Gesamtschuld auf RK9 sparen!
			 */
			if (primitives.klWk)
				primitives.wbkZahl = 1;
			else
				primitives.wbkZahl = 0;
			primitives.wbkZahl += primitives.dwbkZahl;
			primitives.tab9ueberspringen = (primitives.wbkZahl < 2);
			// wenn der Assistent RK9 _nicht_ überspringt, Gesamtschuld vorbelegen.
			primitives.wkGesamt = !primitives.tab9ueberspringen;
			/*
			 * Beim markierten Beklagten anWiderklageBeteiligt auf true setzen (alle anderen
			 * müssen auf false, falls früher mal gesetzt).
			 */
			primitives.bkWiderklaegerIndex = liBkWk.getSelectedIndex();
			for (int i = 0; i < primitives.beklagtenZahl; i++) {
				BaumbachBeteiligter bk = rueckgabewert.beklagte.get(i);
				if (i == primitives.bkWiderklaegerIndex)
					bk.setAnWiderklageBeteiligt(true);
				else
					bk.setAnWiderklageBeteiligt(false);
			}
			break;
		case 7:
			// Überspringbar!
			// genus bei Drittwiderbeklagten steht schon in bbl
			break;
		case 8:
			/*
			 * Streitwert Kl/Dwbk: einheitlich oder einzeln: Wurden Widerbeklagte in
			 * derselben Höhe verklagt? Welche (jeweils): Hier achtet die Prüfung immer
			 * darauf, dass es einen Kläger geben kann, der nicht widerbeklagt ist bzw. dass
			 * es möglicherweise keine Dwbk gibt.
			 */
			primitives.wkSwEinheitlich = chbWkEinheitlich.isSelected();
			if (primitives.wkSwEinheitlich) {
				primitives.wkStreitwert = new NumberExtractor(ftfWkSw,
						ftfWkSw.getAccessibleContext().getAccessibleName()).extractDouble();
				// ggfs. Streitwert zum Kl schreiben
				if (primitives.klWk)
					rueckgabewert.klaeger.get(0).setStreitwert(primitives.wkStreitwert);
				// Streitwert bei Dwbken setzen, falls welche vorhanden
				if (primitives.dwbkZahl > 0)
					for (int row = 0; row < primitives.dwbkZahl; row++)
						rueckgabewert.drittwiderbeklagte.get(row).setStreitwert(primitives.wkStreitwert);
			} else {
				/* wenn nicht einheitlich */
				double klWkStreitwert = 0;
				double dwbkStreitwert = 0;
				if (primitives.klWk) {
					// Wenn nicht bei Wk dabei, dann auch nichts lesen
					klWkStreitwert = rueckgabewert.klaeger.getHoechstenStreitwert(Beteiligter.KLAEGER);
					// Streitwert steht schon in bbl
				}
				if (primitives.dwbkZahl > 0) {
					// Man kann Dwbk auch weglassen
					dwbkStreitwert = rueckgabewert.drittwiderbeklagte.getHoechstenStreitwert(Beteiligter.BEKLAGTE);
					// einzelne Streitwerte stehen schon in bbl
				}
				// Höheren Streitwert von beiden speichern
				if (klWkStreitwert > dwbkStreitwert)
					primitives.wkStreitwert = klWkStreitwert;
				else
					primitives.wkStreitwert = dwbkStreitwert;
			}
			break;
		case 9:
			/*
			 * Kl/Dwbk Ges.sch. bauen, der bbl Bk hinzfügen: Wurden die Widerbeklagten z.T.
			 * gesamtschuldnerisch verurteilt? Welche Widerbeklagten, in welcher Höhe?
			 */
			primitives.wkGesamt = chbWkGesamtschuld.isSelected();
			if (primitives.wkGesamt) {
				primitives.wkGesamtIndices = liWkGesamtschuldner.getSelectedIndices();
				primitives.wkGesamtUnterliegen = new NumberExtractor(ftfWkUnterliegen,
						ftfWkUnterliegen.getAccessibleContext().getAccessibleName()).extractDouble();
				// Dwbk Ges.sch. bauen,
				BaumbachGesamtschuldnerschaft gessch = new BaumbachGesamtschuldnerschaft();
				gessch.setTyp(Beteiligter.DRITTWIDERBEKLAGTE);
				gessch.setGenusNumerus(rueckgabewert.drittwiderbeklagte.get(primitives.dwbkZahl - 1).getGenusNumerus()
						+ Beteiligter.PLURAL);
				// anWiderklageBeteiligt wird bei Drittwiderbeklagten immer gesetzt
				gessch.setAnWiderklageBeteiligt(true);
				gessch.setUnterliegen(primitives.wkGesamtUnterliegen);
				gessch.setAufzaehlung(primitives.wkGesamtIndices);
				// Gemeinsame bbl bauen und in der Gesamtschuld setzen
				BaumbachBeteiligtenListe zwischen = new BaumbachBeteiligtenListe();
				zwischen.addAll(rueckgabewert.klaeger);
				zwischen.addAll(rueckgabewert.drittwiderbeklagte);
				gessch.setBaumbachBeteiligtenListe(zwischen);
				// Kl/Dwbk Ges.sch. der bbl Dwbk hinzfügen
				rueckgabewert.drittwiderbeklagte.add(gessch);
			}
			break;
		case 10:
			/*
			 * Kl/Dwbk unterliegen (einzeln): Wurden einzelne Widerbeklagten darüber hinaus
			 * einzeln verurteilt? Welche, in welcher Höhe?
			 */
			if (primitives.klWk) {
				// Unterliegen steht schon in bbl
				if (rueckgabewert.klaeger.get(0).getUnterliegen() > primitives.wkUnterliegen)
					primitives.wkUnterliegen = rueckgabewert.klaeger.get(0).getUnterliegen();
			}
			if (primitives.dwbkZahl > 0) {
				// Dwbk unterliegen (einzeln) steht schon in bbl
				for (int row = 0; row < primitives.dwbkZahl; row++) {
					if (rueckgabewert.drittwiderbeklagte.get(row).getUnterliegen() > primitives.wkUnterliegen)
						primitives.wkUnterliegen = rueckgabewert.drittwiderbeklagte.get(row).getUnterliegen();
				}
			}
			break;
		}
		return rueckgabewert;
	}

	/**
	 * Die Methode prüft, ob in einem Array int[] ein bestimmter int enthalten ist.
	 * 
	 * @param indices das zu durchsuchende Array int[]
	 * @param index   der int-Wert, der enthalten sein könnte
	 * @return true, wenn index in indices enthalten ist.
	 */
	private boolean contains(int[] indices, int index) {
		boolean contains = false;
		for (int i : indices)
			if (i == index)
				contains = true;
		return contains;
	}

}
//public ParsendUndBauend<Streitgenossen> setzeSonstigeWerte(double bkStreitwert, double bkGesamtUnterliegen,
//double bkUnterliegen, boolean mitWiderklage, boolean ohneXjustiz, double wkStreitwert,
//double wkGesamtUnterliegen, double wkUnterliegen) {
//this.bkStreitwert = bkStreitwert;
//this.bkGesamtUnterliegen = bkGesamtUnterliegen;
//this.bkUnterliegen = bkUnterliegen;
//this.mitWiderklage = mitWiderklage;
//this.ohneXjustiz = ohneXjustiz;
//this.wkStreitwert = wkStreitwert;
//this.wkGesamtUnterliegen = wkGesamtUnterliegen;
//this.wkUnterliegen = wkUnterliegen;
//return this;
//}
// setzeSonstigeWerte(bkStreitwert, bkGesamtUnterliegen, bkUnterliegen,
// isMitWiderklage(), getVerfahrensDatenContainer() == null, getWkStreitwert(),
// getWkGesamtUnterliegen(),
// wkUnterliegen);

//else if ("".equals(compName)) {
//	Component c = comp;
//	System.out.printf("Hä: %s ", compName);
//} 
// switch (tab) {
// default:
// break;
// }
// case 0:
// default:
// // Willkommen: nix zu prüfen
// b = true;
// break;
// case 1:
// // Anzahl Kläger und Beklagte
// try {
// klaegerZahl = ((SpinnerNumberModel)
// spiKlaegerZahl.getModel()).getNumber().intValue();
// // Integer.parseInt(spiKlaegerZahl.getText());
// beklagtenZahl = ((SpinnerNumberModel)
// spiBeklagtenZahl.getModel()).getNumber().intValue();
// // Integer.parseInt(spBeklagtenZahl.getText());
// if (klaegerZahl == 1 && beklagtenZahl > 0) {
// objectsKlaeger = new Object[klaegerZahl][4];
// objectsBeklagte = new Object[beklagtenZahl][4];
// sbtmKlaeger = new ScalableBeteiligtenTableModel(objectsKlaeger,
// Beteiligter.KLAEGER);
// tabKlaegerGenus.setModel(sbtmKlaeger);
// // ((BeteiligtenTable) tabKlaegerGenus).setWidth();
// sbtmBeklagte = new ScalableBeteiligtenTableModel(objectsBeklagte,
// Beteiligter.BEKLAGTE);
// tabBeklagtenGenus.setModel(sbtmBeklagte);
// // ((BeteiligtenTable) tabBeklagtenGenus).setWidth();
// b = true;
// } else
// b = false;
// } catch (IllegalArgumentException e) {
// if (e instanceof NumberFormatException)
// FehlerDialog.zeigeFehler(FehlerDialog.FEHLER_BLANKO,
// HinzufuegenDialog.FEHLER_ZAHLENFORMAT_KL_BK,
// e);
// else
// FehlerDialog.zeigeFehler(FehlerDialog.FEHLER_BLANKO,
// HinzufuegenDialog.FEHLER_BEIM_ANLEGEN_KL_BK,
// e);
// // e.printStackTrace();
// b = false;
// }
// break;
// case 2:
// /* Bezeichnung von Kläger und Beklagten (nix zu beachten) */
// objectsKlaeger = sbtmKlaeger.getAllValues();
// objectsBeklagte = sbtmBeklagte.getAllValues();
// ftfBkStreitwert.setText(Double.toString(bkStreitwert));
// sbtmBeklagte = new ScalableBeteiligtenTableModel(objectsBeklagte,
// Beteiligter.BEKLAGTE, bkStreitwert);
// tabBeklagtenSw.setModel(sbtmBeklagte);
// chbBkSwEinheitlich.setSelected(false);
// chbBkSwEinheitlich.setSelected(true);
// // Kläger werden für eine der Widerklage-Tabs gebraucht!
// b = true;
// break;
// case 3:
// /*
// * Streitwert der Beklagten (für alle einheitlich?) Prüfen, ob einheitlich
// oder
// * nicht
// */
// boolean bkEinheitlich = chbBkSwEinheitlich.isSelected();
// try {
// if (bkEinheitlich) {
// /*
// * wenn einheitlich: speichern und auf alle Bk übernehmen und alle Werte
// ziehen
// * --> für Panel 5
// */
// bkStreitwert = doubleAusFormatter(ftfBkStreitwert,
// lblBkStreitwert.getText()); // Double.parseDouble(ftfBkStreitwert.getText());
// sbtmBeklagte.setEachStreitwert(bkStreitwert, true);
// sbtmBeklagte.fireTableRowsUpdated(0, sbtmBeklagte.getRowCount() - 1);
// objectsBeklagte = sbtmBeklagte.getAllValues();
// } else {
// /*
// * sonst: getValues und höchsten Streitwert für schlechte Tage zur Seite legen
// * --> für den zurückzugebenden KostenTenorContainer
// */
// objectsBeklagte = sbtmBeklagte.getAllValues();
// bkStreitwert = sbtmBeklagte.getHoechstenStreitwert();
// ftfBkStreitwert.setText(Double.toString(bkStreitwert));
// }
// /*
// * Panel 4 vorbereiten: Array Beteiligter[], nur erste Spalte --> wird auch
// für
// * Panel 6 gebraucht
// */
// beteiligteBeklagte = sbtmBeklagte.getBeteiligterColumn();
// liBkGesamtschuldner.setListData(beteiligteBeklagte);
// ftfBkGesamtUnterliegen.setText(Double.toString(bkGesamtUnterliegen));
// /* Panel 5 schon einmal vorbereiten */
// sbtmBeklagte = new ScalableBeteiligtenTableModel(objectsBeklagte,
// Beteiligter.BEKLAGTE, bkStreitwert,
// bkUnterliegen);
// tbBkUnt.setModel(sbtmBeklagte);
// b = true;
// } catch (NumberFormatException e) {
// FehlerDialog.zeigeFehler(FehlerDialog.FEHLER_BLANKO,
// HinzufuegenDialog.FEHLER_ZAHLENFORMAT_SW, e);
// // e.printStackTrace();
// b = false;
// }
// break;
// case 4:
// bkGesamt = chBkGesamtschuldner.isSelected();
// /*
// * Prüfen: Häkchen bei gesamtschuldnerischer Verurteilung gesetzt? Wenn ja,
// dann
// * muss das Unterliegen auch über 0 sein, sonst ist die Eingabe kappes.
// */
// if (bkGesamt) {
// try {
// bkGesamtIndices = liBkGesamtschuldner.getSelectedIndices();
// bkGesamtUnterliegen = doubleAusFormatter(ftfBkGesamtUnterliegen,
// lblBkGesamtUnterliegen.getText()); //
// Double.parseDouble(ftfBkGesamtUnterliegen.getText());
// if (bkGesamtIndices != null && bkGesamtIndices.length > 1 &&
// bkGesamtUnterliegen >= 0
// && bkGesamtUnterliegen <= bkStreitwert)
// b = true;
// else {
// FehlerDialog.zeigeFehler(FehlerDialog.FEHLER_BLANKO,
// HinzufuegenDialog.FEHLER_BK_GESAMTSCHULDNERSCHAFT,
// new
// IllegalArgumentException(HinzufuegenDialog.FEHLER_BK_GESAMTSCHULDNERSCHAFT));
// }
// } catch (NumberFormatException e) {
// FehlerDialog.zeigeFehler(FehlerDialog.FEHLER_BLANKO,
// HinzufuegenDialog.FEHLER_ZAHLENFORMAT, e);
// // e.printStackTrace();
// }
// } else {
// // sonst: alles egal, keine Plausi zu prüfen
// b = true;
// }
// break;
// case 5:
// /*
// * Wurden einige Beklagte darüber hinaus verurteilt? Welche, zu wie viel?
// */
// /*
// * Prüfen, ob einer der Beteiligten ein höheres Unterliegen als der streitwert
// * hat (ggfs. zusammen mit der gesamtschuldnerischen Verurteilung)
// */
// b = true;
// for (int row = 0; row < sbtmBeklagte.getRowCount(); row++) {
// double d = ((Double) sbtmBeklagte.getValueAt(row, 3)).doubleValue();
// if (d > bkStreitwert)
// b = false;
// if (chBkGesamtschuldner.isSelected()
// // && Arrays.asList(bkGesamtIndices).contains(row)
// && contains(bkGesamtIndices, row) && (d + bkGesamtUnterliegen) >
// bkStreitwert)
// b = false;
// }
// if (!b)
// FehlerDialog.zeigeFehler(FehlerDialog.FEHLER_BLANKO,
// HinzufuegenDialog.FEHLER_UNTERLIEGEN_GT_STREITWERT,
// new
// IllegalArgumentException(HinzufuegenDialog.FEHLER_UNTERLIEGEN_GT_STREITWERT));
// else {
// /*
// * Für die Rückgabe muss anWiderklageBeteiligt erst einmal auf false gesetzt
// * werden, wo es nicht gesetzt wurde
// */
// sbtmBeklagte.setEachAnwiderklagebeteiligt(false, false);
// // Dann Werte auslesen
// objectsBeklagte = sbtmBeklagte.getAllValues();
// if (isMitWiderklage()) {
// // Wenn Widerklage
// if (getVerfahrensDatenContainer() == null) {
// // (und sofern nicht xjustiz-import) Panel 6 bauen
// liBkWk.setListData(beteiligteBeklagte);
// // spiDwbkAnz.setText(Integer.toString(dwbkZahl));
// ((SpinnerNumberModel) spiDwbkAnz.getModel()).setValue(new Integer(dwbkZahl));
// } else {
// // Wenn xjustiz-import: nix, Panel 6 schon angelegt
// }
// } else {
// // wenn keine Widerklage vorhanden => hier fertig!
// }
// }
// break;
// case 6:
// // Welcher Beklagte klagt? Wie viele Drittwiderbeklagten?
// try {
// bkWiderklaegerIndex = liBkWk.getSelectedIndex();
// dwbkZahl = ((SpinnerNumberModel)
// spiDwbkAnz.getModel()).getNumber().intValue();
// // Integer.parseInt(spiDwbkAnz.getText());
// setKlWk(chbKlWk.isSelected());
// /*
// * Mindestens ein Bk muss ausgewählt sein, dwbkZahl darf nicht negativ sein
// und
// * wenn schon mitWiderklage, dann muss der Kläger widerbeklagt sein oder
// * wenigstens ein Dwbk vorhanden sein
// */
// if (bkWiderklaegerIndex > -1 && dwbkZahl > -1 && (isKlWk() || dwbkZahl > 0))
// {
// /*
// * Beim markierten Beklagten die vierte Spalte (index 3,
// anWiderklageBeteiligt)
// * auf true setzen. Hier gehe ich direkt über die Werte und nicht den Umweg
// über
// * ein sbtm, weil an dieser Stelle keine Tabelle angezeigt wird.
// */
// objectsBeklagte[bkWiderklaegerIndex][3] = Boolean.TRUE;
// /*
// * Schreiben, ob Wk auch gegen den Kläger geht. Auch hier direkt über die
// Werte,
// * weil eine Tabelle dazu gerade nicht angezeigt wird.
// */
// objectsKlaeger[0][3] = new Boolean(isKlWk());
// objectsDwbk = new Object[dwbkZahl][4];
// sbtmDrittwiderbeklagte = new ScalableBeteiligtenTableModel(objectsDwbk,
// Beteiligter.DRITTWIDERBEKLAGTE);
// // Drittwiderbeklagte sind immer an der Wk beteiligt
// sbtmDrittwiderbeklagte.setEachAnwiderklagebeteiligt(true, false);
// tbDwbkGenus.setModel(sbtmDrittwiderbeklagte);
// b = true;
// } else {
// FehlerDialog.zeigeFehler(FehlerDialog.FEHLER_BLANKO,
// HinzufuegenDialog.FEHLER_NIX_WIDERKLAEGER_WIDERBEKLAGTER,
// new
// IllegalArgumentException(HinzufuegenDialog.FEHLER_NIX_WIDERKLAEGER_WIDERBEKLAGTER));
// b = false;
// }
// } catch (NumberFormatException e) {
// FehlerDialog.zeigeFehler(FehlerDialog.FEHLER_BLANKO,
// HinzufuegenDialog.FEHLER_ZAHLFORMAT_DWBK, e);
// // e.printStackTrace();
// b = false;
// }
// break;
//case 7:
///* Bezeichnung der Drittwiderbeklagten (nix zu prüfen) 
// * 
// * Denkt auch daran, dass man Dwbk auch weglassen kann
// */
//if (dwbkZahl > 0) {
//	objectsDwbk = sbtmDrittwiderbeklagte.getAllValues();
//	sbtmDrittwiderbeklagte = new ScalableBeteiligtenTableModel(objectsDwbk, Beteiligter.DRITTWIDERBEKLAGTE,
//			getWkStreitwert());
//	tabDwbkSw.setModel(sbtmDrittwiderbeklagte);
//} else {
//	// schon gut
//}
//b = true;
///*
// * Für Panel 8 werden dann wieder die Klägerdaten gebraucht: angezeigt werden
// * sollen sie immer. Der ChangeListener der checkbox sorgt dafür, dass die
// * Kläger nur bearbeitet werden können, wenn der Kläger was mit der Wk am Hut
// * hat.
// */
//if (isKlWk()) {
//	sbtmKlaeger = new ScalableBeteiligtenTableModel(objectsKlaeger, Beteiligter.KLAEGER, getWkStreitwert());
//} else {
//	sbtmKlaeger = new ScalableBeteiligtenTableModel(objectsKlaeger, Beteiligter.KLAEGER, 0);
//}
//tabKlSw.setModel(sbtmKlaeger);
///*
// * Einmal den ChangeListener aufmerksam machen, der schreibt auch den
// * wkStreitwert ins Textfeld.
// */
//chbWkEinheitlich.setSelected(false);
//chbWkEinheitlich.setSelected(true);
//break;
//case 8:
// /*
// * Wurden Widerbeklagte in derselben Höhe verklagt? Welche (jeweils): Hier
// * achtet die Prüfung immer darauf, dass es einen Kläger geben kann, der nicht
// * widerbeklagt ist bzw. dass es möglicherweise keine Dwbk gibt.
// */
// boolean wkEinheitlich = chbWkEinheitlich.isSelected();
// try {
// if (wkEinheitlich) {
// /* wenn einheitlich */
// setWkStreitwert(doubleAusFormatter(ftfWkSw, lblWkSw.getText()));
// // Double.parseDouble(ftfWkSw.getText()));
// if (isKlWk()) {
// // Wenn nicht bei Wk dabei, dann auch nichts setzen
// sbtmKlaeger.setEachStreitwert(getWkStreitwert(), true);
// objectsKlaeger = sbtmKlaeger.getAllValues();
// }
// if (dwbkZahl > 0) {
// // Man kann Dwbk auch weglassen
// sbtmDrittwiderbeklagte.setEachStreitwert(getWkStreitwert(), true);
// objectsDwbk = sbtmDrittwiderbeklagte.getAllValues();
// }
// } else {
// /* wenn nicht einheitlich */
// double klWkStreitwert = 0;
// double dwbkStreitwert = 0;
// if (isKlWk()) {
// // Wenn nicht bei Wk dabei, dann auch nichts lesen
// objectsKlaeger = sbtmKlaeger.getAllValues();
// klWkStreitwert = sbtmKlaeger.getHoechstenStreitwert();
// }
// if (dwbkZahl > 0) {
// // Man kann Dwbk auch weglassen
// objectsDwbk = sbtmDrittwiderbeklagte.getAllValues();
// dwbkStreitwert = sbtmDrittwiderbeklagte.getHoechstenStreitwert();
// }
// // Höheren Streitwert von beiden speichern
// setWkStreitwert((klWkStreitwert > dwbkStreitwert) ? klWkStreitwert :
// dwbkStreitwert);
// ftfWkSw.setText(Double.toString(getWkStreitwert()));
// }
// if (getWkStreitwert() <= 0) {
// FehlerDialog.zeigeFehler(FehlerDialog.FEHLER_BLANKO,
// HinzufuegenDialog.FEHLER_WK_STREITWERT,
// new IllegalArgumentException(HinzufuegenDialog.FEHLER_WK_STREITWERT));
// b = false;
// break;
// } else {
// b = true;
// }
// /*
// * Immer eine Liste aus Kl (auch wenn nicht wbk) und Dwbk bauen, damit die
// * Indexe beim Bauen einer BaumbachGesamtschuldnerschaft stimmen.
// Umständlicher
// * Weg mit einer for-each-Schleife, aber Arrays.asList() gibt eine
// * unveränderliche Liste zurück, sodass der erste Weg nicht geklappt hat...
// */
// ArrayList<Beteiligter> tempWiderbeklagte = new ArrayList<Beteiligter>();
// for (Beteiligter bb : sbtmKlaeger.getBeteiligterColumn())
// tempWiderbeklagte.add(bb);
// for (Beteiligter bb : sbtmDrittwiderbeklagte.getBeteiligterColumn())
// tempWiderbeklagte.add(bb);
// beteiligteWiderbeklagte = tempWiderbeklagte.toArray(new
// Beteiligter[tempWiderbeklagte.size()]);
// liWkGesamtschuldner.setListData(beteiligteWiderbeklagte);
// ftfWkUnterliegen.setText(Double.toString(getWkGesamtUnterliegen()));
// if (isKlWk()) {
// sbtmKlaeger = new ScalableBeteiligtenTableModel(objectsKlaeger,
// Beteiligter.KLAEGER,
// getWkStreitwert(), wkUnterliegen);
// tbKlDarueber.setModel(sbtmKlaeger);
// tbKlDarueber.setEnabled(true);
// } else {
// /*
// * Wenn Kl nicht wbk, dann darf man für ihn auch nichts eingeben
// */
// tbKlDarueber.setEnabled(false);
// }
// if (dwbkZahl > 0) {
// sbtmDrittwiderbeklagte = new ScalableBeteiligtenTableModel(objectsDwbk,
// Beteiligter.DRITTWIDERBEKLAGTE, getWkStreitwert(), wkUnterliegen);
// tbDwbkDarueber.setModel(sbtmDrittwiderbeklagte);
// tbDwbkDarueber.setEnabled(true);
// } else {
// tbDwbkDarueber.setEnabled(false);
// }
// chbWkGesamtschuld.setSelected(false);
// chbWkGesamtschuld.setSelected(true);
// chbWkGesamtschuld.setSelected(beteiligteWiderbeklagte.length > 1);
// } catch (NumberFormatException e) {
// b = false;
// FehlerDialog.zeigeFehler(FehlerDialog.FEHLER_BLANKO,
// HinzufuegenDialog.FEHLER_ZAHLENFORMAT_SW, e);
// // e.printStackTrace();
// }
// break;
//case 9:
// /*
// * Wurden die Widerbeklagten z.T. gesamtschuldnerisch verurteilt? Welche
// * Widerbeklagten, in welcher Höhe? Prüfen: Häkchen bei gesamtschuldnerischer
// * Verurteilung gesetzt? Wenn ja, dann muss das Unterliegen auch über 0 sein,
// * sonst ist die Eingabe kappes.
// */
// wkGesamt = chbWkGesamtschuld.isSelected();
// if (wkGesamt) {
// try {
// wkGesamtIndices = liWkGesamtschuldner.getSelectedIndices();
// setWkGesamtUnterliegen(doubleAusFormatter(ftfWkUnterliegen,
// lblWkUnterliegen.getText()));
// // Double.parseDouble(ftfWkUnterliegen.getText()));
// if (wkGesamtIndices != null && wkGesamtIndices.length > 1 &&
// getWkGesamtUnterliegen() > 0
// && getWkGesamtUnterliegen() <= getWkStreitwert())
// b = true;
// else {
// b = false;
// FehlerDialog.zeigeFehler(FehlerDialog.FEHLER_BLANKO,
// HinzufuegenDialog.FEHLER_WK_GESAMTSCHULDNERSCHAFT,
// new
// IllegalArgumentException(HinzufuegenDialog.FEHLER_WK_GESAMTSCHULDNERSCHAFT));
// }
// /*
// * Wenn der Kläger nicht widerbeklagt ist, darf er nicht ausgewählt werden
// */
// if (!isKlWk() && contains(wkGesamtIndices, 0)) {
// b = false;
// FehlerDialog.zeigeFehler(FehlerDialog.FEHLER_BLANKO,
// HinzufuegenDialog.FEHLER_KLAEGER_NIX_WBK,
// new IllegalArgumentException(HinzufuegenDialog.FEHLER_KLAEGER_NIX_WBK));
// }
// } catch (NumberFormatException e) {
// b = false;
// FehlerDialog.zeigeFehler(FehlerDialog.FEHLER_BLANKO,
// HinzufuegenDialog.FEHLER_ZAHLENFORMAT, e);
// // e.printStackTrace();
// }
// } else {
// /*
// * Wenn keine Gesamtschuldnerschaft angehakt, keine Plausis zu prüfen.
// */
// b = true;
// }
// /* Panel 10 ist schon unter case 8 gebaut worden. */
// break;
//case 10:
///*
// * Wurden einzelne Widerbeklagten darüber hinaus einzeln verurteilt? Welche, in
// * welcher Höhe?
// */
//
///*
// * Prüfen, ob einer der Beteiligten ein höheres Unterliegen als der streitwert
// * hat (ggfs. zusammen mit der gesamtschuldnerischen Verurteilung)
// */
//b = true;
///*
// * Wenn Kl nicht wbk, dann darf man für ihn auch nichts eingeben, sonst prüfen,
// * ob Unterliegen den Streitwert überschreitet
// */
//if (primitives.klWk)
//	for (int row = 0; row < sbtmKlaeger.getRowCount(); row++) {
//		double d = ((Double) sbtmKlaeger.getValueAt(row, 3)).doubleValue();
//		if (d > primitives.wkStreitwert)
//			b = false;
//		else if (chbWkGesamtschuld.isSelected() && contains(primitives.wkGesamtIndices, row)
//				&& (d + primitives.wkGesamtUnterliegen) > primitives.wkStreitwert)
//			b = false;
//	}
///* Dran denken, dass man Dwbk auch weglassen kann */
//if (primitives.dwbkZahl > 0)
//	for (int row = 0; row < sbtmDrittwiderbeklagte.getRowCount(); row++) {
//		int absRow = row + primitives.klaegerZahl;
//		double d = ((Double) sbtmDrittwiderbeklagte.getValueAt(row, 3)).doubleValue();
//		if (d > primitives.wkStreitwert)
//			b = false;
//		else if (chbWkGesamtschuld.isSelected() && contains(primitives.wkGesamtIndices, absRow)
//				&& (d + primitives.wkGesamtUnterliegen) > primitives.wkStreitwert)
//			b = false;
//	}
//if (!b)
//	FehlerDialog.zeigeFehler(FehlerDialog.FEHLER_BLANKO, HinzufuegenDialog.FEHLER_UNTERLIEGEN_GT_STREITWERT,
//			new IllegalArgumentException(HinzufuegenDialog.FEHLER_UNTERLIEGEN_GT_STREITWERT));
//// Wenn Widerklage erhoben: hier fertig!
//break;
/*
 * Streitwert der Beklagten (für alle einheitlich?) Prüfen, ob einheitlich oder
 * nicht
 */
// boolean primitives.bkSwEinheitlich = chbBkSwEinheitlich.isSelected();
// sbtm = (ScalableBeteiligtenTableModel) tabBeklagtenSw.getModel();
// Object[][] objects;
// if (primitives.bkSwEinheitlich) {
// /*
// * wenn einheitlich: speichern und auf alle Bk übernehmen und alle Werte
// ziehen
// * --> für Panel 5
// */
// sbtm.setEachStreitwert(primitives.bkStreitwert, true);
// sbtm.fireTableRowsUpdated(0, sbtm.getRowCount() - 1);
// objects = sbtm.getAllValues();
// } else {
// /*
// * sonst: getValues und höchsten Streitwert für schlechte Tage zur Seite legen
// * --> für den zurückzugebenden KostenTenorContainer
// */
// objects = sbtm.getAllValues();
// ftfBkStreitwert.setText(Double.toString(primitives.bkStreitwert));
// }
//sbtm = (ScalableBeteiligtenTableModel) tbBkUnt.getModel();
//sbtm.getRowCount()
// sbtm = (ScalableBeteiligtenTableModel) tabBeklagtenSw.getModel();
// objectsBeklagte = sbtm.getAllValues();
/*
 * Hier gehe ich direkt über die Werte und nicht den Umweg über ein sbtm, weil
 * an dieser Stelle keine Tabelle angezeigt wird.
 */
// objectsBeklagte[primitives.bkWiderklaegerIndex][3] = Boolean.TRUE;
/*
 * Auch hier direkt über die Werte, weil eine Tabelle dazu gerade nicht
 * angezeigt wird. ((ScalableBeteiligtenTableModel)
 * tabKlaegerGenus.getModel()).setValueAt(new Boolean(primitives.klWk), 0, 3);
 */
// ((Double) btmBk.getValueAt(row, 3)).doubleValue();
//ScalableBeteiligtenTableModel sbtm;
//ScalableBeteiligtenTableModel sbtmDrittwiderbeklagte;
// eigentliches Auslesen der Felder
/*
 * , die in Panel 2 hierhin geschrieben wurden
 */
//primitives.wkSwEinheitlich = chbWkEinheitlich.isSelected();
//if (primitives.wkSwEinheitlich) {
///* wenn einheitlich */
//if (primitives.klWk) {
//	// Wenn nicht bei Wk dabei, dann auch nichts setzen
//	((ScalableBeteiligtenTableModel) tabKlSw.getModel()).setEachStreitwert(primitives.wkStreitwert,
//			true);
//}
//if (primitives.dwbkZahl > 0) {
//	// Man kann Dwbk auch weglassen
//	((ScalableBeteiligtenTableModel) tabDwbkSw.getModel()).setEachStreitwert(primitives.wkStreitwert,
//			true);
//}
//} else {
///* wenn nicht einheitlich */
//double klWkStreitwert = 0;
//double dwbkStreitwert = 0;
//if (primitives.klWk) {
//	// Wenn nicht bei Wk dabei, dann auch nichts lesen
//	klWkStreitwert = ((ScalableBeteiligtenTableModel) tabKlSw.getModel()).getHoechstenStreitwert();
//}
//if (primitives.dwbkZahl > 0) {
//	// Man kann Dwbk auch weglassen
//	dwbkStreitwert = ((ScalableBeteiligtenTableModel) tabDwbkSw.getModel()).getHoechstenStreitwert();
//}
//// Höheren Streitwert von beiden speichern
//if (klWkStreitwert > dwbkStreitwert)
//	primitives.wkStreitwert = klWkStreitwert;
//else
//	primitives.wkStreitwert = dwbkStreitwert;
//ftfWkSw.setText(Double.toString(primitives.wkStreitwert));
//}
//tbKlDarueber.setModel(new ScalableBeteiligtenTableModel( ((ScalableBeteiligtenTableModel) tabKlSw.getModel()).getAllValues(), Beteiligter.KLAEGER, primitives.wkStreitwert, primitives.wkUnterliegen));
//tbDwbkDarueber.setModel(new ScalableBeteiligtenTableModel( ((ScalableBeteiligtenTableModel) tabDwbkSw.getModel()).getAllValues(),Beteiligter.DRITTWIDERBEKLAGTE, primitives.wkStreitwert, primitives.wkUnterliegen));
/*
 * Die Methode setzt Daten aus dem {@link VerfahrensDatenContainer}, genauer dem
 * {@link UnterContainerKlasse} {@link Streitgenossen}
 * 
 * Zunächst Klägerdaten auslesen (werden an der Oberfläche dann erst mal nicht
 * gebraucht), Beklagtendaten auslesen und für Panel 3 setzen.
 * 
 * Wenn es soweit ist: Drittwiderbeklagtendaten auslesen, Klägerdaten und
 * Drittwiderbeklagtendaten für Panel 8 setzen.
 * 
 * 
 * @param streitgenossen eine {@link UnterContainerKlasse} {@link
 * Streitgenossen}
 * 
 * @param schonWiderklage Soll ich schon zu den speziellen Daten für die
 * Widerklage springen? <code>true</code>: Es ist soweit, Klägerdaten und
 * Drittwiderbeklagtendaten für Panel 8 zu setzen. Spricht was dagegen, das
 * direkt zu tun, wenn Widerklage aktiviert ist (allenfalls, dass man noch nicht
 * weiß, ob der Kläger auch Widerbeklagter ist)?
 * 
 * public void setzeWerte(Streitgenossen streitgenossen, boolean
 * schonWiderklage) { if (!schonWiderklage) { objectsKlaeger =
 * ScalableBeteiligtenTableModel.konvertiere(streitgenossen.klaeger);
 * objectsBeklagte =
 * ScalableBeteiligtenTableModel.konvertiere(streitgenossen.beklagte);
 * ftfBkStreitwert.setText(Double.toString(primitives.bkStreitwert));
 * sbtmBeklagte = new ScalableBeteiligtenTableModel(objectsBeklagte,
 * Beteiligter.BEKLAGTE, primitives.bkStreitwert);
 * tabBeklagtenSw.setModel(sbtmBeklagte); chbBkSwEinheitlich.setSelected(false);
 * chbBkSwEinheitlich.setSelected(true); } else {
 * ftfWkSw.setText(Double.toString(primitives.wkStreitwert)); if
 * (primitives.klWk) { sbtmKlaeger = new
 * ScalableBeteiligtenTableModel(objectsKlaeger, Beteiligter.KLAEGER,
 * primitives.wkStreitwert); } else { sbtmKlaeger = new
 * ScalableBeteiligtenTableModel(objectsKlaeger, Beteiligter.KLAEGER, 0); }
 * tabKlSw.setModel(sbtmKlaeger); if
 * (VerfahrensDatenContainerUtil.gibtsDrittwiderbeklagte(streitgenossen))
 * objectsDwbk =
 * ScalableBeteiligtenTableModel.konvertiere(streitgenossen.drittwiderbeklagte);
 * else objectsDwbk = new Object[0][4]; sbtmDrittwiderbeklagte = new
 * ScalableBeteiligtenTableModel(objectsDwbk, Beteiligter.DRITTWIDERBEKLAGTE,
 * primitives.wkStreitwert); tabDwbkSw.setModel(sbtmDrittwiderbeklagte);
 * chbWkEinheitlich.setSelected(false); chbWkEinheitlich.setSelected(true); } }
 */
/*
 * tabKlaegerGenus.setModel(new ScalableBeteiligtenTableModel( new
 * Object[primitives.klaegerZahl][MAX_SPALTEN], Beteiligter.KLAEGER));
 * tabBeklagtenGenus.setModel(new ScalableBeteiligtenTableModel( new
 * Object[primitives.beklagtenZahl][MAX_SPALTEN], Beteiligter.BEKLAGTE));
 */
/*
 * Object[][] objectsBeklagte = ((ScalableBeteiligtenTableModel)
 * tabBeklagtenGenus.getModel()).getAllValues();
 * ftfBkStreitwert.setText(Double.toString(primitives.bkStreitwert));
 * tabBeklagtenSw.setModel( new ScalableBeteiligtenTableModel(objectsBeklagte,
 * Beteiligter.BEKLAGTE, primitives.bkStreitwert));
 */
/*
 * liBkGesamtschuldner.setListData(sbtm.getBeteiligterColumn());
 * ftfBkGesamtUnterliegen.setText(Double.toString(primitives.bkGesamtUnterliegen
 * ));
 */
// tbBkUnt.setModel(new ScalableBeteiligtenTableModel(objects,
// Beteiligter.BEKLAGTE, primitives.bkStreitwert,
// primitives.bkUnterliegen));
// //
// // Dann Werte auslesen
// //objectsBeklagte = sbtm.getAllValues();
// if (primitives.mitWiderklage) {
// // wenn Widerklage
// if (primitives.ohneXjustiz) {
// // (und sofern nicht xjustiz-import) Panel 6 bauen
// liBkWk.setListData(sbtm.getBeteiligterColumn());
// primitives.dwbkZahl = ((SpinnerNumberModel)
// spiDwbkAnz.getModel()).getNumber().intValue();
// } else {
// // Wenn xjustiz-import: nix, Panel 6 schon angelegt
// }
// } else {
// // wenn keine Widerklage vorhanden => hier fertig!
// }
// } else {
/*
 * sbtmDrittwiderbeklagte = new ScalableBeteiligtenTableModel(new
 * Object[primitives.dwbkZahl][4], Beteiligter.DRITTWIDERBEKLAGTE);
 * sbtmDrittwiderbeklagte.setEachAnwiderklagebeteiligt(true, false);
 * tbDwbkGenus.setModel(sbtmDrittwiderbeklagte);
 */
/*
 * sbtm = (ScalableBeteiligtenTableModel) tabKlSw.getModel(); if
 * (primitives.klWk) { sbtm.setEachStreitwert(primitives.wkStreitwert, true); }
 * else { sbtm.setEachStreitwert(0.0, true); } tabKlSw.setModel(sbtm);
 */
/*
 * if (primitives.mitWiderklage) { // schon einmal auf Panel 8 setzen Object[][]
 * objectsKlaeger = ((ScalableBeteiligtenTableModel)
 * tabKlaegerGenus.getModel()).getAllValues(); tabKlSw.setModel(new
 * ScalableBeteiligtenTableModel(objectsKlaeger, Beteiligter.KLAEGER,
 * primitives.wkStreitwert)); } // Kläger werden für eine der Widerklage-Tabs
 * gebraucht! später lesen? // jetzt schon setzen?
 */
// objects = ((ScalableBeteiligtenTableModel)
// tbDwbkGenus.getModel()).getAllValues();
// tabDwbkSw.setModel(new ScalableBeteiligtenTableModel(objects,
// Beteiligter.DRITTWIDERBEKLAGTE,
// primitives.wkStreitwert));
// } else {
// schon gut
// }
// .addAll(Arrays.asList(((ScalableBeteiligtenTableModel)
// tabKlSw.getModel()).getBeteiligterColumn()));
// Arrays.asList(((ScalableBeteiligtenTableModel)
// tabDwbkSw.getModel()).getBeteiligterColumn())
// for (int row = 0; row < primitives.beklagtenZahl; row++)
// rueckgabewert.beklagte.get(row)
// .setGenusNumerus(((Beteiligter) tabBeklagtenGenus.getValueAt(row,
// 1)).getGenusNumerus());
// for (int row = 0; row < primitives.beklagtenZahl; row++)
// rueckgabewert.beklagte.get(row).setStreitwert((Double)
// tabBeklagtenSw.getValueAt(row, 2));
// ((ScalableBeteiligtenTableModel) tabBeklagtenSw.getModel())
// .getHoechstenStreitwert();
// primitives.bkStreitwert = sbtm.getHoechstenStreitwert();
//for (int row = 0; row < primitives.beklagtenZahl; row++)
//rueckgabewert.beklagte.get(row).setUnterliegen((Double) tbBkUnt.getValueAt(row, 3));
//for (int row = 0; row < primitives.dwbkZahl; row++)
//rueckgabewert.drittwiderbeklagte.get(row)
// .setGenusNumerus(((Beteiligter) tbDwbkGenus.getValueAt(row,
// 1)).getGenusNumerus());
//rueckgabewert.klaeger.get(0).setStreitwert(klWkStreitwert);
//for (int row = 0; row < primitives.dwbkZahl; row++)
// rueckgabewert.drittwiderbeklagte.get(row).setStreitwert((Double)
// tabDwbkSw.getValueAt(row, 2));
//((ScalableBeteiligtenTableModel) tabDwbkSw.getModel()).getHoechstenStreitwert(); 
//((ScalableBeteiligtenTableModel) tabKlSw.getModel()).getHoechstenStreitwert();
//rueckgabewert.klaeger.get(0).setUnterliegen(((Double) tbKlDarueber.getValueAt(0, 3)).doubleValue());
//rueckgabewert.drittwiderbeklagte.get(row)
//.setUnterliegen(((Double) tbDwbkDarueber.getValueAt(row, 3)).doubleValue());

/*
 * Die Methode setzt Daten aus dem {@link VerfahrensDatenContainer}, genauer dem
 * {@link UnterContainerKlasse} {@link Streitgenossen}
 * 
 * Zunächst Klägerdaten auslesen (werden an der Oberfläche dann erst mal nicht
 * gebraucht), Beklagtendaten auslesen und für Panel 3 setzen.
 * 
 * Wenn es soweit ist: Drittwiderbeklagtendaten auslesen, Klägerdaten und
 * Drittwiderbeklagtendaten für Panel 8 setzen.
 * 
 * 
 * @param streitgenossen eine {@link UnterContainerKlasse} {@link
 * Streitgenossen}
 * 
 * @param schonWiderklage Soll ich schon zu den speziellen Daten für die
 * Widerklage springen? <code>true</code>: Es ist soweit, Klägerdaten und
 * Drittwiderbeklagtendaten für Panel 8 zu setzen. Spricht was dagegen, das
 * direkt zu tun, wenn Widerklage aktiviert ist (allenfalls, dass man noch nicht
 * weiß, ob der Kläger auch Widerbeklagter ist)?
 * 
 * public void setzeWerte(Streitgenossen streitgenossen, boolean
 * schonWiderklage) { if (!schonWiderklage) { objectsKlaeger =
 * ScalableBeteiligtenTableModel.konvertiere(streitgenossen.klaeger);
 * objectsBeklagte =
 * ScalableBeteiligtenTableModel.konvertiere(streitgenossen.beklagte);
 * ftfBkStreitwert.setText(Double.toString(primitives.bkStreitwert));
 * sbtmBeklagte = new ScalableBeteiligtenTableModel(objectsBeklagte,
 * Beteiligter.BEKLAGTE, primitives.bkStreitwert);
 * tabBeklagtenSw.setModel(sbtmBeklagte); chbBkSwEinheitlich.setSelected(false);
 * chbBkSwEinheitlich.setSelected(true); } else {
 * ftfWkSw.setText(Double.toString(primitives.wkStreitwert)); if
 * (primitives.klWk) { sbtmKlaeger = new
 * ScalableBeteiligtenTableModel(objectsKlaeger, Beteiligter.KLAEGER,
 * primitives.wkStreitwert); } else { sbtmKlaeger = new
 * ScalableBeteiligtenTableModel(objectsKlaeger, Beteiligter.KLAEGER, 0); }
 * tabKlSw.setModel(sbtmKlaeger); if
 * (VerfahrensDatenContainerUtil.gibtsDrittwiderbeklagte(streitgenossen))
 * objectsDwbk =
 * ScalableBeteiligtenTableModel.konvertiere(streitgenossen.drittwiderbeklagte);
 * else objectsDwbk = new Object[0][4]; sbtmDrittwiderbeklagte = new
 * ScalableBeteiligtenTableModel(objectsDwbk, Beteiligter.DRITTWIDERBEKLAGTE,
 * primitives.wkStreitwert); tabDwbkSw.setModel(sbtmDrittwiderbeklagte);
 * chbWkEinheitlich.setSelected(false); chbWkEinheitlich.setSelected(true); } }
 */
//for (int row = 0; row < primitives.klaegerZahl; row++)
//rueckgabewert.klaeger.get(row)
// .setGenusNumerus(((Beteiligter) tabKlaegerGenus.getValueAt(row,
// 1)).getGenusNumerus());
// throw new IllegalArgumentException("Datei enthält keine Beklagten!");
// throw new IllegalArgumentException("Datei enthält Widerkläger ohne
// Widerbeklagte!");
// throw new IllegalArgumentException("Datei enthält Widerbeklagte ohne
// Widerkläger!");