/*
 * StreitgenossenPruefer.java
 * eu.gronos.kostenrechner.controller.baumbach (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.baumbach;

import java.awt.Component;

import javax.swing.JComponent;
import javax.swing.JFormattedTextField;
import javax.swing.JTable;

import eu.gronos.kostenrechner.controller.NumberExtractor;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.interfaces.ParsendUndBauend;
import eu.gronos.kostenrechner.interfaces.UnterContainerKlasse;
import eu.gronos.kostenrechner.logic.TenorTexter;
import eu.gronos.kostenrechner.model.baumbach.BaumbachBeteiligtenListe;
import eu.gronos.kostenrechner.model.baumbach.BaumbachBeteiligter;
import eu.gronos.kostenrechner.model.baumbach.BaumbachGesamtschuldnerschaft;
import eu.gronos.kostenrechner.model.baumbach.BeteiligtenTableModel;
import eu.gronos.kostenrechner.model.baumbach.Streitgenossen;
import eu.gronos.kostenrechner.model.baumbach.Widerklage;
import eu.gronos.kostenrechner.model.gebuehren.Teilklageruecknahme;
import eu.gronos.kostenrechner.model.tenordaten.VerfahrensDatenContainer;

/**
 * Ein {@link ParsendUndBauend} für die {@link UnterContainerKlasse}
 * {@link Streitgenossen}
 *
 * @author Peter Schuster (setrok)
 * @date 31.08.2019
 *
 */
public class StreitgenossenPruefer implements ParsendUndBauend<Streitgenossen> {

	private Streitgenossen rueckgabewert;
	private Widerklage mitWiderklage;
	private JTable tabKlaeger;
	private JTable tabBeklagte;
	private JTable tabDrittwiderbeklagte;
	/**
	 * wenn <code>true</code>, dann prüft der {@link ParsendUndBauend} noch weitere
	 * Plausibilitäten
	 */
	private final boolean streng;

	/**
	 * Konstruktor, der nur die Standard-Plausibilitäten prüft.
	 * 
	 */
	public StreitgenossenPruefer() {
		this(false);
	}

	/**
	 * Konstruktor.
	 * 
	 * @param streng boolean wenn <code>true</code>, dann prüft der
	 *               {@link ParsendUndBauend} noch weitere Plausibilitäten
	 */
	public StreitgenossenPruefer(boolean streng) {
		super();
		this.streng = streng;
	}

	/**
	 * @param components die benötigten {@link JComponent}s, namentlich die drei
	 *                   {@link JTable}s
	 * @return diesen {@link StreitgenossenPruefer}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#leseComponents(Component[])
	 */
	@Override
	public ParsendUndBauend<Streitgenossen> leseComponents(Component... components) {
		if (components != null) {
			for (Component comp : components) {
				JComponent component = (JComponent) comp;
				if ("31011".equals(component.getName())) {
					tabKlaeger = (JTable) component;
				} else if ("32011".equals(component.getName())) {
					tabBeklagte = (JTable) component;
				} else if ("33111".equals(component.getName())) {
					tabDrittwiderbeklagte = (JTable) component;
				} else
					System.out.println("Hä? " + component.getName());
			}
		}
		return this;
	}

	/**
	 * Die Methode überprüft den {@link #getRueckgabewert()}, nachdem er Felder
	 * ausgelesen hat.
	 * 
	 * @return <code>true</code>, wenn die Plausibilitäten zufrieden sind
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#parseEingabe()
	 */
	@Override
	public boolean parseEingabe() {
		try {
			rueckgabewert = leseFelder();
			// nur strenge Variante ruft parseEingabe(rueckgabewert) auf
			if (streng)
				return parseEingabe(rueckgabewert);
		} catch (NumberFormatException | NullPointerException ne) {
			FehlerHelper.zeigeFehler(ne.getLocalizedMessage(), ne);
			return false;
		}
		return true;
	}

	/**
	 * Die Methode überprüft den {@link #getRueckgabewert()}, ohne Felder
	 * auszulesen, indem er ihr übergeben wird.
	 * 
	 * @param rueckgabewert die zu prüfende {@link UnterContainerKlasse} vom Typ
	 *                      {@link Teilklageruecknahme}
	 * @return <code>true</code>, wenn die Plausibilitäten zufrieden sind
	 * @throws NumberFormatException    wenn der {@link NumberExtractor} bei einem
	 *                                  {@link JFormattedTextField} eine solche
	 *                                  Exception wirft.
	 * @throws NullPointerException     nur zur Sicherheit...
	 * @throws IllegalArgumentException wird geworfen, wenn eine der folgenden
	 *                                  Vorgaben nicht eingehalten wurde. a) Kläger
	 *                                  und Beklagte vorhanden; b) nicht mehr als 1
	 *                                  Kläger und 1 Widerkläger; c) die Klägerliste
	 *                                  darf nur Kläger, die Beklagtenliste nur
	 *                                  Beklagte, die Drittwiderbeklagtenliste nur
	 *                                  Drittwiderbeklagte enthalten; d) wenn es
	 *                                  einen Widerkläger gibt, muss es auch
	 *                                  Widerbeklagte geben; wenn es Widerbeklagte
	 *                                  gibt, muss es auch einen Widerkläger geben.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#parseEingabe(java.lang.Object)
	 */
	@Override
	public boolean parseEingabe(Streitgenossen rueckgabewert) throws NumberFormatException, NullPointerException {
		// nur strenge Variante ruft parseEingabe(rueckgabewert) auf
		this.rueckgabewert = rueckgabewert;

		// TODO auch BeteiligtenPruefer und GesamtschuldPruefer aufrufen

		/* Kläger hinzufügen (wenn vorhanden), sonst Fehler */
		if (rueckgabewert.klaeger == null || rueckgabewert.klaeger.isEmpty())
			throw new IllegalArgumentException(TenorTexter.FEHLER_NIX_PARTEIEN);
		/* Beklagte hinzufügen, wenn vorhanden, sonst Fehler */
		if (rueckgabewert.beklagte == null || rueckgabewert.beklagte.isEmpty())
			throw new IllegalArgumentException(TenorTexter.FEHLER_NIX_PARTEIEN);

		/* Nur einen Kläger und nur einen Widerkläger! */
		if (rueckgabewert.klaeger.zaehleKlaeger() > 1 || rueckgabewert.beklagte.zaehleAnWiderklageBeteiligte() > 1)
			throw new IllegalArgumentException(TenorTexter.FEHLER_NUR_EINER);

		/*
		 * Wenn die Listen Beteiligte von einem unpassenden Typ enthalten, kann die
		 * Reihenfolge nicht eingehalten werden.
		 */
		if (rueckgabewert.klaeger.zaehleBeklagte() > 0 || rueckgabewert.klaeger.zaehleDrittwiderbeklagte() > 0
				|| rueckgabewert.beklagte.zaehleKlaeger() > 0 || rueckgabewert.beklagte.zaehleDrittwiderbeklagte() > 0
				|| rueckgabewert.drittwiderbeklagte.zaehleKlaeger() > 0
				|| rueckgabewert.drittwiderbeklagte.zaehleBeklagte() > 0)
			throw new IllegalArgumentException(TenorTexter.FEHLER_REIHENFOLGE_BETEILIGTE);

		/*
		 * Wenn es einen Widerkläger gibt, muss es mindestens einen Widerbeklagten geben
		 */
		if (rueckgabewert.beklagte.zaehleAnWiderklageBeteiligte() > 0
				&& (rueckgabewert.klaeger.zaehleAnWiderklageBeteiligte()
						+ rueckgabewert.drittwiderbeklagte.zaehleAnWiderklageBeteiligte()) <= 0)
			throw new IllegalArgumentException(TenorTexter.FEHLER_NIX_WIDERBEKLAGTE);
		/* Wenn es Widerbeklagte gibt, muss es auch einen Widerkläger geben */
		if (rueckgabewert.beklagte.zaehleAnWiderklageBeteiligte() <= 0
				&& (rueckgabewert.klaeger.zaehleAnWiderklageBeteiligte()
						+ rueckgabewert.drittwiderbeklagte.zaehleAnWiderklageBeteiligte()) > 0)
			throw new IllegalArgumentException(TenorTexter.FEHLER_NIX_WIDERKLAEGER);

		/* jetzt noch die Einzelkomponenten prüfen */
		final BeteiligtenPruefer betPruefer = new BeteiligtenPruefer();
		final GesamtschuldPruefer gesPruefer = new GesamtschuldPruefer();
		for (BaumbachBeteiligtenListe liste : new BaumbachBeteiligtenListe[] { rueckgabewert.klaeger,
				rueckgabewert.beklagte, rueckgabewert.drittwiderbeklagte }) {
			for (BaumbachBeteiligter beteiligter : liste) {
				if (beteiligter instanceof BaumbachGesamtschuldnerschaft)
					gesPruefer.parseEingabe((BaumbachGesamtschuldnerschaft) beteiligter);
				else
					betPruefer.parseEingabe(beteiligter);
			}
		}

		return true;
	}

	/**
	 * Die Methode dient dazu, die zusammen gepackten Programmzustände in einem
	 * {@link VerfahrensDatenContainer} zurückzugeben
	 * 
	 * @return den Untercontainer von
	 *         {@link VerfahrensDatenContainer#streitgenossen}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#getRueckgabewert()
	 */
	@Override
	public Streitgenossen getRueckgabewert() {
		return rueckgabewert;
	}

	/**
	 * @param mitWiderklage d. {@link #mitWiderklage}, d. gesetzt werden soll als
	 *                      {@link Widerklage}.
	 * @return diesen {@link StreitgenossenPruefer}
	 */
	public ParsendUndBauend<Streitgenossen> setzeSonstigeWerte(Widerklage mitWiderklage) {
		this.mitWiderklage = mitWiderklage;
		return this;
	}

	/**
	 * Die Methode packt alle Programmzustände in einen
	 * {@link VerfahrensDatenContainer}
	 * 
	 * @return den Untercontainer von
	 *         {@link VerfahrensDatenContainer#streitgenossen}
	 */
	private Streitgenossen leseFelder() throws NumberFormatException, NullPointerException {
		rueckgabewert = new Streitgenossen();
		rueckgabewert.widerklage = mitWiderklage.isVorhanden();
		rueckgabewert.wirtschaftlichIdentisch = mitWiderklage.isWirtschaftlichIdentisch();
		BeteiligtenTableModel btmKlaeger = (BeteiligtenTableModel) tabKlaeger.getModel();
		rueckgabewert.klaeger = btmKlaeger.getAllValues();
		BeteiligtenTableModel btmBeklagte = (BeteiligtenTableModel) tabBeklagte.getModel();
		rueckgabewert.beklagte = btmBeklagte.getAllValues();
		if (mitWiderklage.isVorhanden()) {
			BeteiligtenTableModel btmDrittwiderbeklagte = (BeteiligtenTableModel) tabDrittwiderbeklagte.getModel();
			rueckgabewert.drittwiderbeklagte = btmDrittwiderbeklagte.getAllValues();
		} else
			rueckgabewert.drittwiderbeklagte = new BaumbachBeteiligtenListe();
		return rueckgabewert;
	}

}
// isWiderKlageAnzeigen();
// isWirtschaftlichIdentisch();
// Streitgenossen container = new Streitgenossen();
// container.widerklage = isWiderKlageAnzeigen();
// container.wirtschaftlichIdentisch = isWirtschaftlichIdentisch();
// container.klaeger = btmKlaeger.getAllValues();
// if (isWiderKlageAnzeigen())
// container.drittwiderbeklagte = btmDrittwiderbeklagte.getAllValues();
// else
// container.drittwiderbeklagte = new BaumbachBeteiligtenListe();
// container.beklagte = btmBeklagte.getAllValues();
// return container;
// baumbachBeteiligtenListe.addAll(streitgenossen.klaeger);
// else
// baumbachBeteiligtenListe.addAll(streitgenossen.beklagte);
//else
