/**
 * DatenXmlLadenSpeichern.java
 * eu.gronos.kostenrechner.util (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.files;

import java.io.File;
import java.io.StringWriter;

import javax.xml.bind.JAXB;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.PropertyException;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.model.baumbach.BaumbachBeteiligtenListe;
import eu.gronos.kostenrechner.model.baumbach.BaumbachGesamtschuldnerschaft;
import eu.gronos.kostenrechner.model.baumbach.Streitgenossen;
import eu.gronos.kostenrechner.model.tenordaten.VerfahrensDatenContainer;

/**
 * Diese Klasse verwaltet die {@link Marshaller}s, die zum Speichern der
 * XML-Datei (zu {@link VerfahrensDatenContainer}n) benötigt werden.
 * 
 * Dazu stellt sie die Methoden für Dateioperationen
 * {@link #speicherXml(VerfahrensDatenContainer, File)} und
 * {@link #ladeXml(File)} zur Verfügung. Zu Debug-Zweck gibt es außerdem noch
 * {@link #toXmlString(VerfahrensDatenContainer)}, der statt in eine Datei in
 * einen {@link StringWriter} schreibt und einen {@link String} zurückgibt.
 *
 * @author Peter Schuster (setrok)
 * @date 01.07.2018
 *
 */
public class BerechnungXmlDatei extends XmlDatei<VerfahrensDatenContainer> {

	private static final String FEHLER_BEIM_LADEN_DER_XML_DATEI = "Fehler beim Laden der XML-Datei.";
	public static final String DATEIENDUNG = "skktx";
	public static final String DATEITYP = "Schusters kleiner Kostentenor XML-Dateien";
	private static final String SCHEMA_LOCATION = "https://www.kostentenor.de schema-skktx.xsd";
	public static final KostenFileFilter DATEI_FILTER = new KostenFileFilter("berechnung",
			String.format("%s (%s)", DATEITYP, DATEIENDUNG), DATEIENDUNG);

	public BerechnungXmlDatei() {
		super();
	}

	/**
	 * Die Methode dient zum Umwandeln der Java Objekte (dargereicht als
	 * {@link VerfahrensDatenContainer}) in XML. In
	 * {@link VerfahrensDatenContainer#allgemein} wird das Speicherdatum auf die
	 * aktuelle Zeit gesetzt.
	 * 
	 * 
	 * @param container der {@link VerfahrensDatenContainer} mit allen zu
	 *                  speichernden Zuständen
	 * @param datei     ein {@link File}, in das zu speichern ist
	 * @throws JAXBException
	 */
	@Override
	public void speicherXml(VerfahrensDatenContainer container, File datei) throws JAXBException {
		// VerfahrensDatenContainerUtil.
		container.allgemein.speicherdatum = new XmlTransferHandler().speicherDatum();
		Marshaller marshaller = BerechnungXmlDatei.createMarshaller(true, getSchemaLocation());
		marshaller.marshal(container, datei);
	}

	/**
	 * Die Methode liest über {@link JAXB#unmarshal(File, Class)} aus der
	 * ausgewählten Datei einen {@link VerfahrensDatenContainer} aus.
	 * 
	 * @param datei ein {@link File}, das zu laden ist
	 * 
	 * @return ein {@link VerfahrensDatenContainer} mit allen gespeicherten
	 *         Zuständen
	 * @see JAXB#unmarshal(File, Class)
	 */
	public VerfahrensDatenContainer ladeXml(File datei) {
		VerfahrensDatenContainer container = null;
		try {
			container = unmarshalXml(datei);
			// System.out.println
			Kostenrechner.getLogger().info(String.format("Geladen: %s", datei.getAbsolutePath()));
		} catch (JAXBException je) {
			FehlerHelper.zeigeFehler(FEHLER_BEIM_LADEN_DER_XML_DATEI, je);
			container = null;
		}
		if (container != null)
			container.streitgenossen = fixGesamtschuldnerschaften(container.streitgenossen);
		return container;
	}

	/**
	 * Die Methode dient dazu, die Programmzustände ohne Datei in einen String zu
	 * konvertieren.
	 * 
	 * @param container der {@link VerfahrensDatenContainer} mit allen zu
	 *                  speichernden Zuständen
	 * @return einen String mit den XML-Daten oder einen Fehlertext, falls die
	 *         Konvertiertung nicht klappt
	 * @see XmlDatei#toXmlString(Object)
	 */
	@Override
	public String toXmlString(VerfahrensDatenContainer container) {
		StringWriter sw = new StringWriter();
		try {
			/* für den XML-String sollen keine Zeilenumbrüche erstellt werden */
			Marshaller marshaller = createMarshaller(false, getSchemaLocation());
			marshaller.marshal(container, sw);
		} catch (JAXBException e) {
			return "Programmzustände konnten nicht in XML konvertiert werden: " + e.getLocalizedMessage();
		}
		return sw.toString();
	}

	/**
	 * 
	 * @see eu.gronos.kostenrechner.controller.files.XmlDatei#getSchemaLocation()
	 */
	@Override
	public String getSchemaLocation() {
		return SCHEMA_LOCATION;
	}

	@Override
	public String getDateiendung() {
		return DATEIENDUNG;
	}

	/**
	 * Die Methode dient dazu, einen {@link Marshaller} zu erstellen, der dann zum
	 * Speichern als XML-Datei benötigt wird.
	 * 
	 * @param formattedOutput schaltet, ob der Marshaller Zeilenumbrüche und
	 *                        Einrückungen bauen soll
	 *                        {@link Marshaller#JAXB_FORMATTED_OUTPUT}
	 * @param schemaLocation  {@link #getSchemaLocation()}
	 * @return einen {@link Marshaller}
	 * @throws JAXBException
	 * @throws PropertyException
	 */
	private static Marshaller createMarshaller(boolean formattedOutput, String schemaLocation)
			throws JAXBException, PropertyException {
		JAXBContext context = JAXBContext.newInstance(VerfahrensDatenContainer.class);
		Marshaller marshaller = context.createMarshaller();
		marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, new Boolean(formattedOutput));
		marshaller.setProperty(Marshaller.JAXB_SCHEMA_LOCATION, schemaLocation);
		return marshaller;
	}

	/**
	 * Die Methode dient zum Umwandeln XML in einen {@link VerfahrensDatenContainer}
	 * 
	 * @param datei ein {@link File}, das zu laden ist
	 * @return ein {@link VerfahrensDatenContainer} mit allen gespeicherten
	 *         Zuständen
	 * @throws JAXBException
	 */
	private VerfahrensDatenContainer unmarshalXml(File datei) throws JAXBException {
		return JAXB.unmarshal(datei, VerfahrensDatenContainer.class);
	}

	/**
	 * Die Methode, die den Container ausliest, muss bei
	 * {@link BaumbachGesamtschuldnerschaft}en die richtige
	 * {@link BaumbachBeteiligtenListe} hinzufügen
	 * 
	 * @param streitgenossen ein Untercontainer {@link Streitgenossen}
	 * 
	 * @return den reparierten Untercontainer {@link Streitgenossen}
	 */
	private Streitgenossen fixGesamtschuldnerschaften(Streitgenossen streitgenossen) {
		BaumbachBeteiligtenListe beklagte = streitgenossen.beklagte;
		if (beklagte.enthaeltGesamtschuldner())
			for (int index = 0; index < beklagte.size(); index++) {
				if (beklagte.get(index) instanceof BaumbachGesamtschuldnerschaft)
					((BaumbachGesamtschuldnerschaft) beklagte.get(index)).setBaumbachBeteiligtenListe(beklagte);
			}
		BaumbachBeteiligtenListe widerbeklagte = new BaumbachBeteiligtenListe();
		widerbeklagte.addAll(streitgenossen.klaeger);
		widerbeklagte.addAll(streitgenossen.drittwiderbeklagte);
		if (widerbeklagte.enthaeltGesamtschuldner())
			for (int index = 0; index < widerbeklagte.size(); index++) {
				if (widerbeklagte.get(index) instanceof BaumbachGesamtschuldnerschaft)
					((BaumbachGesamtschuldnerschaft) widerbeklagte.get(index))
							.setBaumbachBeteiligtenListe(widerbeklagte);
			}
		return streitgenossen;
	}
}

//public static final String DATEIENDUNG_GANZ = "." + DATEIENDUNG;
// FileNameExtensionFilter filter = new FileNameExtensionFilter(