/*
 * KostenFileFilter.java
 * eu.gronos.kostenrechner.controller.files (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.files;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JFileChooser;
import javax.swing.filechooser.FileFilter;
import javax.swing.filechooser.FileNameExtensionFilter;

import eu.gronos.kostenrechner.model.beschriftungen.Beschriftung;
import eu.gronos.kostenrechner.view.KostenFileChooser;

/**
 * Von {@link FileFilter} abgeleitete Klasse für {@link KostenFileChooser}, die
 * außer der Funktionalität des {@link FileNameExtensionFilter}s noch die
 * Methode {@link #pruefeEndung(File)} übernommen hat, um gegebenfalls noch die
 * Endung zu ergänzen.
 *
 * @author Peter Schuster (setrok)
 * @date 08.02.2020
 *
 */
public class KostenFileFilter extends FileFilter {

	private final String description;
	private final List<String> extensions;
	private final List<String> lowerCaseExtensions;
	private final String preSelected;

	/**
	 * Konstruktor mit allen Feldern
	 * 
	 * @param preSelected benennt einen Namensvorschlag für Dateien dieser Art, z.B.
	 *                    "tenor" oder "berechnung"
	 * @param description {@link #getDescription()}
	 * @param extensions  {@link #getExtensions()}, aber als Aufzählung der
	 *                    Erweiterungen
	 */
	public KostenFileFilter(String preSelected, String description, String... extensions) {
		this.preSelected = preSelected;
		this.description = description;
		this.extensions = new ArrayList<String>();// Arrays.asList(extensions)
		this.lowerCaseExtensions = new ArrayList<String>();
		for (String extension : extensions) {
			if (extension == null || extension.length() < 1)
				throw new IllegalArgumentException("Extensions must not be null or empty!");
			this.extensions.add(extension);
			lowerCaseExtensions.add(extension.toLowerCase());
		}
	}

	/**
	 * Die Methode prüft das angegebene {@link File}, ob die Dateinamenserweiterung
	 * in der Liste ist oder es ein Verzeichnis ist.
	 * 
	 * @param datei das zu prüfende {@link File}
	 * @return <code>true</code>, wenn Verzeichnis oder brave Erweiterung, sonst
	 *         <code>false</code>.
	 * 
	 * @see javax.swing.filechooser.FileFilter#accept(java.io.File)
	 */
	@Override
	public boolean accept(File datei) {
		if (datei != null) {
			if (datei.isDirectory()) {
				return true;
			}
			if (acceptExtension(datei))
				return true;
		}
		return false;
	}

	/**
	 * Die Methode prüft das angegebene {@link File}, ob die Dateinamenserweiterung
	 * in der Liste ist (Verzeichnis gildet hier nicht)
	 * 
	 * @param datei das zu prüfende {@link File}
	 * @return <code>true</code>, wenn brave Erweiterung, sonst <code>false</code>.
	 */
	public boolean acceptExtension(File datei) {
		if (datei == null)
			return false;
		String dateiName = datei.getName();
		int i = dateiName.lastIndexOf('.');
		if (i > 0 && i < dateiName.length() - 1) {
			String dateiErweiterung = dateiName.substring(i + 1);
			for (String extension : lowerCaseExtensions)
				if (extension.equals(dateiErweiterung))
					return true;
		}
		return false;
	}

	/**
	 * Die Methode kümmert sich darum, dass der eingegebene Dateiname die Endung aus
	 * {@link KostenFileChooser#KostenFileChooser(FileFilter, Beschriftung)} bekommt.
	 * 
	 * @param datei die gewählte Datei
	 * 
	 * @return die Datei mit dem gewählten Dateinamen
	 */
	public File pruefeEndung(File datei) {
		if (!datei.getName().endsWith(ganzeEndung())) {
			datei = new File(datei.getAbsolutePath() + ganzeEndung());
		}
		return datei;
	}

	/**
	 * @return gibt {@link #preSelected} als {@link String} zurück, also einen
	 *         Namensvorschlag für Dateien dieser Art, z.B. "tenor" oder
	 *         "berechnung", die {@link KostenFileChooser} dann in einem
	 *         Speicher-Dialog {@link JFileChooser#setSelectedFile(File)} setzen
	 *         kann.
	 */
	public String getPreSelected() {
		return preSelected;
	}

	/**
	 * Eine Beschreibung für den Dateityp
	 * 
	 * @return gibt {@link #description} als {@link String} zurück.
	 * 
	 * @see javax.swing.filechooser.FileFilter#getDescription()
	 */
	@Override
	public String getDescription() {
		return description;
	}

	/**
	 * @return gibt {@link #extensions} als {@link List<String>} zurück.
	 */
	public List<String> getExtensions() {
		return extensions;
	}

	@Override
	public String toString() {
		return super.toString() + "[description=" + getDescription() + " extensions=" + extensions + "]";
	}

	/**
	 * Die Methode gibt die vollständige Dateiendung aus {@link #getExtensions()}
	 * zurück, d.h. mit Punkt "."
	 * 
	 * @return einen {@link String} "." + Dateiendung {@link #getExtensions()}
	 */
	public String ganzeEndung() {
		return "." + getExtensions().get(0);
	}

}

// kostenFileChooser.setSelectedFile(datei);
// public String getDescription() {
// Auto-generated method stub
// return null;
// }
/*
 * 
 * @return gibt {@link #lowerCaseExtensions} als {@link List<String>} zurück.
 */
// public List<String> getLowerCaseExtensions() {
// return lowerCaseExtensions;
// }