/*
 * RtfWerkzeugKasten.java
 * eu.gronos.kostenrechner.controller.files (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.files;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.ClipboardOwner;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.io.BufferedWriter;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.nio.file.Path;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Map;

import javax.swing.JComponent;
import javax.swing.JEditorPane;
import javax.swing.JFileChooser;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextPane;
import javax.swing.UIDefaults;
import javax.swing.UIManager;
import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.DefaultStyledDocument;
import javax.swing.text.Document;
import javax.swing.text.EditorKit;
import javax.swing.text.MutableAttributeSet;
import javax.swing.text.Style;
import javax.swing.text.StyleConstants;
import javax.swing.text.StyledDocument;
import javax.swing.text.TabSet;
import javax.swing.text.TabStop;
import javax.swing.text.html.HTMLEditorKit;
import javax.swing.text.rtf.RTFEditorKit;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.interfaces.Tabulierend;
import eu.gronos.kostenrechner.interfaces.Tenorierend;
import eu.gronos.kostenrechner.logic.TenorTexter;
import eu.gronos.kostenrechner.model.beschriftungen.Beschriftung;
import eu.gronos.kostenrechner.model.beschriftungen.NameContainerSammlung;
import eu.gronos.kostenrechner.model.tenordaten.CaretRange;
import eu.gronos.kostenrechner.model.tenordaten.TabulierendTableModel;
import eu.gronos.kostenrechner.view.FontHelfer;
import eu.gronos.kostenrechner.view.KostenFileChooser;
import eu.gronos.kostenrechner.view.result.NachkommastellenRenderer;
import eu.gronos.kostenrechner.view.result.StringButtonRenderer;

/**
 * Die Klasse ist ein {@link RTFEditorKit}, das zusätzlich Methoden
 * bereitstellt, um RTF-Daten in die Zwischenablage zu kopieren:
 * {@link #kopiereRtfInZwischenablage()}, und in eine Datei zu speichern:
 * {@link #speichereRTF()}. Zudem eine Klasse, damit ein RTF in die
 * Zwischenablage kopiert werden kann. Man kann auch Formatierungen setzen.
 * 
 * Defines the interface for classes that can be used to provide data for a
 * transfer operation. For information on using data transfer with Swing, see
 * How to Use Drag and Drop and Data Transfer, a section in The Java Tutorial,
 * for more information.
 *
 * @author Peter Schuster (setrok)
 * @date 15.08.2018
 * 
 * @url "http://lists.apple.com/archives/java-dev/2004/Jul/msg00359.html"
 * @url "http://docs.oracle.com/javase/tutorial/uiswing/dnd/index.html"
 *
 */
public class RtfWerkzeugKasten extends RTFEditorKit implements ClipboardOwner, Transferable {

	private static final String[] ARIALS = new String[] { "Arial", "Helvetica", "Helv" };

	private static final long serialVersionUID = 2201251699302376878L;

	/**
	 * flavors speichert ein {@link DataFlavor}[] mit den DataFlavors. Das Array
	 * muss so zusammengesetzt sein, dass df[i] den passenden DataFlavor zum String
	 * s[i] enthält.
	 */
	private final DataFlavor[] flavors = new DataFlavor[] { new DataFlavor("text/rtf", "Rich Formatted Text"),
			new DataFlavor("text/plain", "Plain Text") };

	private String[] data;
	private KostenFileChooser fileChooser;

	private Document doc;

	private String tenor;
	private Tenorierend tenorierend;

	private String gruende;

	public static final KostenFileFilter DATEI_FILTER = new KostenFileFilter("tenor", "Rich Text Format (RTF)", "rtf");

	private static final String RTF_CREATIM_FORMAT = "{\\creatim\\yr%d\\mo%d\\dy%d\\hr%d\\min%d}%n";
	private static final String RTF_DOCCOMM_START = "{\\doccomm\n";
	private static final String RTF_TITLE_START = "{\\title\n";
	private static final String RTF_TAG_END = "}\n";
	private static final String RTF_INFO_START = "{\\info\n";
	private static final String RTF_FONT_DEF_START = "\\deff0{\\fonttbl{\\f0\\fswiss ";
	private static final String RTF_FONT_DEF_END = ";}}\n";
	private static final String RTF_STYLESHEET_START = "{\\stylesheet";
	private static final String RTF_STYLE_START = "{\\s";
	private static final String RTF_STYLE_BASEDON = "\\sbasedon0 ";
	private static final String RTF_STYLE_END = ";}";
	private static final String RTF_TABLE_ROW_START = "{\\trowd\\trgaph180";
	private static final String RTF_CELL_WIDTH_TAG = "\\cellx";
	private static final String RTF_TABLE_ROW_END = "\\row}\n";
	private static final String RTF_TABLE_CELL_START = "{\\pard\\intbl";
	private static final String RTF_TABLE_CELL_END = "\\cell}\n";
	// private static final String RTF_BASE_STYLE_2 =
	// "\\sa142\\sl320\\slmult1\\f0\\fs24\\qj";
	private static final String RTF_BODY_START = "\\deflang1031\\plain\\fs24\\widowctrl\\hyphauto\\ftnbj";
	private static final String RTF_BASE_STYLE = "\\sa142\\sl320\\slmult1\\f0\\fs24\\qj";
	private static final String RTF_LINE_INDENT = "\\li1133";
	private static final String RTF_TENOR_STYLE = RTF_LINE_INDENT + RTF_BASE_STYLE;
	private static final String RTF_BOLD_CENTERED = "\\keepn\\qc\\b";
	private static final String RTF_HEADING_STYLE = RTF_BASE_STYLE + RTF_BOLD_CENTERED;
	private static final String RTF_PARAGRAPH_START = "\n{\\pard";
	private static final String RTF_PARAGRAPH_END = "\\par}";
	private static final String RTF_TAB = "\\\\tab ";

	public RtfWerkzeugKasten() {
		super();
	}

	/*
	 * Der Konstruktor braucht aus der GUI das EditorKit und das StyledDocument
	 * 
	 * @param doc ein {@link StyledDocument} public RtfWerkzeugKasten(StyledDocument
	 * doc) { // this.editorKit = editorKit; this.doc = doc; }
	 */
	// RTFEditorKit editorKit,
	/*
	 * @param editorKit es muss schon ein {@link RTFEditorKit} sein!
	 */

	/**
	 * Die Methode fügt den Text der JTextPane in die Zwischenablage ein. Versucht
	 * das jetzt mit text/rtf!
	 * 
	 * @fixed auch hier funktionierte der Zeichensatz nicht.
	 */
	public void kopiereRtfInZwischenablage() {
		Clipboard zwischenAblage = Toolkit.getDefaultToolkit().getSystemClipboard();
		try {
			setTransferData(createRTFString(), getAllText());
			zwischenAblage.setContents(this, this);
		} catch (BadLocationException | UnsupportedEncodingException e) {
			FehlerHelper.zeigeFehler(e.getLocalizedMessage(), e);
		}
	}

	/**
	 * 
	 * @see java.awt.datatransfer.ClipboardOwner#lostOwnership(java.awt.datatransfer.Clipboard,
	 *      java.awt.datatransfer.Transferable)
	 */
	@Override
	public void lostOwnership(Clipboard clipboard, Transferable contents) {
		// System.out.println
		Kostenrechner.getLogger()
				.info(String.format("Och nö, gehört mir doch die Zwischeablage (%s) nicht mehr!", clipboard.getName()));
	}

	/**
	 * Die Methode setzt in einer {@link JTextPane} das {@link HTMLEditorKit} und
	 * das {@link StyledDocument} und lädt
	 * 
	 * @param textPane die {@link JTextPane}
	 * @return das {@link StyledDocument}, namentlich das
	 *         {@link EditorKit#createDefaultDocument()} des {@link RTFEditorKit}s
	 * 
	 * @see HtmlWerkzeugKasten#createDefaultDocument(JTextPane)
	 * @see javax.swing.text.StyledEditorKit#install(JEditorPane)
	 */
	@Override
	public void install(JEditorPane textPane) {
		super.install(textPane);
		this.doc = this.createDefaultDocument();

		textPane.setEditable(false);

		createStyles();
	}

	/**
	 * Die Methode erstellt ein leeres neues {@link StyledDocument} und merkt es
	 * sich. Wenn bereits eines erstellt wurde, nimmt sie das zuvor erstellte (und
	 * erstellt kein neues).
	 * 
	 * @return das {@link StyledDocument} als {@link Document}
	 * 
	 * @see javax.swing.text.StyledEditorKit#createDefaultDocument()
	 */
	@Override
	public Document createDefaultDocument() {
		if (getDefaultStyledDocument() != null) {
			return getDefaultStyledDocument();
		}
		doc = super.createDefaultDocument();
		return getDefaultStyledDocument();
	}

	// Document
	// return doc;
	// DefaultStyledDocument doc, final boolean tabulierend
	// helper.
	// helper.
	// helper.

	/**
	 * @return gibt das {@link #doc} als {@link StyledDocument} zurück
	 */
	public DefaultStyledDocument getDefaultStyledDocument() {
		return (DefaultStyledDocument) getDocument();
	}

	/**
	 * Die Methode speichereRTF dient dazu, einen {@link KostenFileChooser} zu
	 * erzeugen, der fürs Speichern von RTF-Dateien vorkonfiguriert ist, um den
	 * Inhalt der JTextPane als RTF zu speichern. Die Variante ohne Parameter soll
	 * vom ActionListener der Schaltfläche selbst aufgerufen werden. Sie ruft
	 * nacheinander den Dateiauswahldialog auf, überprüft ob die Datei schon
	 * existiert und ruft dann die eigentliche Methode zum Speichern auf.
	 * 
	 * Write content from a document to the given stream in a format appropriate for
	 * this kind of content handler.
	 * 
	 * @return bei Erfolg das {@link File}, sonst <code>null</code>
	 */
	public File speichereRTF(Component parent) {
		boolean erfolg = false;
		File datei = null;
		fileChooser = new KostenFileChooser(DATEI_FILTER,
				(Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(75000));
		// RtfFileChooser();
		// dateiAuswahl.showSaveDialog(dateiAuswahl);
		int option = fileChooser.showSaveDialog(parent);// fileChooser
		if (option == JFileChooser.APPROVE_OPTION) {
			datei = fileChooser.getSelectedFile().getAbsoluteFile();
			// Existenz wird schon vom fileChooser geprüft
			erfolg = write(datei);
		}
		fileChooser = null;
		if (erfolg)
			return datei;
		else
			return null;
	}

	/**
	 * Returns an array of DataFlavor objects indicating the flavors the data can be
	 * provided in. The array should be ordered according to preference for
	 * providing the data (from most richly descriptive to least descriptive).
	 * 
	 * @return an array of data flavors in which this data can be transferred
	 * 
	 * @see java.awt.datatransfer.Transferable#getTransferDataFlavors()
	 */
	@Override
	public DataFlavor[] getTransferDataFlavors() {
		return flavors;
	}

	/**
	 * @return gibt {@link #doc} als {@link Document} zurück.
	 */
	public Document getDocument() {
		return doc;
	}

	/**
	 * Returns whether or not the specified data flavor is supported for this
	 * object.
	 * 
	 * @param flavor the requested flavor for the data
	 * @return boolean indicating whether or not the data flavor is supported
	 * 
	 * @see java.awt.datatransfer.Transferable#isDataFlavorSupported(java.awt.datatransfer.DataFlavor)
	 */
	@Override
	public boolean isDataFlavorSupported(DataFlavor flavor) {
		for (DataFlavor df : getTransferDataFlavors())
			if (df.isMimeTypeEqual(flavor))
				return true;
		return false;
	}

	/**
	 * Returns an object which represents the data to be transferred. The class of
	 * the object returned is defined by the representation class of the flavor.
	 * 
	 * @param flavor the requested flavor for the data
	 * @return an object which represents the data to be transferred. The class of
	 *         the object returned is defined by the representation class of the
	 *         flavor.
	 * @throws UnsupportedFlavorException
	 * @throws IOException
	 * 
	 * @see java.awt.datatransfer.Transferable#getTransferData(java.awt.datatransfer.DataFlavor)
	 */
	@Override
	public Object getTransferData(DataFlavor flavor) throws UnsupportedFlavorException, IOException {
		for (int i = 0; i < flavors.length; i++)
			if (flavors[i].isMimeTypeEqual(flavor))
				return new ByteArrayInputStream(data[i].getBytes());
		throw new UnsupportedFlavorException(flavor);
	}

	/**
	 * @return gibt {@link #fileChooser} als {@link KostenFileChooser} zurück.
	 */
	public KostenFileChooser getFileChooser() {
		return fileChooser;
	}

	/**
	 * Die Methode dient dazu, den Tenor der übergegebenen Tenorklasse im JTextPane
	 * mit der passenden Schriftart und Formatierungen zu setzen. Als Dialogtitel
	 * wird die Tenorbeschreibung des Tenorierend gesetzt.
	 * 
	 * @param str               den Text des Tenors als {@link String}
	 * @param title             {@link Tenorierend#getTenorBeschreibung()} als
	 *                          {@link String}
	 * @param grosseTabulatoren ein <code>boolean</code>, ob große Tabulatorabstände
	 *                          eingebaut werden sollen
	 * @throws BadLocationException TODO (kommentieren)
	 * 
	 * @see javax.swing.text.Document#insertString(int offset, String str,
	 *      AttributeSet a)
	 * @see javax.swing.text.Document#putProperty(Object key, Object value)
	 */
	public void insertString(String str, String title, boolean grosseTabulatoren) throws BadLocationException {
		this.tenor = str;
		doc.remove(0, doc.getLength());
		doc.insertString(0, str, ((StyledDocument) doc).getStyle("base"));
		// Die Beschreibung auch beim Speichern als Titel setzen
		doc.putProperty(Document.TitleProperty, title);
		setParagraphAttributes(0, doc.getLength(), grosseTabulatoren);
	}

	// public void applyStyles(FontHelferForWindow helper, JTextPane textPane,
	// boolean grosseTabulatoren) {
	// setParagraphAttributes(textPane, getDefaultStyledDocument(), 0,
	// doc.getLength(), grosseTabulatoren);
	// }

	/**
	 * Die Methode fügt weiteren Text ins {@link #getDefaultStyledDocument()} ein,
	 * vorzugsweise die Begründung.
	 * 
	 * @param str         die Gründe
	 * @param tenorierend ein {@link Tenorierend}. Wenn es auch ein
	 *                    {@link Tabulierend} ist, wird auch die Begründungs-Tabelle
	 *                    eingefügt.
	 * @param paneSize    eine {@link Dimension} mit der
	 *                    {@link JComponent#getPreferredSize()} der {@link JTable}
	 * @throws BadLocationException TODO (kommentieren)
	 */
	public int insertString(String str, Tenorierend tenorierend, Dimension paneSize) throws BadLocationException {
		this.tenorierend = tenorierend;
		this.gruende = str;
		int anfang = doc.getLength();
		doc.insertString(anfang, TenorTexter.GRUENDE_UEBERSCHRIFT, getDefaultStyledDocument().getStyle("bold"));

		int nachUeberschrift = doc.getLength();
		doc.insertString(nachUeberschrift, str, getDefaultStyledDocument().getStyle("base"));

		Kostenrechner.getLogger().info(String.format("doc.Length()-anfang: %04d", (doc.getLength() - anfang)));

		// an dieser Stelle ist das vorerst besser
		if (tenorierend instanceof Tabulierend) {
			createTableStyle(baueGruendeTabelle((Tabulierend) tenorierend, paneSize));
			gruendeTabelleEinpassen(str, (Tabulierend) tenorierend, nachUeberschrift);
		}

		return nachUeberschrift;
	}

	/**
	 * Die Methode schreibt den RTF-Inhalt des StyledDocument in einen
	 * ByteArrayOutputStream.
	 * 
	 * @return den Inhalt des ByteArrayOutputStream mittels toString()
	 * @throws UnsupportedEncodingException
	 * 
	 * 
	 * 
	 * @fixed: auch hier funktionierte der Zeichensatz nicht, jetzt schleust die
	 *         Methoden den {@link String} durch
	 *         {@link #unicodeReplace(String, Hashtable)} und gibt ihn erst dann
	 *         zurück.
	 */
	private String createRTFString() throws UnsupportedEncodingException {
		// try (ByteArrayOutputStream bos = new ByteArrayOutputStream(doc.getLength());)
		// {
		// super.write(bos, doc, doc.getStartPosition().getOffset(), doc.getLength());
		// final String str = appendRtfHeader(bos.toString("UTF-8"));
		final StringBuilder rtfBuilder = createRtfStart();
		appendRtfHeader(rtfBuilder);
		appendRtfBody(rtfBuilder);
		// wenn Begründung, dann reasons
		if (tenorierend != null && tenorierend instanceof Tabulierend) {
			appendRtfReasons(rtfBuilder);
		}
		final String str = closeRtfBuilder(rtfBuilder);
		final Map<String, String> dict = createDictionary(str);
		return unicodeReplace(str, dict);
		// } catch (IOException e) {
		// FehlerHelper.zeigeFehler(e.getLocalizedMessage(), e);
		// } catch (BadLocationException e) {
		// FehlerHelper.zeigeFehler(e.getLocalizedMessage(), e);
		// }
		// return null;
	}
	// editorKit
	// ByteArrayOutputStream bos = null;
	// bos = new ByteArrayOutputStream(doc.getLength());
	// finally {
	// der ByteArrayOutputStream muss nicht geschlossen werden
	// }
	// if (bos == null || bos.size() < 1)
	// return null;
	//
	// else
	// return null;

	// return bos != null ? bos.toString("UTF-8") : null;

	/**
	 * Die Methode erzeugt einen {@link StringBuilder} zur RTF-Erzeugung
	 * 
	 * @return den durchgereichten {@link StringBuilder}
	 */
	private StringBuilder createRtfStart() {
		// Start-Klammer, Magic number + encoding
		// Nach der Klammer darf kein Zeilenumbruch sein, sonst akzeptiert Word die
		// Datei nicht als RTF.
		return new StringBuilder("{\\rtf1\\ansi\n");
	}

	/**
	 * Die Methode erzeugt den RTF-Header und hängt ihn an den {@link StringBuilder}
	 * an. Die Methode fügt Metadaten ein, wie etwa den Titel des Dokuments
	 * 
	 * @param rtfBuilder den durchgereichten {@link StringBuilder}
	 * @return den durchgereichten {@link StringBuilder}
	 */
	private StringBuilder appendRtfHeader(StringBuilder rtfBuilder) {
		final FontHelfer helper = new FontHelfer();

		// Font table
		String arial = baseFont().getFamily();
		for (String a : ARIALS) {
			if (helper.getFontForName(a) != null) {
				arial = a;
			}
		}
		rtfBuilder.append(RTF_FONT_DEF_START + arial + RTF_FONT_DEF_END);

		// Styles
		rtfBuilder.append(RTF_STYLESHEET_START + "\n");
		rtfBuilder.append(RTF_STYLE_START + "0" + " " + "Standard" + RTF_STYLE_END);
		rtfBuilder.append(RTF_STYLE_START + "1" + RTF_STYLE_BASEDON + "Tenor" + RTF_STYLE_END);
		rtfBuilder.append(RTF_STYLE_START + "2" + RTF_STYLE_BASEDON + "Ü berschrift 1" + RTF_STYLE_END);
		rtfBuilder.append(RTF_TAG_END);

		// Infos: Title usw
		final String title = ((String) doc.getProperty(Document.TitleProperty)).replaceAll("\\\\", " ");
		rtfBuilder.append(RTF_INFO_START);
		rtfBuilder.append(RTF_TITLE_START + title + RTF_TAG_END);

		// {\creatim\yr1969\mo7\dy20\hr22\min56}
		LocalDateTime jetzt = LocalDateTime.now();
		rtfBuilder.append(String.format(RTF_CREATIM_FORMAT, jetzt.getYear(), jetzt.getMonthValue(),
				jetzt.getDayOfMonth(), jetzt.getHour(), jetzt.getMinute()));

		rtfBuilder.append(RTF_DOCCOMM_START + Kostenrechner.getInstance().getTitle() + RTF_TAG_END);
		rtfBuilder.append(RTF_TAG_END);

		return rtfBuilder;
	}

	/**
	 * Die Methode dient hängt den RTF-"Body" an den {@link StringBuilder} an, also
	 * den eingerückten Tenor-Text
	 * 
	 * @param rtfBuilder den durchgereichten {@link StringBuilder}
	 * @return den durchgereichten {@link StringBuilder}
	 */
	private StringBuilder appendRtfBody(StringBuilder rtfBuilder) {
		// 1 TWIP = 1/20 Pt. ("TWentIeth of a Point") = 1/1440 Zoll = 0,01763889 mm
		// (1/57 mm), 1mm=56,6929133858twips,

		// Los geht's mit dem RTF-"Body"
		rtfBuilder.append("\n" + RTF_BODY_START + "\n");
		rtfBuilder.append("{\\header\\pard\\plain " + RTF_BASE_STYLE + "\\qr\\chpgn" + RTF_PARAGRAPH_END + "\n");

		// Tenor nach Absätzen auftrennen und als Absätze einfügen
		appendParagraphs(rtfBuilder, tenor, "\\s1" + RTF_TENOR_STYLE);

		return rtfBuilder;
	}

	/**
	 * Die Methode hängt die Begründung an den {@link StringBuilder} an.
	 * 
	 * @param rtfBuilder den durchgereichten {@link StringBuilder}
	 * @return den durchgereichten {@link StringBuilder}
	 */
	private StringBuilder appendRtfReasons(StringBuilder rtfBuilder) {
		// Aus dem tenorierend nehmen wir nicht den Text, aber die Zeilen und den caret
		final Tabulierend tabulierend = (Tabulierend) tenorierend;

		// final String formattingBold = ;
		rtfBuilder.append(RTF_PARAGRAPH_START + "\\s2" + RTF_HEADING_STYLE + "\n");
		rtfBuilder.append(TenorTexter.GRUENDE_UEBERSCHRIFT);
		rtfBuilder.append(RTF_PARAGRAPH_END);

		// Tabelle aus Gründen heraustrennen mit caret
		final CaretRange range = tabulierend.getRange();
		final String gruendeVorTabelle = gruende.substring(0, range.beginn);
		appendParagraphs(rtfBuilder, gruendeVorTabelle, "\\s0" + RTF_BASE_STYLE);

		// in RTF-Tabelle wandeln
		final String gruendeTabelle = gruende.substring(range.beginn, range.ende);
		// appendParagraphs(rtfBuilder, gruendeTabelle, formatting);
		appendTableCells(rtfBuilder, gruendeTabelle, "\\s0" + RTF_BASE_STYLE, tabulierend);

		final String gruendeNachTabelle = gruende.substring(range.ende, gruende.length());
		appendParagraphs(rtfBuilder, gruendeNachTabelle, "\\s0" + RTF_BASE_STYLE);

		return rtfBuilder;
	}

	/**
	 * Die Methode hängt einen Abschluss an den {@link StringBuilder} und gibt ihn
	 * insgesamt als {@link String} zurück.
	 * 
	 * @param rtfBuilder den durchgereichten {@link StringBuilder}
	 * @return den durchgereichten {@link StringBuilder}
	 */
	private String closeRtfBuilder(StringBuilder rtfBuilder) {
		// Zum Schluss die geschweifte Klammer schließen
		rtfBuilder.append("\n}");
		return rtfBuilder.toString();
	}

	/**
	 * Die Methode zerlegt den übergebenen {@link String} in einzelne Absätze und
	 * hängt diese an den {@link StringBuilder} an.
	 * 
	 * @param rtfBuilder den durchgereichten {@link StringBuilder}
	 * @param string     den {@link String} mit dem unformattierten Text
	 * @param formatting der RTF-Formattierungsstring für jeden Absatz
	 * @return den durchgereichten {@link StringBuilder}
	 */
	private StringBuilder appendParagraphs(StringBuilder rtfBuilder, String string, final String formatting) {
		final String[] paragraphs = string.split("\n");
		for (String paragraph : paragraphs) {
			if (paragraph != null && !paragraph.isEmpty()) {
				rtfBuilder.append(RTF_PARAGRAPH_START + formatting + "\n");
				rtfBuilder.append(paragraph.replaceAll("\t", RTF_TAB).replaceAll("§ ", "§\\\\~"));
				//
				rtfBuilder.append(RTF_PARAGRAPH_END);
			}
		}
		return rtfBuilder;
	}// "\n{" + "\\pard" "\n{\\pard";

	/**
	 * Die Methode dient dazu, die Begründungstabelle als RTF-Tabelle zu erstellen
	 * und an den {@link StringBuilder} anzuhängen.
	 * 
	 * @param rtfBuilder  den durchgereichten {@link StringBuilder}
	 * @param string      den {@link String} aus den {@link #gruende}n, der die
	 *                    \t-getrennte Tabelle enthält
	 * @param formatting  der RTF-Formattierungsstring für jede Zelle
	 * @param tabulierend das {@link Tabulierend}e {@link Tenorierend}
	 * @return den durchgereichten {@link StringBuilder}
	 */
	private StringBuilder appendTableCells(StringBuilder rtfBuilder, String string, String formatting,
			final Tabulierend tabulierend) {
		final String startOfRow = cellWidthString(tabulierend.getColumnClasses().length);
		// Erstmal am \n nach Zeilen aufteilen - wie gehabt.
		final String[] paragraphs = string.split("\n");
		for (String paragraph : paragraphs) {
			if (paragraph != null && !paragraph.isEmpty()) {
				// Jede Zeile bekommt ein \trowd \tgraph180 und so viele \cellx<n> wie Spalten
				rtfBuilder.append(startOfRow);

				// Dann die jeweilige Zeile (Tabellen-Zeilen-String) anhand des Tabs \t nach
				// Spalten aufteilen,
				final String[] cells = paragraph.split("\t");

				// Jeweils {\pard\intbl …Formatierung……Text… \cell}
				for (String cell : cells) {
					// Hier muss man leere Spalten wohl zulassen, sonst fehlen welche
					rtfBuilder.append(RTF_TABLE_CELL_START + formatting);
					// Aber hier muss man trotzdem gegen null prüfen
					if (cell != null) {
						rtfBuilder.append(" " + cell.replaceAll("§ ", "§\\\\~"));
					}
					rtfBuilder.append(RTF_TABLE_CELL_END);
				}
				// Nach allen Spalten, am Ende der Zeile: \row}
				rtfBuilder.append(RTF_TABLE_ROW_END);
			}
		}
		// z.B
		// \pard\intbl {Text der ersten Spalte}\cell \pard\intbl {Text der zweiten
		// Spalte}\cell \row }
		return rtfBuilder;
	}

	/**
	 * Die Methode baut einen {@link String} für die RTF-Tabellenzeile zusammen. Der
	 * muss jeweils die Breite jeder Spalte enthalten.
	 * 
	 * @param spalten Anzahl der Spalten als <code>int</code>
	 * @return einen {@link String} für den Start einer Tabellenzeile, z.B.
	 *         <code>{\trowd \trgaph180 \cellx4800\cellx9600</code> + Zeilenwechsel
	 */
	private String cellWidthString(final int spalten) {
		final String tableRowStart = RTF_TABLE_ROW_START;
		final StringBuilder builder = new StringBuilder(tableRowStart);
		final String cellWidthBeginning = RTF_CELL_WIDTH_TAG;
		/*
		 * Breite einer A4-Seite (ca. 9600 TWIPs) nehmen und durch die Zahl der Spalten
		 * teilen. \\cellx für cellWidth, muss jeweils mit der Nummer der Spalte
		 * (1-based, also index+1) multipliziert werden
		 */
		final int maxTableWidthTwips = 9600;
		final int cellWidth = maxTableWidthTwips / spalten;
		// Dabei eine for-Schleife zum Erstellen der Zeilenbreiten nutzen:
		// \cellx<cellWidth*(i+1)>";
		for (int i = 0; i < spalten; i++) {
			final int rightCellPos = cellWidth * (i + 1);
			builder.append(cellWidthBeginning + rightCellPos);
		}

		builder.append("\n");
		return builder.toString();
	}

	// private String appendRtfHeader(String str) {
	// {\info{\<...> ...}
	// final String regex = "^(\\{\\\\rtf.*\\s+\\{.*\\})(\\s+)";
	// final String title = ((String)
	// doc.getProperty(Document.TitleProperty)).replaceAll("[\\[\\]\\?\\.
	// \\\\\\\\]",
	// " ");
	// {\creatim\yr2020\mo3\dy8\hr12\min43}
	// final String replacement = "$1" + "\\{\\" + "\\info" + "\\{\\" + "\\title " +
	// title + "\\}" + "\\}" + "$2";
	// return str.replaceAll(regex, replacement);
	// }

	/**
	 * Die Methode übersetzt nach den Vorgaben des <code>dict</code> die
	 * verunglückten Escape-Versuche in <code>str</code>.
	 * 
	 * @param str  der zu bereinigende {@link String}
	 * @param dict eine {@link Map}&lt;{@link String}, {@link String}&gt; mit den
	 *             Ersetzungsvorgaben
	 * @return der bereinigte <code>str</code> als {@link String}
	 */
	private String unicodeReplace(String str, Map<String, String> dict) {
		String value;

		// Alle Vorkommen aller Einträge ersetzen
		for (String key : dict.keySet()) {
			value = (String) dict.get(key);
			str = str.replaceAll(key, value);
		}

		System.out.println(str);

		return str;

	}

	// Hashtable
	// Enumeration<String> e = replacements.keys();
	// String key, value;
	// while (e.hasMoreElements()) {
	// key = (String) e.nextElement();
	// value = (String) replacements.get(key);
	// str = str.replaceAll(key, value);
	// }

	/**
	 * Die Methode liefert die Ersetzungstabelle für
	 * {@link #unicodeReplace(String, Map)}, indem sie alle über den
	 * 7-Bit-ASCII-Bereich hinausgehenden Zeichen sammelt und mitsamt einem
	 * RTF-Unicode-Ersatz-Tag-RegEx in die {@link Map} einfügt.
	 * 
	 * @param str den {@link String} mit Unicode-Zeichen
	 * @return eine Ersetzungstabelle (&quot;Dictionary&quot;) als
	 *         {@link Map}&lt;{@link String}, {@link String}>
	 */
	private Map<String, String> createDictionary(String str) {
		// HashMap als Wörterbuch für die Ersetzungswerte erzeugen
		Map<String, String> replace = new HashMap<String, String>();

		int value;
		String bit;
		String hex;
		for (int i = 0; i < str.length(); i++) {
			// ASCII-Wert abfragen
			bit = str.substring(i, i + 1);
			value = str.codePointAt(i);

			// Wenn der ASCII-Wert mehr als der 7-bit Rahmen von RTF ASCII ist, in die
			// Hashmap einfügen (Schritt 1)
			if (value > 127) {
				replace.put("(" + bit + ")", "\\\\u" + value + "\\?");
			}

			// Anscheinend werden Umlaute bereits vorher umgewandelt, aber seltsam,
			// mit "\'" + zweistelligen Hexadezimalwert,
			// deshalb umwandeln in die oben genutzte \\u + Dezimalwert + ? Syntax (Schritt
			// 2)
			if ("\\".equals(bit) && (i + 3) < str.length() && "\\\'".equals(str.substring(i, i + 2))) {
				bit = str.substring(i, i + 4);// zB: "\'fc" für "ü"
				hex = str.substring(i + 2, i + 4);// der zweitstellige Hex-Wert, zB "fc"
				try {
					value = Integer.parseInt(hex, 16);
					replace.put("\\\\" + "\\\'" + hex, "\\\\" + "u" + value + "?");
					// z.B.: replace.put("\\\\" + "\\\'" + "fc", "\\\\" + "u" + 252 + "?");
				} catch (NumberFormatException e) {
				}
			}
		}

		return replace;
	}
	// Hashtable Hashtable Hashtable//
	// System.out.printf("bit: %s; length: %d;%n", bit, bit.length());

	/**
	 * Die Methode speichereRTF dient dazu, die <code>datei</code> wirklich zu
	 * speichern. Dazu wird
	 * {@link javax.swing.text.rtf.RTFEditorKit#write(OutputStream,Document,int,int)}
	 * benutzt.
	 * 
	 * @param datei die Datei, in die gespeichert werden soll
	 * @return hat's geklappt, dann true TODO Muss {@link File} oder {@link Path}
	 *         zurückgeben
	 * 
	 * @url {@link http://stackoverflow.com/questions/2725141/java-jtextpane-rtf-save}
	 * @url {@link http://openbook.rheinwerk-verlag.de/javainsel/07_006.html#u7.6.3}
	 *      (try with resources)
	 * 
	 * @see javax.swing.text.rtf.RTFEditorKit#write(OutputStream,Document,int,int)
	 */
	private boolean write(File datei) {
		// TODO auch hier createRtfString nutzen!
		boolean erfolg = false;
		try (FileWriter fw = new FileWriter(datei); BufferedWriter bw = new BufferedWriter(fw);
		// FileOutputStream fos = new FileOutputStream(datei);
		// BufferedOutputStream bos = new BufferedOutputStream(fos);
		) {
			// super.write(bos, doc, doc.getStartPosition().getOffset(), doc.getLength());
			bw.write(createRTFString());
			erfolg = true;
		} catch (FileNotFoundException e) {
			FehlerHelper.zeigeFehler(e.getLocalizedMessage(), e);
		} catch (IOException e) {
			FehlerHelper.zeigeFehler(e.getLocalizedMessage(), e);
		} // catch (BadLocationException e) {
			// FehlerHelper.zeigeFehler(e.getLocalizedMessage(), e);
			// }
		return erfolg;
	}
	// editorKit

	/**
	 * Die Methode holt sich dem vollständigen Text der JTextPane
	 * 
	 * @return den gesamten Text der {@link JTextPane} als {@link String}
	 * @throws BadLocationException
	 */
	private String getAllText() throws BadLocationException {
		return doc.getText(doc.getStartPosition().getOffset(), doc.getLength());
	}

	/**
	 * Die Methode, die die String[] aufnehmen kann.
	 * 
	 * @param data ein String[] mit den Inhalten. Das Array muss so zusammengesetzt
	 *             sein, dass flavors[i] den passenden DataFlavor zum String data[i]
	 *             enthält.
	 */
	private void setTransferData(String... data) {
		this.data = data;
	}

	// public void applyStyles(FontHelferForWindow helper, JTextPane textPane,
	// boolean grosseTabulatoren) {
	// setParagraphAttributes(textPane, getDefaultStyledDocument(), 0,
	// doc.getLength(), grosseTabulatoren);
	// }

	/**
	 * Die Methode erstellt die {@link Style}s für das {@link StyledDocument}
	 * 
	 * @return den baseStyle
	 */
	private Style createStyles() {
		DefaultStyledDocument doc = getDefaultStyledDocument();

		Style baseStyle = createBaseStyle();
		createBoldStyle(doc, baseStyle);
		createTabStoppedStyle(doc, baseStyle);
		return baseStyle;
	}

	/**
	 * Die Methode definiert erstmal den grundlegenden Style ("base"), wobei diese
	 * direkt ins übergebene {@link StyledDocument} geschrieben werden. Zusätzlich
	 * wird der Style noch zurückgegeben.
	 * 
	 * @return ein {@link Style} mit den grundlegenden Formatierungen
	 * @see #createBaseStyle()
	 */
	private Style createBaseStyle() {
		// final Font font = helper.baseFont();
		Style baseStyle = getDefaultStyledDocument().addStyle("base", null);
		setFontArial(baseStyle);
		setFontSize(baseStyle);

		StyleConstants.setLineSpacing(baseStyle, .33f);
		StyleConstants.setSpaceBelow(baseStyle, 7f);

		return baseStyle;
	}

	// istTenorierendTabulierend()

	/*
	 * Die Methode findet die Schriftgröße heraus. Die Schriftgröße wird auf 12px
	 * gesetzt, solange nicht die Systemschriftgröße größer ist.
	 * 
	 * @param attrib ein {@link MutableAttributeSet} oder {@link Style}
	 * 
	 * @param font der {@link Font}
	 */
	// public void setFontSize(MutableAttributeSet attrib, final Font font) {
	// }

	/**
	 * Die Methode baut einen weiteren Style für fetten Text und schreibt ihn ins
	 * übergebene {@link StyledDocument}. Zusätzlich wird der Style noch
	 * zurückgegeben.
	 * 
	 * @param doc       ein {@link StyledDocument}
	 * @param baseStyle ein {@link Style} mit den grundlegenden Formatierungen
	 * @return ein {@link Style} mit den neuen Formatierungen
	 * @see #createBaseStyle()
	 */
	private Style createBoldStyle(StyledDocument doc, Style baseStyle) {

		Style style = doc.addStyle("bold", baseStyle);
		StyleConstants.setBold(style, true);

		return style;
	}

	/*
	 * Die Methode findet die Schriftgröße heraus. Die Schriftgröße wird auf 12px
	 * gesetzt, solange nicht die Systemschriftgröße größer ist.
	 * 
	 * @param attrib ein {@link MutableAttributeSet} oder {@link Style}
	 * 
	 * @param font der {@link Font}
	 */
	// public void setFontSize(MutableAttributeSet attrib, final Font font) {
	// }

	/**
	 * Die Methode baut einen Style mit Tabstopps ("tabulatoren") und schreibt ihn
	 * ins übergebene {@link StyledDocument}. Zusätzlich wird der Style noch
	 * zurückgegeben.
	 * 
	 * @param doc       ein {@link StyledDocument}
	 * @param baseStyle ein {@link Style} mit den Formatierungen
	 * @return ein {@link Style} mit den neuen Formatierungen
	 * @url "http://www.java2s.com/Tutorial/Java/0240__Swing/TabStopandTabSetClasses.htm"
	 *      "http://www.java2s.com/Code/Java/Swing-JFC/Createadecimalalignedtabstopat400pixelsfromtheleftmargin.htm"
	 * @see #createBaseStyle()
	 */
	private Style createTabStoppedStyle(StyledDocument doc, Style baseStyle) {
		Style style = doc.addStyle("tabulatoren", baseStyle);
		StyleConstants.setTabSet(style,
				new TabSet(new TabStop[] { new TabStop(600, TabStop.ALIGN_RIGHT, TabStop.LEAD_DOTS),
						new TabStop(950, TabStop.ALIGN_RIGHT, TabStop.LEAD_DOTS) }));

		return style;
	}

	/**
	 * Die Methode definiert einen Style mit einer JTable ("table"), sofern das
	 * übergebene {@link Tenorierend} auch das Interface {@link Tabulierend}
	 * implementiert
	 * 
	 * @param table eine {@link JTable}, die in den {@link Style} eingefügt wird
	 * @return der {@link Style} mit {@link JTable}
	 * @url "http://www.java-forums.org/awt-swing/2260-how-insert-tables-into-jtextpane.html#post4296"
	 */
	private Style createTableStyle(JTable table) {
		Style style = getDefaultStyledDocument().addStyle("table", getDefaultStyledDocument().getStyle("base"));
		JScrollPane jScrollPane = new JScrollPane(table);
		StyleConstants.setComponent(style, jScrollPane);
		return style;
	}
	// StyledDocument doc, Style baseStyle,
	// baseStyle
	// JTable table = baueGruendeTabelle();

	/**
	 * Die Methode dient dazu, eine Begründungstabelle einzufügen, wenn möglich
	 * 
	 * @param gruende die Begründung als {@link String}
	 * @param offset  hier soll das Einsetzen anfangen als <code>int</code>
	 * 
	 * @throws BadLocationException
	 * 
	 * @FIXED: Folgendes sorgte für eine NullPointerExeption beim
	 *         Speichern/Kopieren: setParagraphAttributes(doc, anfang,
	 *         doc.getLength() - anfang, false); Wohl wegen doc.getLenght-anfang
	 * 
	 *         {@url http://stackoverflow.com/questions/2600960/nullpointerexception-in-javax-swing-text-simpleattributeset-addattribute}
	 */
	private void gruendeTabelleEinpassen(final String gruende, Tabulierend tab, int offset)
			throws BadLocationException {
		// Wenn eine Begründungstabelle erzeugt werden kann, dann diese einfügen.
		final CaretRange range = tab.getRange();
		int anfangGruende = offset + range.beginn + 1;
		int laenge = range.ende - range.beginn;
		// Damit es keine Dopplungen gibt, erst einmal den alten Text herausnehmen
		getDefaultStyledDocument().remove(anfangGruende, laenge - 1);
		// Dann den neuen Text mit der richtigen Formatierung (d.h. mit JTable in
		// JScrollpane) einfügen
		getDefaultStyledDocument().insertString(anfangGruende, gruende.substring(range.beginn + 1, range.ende),
				getDefaultStyledDocument().getStyle("table"));
		Kostenrechner.getLogger()
				.info(String.format("anfangGruende: %04d, doc.getLength(): %04d, range: (%04d, %04d); laenge: %04d",
						anfangGruende, getDefaultStyledDocument().getLength(), range.beginn, range.ende, laenge));
	}
	// Tabulierend tab = (Tabulierend) tenorierend;

	/**
	 * Die Methode dient dazu, mit den Tabellendaten eines {@link Tabulierend} eine
	 * JTable zu bauen und mit den passenden Renderern zu versehen.
	 * 
	 * @param tabulierend das {@link Tabulierend}, in dem die Begründung steckt
	 * @param paneSize    die bevorzugte Größe ({@link Dimension}) des
	 *                    {@link JTextPane}s
	 * @return die gebaute JTable
	 */
	private JTable baueGruendeTabelle(Tabulierend tabulierend, Dimension paneSize) {
		JTable table = new JTable(new TabulierendTableModel(tabulierend));
		// Dimension d = table.getMinimumSize();
		Dimension d = table.getPreferredSize();
		d.width = paneSize.width;
		table.setPreferredScrollableViewportSize(d);
		table.setDefaultRenderer(Double.class, new NachkommastellenRenderer());
		table.setDefaultRenderer(String.class, new StringButtonRenderer());
		return table;
	}

	/**
	 * Die Methode dient dazu, die Schriftart Arial , sonst eine verwandte
	 * Schriftart aus den Schriftarten des Systems zu finden. Sie geht nacheinander
	 * die Schriftnamen durch. Ist einer davon der aktuelle Font, gewinnt dieser
	 * immer, sonst wird einer der anderen aus der Liste genommen, sofern er
	 * existiert.
	 * 
	 * @param attrib ein {@link MutableAttributeSet} oder {@link Style}
	 */
	private boolean setFontArial(MutableAttributeSet attrib) {
		final FontHelfer helper = new FontHelfer();
		boolean ok = false;
		final String ff = StyleConstants.getFontFamily(attrib);
		for (String arial : ARIALS)
			if (arial.equals(ff)) {
				StyleConstants.setFontFamily(attrib, arial);
				ok = true;
			} else if (!ok && helper.getFontForName(arial) != null) {
				StyleConstants.setFontFamily(attrib, arial);
				ok = true;
			}
		return ok;
	}

	/**
	 * Die Methode findet die Schriftgröße heraus. Die Schriftgröße wird auf 12px
	 * gesetzt, solange nicht die Systemschriftgröße größer ist.
	 * 
	 * @param attrib ein {@link MutableAttributeSet} oder {@link Style}
	 */
	private void setFontSize(MutableAttributeSet attrib) {
		final Font font = baseFont();

		// TODO Kann man herausfinden, ob die Schriftart die Größe unterstützt?
		// Nimmt jetzt auch den Zoom-Faktor
		final int fs = (int) (StyleConstants.getFontSize(attrib) * FontHelfer.zoom);
		StyleConstants.setFontSize(attrib, fs > font.getSize() ? fs : font.getSize());
	}
	// helper.
	// setFontSize(attrib, font);

	/**
	 * @return gibt den UI-Standard für EditorPane.font als {@link Font} zurück
	 */
	private Font baseFont() {
		UIDefaults uid = UIManager.getLookAndFeel().getDefaults();
		return uid.getFont("EditorPane.font");
	}

	/*
	 * Die Methode findet die Schriftgröße heraus. Die Schriftgröße wird auf 12px
	 * gesetzt, solange nicht die Systemschriftgröße größer ist.
	 * 
	 * @param attrib ein {@link MutableAttributeSet} oder {@link Style}
	 * 
	 * @param font der {@link Font}
	 */
	// public void setFontSize(MutableAttributeSet attrib, final Font font) {
	// }

	/**
	 * Die Methode setzt die Absatzformatierung auf anderthalbfachen
	 * Zeilenabsabstand (ok, 1,33-fach) und 7px Abstand danach.
	 * 
	 * @param offset            the start of the change >= 0
	 * @param length            the length of the change >= 0
	 * @param grosseTabulatoren sollen große Tabulatorabstände eingebaut werden?
	 * 
	 * @see javax.swing.JTextPane#getParagraphAttributes()
	 */
	private void setParagraphAttributes(int offset, int length, boolean grosseTabulatoren) {
		if (grosseTabulatoren)
			getDefaultStyledDocument().setParagraphAttributes(offset, length,
					getDefaultStyledDocument().getStyle("tabulatoren"), false);
		else
			getDefaultStyledDocument().setParagraphAttributes(offset, length,
					getDefaultStyledDocument().getStyle("base"), false);
	}
	// (JTextPane textPane, @param textPane ein {@link JTextPane}
	// int caret = textPane.getCaretPosition();
	// textPane.setCaretPosition(offset);
	// textPane.setCaretPosition(caret);

}
