/**
 * XmlDatei.java
 * eu.gronos.kostenrechner.controller.files (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.files;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;

import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;

import eu.gronos.kostenrechner.model.tenordaten.TenorDatenContainer;
import eu.gronos.kostenrechner.model.tenordaten.VerfahrensDatenContainer;

/**
 * Diese Klasse verwaltet die {@link Marshaller}s, die zum Speichern der
 * XML-Datei (zu {@link VerfahrensDatenContainer}n) benötigt werden.
 *
 * @author Peter Schuster (setrok)
 * @date 10 Jan 2019
 *
 * @param <E> entweder {@link VerfahrensDatenContainer} oder
 *            {@link TenorDatenContainer}
 */
public abstract class XmlDatei<E> {

	/**
	 * Die Methode soll die Programmzustände ohne Datei in einen String zu
	 * konvertieren.
	 * 
	 * @param container ein E
	 * @return
	 */
	abstract public String toXmlString(E container);

	/**
	 * Die Methode dient zum Umwandeln der Java Objekte in XML
	 * 
	 * @param container ein E
	 * @param datei     ein {@link File}
	 * @throws JAXBException
	 */
	abstract public void speicherXml(E container, File datei) throws JAXBException;

	/**
	 * Die Methode erstellt über {@link #speicherXml(Object, File)} eine temporäre
	 * Datei im Standard-Temp-Verzeichnis und gibt sie zurück.
	 * 
	 * @param container einen passenden Container
	 * @param suffix    das Datei-Suffix mit Punkt davor (.SKKTX oder .XML)
	 * @return ein {@link File}
	 * @throws JAXBException
	 * @throws IOException
	 */
	public File speicherTempXml(E container, String suffix) throws JAXBException, IOException {
		if (suffix == null) {
			suffix = "." + getDateiendung();
		}
		Path path = Files.createTempFile(null, suffix);
		File datei = path.toFile();
		speicherXml(container, datei);
		return datei;
	}

	/**
	 * Die Methode solll die W3C XML Schema referenzienren und als String
	 * zurückgeben, etwa
	 * <code>marshaller.setProperty(Marshaller.JAXB_SCHEMA_LOCATION, "http://es.indra.transporte.configuration StationNetwork.xsd");</code>
	 * 
	 * @return etwa "https://www.kostentenor.de schema-skktx.xsd"
	 * 
	 */
	abstract public String getSchemaLocation();

	/**
	 * Die Methode soll die typische Dateiendung für Dateien des behandelten
	 * Dateityps zurückgeben.
	 * 
	 * @return einen String ohne (!) führenden Punkt (&quot;.&quot;)
	 */
	abstract public String getDateiendung();
}
