/**
 * XjustizTransferHandler.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.files;

import java.awt.Image;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.io.File;
import java.io.IOException;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Locale;

import javax.swing.SwingUtilities;
import javax.swing.TransferHandler;
import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenXJustizLesenAction;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.model.baumbach.BaumbachBeteiligtenListe;
import eu.gronos.kostenrechner.model.baumbach.BaumbachBeteiligter;
import eu.gronos.kostenrechner.model.baumbach.Streitgenossen;
import eu.gronos.kostenrechner.model.beschriftungen.LangBeschriftung;
import eu.gronos.kostenrechner.model.beschriftungen.NameContainerSammlung;
import eu.gronos.kostenrechner.model.gebuehren.Teilklageruecknahme;
import eu.gronos.kostenrechner.model.tenordaten.Allgemein;
import eu.gronos.kostenrechner.model.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.model.tenordaten.Dateikopf;
import eu.gronos.kostenrechner.model.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.view.RechnerhammerIcons;
import eu.gronos.kostenrechner.view.baumbach.BaumbachAssistentDialog;
import eu.gronos.kostenrechner.view.baumbach.StreitgenossenPanel;
import eu.gronos.kostenrechner.view.forderungen.ForderungsStaffelungPanel;
import eu.gronos.kostenrechner.view.gebuehren.GebuehrenBerechnungPanel;
import eu.gronos.kostenrechner.view.gebuehren.TeilklageruecknahmePanel;
import eu.gronos.kostenrechner.view.menus.ExtrasJMenu;

/**
 * Die Klasse stellt Methoden bereit, um die Werte einem
 * {@link VerfahrensDatenContainer} in die einzelnen JPanels zu schreiben, um
 * umgekehrt die Werte aus den JPanels in einen {@link VerfahrensDatenContainer}
 * zu schreiben und auch, um diese Werte aus einer XML-Datei zu nehmen.
 * 
 * Hält auch Methoden und Konstanten zum Lesen, Schreiben und Verwalten von
 * VerfahrensDatenContainern bereit.
 * 
 * Außerdem TransferHandler zum Drag-and-Drop-Support für Berechnungs-XML- und
 * Xjustiz-XML-Dateien
 * 
 * @author Peter Felix Schuster (SchusterP2)
 * @date 05.08.2014
 */
public class XmlTransferHandler extends TransferHandler {

	private static final long serialVersionUID = 8733776081066774904L;
	private static final String FEHLER_KEINE_XML_DATEI = "Keine XML-Datei!";
	private static final String FEHLER_MEHRERE_DATEIEN = "Mehr als eine oder keine Datei im Zugriff!";
	// public static final String XML_SUFFIX = ".xml";

	/**
	 * Erzeugt einen TransferHandler
	 * 
	 */
	public XmlTransferHandler() {
		super();
	}

	/**
	 * This method is called repeatedly during a drag gesture and returns true if
	 * the area below the cursor can accept the transfer, or false if the transfer
	 * will be rejected. For example, if a user drags a color over a component that
	 * accepts only text, the canImport method for that component's TransferHandler
	 * should return false.
	 * 
	 * @param support the object containing the details of the transfer, not null.
	 * @return true if the import can happen, false otherwise
	 * 
	 * @see javax.swing.TransferHandler#canImport(javax.swing.TransferHandler.TransferSupport)
	 */
	@Override
	public boolean canImport(TransferSupport support) {
		if (support.isDrop())
			support.setShowDropLocation(false);
		if (!support.isDataFlavorSupported(DataFlavor.javaFileListFlavor)) {
			Kostenrechner.getLogger().info("Flavor not supported: javaFileListFlavor!");
			return false;
		}
		boolean copySupported = (COPY & support.getSourceDropActions()) == COPY;
		if (!copySupported) {
			Kostenrechner.getLogger().info("COPY not supported!");
			return false;
		}
		support.setDropAction(COPY);
		return true;
	}

	/**
	 * Convenience-Methode, die für Programmteile ein neues Parameterobjekt für die
	 * Verfahrensdaten erzeugt, die früher VerfahrensDatenContainer genutzt haben,
	 * vor allem für den {@link BaumbachAssistentDialog}.
	 * 
	 * @param klaeger             die BaumbachBeteiligtenListe mit den Klägern
	 * @param drittwiderbeklagten die BaumbachBeteiligtenListe mit den
	 *                            Drittwiderbeklagten
	 * @param beklagten           die BaumbachBeteiligtenListe mit den Beklagten
	 * @param streitwert          den Streitwert als double
	 * @param aktenzeichen        das Aktenzeichen als String
	 */
	public VerfahrensDatenContainer erstelleContainer(BaumbachBeteiligtenListe klaeger,
			BaumbachBeteiligtenListe drittwiderbeklagte, BaumbachBeteiligtenListe beklagte, double streitwert,
			String aktenzeichen) {
		VerfahrensDatenContainer c = new VerfahrensDatenContainer();
		/* Erst die BaumbachBeteiligtenListen einpacken */
		Streitgenossen s = c.streitgenossen;
		s.klaeger = klaeger;
		s.drittwiderbeklagte = drittwiderbeklagte;
		s.beklagte = beklagte;
		/*
		 * Vorher gab's kein Feld fürs Vorhandensein einer Widerklage, jetzt gibt's
		 * eins, dann muss es auch richtig gefüllt werden. Die Methode bleibt dieselbe.
		 */
		s.widerklage = gibtsDrittwiderbeklagte(s) || (gibtsKlaeger(c) && s.klaeger.enthaeltAnWiderklageBeteiligte())
				|| (gibtsBeklagte(c) && s.beklagte.enthaeltAnWiderklageBeteiligte());
		/* Dann die allgemeinen Teile einpacken */
		Allgemein a = c.allgemein;
		a.speicherdatum = speicherDatum();
		a.aktenzeichen = aktenzeichen;
		a.streitwert = streitwert;
		a.selectedPanel = StreitgenossenPanel.TAB_STREITGENOSSEN;
		/* Damit der Rest nicht leer ist */
		c = setzeAlleStreitwerte(setzeAlleBeteiligte(c));
		return c;
	}

	/**
	 * This method is called on a successful drop (or paste) and initiates the
	 * transfer of data to the target component. This method returns true if the
	 * import was successful and false otherwise.
	 * 
	 * Causes a transfer to occur from a clipboard or a drag and drop operation. The
	 * Transferable to be imported and the component to transfer to are contained
	 * within the TransferSupport. While the drag and drop implementation calls
	 * canImport to determine the suitability of a transfer before calling this
	 * method, the implementation of paste does not. As such, it cannot be assumed
	 * that the transfer is acceptable upon a call to this method for paste. It is
	 * recommended that canImport be explicitly called to cover this case.
	 * 
	 * Note: The TransferSupport object passed to this method is only valid for the
	 * duration of the method call. It is undefined what values it may contain after
	 * this method returns.
	 * 
	 * @param support the object containing the details of the transfer, not null.
	 * @return true if the data was inserted into the component, false otherwise
	 * @throws NullPointerException if <code>support</code> is null
	 * 
	 * @see javax.swing.TransferHandler#importData(javax.swing.TransferHandler.TransferSupport)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public boolean importData(TransferSupport support) throws NullPointerException {
		if (!canImport(support)) {
			return false;
		}
		Transferable t = support.getTransferable();
		try {
			Object o = (List<File>) t.getTransferData(DataFlavor.javaFileListFlavor);
			List<File> list;
			if (o instanceof List<?>) {
				list = (List<File>) o;
			} else {
				return false;
			}
			if (list == null || list.size() != 1) {
				FehlerHelper.zeigeFehler(FEHLER_MEHRERE_DATEIEN, new IOException(FEHLER_MEHRERE_DATEIEN));
				return false;
			}
			final File xml = list.get(0);
			if (isXml(xml) || isBerechnung(xml)) {
				Kostenrechner.getLogger().info("(isXml(xml) || isBerechnung(xml))");
				SwingUtilities.invokeLater(() -> nimmXmlDatei(xml));
				Kostenrechner.getLogger().info("return true;");
				return true;
			} else {
				FehlerHelper.zeigeFehler(FEHLER_KEINE_XML_DATEI, new IOException(FEHLER_KEINE_XML_DATEI));
				return false;
			}
		} catch (UnsupportedFlavorException e) {
			FehlerHelper.zeigeFehler("Flavor not supported. " + e.getLocalizedMessage(), e);
			return false;
		} catch (IOException e) {
			FehlerHelper.zeigeFehler("Data is no longer available in the requested flavor. " + e.getLocalizedMessage(),
					e);
			return false;
		}
	}

	@Override
	public Image getDragImage() {
		return RechnerhammerIcons.getInstance().get(4);
	}

	/**
	 * @param container der zu prüfende {@link VerfahrensDatenContainer}
	 * @return <code>true</code>, wenn es darin eine nicht-leere
	 *         Drittwiderbeklagten-Liste gibt
	 */
	public boolean gibtsDrittwiderbeklagte(Streitgenossen streitgenossen) {
		return streitgenossen.drittwiderbeklagte != null && !streitgenossen.drittwiderbeklagte.isEmpty();
	}

	/**
	 * @param container der zu prüfende {@link VerfahrensDatenContainer}
	 * @return <code>true</code>, wenn es darin eine nicht-leere Beklagten-Liste
	 *         gibt
	 */
	public boolean gibtsBeklagte(VerfahrensDatenContainer container) {
		return container.streitgenossen.beklagte != null && !container.streitgenossen.beklagte.isEmpty();
	}

	/**
	 * @param container der zu prüfende {@link VerfahrensDatenContainer}
	 * @return <code>true</code>, wenn der {@link VerfahrensDatenContainer} eine
	 *         nicht-leere Klägerliste enthält.
	 */
	public boolean gibtsKlaeger(VerfahrensDatenContainer container) {
		return container.streitgenossen.klaeger != null && !container.streitgenossen.klaeger.isEmpty();
	}

	/**
	 * Die Methode dient dazu, alle Programmzustände in einen
	 * {@link VerfahrensDatenContainer} zu packen
	 * 
	 * @return einen {@link VerfahrensDatenContainer}
	 */
	public VerfahrensDatenContainer leseAlleWertefuerContainer() {
		VerfahrensDatenContainer container = new VerfahrensDatenContainer();

		/*
		 * Allgemeine Programmeinstellungen, die nicht vom Container selbst gesetzt
		 * werden
		 */
		container.allgemein = ExtrasJMenu.getInstance().baueRueckgabewert();

		/* Erste Registerkarte Teilklagerücknahme */
		container.teilklageruecknahme = TeilklageruecknahmePanel.getInstance().baueRueckgabewert();

		/* Zweite Registerkarte Streitgenossen */
		container.streitgenossen = StreitgenossenPanel.getInstance().baueRueckgabewert();

		/* Dritte Registerkarte Forderungsstaffelung */
		container.staffelung = ForderungsStaffelungPanel.getInstance().baueRueckgabewert();

		/* Vierte Registerkarte Freie Gebührenberechnung */
		container.gebuehrenBerechnung = GebuehrenBerechnungPanel.getInstance().baueRueckgabewert();

		/* Höchster Wert im Verfahren */
		setzeAllgemeinStreitwert(container);
		// VerfahrensDatenContainerUtil.

		/* und weg damit */
		return container;
	}

	/**
	 * Die Methode dient zum Xjustiz-Import bzw Laden von Xml-Berechnungen aus
	 * {@link Kostenrechner#main(String[]) main} und aus anderen Klassen.
	 * 
	 * Wenn <code>null</code> übergeben wird, macht sie einfach nix.
	 * 
	 * @param xml ein {@link File}
	 */
	public void nimmXmlDatei(File xml) {
		VerfahrensDatenContainer container = null;
		if (xml != null) {
			Kostenrechner.getLogger().info("nimmXmlDatei");
			if (isXml(xml)) {
				container = new StreitgenossenXJustizLesenAction(StreitgenossenPanel.getInstance(),
						Kostenrechner.getInstance(), (LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(39211))
								.xJustizLesen(xml);
				// Beim XJustiz-Import dürfen nur der allgemeine Teil und die Streitgenossen
				// eingelesen werden
				if (container != null) {
					if (container.allgemein != null)
						ExtrasJMenu.getInstance().setzeWerte(container.allgemein);
					if (container.streitgenossen != null)
						StreitgenossenPanel.getInstance().setzeWerte(container.streitgenossen);
				}
			} else if (isBerechnung(xml)) {
				container = new BerechnungXmlDatei().ladeXml(xml);
				// Beim Import einer Berechnung alle Werte setzen!
				if (container != null)
					setzeAlleWerteAusContainer(container);
			}
		}
	}

	/**
	 * Die Methode dient dazu, alle Programmzustände dem
	 * {@link VerfahrensDatenContainer} anzupassen.
	 * 
	 * @param container ein {@link VerfahrensDatenContainer}
	 */
	public void setzeAlleWerteAusContainer(VerfahrensDatenContainer container) {
		Kostenrechner.getLogger().info(allgemeinContainerToString(container));

		/* Allgemeine Einstellungen */
		Kostenrechner.getLogger().info("container.allgemein");
		ExtrasJMenu.getInstance().setzeWerte(container.allgemein);

		/* Erste Registerkarte Teilklagerücknahme */
		Kostenrechner.getLogger().info("container.teilklageruecknahme");
		TeilklageruecknahmePanel.getInstance().setzeWerte(container.teilklageruecknahme);

		/* Zweite Registerkarte Streitgenossen */
		Kostenrechner.getLogger().info("container.streitgenossen");
		StreitgenossenPanel.getInstance().setzeWerte(container.streitgenossen);

		/* Dritte Registerkarte Forderungsstaffelung */
		Kostenrechner.getLogger().info("container.staffelung");
		ForderungsStaffelungPanel.getInstance().setzeWerte(container.staffelung);

		/* Vierte Registerkarte Freie Gebührenberechnung */
		Kostenrechner.getLogger().info("container.gebuehrenBerechnung");
		GebuehrenBerechnungPanel.getInstance().setzeWerte(container.gebuehrenBerechnung);

	}

	/**
	 * Die Methode dient dazu, das aktuelle Systemdatum zum Speichern zu ermitteln
	 * 
	 * @return einen {@link XMLGregorianCalendar} mit dem aktuellen Systemdatum
	 * @throws DatatypeConfigurationException
	 */
	public XMLGregorianCalendar speicherDatum() {
		try {
			DatatypeFactory factory = DatatypeFactory.newInstance();
			return factory.newXMLGregorianCalendar(new GregorianCalendar());
		} catch (DatatypeConfigurationException dce) {
			FehlerHelper.zeigeFehler("Schwerer Fehler beim Konfigurieren der Zeit!", dce);
			return null;
		}
	}

	/**
	 * Die Methode dient dazu, die Daten aus den Untercontainern {@link Dateikopf}
	 * und {@link Allgemein} zu einem String zusammenzufassen
	 * 
	 * @param container ein {@link VerfahrensDatenContainer}
	 * @return ein String
	 */
	private String allgemeinContainerToString(VerfahrensDatenContainer container) {
		GregorianCalendar gregorianCalendar = new GregorianCalendar();
		if (container.allgemein.speicherdatum != null) {
			gregorianCalendar = container.allgemein.speicherdatum.toGregorianCalendar();
		}
		return String.format(
				"Datei-/Systeminfo [speicherdatum=%s, aktenzeichen=%s, version=%s, osName=%s, osArch=%s, osVersion=%s, javaVendor=%s, javaName=%s, javaVersion=%s]",
				DateFormat.getDateInstance(DateFormat.MEDIUM, Locale.GERMAN).format(gregorianCalendar.getTime()),
				container.allgemein.aktenzeichen, container.dateikopf.version, container.dateikopf.osName,
				container.dateikopf.osArch, container.dateikopf.osVersion, container.dateikopf.javaVendor,
				container.dateikopf.javaName, container.dateikopf.javaVersion);
	}

	/**
	 * Die Methode dient dazu, den Genus/Numerus zu ermitteln
	 * 
	 * @param liste eine {@link BaumbachBeteiligtenListe}
	 * @param typ   {@link Beteiligter#KLAEGER} oder {@link Beteiligter#BEKLAGTE}
	 * @return einen {@link Beteiligter Beteiligten} mit dem richtigen genusNumerus:
	 *         Wenn die jeweilige Liste mehrere Beteiligte hat, ist Plural angesagt.
	 *         Wenn männliche Beteiligte in der Liste sind, dann ist die Liste
	 *         grammatikalisch männlich; wenn's nur weibliche sind, dann weiblich
	 */
	private Beteiligter ermittleGenusNumerus(BaumbachBeteiligtenListe liste, int typ) {
		Beteiligter bt = new Beteiligter(typ, Beteiligter.MAENNLICH);
		if (liste.size() > 0 && liste.size() < 2)
			bt.setGenusNumerus(liste.getGenusNumerusFor(0));
		else {
			int genus = Beteiligter.PLURAL;
			boolean maenner = false;
			for (BaumbachBeteiligter bb : liste)
				if (bb.getGenusNumerus() == Beteiligter.MAENNLICH)
					maenner = true;
			if (!maenner)
				genus += Beteiligter.WEIBLICH;
			bt.setGenusNumerus(genus);
			/*
			 * die laufende Nummer entspricht bei den Auswahllisten an der Oberfläche dem
			 * genusNumerus
			 */
			bt.setLfdNr(genus);
		}
		return bt;
	}

	/**
	 * Die Methode dient dazu, anhand der Dateiendung zu prüfen, ob es sich um eine
	 * gespeicherte Berechnung im skktx-XML-Format handelt
	 * 
	 * @param datei das zu prüfende {@link File}
	 * @return <code>true</code>, wenn der Dateiname auf
	 *         {@link BerechnungXmlDatei#DATEIENDUNG} endet und deswegen von
	 *         {@link BerechnungXmlDatei#DATEI_FILTER} akzeptiert wird.
	 */
	private boolean isBerechnung(final File datei) {
		return BerechnungXmlDatei.DATEI_FILTER.acceptExtension(datei);
		// datei.getName().toLowerCase().endsWith(BerechnungXmlDatei.DATEIENDUNG_GANZ);
	}

	/**
	 * Die Methode dient dazu, anhand der Dateiendung zu prüfen, ob es sich um eine
	 * Xjustiz-XML-Datei handelt
	 * 
	 * @param datei das zu prüfende {@link File}
	 * @return <code>true</code>, wenn der Dateiname vom
	 *         {@link XjustizContentHandler#XJUSTIZ_FILTER} akzeptiert wird, also
	 *         auf .xml endet
	 */
	private boolean isXml(final File datei) {
		return XjustizContentHandler.XJUSTIZ_FILTER.acceptExtension(datei);
		// datei.getName().toLowerCase().endsWith(XML_SUFFIX);
	}

	/**
	 * Die Methode sucht den höchsten Streitwert aus den UnterContainern heraus und
	 * schreibt ihn in {@link Allgemein#streitwert}
	 *
	 */
	private void setzeAllgemeinStreitwert(VerfahrensDatenContainer container) {
		container.allgemein.streitwert = Math.max(
				container.streitgenossen.klaeger.getHoechstenStreitwert(Beteiligter.KLAEGER),
				Math.max(
						container.streitgenossen.drittwiderbeklagte
								.getHoechstenStreitwert(Beteiligter.DRITTWIDERBEKLAGTE),
						Math.max(container.streitgenossen.beklagte.getHoechstenStreitwert(Beteiligter.BEKLAGTE),
								Math.max((double) container.gebuehrenBerechnung.streitwert,
										container.teilklageruecknahme.streitwerteUndObsiegen.get(0).doubleValue()))));
	}

	/**
	 * Die Methode dient dazu, alle Streitwertangaben auf den im
	 * {@link VerfahrensDatenContainer} angegebenen höchsten Streitwert der
	 * Streitgenossen zu setzen.
	 * 
	 * @param container ein {@link VerfahrensDatenContainer}
	 * @return den "aufgefüllten" {@link VerfahrensDatenContainer}
	 */
	private VerfahrensDatenContainer setzeAlleStreitwerte(VerfahrensDatenContainer container) {
		final double streitwert = container.allgemein.streitwert;
		if (streitwert <= 0)
			return container;
		/* Zuerst auf Registerkarte 1 verteilen */
		Teilklageruecknahme t = container.teilklageruecknahme;
		t.streitwerteUndObsiegen = new ArrayList<Long>();
		t.streitwerteUndObsiegen.add((long) streitwert);
		t.streitwerteUndObsiegen.add((long) streitwert);
		t.streitwerteUndObsiegen.add((long) streitwert);
		/* Dann auf Registerkarte 4 */
		container.gebuehrenBerechnung.streitwert = (long) streitwert;
		/* und Rückgabe */
		return container;
	}

	/**
	 * Die Methode dient dazu, die Beteiligten hinsichtlich genusNumerus anhand des
	 * Inhalts der {@link BaumbachBeteiligtenListe BaumbachBeteiligtenListen} der
	 * Streitgenossen zu setzen.
	 * 
	 * @param container ein {@link VerfahrensDatenContainer}
	 * @return den "aufgefüllten" {@link VerfahrensDatenContainer}
	 */
	private VerfahrensDatenContainer setzeAlleBeteiligte(VerfahrensDatenContainer container) {
		/* Erst den Genus/Numerus zu ermitteln */
		BaumbachBeteiligtenListe klaegerseite = new BaumbachBeteiligtenListe();
		klaegerseite.addAll(container.streitgenossen.klaeger);
		if (container.streitgenossen.drittwiderbeklagte != null)
			klaegerseite.addAll(container.streitgenossen.drittwiderbeklagte);
		Beteiligter kl = ermittleGenusNumerus(klaegerseite, Beteiligter.KLAEGER);
		BaumbachBeteiligtenListe beklagtenseite = new BaumbachBeteiligtenListe();
		beklagtenseite.addAll(container.streitgenossen.beklagte);
		Beteiligter bk = ermittleGenusNumerus(beklagtenseite, Beteiligter.BEKLAGTE);
		ArrayList<Beteiligter> parteien = new ArrayList<Beteiligter>();
		/* Dann das Ergebnis einsetzen */
		parteien.add(kl);
		parteien.add(bk);
		container.teilklageruecknahme.parteien = parteien;
		container.staffelung.parteien = parteien;
		/* Und zurückgeben */
		return container;
	}
}
//static
//private static final long serialVersionUID = -5507906941519939902L;
// private Kostenrechner kostenRechner;
// this.kostenRechner = kostenRechner;
//public static
// LadeXjustizController
// @SuppressWarnings("unchecked")
// List<File> l = (List<File>) t.getTransferData(DataFlavor.javaFileListFlavor);
// new VerfahrensDatenContainer2GUI().
// public static//public static//private static// public static//static// static
// public static// static//static//private static
// + "\nBerechnung vom "+ ", zu Az. " +final String string = string;
// container. container. VerfahrensDatenContainer container
// return super.importData(support);
// return super.canImport(support);
// System.out.println
// System.out.println
// VerfahrensDatenContainerUtil.