/*
 * ForderungsStaffelungsPruefer.java
 * eu.gronos.kostenrechner.view.forderungen (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.forderungen;

import java.awt.Component;
import java.util.ArrayList;

import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JTable;

import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.interfaces.ParsendUndBauend;
import eu.gronos.kostenrechner.interfaces.UnterContainerKlasse;
import eu.gronos.kostenrechner.interfaces.WerteLesbar;
import eu.gronos.kostenrechner.logic.TenorTexter;
import eu.gronos.kostenrechner.model.forderungen.AufrechnungsForderung;
import eu.gronos.kostenrechner.model.forderungen.Forderung;
import eu.gronos.kostenrechner.model.forderungen.ForderungenTableModel;
import eu.gronos.kostenrechner.model.forderungen.ForderungsStaffelung;
import eu.gronos.kostenrechner.model.forderungen.KlageForderung;
import eu.gronos.kostenrechner.model.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.model.tenordaten.VerfahrensDatenContainer;

/**
 * Der Prüfer kontrolliert alle Bestandteile einer {@link ForderungsStaffelung}
 *
 * @author Peter Schuster (setrok)
 * @date 30.08.2019
 *
 */
public class ForderungsStaffelungPruefer implements ParsendUndBauend<ForderungsStaffelung> {

	private ForderungsStaffelung rueckgabewert;
	private JComboBox<Beteiligter> cbAngriffsKlaeger;
	private JComboBox<Beteiligter> cbVerteidigungsBeklagter;
	private JTable tbForderungen;
	/**
	 * wenn <code>true</code>, dann prüft der {@link ParsendUndBauend} noch weitere
	 * Plausibilitäten
	 */
	private final boolean streng;

	/**
	 * Konstruktor, der nur die Standard-Plausibilitäten prüft.
	 * 
	 */
	public ForderungsStaffelungPruefer() {
		this(false);
	}

	/**
	 * Konstruktor.
	 * 
	 * @param streng boolean wenn <code>true</code>, dann prüft der
	 *               {@link ParsendUndBauend} noch weitere Plausibilitäten
	 */
	public ForderungsStaffelungPruefer(boolean streng) {
		super();
		this.streng = streng;
	}

	/**
	 * @param components zwei {@link JComboBox}es und eine {@link JTable}
	 * @return diesen {@link ForderungsStaffelungPruefer}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#leseComponents(Component[])
	 */
	@SuppressWarnings("unchecked")
	@Override
	public ParsendUndBauend<ForderungsStaffelung> leseComponents(Component... components) {
		if (components != null) {
			for (Component comp : components) {
				JComponent component = (JComponent) comp;
				if ("41005".equals(component.getName())) {
					cbAngriffsKlaeger = (JComboBox<Beteiligter>) component;
				} else if ("41015".equals(component.getName())) {
					cbVerteidigungsBeklagter = (JComboBox<Beteiligter>) component;
				} else if ("42015".equals(component.getName())) {
					tbForderungen = (JTable) component;
				}
			}
		}
		return this;
	}

	/**
	 * Die Methode wird von {@link WerteLesbar#baueRueckgabewert()} aufgerufen, um
	 * mit {@link #leseFelder()} die Felder auszulesen und zu prüfen, ob der einen
	 * Felder wirft.
	 * 
	 * @return <code>true</code>, wenn die Plausibilitäten zufrieden sind
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#parseEingabe()
	 */
	@Override
	public boolean parseEingabe() {
		try {
			rueckgabewert = leseFelder();
			// nur strenge Variante ruft parseEingabe(rueckgabewert) auf
			if (streng)
				return parseEingabe(rueckgabewert);
			else
				return true;
		} catch (IllegalArgumentException | NullPointerException ne) {
			FehlerHelper.zeigeFehler(ne.getLocalizedMessage(), ne);
			return false;
		}
	}

	/**
	 * Die Methode und überprüft den {@link #getRueckgabewert()}. ob alle Felder der
	 * {@link UnterContainerKlasse} mit gültigen Werten gefüllt sind und diese
	 * zueinander nicht im Widerspruch stehen.
	 * 
	 * @param forderung den {@link #getRueckgabewert()}
	 * @return <code>true</code>, wenn die Plausibilitäten zufrieden sind
	 * 
	 * @throws IllegalArgumentException Wenn forderungen null oder leer ist; oder
	 *                                  wenn die Reihenfolge (erst Klageforderungen,
	 *                                  dann Aufrechnungsforderungen) nicht
	 *                                  eingehalten ist.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#parseEingabe(java.lang.Object)
	 */
	@Override
	public boolean parseEingabe(ForderungsStaffelung rueckgabewert)
			throws IllegalArgumentException, NullPointerException {
		this.rueckgabewert = rueckgabewert;

		if (rueckgabewert.forderungen == null || rueckgabewert.forderungen.size() < 1)
			throw new IllegalArgumentException(TenorTexter.FEHLER_FORDERUNG_LEER);
		if (!isReihenfolgeEingehalten(rueckgabewert.forderungen))
			throw new IllegalArgumentException(TenorTexter.FEHLER_REIHENFOLGE_FORDERUNGEN);

		/* Jetzt auch noch die einzelnen Forderungen prüfen */
		final ForderungPruefer pruefer = new ForderungPruefer();
		for (Forderung forderung : rueckgabewert.forderungen) {
			pruefer.parseEingabe(forderung);
		}

		return true;
	}

	/**
	 * Die Methode dient dazu, alle Programmzustände in einen
	 * {@link VerfahrensDatenContainer} zu packen
	 * 
	 * @return den Untercontainer von {@link VerfahrensDatenContainer#staffelung}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#getRueckgabewert()
	 */
	@Override
	public ForderungsStaffelung getRueckgabewert() {
		return rueckgabewert;
	}

	private ForderungsStaffelung leseFelder() throws NullPointerException {
		rueckgabewert = new ForderungsStaffelung();
		rueckgabewert.parteien.add((Beteiligter) cbAngriffsKlaeger.getSelectedItem());
		rueckgabewert.parteien.add((Beteiligter) cbVerteidigungsBeklagter.getSelectedItem());
		ForderungenTableModel ftm = (ForderungenTableModel) tbForderungen.getModel();
		rueckgabewert.forderungen = ftm.getAllValues();
		return rueckgabewert;
	}

	/**
	 * Die Methode dient dazu, die Reihenfolge der Forderungen zu prüfen!
	 * 
	 * @param forderungen die zu prüfende ArrayList<Forderung>
	 * @return true, wenn die Reihenfolge eingehalten ist: Erste muss Klageforderung
	 *         sein, dann alle übrigen Klageforderungen, dann Aufrechnungen (und
	 *         keine Klageforderungen mehr)
	 */
	private boolean isReihenfolgeEingehalten(ArrayList<Forderung> forderungen) {
		boolean antwort = true;
		boolean umstiegAufAufrechnung = false;
		if (!(forderungen.get(0) instanceof KlageForderung))
			antwort = false;
		for (Forderung forderung : forderungen) {
			if (forderung instanceof AufrechnungsForderung)
				umstiegAufAufrechnung = true;
			if (umstiegAufAufrechnung && forderung instanceof KlageForderung)
				antwort = false;
		}
		return antwort;
	}
}
// ForderungsStaffelung container = new ForderungsStaffelung();
// container.parteien.add((Beteiligter) cbAngriffsKlaeger.getSelectedItem());
// container.parteien.add((Beteiligter)
// cbVerteidigungsBeklagter.getSelectedItem());
// container.forderungen = ftm.getAllValues();
// return container;
