/*
 * TeilklageruecknahmePruefer.java
 * eu.gronos.kostenrechner.controller.gebuehren (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.gebuehren;

import java.awt.Component;

import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JFormattedTextField;
import javax.swing.JTable;

import eu.gronos.kostenrechner.controller.NumberExtractor;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.interfaces.ParsendUndBauend;
import eu.gronos.kostenrechner.interfaces.UnterContainerKlasse;
import eu.gronos.kostenrechner.logic.TenorTexter;
import eu.gronos.kostenrechner.model.gebuehren.GebuehrenTableModel;
import eu.gronos.kostenrechner.model.gebuehren.GebuehrenTatbestand;
import eu.gronos.kostenrechner.model.gebuehren.GebuehrenVerzeichnis;
import eu.gronos.kostenrechner.model.gebuehren.Teilklageruecknahme;
import eu.gronos.kostenrechner.model.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.model.tenordaten.VerfahrensDatenContainer;

/**
 * Ein {@link ParsendUndBauend} für eine {@link UnterContainerKlasse}
 * {@link Teilklageruecknahme}
 *
 * @author Peter Schuster (setrok)
 * @date 30.08.2019
 *
 */
public class TeilklageruecknahmePruefer implements ParsendUndBauend<Teilklageruecknahme> {

	private Teilklageruecknahme rueckgabewert;
	private JComboBox<Beteiligter> cbKlaeger;
	private JComboBox<Beteiligter> cbBeklagte;
	private JFormattedTextField ftfStreitwertVorher;
	private JTable tabVorher;
	private JFormattedTextField ftfStreitwertNachher;
	private JTable tabNachher;
	private JFormattedTextField ftfObsiegen;
	/**
	 * wenn <code>true</code>, dann prüft der {@link ParsendUndBauend} noch weitere
	 * Plausibilitäten
	 */
	private final boolean streng;

	/**
	 * Konstruktor, der nur die Standard-Plausibilitäten prüft.
	 * 
	 */
	public TeilklageruecknahmePruefer() {
		this(false);
	}

	/**
	 * Konstruktor.
	 * 
	 * @param streng boolean wenn <code>true</code>, dann prüft der
	 *               {@link ParsendUndBauend} noch weitere Plausibilitäten
	 */
	public TeilklageruecknahmePruefer(boolean streng) {
		super();
		this.streng = streng;
	}

	/**
	 * 
	 * @param components zwei {@link JComboBox}es, drei {@link JFormattedTextField}s
	 *                   und zwei {@link JTable}s.
	 * @return diesen {@link TeilklageruecknahmePruefer}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#leseComponents(Component[])
	 */
	@SuppressWarnings("unchecked")
	@Override
	public ParsendUndBauend<Teilklageruecknahme> leseComponents(Component... components) {
		if (components != null) {
			for (Component comp : components) {
				JComponent component = (JComponent) comp;
				if ("11005".equals(component.getName())) {
					cbKlaeger = (JComboBox<Beteiligter>) component;
				} else if ("11015".equals(component.getName())) {
					cbBeklagte = (JComboBox<Beteiligter>) component;
				} else if ("12005".equals(component.getName())) {
					ftfStreitwertVorher = (JFormattedTextField) component;
				} else if ("12011".equals(component.getName())) {
					tabVorher = (JTable) component;
				} else if ("13005".equals(component.getName())) {
					ftfStreitwertNachher = (JFormattedTextField) component;
				} else if ("13011".equals(component.getName())) {
					tabNachher = (JTable) component;
				} else if ("14005".equals(component.getName())) {
					ftfObsiegen = (JFormattedTextField) component;
				} else
					System.out.println("Hä? " + component.getName());
			}
		}
		return this;
	}

	/**
	 * Die Methode überprüft den {@link #getRueckgabewert()}, nachdem er Felder
	 * ausgelesen hat.
	 * 
	 * @return <code>true</code>, wenn die Plausibilitäten zufrieden sind
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#parseEingabe()
	 */
	@Override
	public boolean parseEingabe() throws NumberFormatException, NullPointerException {
		try {
			rueckgabewert = leseFelder();
			// nur strenge Variante ruft parseEingabe(rueckgabewert) auf
			if (streng)
				parseEingabe(rueckgabewert);
		} catch (NumberFormatException | NullPointerException ne) {
			FehlerHelper.zeigeFehler(ne.getLocalizedMessage(), ne);
			return false;
		}
		return true;
	}

	/**
	 * Die Methode überprüft den {@link #getRueckgabewert()}, ohne Felder
	 * auszulesen, indem er ihr übergeben wird.
	 * 
	 * @param rueckgabewert die zu prüfende {@link UnterContainerKlasse} vom Typ
	 *                      {@link Teilklageruecknahme}
	 * @return <code>true</code>, wenn die Plausibilitäten zufrieden sind
	 * @throws NumberFormatException    wenn der {@link NumberExtractor} bei einem
	 *                                  {@link JFormattedTextField} eine solche
	 *                                  Exception wirft.
	 * @throws NullPointerException     nur zur Sicherheit...
	 * @throws IllegalArgumentException wird geworfen, wenn eine der beiden
	 *                                  folgenden Plausibilitäten verletzt ist: a)
	 *                                  der frühere Streitwert muss größer als der
	 *                                  spätere (oder gleich) sein, b) der spätere
	 *                                  Streitwert muss größer als die Verurteilung
	 *                                  (oder gleich) sein.
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#parseEingabe(java.lang.Object)
	 */
	@Override
	public boolean parseEingabe(Teilklageruecknahme rueckgabewert)
			throws NumberFormatException, NullPointerException, IllegalArgumentException {
		this.rueckgabewert = rueckgabewert;

		/* Prüfen, ob die Reihenfolge der Streitwerte eingehalten ist */
		if (rueckgabewert.streitwerteUndObsiegen == null || rueckgabewert.streitwerteUndObsiegen.isEmpty())
			throw new IllegalArgumentException("Keine Streitwerte angegeben!");
		if (rueckgabewert.streitwerteUndObsiegen.get(1) > rueckgabewert.streitwerteUndObsiegen.get(0))
			throw new IllegalArgumentException(TenorTexter.FEHLER_REIHENFOLGE_STREITWERTE);
		/* Prüfen, ob Verurteilung nicht höher als die Streitwerte ist */
		if (rueckgabewert.streitwerteUndObsiegen.get(2) > rueckgabewert.streitwerteUndObsiegen.get(1))
			throw new IllegalArgumentException(TenorTexter.FEHLER_REIHENFOLGE_VERURTEILUNG);

		/* Jetzt noch die Einzelkomponenten prüfen */
		final GebuehrenPruefer pruefer = new GebuehrenPruefer();
		for (GebuehrenVerzeichnis verzeichnis : new GebuehrenVerzeichnis[] { rueckgabewert.vorher,
				rueckgabewert.nachher }) {
			for (GebuehrenTatbestand gt : verzeichnis) {
				pruefer.parseEingabe(gt);
			}
		}

		return true;
	}

	/**
	 * Die Methode dient dazu, die zusammen gepackten Programmzustände in einem
	 * {@link VerfahrensDatenContainer} zurückzugeben
	 * 
	 * @return den Untercontainer von
	 *         {@link VerfahrensDatenContainer#teilklageruecknahme}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#getRueckgabewert()
	 */
	@Override
	public Teilklageruecknahme getRueckgabewert() {
		return rueckgabewert;
	}

	/**
	 * Die Methode packt alle Programmzustände in einen
	 * {@link VerfahrensDatenContainer}
	 * 
	 * @return den Untercontainer von
	 *         {@link VerfahrensDatenContainer#teilklageruecknahme}
	 */
	private Teilklageruecknahme leseFelder() throws NumberFormatException, NullPointerException {
		rueckgabewert = new Teilklageruecknahme();
		rueckgabewert.parteien.add((Beteiligter) cbKlaeger.getSelectedItem());
		rueckgabewert.parteien.add((Beteiligter) cbBeklagte.getSelectedItem());
		NumberExtractor xtractor = new NumberExtractor(ftfStreitwertVorher,
				ftfStreitwertVorher.getAccessibleContext().getAccessibleName());
		rueckgabewert.streitwerteUndObsiegen.add(xtractor.extractLong());
		xtractor = new NumberExtractor(ftfStreitwertNachher,
				ftfStreitwertNachher.getAccessibleContext().getAccessibleName());
		rueckgabewert.streitwerteUndObsiegen.add(xtractor.extractLong());
		xtractor = new NumberExtractor(ftfObsiegen, ftfObsiegen.getAccessibleContext().getAccessibleName());
		rueckgabewert.streitwerteUndObsiegen.add(xtractor.extractLong());
		GebuehrenTableModel gtmVorher = (GebuehrenTableModel) tabVorher.getModel();
		rueckgabewert.vorher = gtmVorher.getAllValues();
		GebuehrenTableModel gtmNachher = (GebuehrenTableModel) tabNachher.getModel();
		rueckgabewert.nachher = gtmNachher.getAllValues();
		return rueckgabewert;
	}

}

/*
 * 
 * Die Methode parst die JFormattedTextFields für die Streitwertreduktion.
 * 
 * @return ein long[] mit den Werten in der Reihenfolge Streitwert vorher,
 * Streitwert nachher, letztlich obsiegt.
 */
// private long[] parseVorherNachherLetztendlich() {
// long[] zwischen = new long[3];
// long vorher;
// long nachher;
// long letztendlich;
// try {
// vorher = longAusFormatter(getFormattedTextField(TABLE_FTF_VORHER), getLabelText(TABLE_FTF_VORHER));
// nachher = longAusFormatter(getFormattedTextField(TABLE_FTF_NACHHER), getLabelText(TABLE_FTF_NACHHER));
// letztendlich = longAusFormatter(getFormattedTextField(FTF_OBSIEGEN), getLabelText(FTF_OBSIEGEN));
// } catch (NumberFormatException nfe) {
// new FehlerDialog("Bitte Ganzzahlen eingeben!",
// "Konnte Eingabe nicht als Ganzzahl lesen: " + nfe.getLocalizedMessage(), nfe).showDialog();
// return new long[] { 0L, 0L, 0L };
// }
// zwischen = new long[] { vorher, nachher, letztendlich };
// return zwischen;
// }
// ftfStreitwertVorher,lblStreitwertVorReduktion.getText()
// getFtfStreitwertNachher(),lblStreitwertNachReduktion.getText()
// getFtfObsiegen(),lblLetztlichObsiegt.getText()

//ArrayList<GebuehrenTatbestand>
// ArrayList<GebuehrenTatbestand>
// Teilklageruecknahme container = new Teilklageruecknahme();
// container.parteien.add((Beteiligter) cbKlaeger.getSelectedItem());
// container.parteien.add((Beteiligter) cbBeklagte.getSelectedItem());
// long[] vorherNachherLetztendlich = parseVorherNachherLetztendlich();
// container.streitwerteUndObsiegen.add(vorherNachherLetztendlich[0]);
// container.streitwerteUndObsiegen.add(vorherNachherLetztendlich[1]);
// container.streitwerteUndObsiegen.add(vorherNachherLetztendlich[2]);
// container.vorher = new GebuehrenVerzeichnis();
// container.vorher = getGttmVorher().getAllValues();
// container.nachher = new GebuehrenVerzeichnis();
// container.nachher = getGttmNachher().getAllValues();
// return container;