/*
 * WiderklagePruefer.java
 * eu.gronos.kostenrechner.controller.gebuehren (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.gebuehren;

import java.awt.Component;

import javax.swing.JCheckBox;
import javax.swing.JComponent;

import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.interfaces.ParsendUndBauend;
import eu.gronos.kostenrechner.model.baumbach.Widerklage;
import eu.gronos.kostenrechner.view.HinzufuegenDialog;
import eu.gronos.kostenrechner.view.baumbach.WiderklageHinzufuegenDialog;

/**
 * Ein {@link ParsendUndBauend} für den {@link WiderklageHinzufuegenDialog}
 *
 * @author Peter Schuster (setrok)
 * @date 19.07.2019
 *
 */
public class WiderklagePruefer implements ParsendUndBauend<Widerklage> {

	private JCheckBox chbWiderklage;
	private JCheckBox chbWirtschaftlichIdentisch;
	private Widerklage rueckgabewert;

	public WiderklagePruefer() {
		super();
	}

	/**
	 * Die Methode soll Werte aus den {@link JComponent}s des
	 * {@link HinzufuegenDialog}s lesen, hier zewi {@link JCheckBox}s.
	 * 
	 * @param components die {@link JComponent}s des Dialogs als varargs
	 * @return ein {@link ParsendUndBauend}&lt;{@link Widerklage}>
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#leseComponents(Component[])
	 */
	@Override
	public ParsendUndBauend<Widerklage> leseComponents(Component... components) {
		if (components != null) {
			for (Component comp : components) {
				JComponent component = (JComponent) comp;
				if ("37001".equals(component.getName()))// chbWiderklage
					chbWiderklage = (JCheckBox) component;
				else if ("37011".equals(component.getName()))// chbWirtschaftlichIdentisch
					chbWirtschaftlichIdentisch = (JCheckBox) component;
			}
		}
		return this;
	}

	/**
	 * Die Methode wird aufgerufen, um zu prüfen, ob alle {@link JComponent}s zur
	 * Eingabe mit gültigen Werten gefüllt sind und diese zueinander nicht im
	 * Widerspruch stehen.
	 * 
	 * @return <code>true</code>, wenn die Plausibilitäten zufrieden sind
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#parseEingabe()
	 */
	@Override
	public boolean parseEingabe() {
		try {
			rueckgabewert = leseFelder();
			return parseEingabe(rueckgabewert);
		} catch (IllegalArgumentException | NullPointerException ne) {
			FehlerHelper.zeigeFehler(ne.getLocalizedMessage(), ne);
			return false;
		}
	}

	/**
	 * Die Methode wird aufgerufen, um zu prüfen, ob der Rückgabewert gültige
	 * Teilwerte hat, die zueinander nicht im Widerspruch stehen. Prüft nicht viel.
	 * Wirtschaftlich identisch ohne aktivierte Widerklage sinnlos.
	 * 
	 * @param rueckgabewert eine {@link Widerklage}
	 * @return <code>true</code>, wenn die Plausibilitäten zufrieden sind
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#parseEingabe(java.lang.Object)
	 */
	@Override
	public boolean parseEingabe(Widerklage rueckgabewert) throws IllegalArgumentException, NullPointerException {
		this.rueckgabewert = rueckgabewert;
		if (rueckgabewert.isWirtschaftlichIdentisch() && !rueckgabewert.isVorhanden()) {
			throw new IllegalArgumentException(
					"Wirtschaftliche Identität wird nicht beachtet, wenn keine Widerklage aktiviert ist.");
		}
		return true;
	}

	/**
	 * Die Methode soll einen Rückgabewert zurückgeben, der aus den Eingabefeldern
	 * des Dialogs zusammengebaut wurde, der dann von {@link #showDialogAndAsk()} an die
	 * aufrufende Maske zurückgegeben werden kann.
	 * 
	 * @return eine {@link Widerklage} mit allen Werten
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#getRueckgabewert()
	 */
	@Override
	public Widerklage getRueckgabewert() {
		return rueckgabewert;
	}

	private Widerklage leseFelder() throws NullPointerException {
		return new Widerklage(chbWiderklage.isSelected(), chbWirtschaftlichIdentisch.isSelected());
	}

	// return new boolean[] { chbWiderklage.isSelected(),
//		// chbWirtschaftlichIdentisch.isSelected() };
//		return new Widerklage(chbWiderklage.isSelected(), chbWirtschaftlichIdentisch.isSelected());
}
