/*
 * BooleanConfigOption.java
 * eu.gronos.kostenrechner.controller.system (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.system;

import eu.gronos.kostenrechner.model.beschriftungen.BooleanConfigSchluessel;

/**
 * Eine Klasse für den Zugriff auf Registry oder Benutzereinstellungen mit
 * {@link Boolean}-Werten. Sie speichert den Ast /eu/gronos/kostentenor und den
 * Schlüssel. Jede Instanz korrespondiert mit einem Boolean-Wert
 * <code>schluessel</code> in der Registry im Unterast
 * <code>"/eu/gronos/kostentenor"</code> zum Java-Ast. Diese wird zunächst
 * eingelesen. Wenn der Schlüssel <code>schluessel</code> noch nicht gesetzt
 * ist, wird der als <code>standardWert</code> übergebene Wahrheitswert
 * genommen.
 *
 * @author Peter Schuster (setrok)
 * @date 24 Apr 2019
 *
 */
public class BooleanConfigOption extends PreferencesConfigOption {

	private final boolean standardWert;

	/**
	 * @param parameter ein {@link BooleanConfigSchluessel}
	 */
	public BooleanConfigOption(BooleanConfigSchluessel parameter) {
		super(parameter.getSchluessel());
		this.standardWert = parameter.getStandardWert();
		setValue(ausRegistryLesen());
	}

	/**
	 * Die Methode schreibt den Wert in die Registry/Preferences und setzt das
	 * Oberflächenelement entsprechend
	 * 
	 * @param value
	 */
	public void setValue(boolean value) {
		inRegistrySchreiben(value);
	}

	/**
	 * 
	 * Die Methode liest den Wert aus der Registry (den Preferences) und setzt das
	 * Oberflächenelement entsprechend
	 * 
	 * @return value als boolean
	 */
	public boolean getValue() {
		boolean value = ausRegistryLesen();
		return value;
	}

	/**
	 * @return gibt {@link #standardWert} als boolean zurück.
	 */
	public boolean getStandardWert() {
		return standardWert;
	}

	/**
	 * Die Methode setzt wieder den Standardwert in die Registry/Preferences.
	 */
	public void reset() {
		setValue(getStandardWert());
	}

	/**
	 * Schreibt den übergebenen Wahrheitswert in den Java-Zweig der Registry, und
	 * zwar in den Schlüssel, der dem Konstruktor übergeben wurde.
	 * 
	 * @param wert der Wahrheitswert, der in die Registry soll (boolean)
	 * 
	 * @url {@link "http://openbook.galileocomputing.de/javainsel/javainsel_11_009.html#dodtp1f33b78d-493f-43a5-be9c-afc95d5f903b"}
	 *      Java ist auch eine Insel Kapitel 11.9.2 Einträge einfügen, auslesen und
	 *      löschen
	 */
	protected void inRegistrySchreiben(boolean wert) {
		getUserprefs().putBoolean(getSchluessel(), wert);
		logInfo(String.format("%s", wert));
	}

	/**
	 * Liest einen Wert aus dem Java-Zweig der Registry, unter dem Schluessel, der
	 * dem Konstruktor mitgegeben wurde.
	 * 
	 * @return den Wahrheitswert aus der Registry oder den Standardwert, der dem
	 *         Konstruktor mitgegeben wurde
	 * @url {@link "http://openbook.galileocomputing.de/javainsel/javainsel_11_009.html#dodtp1f33b78d-493f-43a5-be9c-afc95d5f903b"}
	 *      Java ist auch eine Insel Kapitel 11.9.2 Einträge einfügen, auslesen und
	 *      löschen
	 */
	protected boolean ausRegistryLesen() {
		return getUserprefs().getBoolean(getSchluessel(), getStandardWert());
	}
}