package eu.gronos.kostenrechner.controller.system;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemListener;

import javax.swing.AbstractButton;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JMenuItem;
import javax.swing.JToggleButton;

import eu.gronos.kostenrechner.controller.ComponentBeschrifter;
import eu.gronos.kostenrechner.interfaces.HorcherWandelbar;
import eu.gronos.kostenrechner.model.beschriftungen.BooleanConfigSchluessel;
import eu.gronos.kostenrechner.model.beschriftungen.LangBeschriftung;
import eu.gronos.kostenrechner.model.beschriftungen.LangVorsorgeBeschriftung;
import eu.gronos.kostenrechner.model.beschriftungen.NameContainer;
import eu.gronos.kostenrechner.model.beschriftungen.SchluesselBeschriftung;

/**
 * Ein {@link ItemListener} für {@link AbstractButton}-Elemente.
 * 
 * Jede Instanz korrespondiert mit einem Boolean-Wert <code>schluessel</code> in
 * der Registry im Unterast <code>"/eu/gronos/kostentenor"</code> zum Java-Ast.
 * Diese wird zunächst eingelesen. Wenn der Schlüssel <code>schluessel</code>
 * noch nicht gesetzt ist, wird der als <code>standardWert</code> übergebene
 * Wahrheitswert genommen.
 *
 * @author Peter Schuster (setrok)
 * @date 18 Feb 2019
 *
 */
public class BooleanConfigOptionHorcher extends BooleanConfigOption
		implements /* ItemListener */ActionListener, HorcherWandelbar {

	private AbstractButton button;
	private SchluesselBeschriftung beschriftung;

	/**
	 * @param beschriftung eine {@link SchluesselBeschriftung}, der auch ein
	 *                     {@link BooleanConfigSchluessel},
	 *                     ({@link javax.swing.Action#NAME}),
	 *                     ({@link javax.swing.Action#SHORT_DESCRIPTION}),
	 *                     {@link javax.swing.Action#MNEMONIC_KEY},
	 *                     {@link javax.swing.Action#ACCELERATOR_KEY} und
	 *                     ({@link javax.swing.Action#ACTION_COMMAND_KEY}) entnommen
	 *                     werden.
	 */
	public BooleanConfigOptionHorcher(SchluesselBeschriftung beschriftung) {
		super((BooleanConfigSchluessel) beschriftung.getConfigSchluessel());
		setBeschriftung(beschriftung);
		setButton(null);
	}

	/**
	 * Die Methode wird aufgerufen, wenn sich der Status
	 * {@link AbstractButton#isSelected()} ändert.
	 * 
	 * Beim Klicken wird der Text des {@link AbstractButton} der Auswahl angepasst
	 * und mit {@link #inRegistrySchreiben(boolean)} der Wert auch gespeichert.
	 * 
	 * @param e ein {@link ActionEvent}
	 * 
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	@Override
	public void actionPerformed(ActionEvent e) {
		Object source = e.getSource();
		if (source instanceof AbstractButton) {
			AbstractButton button = (AbstractButton) source;
			// ((AbstractButton) source)
			boolean value = button.isSelected();
			inRegistrySchreiben(value);
			setButtonText(button);
		}
	}

	/**
	 * Die Methode schreibt den Wert in die Registry/Preferences und setzt das
	 * Oberflächenelement entsprechend
	 * 
	 * @param value
	 */
	public void setValue(boolean value) {
		super.setValue(value);
		adjustButton(value);
	}

	/**
	 * 
	 * Die Methode liest den Wert aus der Registry (den Preferences) und setzt das
	 * Oberflächenelement entsprechend
	 * 
	 * @return value als boolean
	 */
	public boolean getValue() {
		boolean value = super.getValue();
		adjustButton(value);
		return value;
	}

	/**
	 * Die Methode setzt wieder den Standardwert in die Registry/Preferences.
	 * 
	 * @see eu.gronos.kostenrechner.controller.system.BooleanConfigOption#reset()
	 */
	@Override
	public void reset() {
		setValue(getStandardWert());
	}

	/**
	 * Die Methode baut einen {@link JToggleButton} mit dem {@link ItemListener} und
	 * setzt zusätzlich {@link JToggleButton#getName()} auf die
	 * {@link NameContainer#getId()}.
	 * 
	 * @return den erstellten {@link JToggleButton} als {@link AbstractButton}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.HorcherWandelbar#toButton()
	 */
	@Override
	public AbstractButton toButton() {
		AbstractButton button = new JToggleButton();
		return toButton(button);
	}

	/**
	 * Die Methode versieht einen {@link AbstractButton} oder ein abgeleitetes
	 * Objekt mit dem {@link ItemListener} und setzt zusätzlich
	 * {@link AbstractButton#getName()} auf die {@link NameContainer#getId()}.
	 * 
	 * @param button einen {@link AbstractButton}
	 * @return den präparierten {@link AbstractButton}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.HorcherWandelbar#toButton(javax.swing.AbstractButton)
	 */
	@Override
	public AbstractButton toButton(AbstractButton button) {
		this.setButton(button);
		// button.addItemListener(this);
		// button.setAction(this);
		button.addActionListener(this);
		getValue();
		//actionPerformed(new ActionEvent(button, ActionEvent.ACTION_FIRST, getBeschriftung().getActionCommandKey()));
		//berechnungsMethodeAction.actionPerformed(new ActionEvent(btnBerechnungsMethode, ActionEvent.ACTION_FIRST,
				//btnBerechnungsMethode.getActionCommand()));
		setButtonText(button);
		return button;
	}

	/**
	 * Die Methode baut ein {@link JMenuItem} mit dem {@link ItemListener} und setzt
	 * zusätzlich {@link JMenuItem#getName()} auf die {@link NameContainer#getId()}.
	 * 
	 * @return das erstellte {@link JMenuItem}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.HorcherWandelbar#toMenuItem()
	 */
	@Override
	public JMenuItem toMenuItem() {
		JCheckBoxMenuItem item = new JCheckBoxMenuItem();
		return toMenuItem(item);
	}

	/**
	 * Die Methode versieht ein {@link JMenuItem} oder ein abgeleitetes Objekt mit
	 * dem {@link ItemListener} und setzt zusätzlich {@link JMenuItem#getName()} auf
	 * die {@link NameContainer#getId()}.
	 * 
	 * @param item ein {@link JMenuItem}
	 * @return das präparierte {@link JMenuItem}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.HorcherWandelbar#toMenuItem(javax.swing.JMenuItem)
	 */
	@Override
	public JMenuItem toMenuItem(JMenuItem item) {
		setButton(item);
		ComponentBeschrifter beschrifter = new ComponentBeschrifter();
		//.applyTo
		beschrifter.beschrifte(item, getBeschriftung());
		// item.addItemListener(this);
		// item.setAction(this);
		item.addActionListener(this);
		getValue();
		setButtonText(button);
		return item;
	}

	/**
	 * @return gibt {@link #beschriftung} als {@link LangBeschriftung} zurück, der
	 *         auch ({@link javax.swing.Action#NAME}),
	 *         ({@link javax.swing.Action#SHORT_DESCRIPTION}),
	 *         {@link javax.swing.Action#MNEMONIC_KEY},
	 *         {@link javax.swing.Action#ACCELERATOR_KEY} und
	 *         ({@link javax.swing.Action#ACTION_COMMAND_KEY}) entnommen werden.
	 */
	public SchluesselBeschriftung getBeschriftung() {
		return beschriftung;
	}

	/**
	 * @param beschriftung d. {@link #beschriftung}, d. gesetzt werden soll als
	 *                     {@link LangBeschriftung}.
	 */
	public void setBeschriftung(SchluesselBeschriftung beschriftung) {
		this.beschriftung = beschriftung;
	}

	/**
	 * Beim Klicken wird der Text des {@link AbstractButton} aus den
	 * {@link LangVorsorgeBeschriftung#getAlternativeTitles()} der Auswahl
	 * angepasst, bei {@link Boolean#TRUE} der index 1, bei {@link Boolean#FALSE}
	 * der index 0.
	 * 
	 * @param button der {@link #toButton()}
	 */
	private void setButtonText(AbstractButton button) {
		int index = button.isSelected() ? 1 : 0;// 0 : 1;
		ComponentBeschrifter beschrifter = new ComponentBeschrifter();
		//getBeschriftung().changeTo(index, beschrifter);
		//.applyTo
		beschrifter.changeTo(getBeschriftung(), index);
		beschrifter.beschrifte(button, getBeschriftung());
	}

	/**
	 * @param button der {@link AbstractButton}, an dem die Option an der Oberfläche
	 *               geändert werden kann und für den die Instanz als
	 *               {@link ItemListener} eingerichtet werden soll.
	 */
	private void setButton(AbstractButton button) {
		this.button = button;
	}

	/**
	 * Die Methode setzt das Oberflächenelement entsprechend
	 * 
	 * @param value <code>true</code> if the button is selected, otherwise false
	 */
	private void adjustButton(boolean value) {
		if (button != null) {
			button.setSelected(value);
		}
	}

}

/*
 * @param schluessel der Schlüssel im Unterast
 * <code>"/eu/gronos/kostentenor"</code> (String)
 * 
 * @param standardWert der Standard-Wert, falls der Schlüssel noch nicht
 * angelegt ist. (boolean)
 * 
 * @param button der {@link AbstractButton}, an dem die Option an der Oberfläche
 * geändert werden kann und für den die Instanz als {@link ItemListener}
 * eingerichtet werden soll.
 */
//public BooleanConfigOptionHorcher(BooleanConfigSchluessel parameter, AbstractButton button) {
//	super(parameter);
//	setButton(button);
//	if (button.getAction() != null && button.getAction() instanceof BeschriebeneAktion) {
//		setBeschriftung(((BeschriebeneAktion) button.getAction()).getBeschriftung());
//	}
//	getValue();
//}
/*
 * String schluessel, boolean standardWert new
 * BooleanConfigSchluessel(schluessel, standardWert)
 */

/*
 * 
 * @return gibt den {@link AbstractButton}, an dem die Option an der Oberfläche
 * geändert werden kann und für den die Instanz als {@link ItemListener}
 * eingerichtet werden soll.
 */
//private AbstractButton getButton() {
//	return button;
//}

/*
 * @param e ein {@link ItemEvent}
 * 
 * @see java.awt.event.ItemListener#itemStateChanged(java.awt.event.ItemEvent)
 */
//@Override
//public void itemStateChanged(ItemEvent e) {
//	Object source = e.getSource();
//	if (source instanceof AbstractButton) {
//		boolean value = ((AbstractButton) source).isSelected();
//		inRegistrySchreiben(value);
//	}
//}
// public BooleanConfigOptionHorcher(BooleanConfigSchluessel parameter,
// LangBeschriftung beschriftung) {
// super(parameter);
// setBeschriftung(beschriftung);
// setButton(null);
// }
// String schluessel, boolean standardWert
// new BooleanConfigSchluessel(schluessel, standardWert)
