/*
 * FehlerHelper.java
 * eu.gronos.kostenrechner.view.helpanderror (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.system;

import java.io.StringWriter;
import java.util.logging.Level;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.BeschriebeneAktion;
import eu.gronos.kostenrechner.controller.files.ThrowableWrapper;
import eu.gronos.kostenrechner.model.beschriftungen.LangBeschriftung;
import eu.gronos.kostenrechner.model.beschriftungen.NameContainerSammlung;
import eu.gronos.kostenrechner.view.helpanderror.FehlerDialog;

/**
 * Klasse dient dazu die Message, LocalizedMessage und den StackTrace eines
 * Fehlers in einen {@link String} zu schreiben.
 * 
 * Außerdem steuert die Klasse über {@link #zeigeFehler(String, Throwable)} den
 * Aufruf des {@link FehlerDialog}s.
 *
 * @author Peter Schuster (setrok)
 * @date 24 Apr 2019
 *
 */
public class FehlerHelper {

	private final Throwable throwable;
	private final String meldung;
	private final BeschriebeneAktion emailSchreibenFehler;
	private static FehlerDialog dialog = null;	

	/**
	 * @param throwable ein {@link Throwable}, um die {@link StackTraceElement}s
	 *                  auszulesen
	 * @param meldung   die Fehlermeldung
	 */
	public FehlerHelper(Throwable throwable, String meldung) {
		this.throwable = throwable;

		this.meldung = meldung;
		// this.
		StackTraceElement stack = throwable.getStackTrace()[0];
		String klasse = stack.getClassName();
		String methode = stack.getMethodName();
		Kostenrechner.getLogger().logp(Level.WARNING, klasse, methode, meldung, throwable);
		emailSchreibenFehler = new SchreibeEmailFehler(
				(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(72045), this);
	}

	/**
	 * Die Methode zeigt über {@link FehlerDialog} einen Fehlertext an. Wenn
	 * {@link Kostenrechner} noch nicht existiert oder noch nicht angezeigt wird,
	 * zeigt sie den Fehlertext auf der Konsole an.
	 * 
	 * @param meldung   den anzuzeigenden Fehlertext
	 * @param throwable das {@link Throwable} (Oberklasse von {@link Exception}),
	 *                  das geworfen wurde.
	 */
	public static void zeigeFehler(String meldung, Throwable throwable) {
		if (Kostenrechner.getInstance() == null || !Kostenrechner.getInstance().isVisible()) {
			if (Kostenrechner.getLogger() == null) {
				System.err.println(meldung);
				throwable.printStackTrace();
			} else {
				StackTraceElement stack = throwable.getStackTrace()[0];
				String klasse = stack.getClassName();
				String methode = stack.getMethodName();
				Kostenrechner.getLogger().logp(Level.SEVERE, klasse, methode, meldung, throwable);
			}
		} else {
			setDialog(new FehlerDialog(meldung, new FehlerHelper(throwable, meldung))).showDialog();
			setDialog(null);
		}
	}

	/**
	 * Die Methode marshallt den {@link Throwable}, also dessen
	 * {@link Throwable#getMessage()}, die {@link Throwable#getLocalizedMessage()}
	 * und {@link Throwable#getStackTrace()}, in einen XML-{@link String}.
	 * 
	 * @param formatted schaltet, ob das XML formatiert sein soll, also ob der
	 *                  Marshaller Zeilenumbrüche und Einrückungen bauen soll, siehe
	 *                  {@link Marshaller#JAXB_FORMATTED_OUTPUT}
	 * @return einen String mit den XML-Daten oder einen Fehlertext, falls die
	 *         Konvertiertung nicht klappt
	 */
	public String baueFehlerXmlString(boolean formatted) {
		StringWriter sw = new StringWriter();
		try {
			JAXBContext context = JAXBContext.newInstance(ThrowableWrapper.class);
			Marshaller marshaller = context.createMarshaller();
			marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, new Boolean(formatted));
			marshaller.marshal(new ThrowableWrapper(throwable), sw);
		} catch (JAXBException e) {
			e.printStackTrace();
			return "Fehler beim Konvertieren in XML: " + e.getLocalizedMessage();
		}
		return sw.toString();
	}

	/**
	 * @return gibt {@link #dialog} als {@link FehlerDialog} zurück.
	 */
	public static FehlerDialog getDialog() {
		return dialog;
	}

	/**
	 * @param dialog d. {@link #dialog}, d. gesetzt werden soll als {@link FehlerDialog}.
	 */
	public static FehlerDialog setDialog(FehlerDialog dialog) {
		FehlerHelper.dialog = dialog;
		return dialog;
	}

	/**
	 * @return gibt {@link #emailSchreibenFehler} als {@link BeschriebeneAktion}
	 *         zurück.
	 */
	public BeschriebeneAktion getEmailAktion() {
		return emailSchreibenFehler;
	}

	/**
	 * @return gibt {@link #meldung} als {@link String} zurück, also die
	 *         Fehlermeldung
	 */
	public String getMeldung() {
		return meldung;
	}

	/**
	 * @return gibt {@link #throwable} als {@link Throwable} zurück.
	 */
	public Throwable getThrowable() {
		return throwable;
	}
}
//for (int index = 0; index < elem.getElementCount(); index++) {
//Element element = elem.getElement(index);
//System.out.println(element.toString() + element.getElementCount());
//doc.removeElement(element);
//}
//doc.insertAfterStart
// doc.getEndPosition().getOffset()
//doc.remove(0, doc.getLength());

//doc.insertString(0, throwable.getMessage() + "\n", doc.getStyle("bold"));
//doc.insertString(doc.getEndPosition().getOffset() - 1, throwable.getLocalizedMessage() + "\n",
//doc.getStyle("base"));
//doc.insertString(doc.getEndPosition().getOffset() - 1, String.format("%-20s%s%n", "", ste.toString()),
//doc.getStyle("base"));

/*
 * Die Methode schreibt den übergebenen {@link Throwable#getStackTrace()} in
 * eine &lt;ul&gt; des {@link HTMLDocument}s.
 * 
 * @param doc das {@link HTMLDocument}, in das der Text geschrieben werden soll.
 * 
 * @param elem das {@link Element} aus dem {@link HTMLDocument}, an dessen Ende
 * die Liste eingefügt werden soll.
 * 
 * @param stackTrace ein Array von {@link StackTraceElement}s
 * 
 * @return das Kind-{@link Element} mit der &lt;ul&gt;
 * 
 * @throws {@link BadLocationException} if the given position does not represent
 * a valid location in the associated document.
 * 
 * @throws {@link IOException} (kommentieren)
 * 
 * @throws {@link IllegalArgumentException} if <code>elem</code> is a leaf
 * 
 * @throws {@link IllegalStateException} if an HTMLEditorKit.Parser has not been
 * set on the document
 */
//protected Element baueStackTraceUL(HTMLDocument doc, Element elem, StackTraceElement[] stackTrace)
//		throws BadLocationException, IOException, IllegalArgumentException, IllegalStateException {
//	doc.insertBeforeEnd(elem, UL_START);
//	/* Um ins <ul> zu schreiben. */
//	Element ul = elem.getElement(elem.getElementCount() - 1);
//	for (StackTraceElement ste : stackTrace) {
//		doc.insertBeforeEnd(ul, String.format(LIST_ITEM, "&lt;" + ste.toString()));
//	}
//	return ul;
//}

/*
 * Die Methode schreibt die übergebenen {@link Throwable#getMessage()} und
 * {@link Throwable#getLocalizedMessage()} in eine Überschrift des &lt;ul&gt;
 * des {@link HTMLDocument}s.
 * 
 * @param doc das {@link HTMLDocument}, in das der Text geschrieben werden soll.
 * 
 * @param message die {@link Throwable#getMessage()}, wird zu einem
 * &lt;h3&gt;-Element
 * 
 * @param localizedMessage die {@link Throwable#getLocalizedMessage()}, wird zu
 * einer Unterüberschrift.
 * 
 * @return das {@link Element}, in das die Überschrift geschrieben wurde
 * 
 * @throws {@link BadLocationException} if the given position does not represent
 * a valid location in the associated document.
 * 
 * @throws {@link IOException} (kommentieren)
 * 
 * @throws {@link IllegalArgumentException} if <code>elem</code> is a leaf
 * 
 * @throws {@link IllegalStateException} if an HTMLEditorKit.Parser has not been
 * set on the document
 */
//protected Element baueFehlerUeberschrift(HTMLDocument doc, String message, String localizedMessage)
//		throws BadLocationException, IOException {
//	Element elem = doc.getRootElements()[0].getElement(0);
//	doc.setInnerHTML(elem, String.format(H3_HEAD, message));
//	if (!message.equals(localizedMessage) && localizedMessage != null) {
//		doc.insertBeforeEnd(elem, String.format(P_ELEMENT, localizedMessage));
//	}
//	return elem;
//}

// private String marshallFehler() {
// StringBuffer sb = new StringBuffer();
// if (throwable != null) {
// sb.append(throwable.getMessage() + "\n");
// sb.append(throwable.getLocalizedMessage() + "\n");
// for (StackTraceElement ste : throwable.getStackTrace()) {
// sb.append("\t" + ste.toString() + "\n");
// }
// }
// return sb;
// }
//
// Element elem =
// baueFehlerUeberschrift(doc, throwable.getMessage(),
// throwable.getLocalizedMessage());
// baueStackTraceUL(doc, elem, throwable.getStackTrace());
//private static final String P_ELEMENT = "<p class=\"text\">%s</p>";
//private static final String H3_HEAD = "<h3 class=\"fehler\">%s</h3>";

// , Throwable throwable @param throwable ein {@link Throwable}, um die {@link
// StackTraceElement}s auszulesen

/*
 * Die Methode schreibt die Message, {@link Throwable#getLocalizedMessage()} und
 * den {@link Throwable#getStackTrace() eines Fehlers in das übergebene {@link
 * HTMLDocument}.
 * 
 * @param doc das {@link HTMLDocument}, in das der Text geschrieben werden soll.
 * 
 * 
 * @return das übergebene {@link HTMLDocument} mit dem eingesetzten Text aus
 * {@link Throwable#getLocalizedMessage()} und {@link StackTraceElement}
 * 
 * @throws {@link BadLocationException} if the given position does not represent
 * a valid location in the associated document.
 * 
 * @throws {@link IOException} (kommentieren)
 * 
 * @throws {@link IllegalArgumentException} if <code>elem</code> is a leaf
 * 
 * @throws {@link IllegalStateException} if an HTMLEditorKit.Parser has not been
 * set on the document
 */
//private HTMLDocument baueFehlerText(HTMLDocument doc)
//		throws BadLocationException, IOException, IllegalArgumentException, IllegalStateException {
//	if (throwable != null) {
//		baueFehlerXmlHtml(doc);
//		throwable.printStackTrace();
//	}
//	return doc;
//}

/*
 * Die Methode gibt das erstellte {@link HTMLDocument} nach {@link System#err}
 * aus.
 * 
 * @param doc das {@link HTMLDocument}, in das der Text geschrieben werden soll.
 * 
 * @param kit das {@link HTMLEditorKit}, das dabei hilft.
 * 
 * @return einen {@link String} mit dem HTML-Code
 * 
 * @throws {@link BadLocationException} if the given position does not represent
 * a valid location in the associated document.
 * 
 * @throws {@link IOException} (kommentieren)
 */
//private String writeHTML(final HTMLDocument doc, HTMLEditorKit kit) throws IOException, BadLocationException {
//	StringWriter sw = new StringWriter();
//	kit.write(sw, doc, 0, doc.getLength() - 1);
//	final String string = sw.toString();
//	Kostenrechner.getLogger().info(string);
//	return string;
//}

/*
 * Die Methode schreibt den in XML konvertierten {@link Throwable} in eine
 * &lt;ul&gt; des {@link HTMLDocument}s.
 * 
 * @param doc das {@link HTMLDocument}, in das der Text geschrieben werden soll.
 * 
 * @return das Kind-{@link Element} mit der &lt;ul&gt;
 * 
 * @throws {@link BadLocationException} if the given position does not represent
 * a valid location in the associated document.
 * 
 * @throws {@link IOException} (kommentieren)
 */
//private Element baueFehlerXmlHtml(HTMLDocument doc)// , String message, String localizedMessage)
//		throws BadLocationException, IOException {
//	Element elem = doc.getRootElements()[0].getElement(0);
//	doc.setInnerHTML(elem, UL_START);
//	/* Um ins <ul> zu schreiben. */
//	Element ul = elem.getElement(elem.getElementCount() - 1);
//	/* nach Whitespace + < (Elementenanfang) splitten */
//	String[] xmls = baueXmlString(true).split("\\s+<");
//	for (String xmle : xmls) {
//		doc.insertBeforeEnd(ul, String.format(LIST_ITEM, "&lt;" + xmle));
//	}
//
//	return elem;
//}
//private static final String UL_START = "<ul class=\"fehler\">";// style=\"list-style-type:disc\">";
//private static final String LIST_ITEM = "<li>%s</li>";

/*
 * Die Methode schreibt die in XML konvertierten Message, {@link
 * Throwable#getLocalizedMessage()} und {@link Throwable#getStackTrace() eines
 * Fehlers in einen {@link StringBuffer}.
 * 
 * @param formatted ob das XML formatiert sein soll, {@link
 * Marshaller#JAXB_FORMATTED_OUTPUT}
 * 
 * @return {@link Throwable#getLocalizedMessage()} und {@link
 * StackTraceElement}s als XML in einem {@link StringBuffer}
 */
//public StringBuffer getFehlerText(boolean formatted) {
//	return new StringBuffer(baueXmlString(formatted).toString());
//}
//public String  getFehlerText(boolean formatted) {
//	return baueXmlString(formatted);
//}

// private String baueXmlString(boolean formatted) {