/*
 * StringConfigOption2.java
 * eu.gronos.kostenrechner.controller.system (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.system;

import eu.gronos.kostenrechner.model.beschriftungen.StringConfigSchluessel;

/**
 * Eine Klasse für den Zugriff auf Registry oder Benutzereinstellungen mit
 * {@link String}-Werten. Sie speichert den Ast
 * <code>/eu/gronos/kostentenor</code> und den Schlüssel. Jede Instanz
 * korrespondiert mit einem Boolean-Wert <code>schluessel</code> in der Registry
 * im Unterast <code>"/eu/gronos/kostentenor"</code> zum Java-Ast. Diese wird
 * zunächst eingelesen. Wenn der Schlüssel <code>schluessel</code> noch nicht
 * gesetzt ist, wird die als <code>standardWert</code> übergebene Zeichenkette
 * genommen.
 *
 * @author Peter Schuster (setrok)
 * @date 14 Jun 2019
 *
 */
public class StringConfigOption extends PreferencesConfigOption {
	private final String standardWert;

	/**
	 * Der Konstruktor erfordert den Wert für den finalen {@link String}
	 * {@link #getSchluessel()}
	 * 
	 * @param parameter ein {@link StringConfigSchluessel}
	 * 
	 */
	public StringConfigOption(StringConfigSchluessel parameter) {
		super(parameter.getSchluessel());
		this.standardWert = parameter.getStandardWert();
		setValue(ausRegistryLesen());
	}

	/**
	 * @return gibt {@link #standardWert} als String zurück.
	 */
	public String getStandardWert() {
		return standardWert;
	}

	/**
	 * Die Methode schreibt den Wert in die Registry/Preferences und setzt das
	 * Oberflächenelement entsprechend
	 * 
	 * @param string
	 */
	public void setValue(String string) {
		inRegistrySchreiben(string);
	}

	/**
	 * 
	 * Die Methode liest den Wert aus der Registry (den Preferences) und setzt das
	 * Oberflächenelement entsprechend
	 * 
	 * @return value als boolean
	 */
	public String getValue() {
		String value = ausRegistryLesen();
		return value;
	}

	/**
	 * Die Methode setzt wieder den Standardwert in die Registry/Preferences.
	 */
	public void reset() {
		setValue(getStandardWert());
	}

	/**
	 * Schreibt den übergebenen Wahrheitswert in den Java-Zweig der Registry, und
	 * zwar in den Schlüssel, der dem Konstruktor übergeben wurde.
	 * 
	 * @param wert
	 */
	private void inRegistrySchreiben(String wert) {
		getUserprefs().put(getSchluessel(), wert);
		logInfo(wert);
	}

	/**
	 * Liest einen Wert aus dem Java-Zweig der Registry, unter dem Schluessel, der
	 * dem Konstruktor mitgegeben wurde.
	 * 
	 * @return einen {@link String}
	 */
	private String ausRegistryLesen() {
		return getUserprefs().get(getSchluessel(), getStandardWert());
	}

}
