/**
 * TenorUtils.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.logic;

import java.util.Arrays;
import java.util.List;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.interfaces.UnterContainerKlasse;
import eu.gronos.kostenrechner.model.gebuehren.GebuehrenGesetzesSammlung;
import eu.gronos.kostenrechner.model.gebuehren.GebuehrenVerzeichnis;
import eu.gronos.kostenrechner.model.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.model.tenordaten.HauptsacheEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.KostenGrundEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.KostenTragungsVerhaeltnis;
import eu.gronos.kostenrechner.model.tenordaten.StreitwertEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.VollstreckbarkeitsEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.VollstreckbarkeitsListe;
import eu.gronos.kostenrechner.model.tenordaten.VollstreckungsVerhaeltnis;

/**
 * Oberklasse für Klassen, die Tenortexte schreibt.
 * 
 * Hält Methoden zur Tenorerzeugung bereit. Sie gibt aber auch noch ein paar
 * Konstanten.
 * 
 * @author Peter Schuster (setrok)
 * @version 19.06.2014
 * 
 */
public abstract class TenorTexter<E extends UnterContainerKlasse> {

	public static final String KLAGEABWEISUNG = "Die Klage wird abgewiesen.";
	public static final String EURO_FORMAT = "%,.2f EUR";
	/**
	 * Kostentatbestände, die üblicherweise auf Klägerseite anfallen:
	 * Gerichtskosten, Verfahrensgebühr, Terminsgebühr.
	 */
	private static final /* GebuehrenTatbestand[] */ GebuehrenVerzeichnis KOSTEN_KLAEGER = new GebuehrenVerzeichnis();
	/**
	 * Kostentatbestände, die üblicherweise auf Beklagtenseite anfallen:
	 * Verfahrensgebühr, Terminsgebühr (keine Gerichtskosten).
	 */
	private static final /* GebuehrenTatbestand[] */ GebuehrenVerzeichnis KOSTEN_BEKLAGTER = new GebuehrenVerzeichnis();
	public static final String KOSTENENTSCHEIDUNG_92_II = "Die Kostenentscheidung beruht auf § 92 Abs. 1 ZPO.\n";
	public static final String KOSTENENTSCHEIDUNG_91_92_II = "Die Kostenentscheidung beruht auf §§ 91, 92 Abs. 2 ZPO.\n";
	public static final String KOSTENENTSCHEIDUNG_91 = "Die Kostenentscheidung beruht auf § 91 ZPO.\n";
	public static final String GRUENDE_UEBERSCHRIFT = "Entscheidungsgründe\n";
	public static final String UND = " und ";
	public static final String KOMMA = ", ";
	public static final String HABEN = " haben ";
	public static final String HAT = " hat ";
	public static final String WERDEN = "werden";
	public static final String WIRD = "wird";
	public static final String FEHLER_FORDERUNG_LEER = "Die Liste der Forderungen darf nicht null oder leer sein!";
	public static final String FEHLER_REIHENFOLGE_FORDERUNGEN = "Die Reihenfolge (erst Klageforderungen, dann Aufrechnungsforderungen) ist nicht eingehalten!";
	public static final String FEHLER_REIHENFOLGE_STREITWERTE = "Der spätere Streitwert darf nicht größer als der frühere sein!";
	public static final String FEHLER_REIHENFOLGE_VERURTEILUNG = "Die Verurteilung darf nicht größer als der spätere Streitwert sein!";
	public static final String FEHLER_GAR_NIX_PARTEIEN = "baumbachBeteiligtenListe darf nicht null und nicht leer sein!";
	public static final String FEHLER_NIX_PARTEIEN = "Die BaumbachBeteiligtenListe baumbachBeteiligtenListe muss mindestens einen KLAEGER und einen BEKLAGTEn enthalten!";
	public static final String FEHLER_NIX_WIDERBEKLAGTE = "Es wurde ein widerklagender Beklagter, aber kein widerbeklagter Kläger / Drittwiderbeklagter gefunden.";
	public static final String FEHLER_NIX_WIDERKLAEGER = "Es wurden widerbeklagte Kläger / Drittwiderbeklagte, aber kein widerklagender Beklagter gefunden.";
	public static final String FEHLER_NUR_EINER = "Derzeit wird nicht mehr als ein Kläger bzw. ein widerklagender Beklagter unterstützt.";
	public static final String FEHLER_REIHENFOLGE_BETEILIGTE = "Die Beteiligten müssen in der Reihenfolge Kläger, Drittwiderbeklagte (sofern vorhanden), Beklagte übergeben werden. \nBitte Programmierer benachrichtigen!";
	public static final String STREITWERT_KLAGE_WIDERKLAGE = "Der Streitwert wird auf %,.2f EUR, für die Klage auf %,.2f EUR, für die Widerklage auf %,.2f EUR festgesetzt.%n";
	public static final String STREITWERT_KLAGE = "Der Streitwert wird auf %,.2f EUR festgesetzt.%n";
	public static final String STREITWERTFESTSETZUNG_FORMAT = "Der Streitwert wird auf %,.2f EUR bis zum ..., danach auf %,.2f EUR festgesetzt.%n";
	public static final String STREITWERT_FESTGESETZT_EINFACH = "Der Streitwert wird auf %,.2f EUR festgesetzt.%n";
	public static final String STREITWERT_FESTGESETZT_VERGANGENHEIT = "Der Streitwert wurde auf %,.2f EUR festgesetzt.%n";
	public static final String ZU_TRAGEN = " zu tragen.\n";
	public static final String PARTEI_ZU_PROZENT = "%s zu %,.2f%%, ";
	public static final String PARTEI_PROZENT_OHNE = "%s %,.2f%%, ";
	public static final String KOSTENTRAGUNG_SELBST = "Im Übrigen haben die Parteien ihre außergerichtlichen Kosten selbst zu tragen.\n";
	public static final String AUSSERGERICHTLICHE_KOSTEN_TEIL = ", die außergerichtlichen Kosten ";
	public static final String DIE_AUSSERG_KOSTEN = "Die außergerichtlichen Kosten %s haben ";
	public static final String DIE_GERICHTSKOSTEN_HABEN = "Die Gerichtskosten haben ";
	public static final String DIE_GERICHTSKOSTEN = "Die Gerichtskosten";
	public static final String DIE_KOSTEN_DES_RECHTSSTREITS_HABEN = "Die Kosten des Rechtsstreits haben ";
	public static final String KOSTENVERTEILUNG_FORMAT = "Die Kosten haben %s zu %,.2f%% und %s zu %,.2f%% zu tragen. %n";
	public static final String KOSTENTRAGUNG_EINSEITIG = "Die Kosten hat %s zu tragen.%n";
	public static final String GESAMTSCHULDNERISCH_RICHTIG = "werden verurteilt, gesamtschuldnerisch";
	public static final String GESAMTSCHULDNERISCH_FALSCH = "gesamtschuldnerisch werden verurteilt,";
	public static final String GESAMTSCHULDNERISCH_DARUEBER_RICHTIG = "werden darüber hinaus verurteilt, gesamtschuldnerisch";
	public static final String GESAMTSCHULDNERISCH_DARUEBER_FALSCH = "gesamtschuldnerisch werden darüber hinaus verurteilt,";
	public static final String DARUEBER_HINAUS = " darüber hinaus";
//	public static final String VERURTEILUNG_ZAHLUNG_AN_GGF_TEILABWEISUNG_FORMAT = "%s wird verurteilt, an %s %,.2f EUR zu zahlen.%s%n";
	public static final String VERURTEILUNG_ZAHLUNG_AN_GGF_TEILABWEISUNG_FORMAT = "%s %s verurteilt, an %s %,.2f EUR zu zahlen.%s%n";
	public static final String VERURTEILUNGS_FORMAT = "%s wird verurteilt, %s %,.2f EUR zu zahlen. ";
	public static final String KLAGEABWEISUNG_IM_UEBRIGEN_LEERSCHRITT = " Im Übrigen wird die Klage abgewiesen.";
	public static final String KLAGEABWEISUNG_IM_UEBRIGEN = "Im Übrigen wird die Klage abgewiesen.";
	public static final String KLAGEABWEISUNG_ZEILE = "Die Klage wird abgewiesen.\n";
	public static final String URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709 = "Das Urteil ist gegen Sicherheitsleistung in Höhe von 110% des jeweils zu vollstreckenden Betrags vorläufig vollstreckbar.\n";
	public static final String URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709_KOMBI = "Gegenüber %s ist das Urteil gegen Sicherheitsleistung in Höhe von 110%% des jeweils zu vollstreckenden Betrags vorläufig vollstreckbar.\n";
	public static final String URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711 = "%s wird nachgelassen, die Vollstreckung gegen Sicherheitsleistung in Höhe von 110%% des aufgrund des Urteils vollstreckbaren Betrags abzuwenden, wenn nicht %s vor der Vollstreckung Sicherheit in Höhe von 110%% des jeweils zu vollstreckenden Betrags leistet.\n";
	public static final String URTEIL_VORLAEUFIG_VOLLSTRECKBAR = "Das Urteil ist vorläufig vollstreckbar.\n";
	public static final int BERUFUNGS_GRENZE = 600;
	public static final String JEWEILIGER_VOLLSTRECKUNGSGLAEUBIGER = "der jeweilige Vollstreckungsgläubiger";
	public static final String JEWEILIGER_VOLLSTRECKUNGSSCHULDNER = "dem jeweiligen Vollstreckungsschuldner";
	public static final String URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711_WERIST = "%s wird %s nachgelassen, die Vollstreckung gegen Sicherheitsleistung in Höhe von 110%% des aufgrund des Urteils vollstreckbaren Betrags abzuwenden, wenn nicht %s vor der Vollstreckung Sicherheit in Höhe von 110%% des jeweils zu vollstreckenden Betrags leistet.\n";
	public static final String URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709_EINZELN = " ist es gegen Sicherheitsleistung in Höhe von 110% des jeweils zu vollstreckenden Betrags vorläufig vollstreckbar.\n";

	/**
	 * Konstruktor: nix Konstruktor (private Sichtbarkeit)
	 */
	protected TenorTexter() {
		// nix Konstruktor
		super();
	}

	/**
	 * Die Methode soll einen Hauptsachetenor verfassen und den übergebenen
	 * {@link HauptsacheEntscheidungsElemente}n.
	 * 
	 * @param hauptsache ein {@link HauptsacheEntscheidungsElemente}-Objekt, in dem
	 *                   {@link HauptsacheEntscheidungsElemente#prozessverhaeltnisse}
	 *                   gefüllt ist.
	 * @return einen Hauptsachetenor als String mit abschließendem Zeilenumbruch für
	 *         {@link HauptsacheEntscheidungsElemente#text}
	 */
	abstract public String texteHauptsache(HauptsacheEntscheidungsElemente hauptsache);

	/**
	 * Die Methode soll die Kostenentscheidung verfassen.
	 * 
	 * @param elemente {@link KostenGrundEntscheidungsElemente}
	 * @return einen {@link String} mit dem Tenortext und abschließendem
	 *         Zeilenumbruch.
	 */
	public String texteKostenentscheidung(KostenGrundEntscheidungsElemente elemente) {
		final List<KostenTragungsVerhaeltnis> verhaeltnisse = elemente.prozessverhaeltnisse;
		if (verhaeltnisse.isEmpty())
			throw new IllegalArgumentException("elemente darf nicht leer sein!");
		if (verhaeltnisse.size() == 1) {
			Beteiligter schuldner = verhaeltnisse.get(0).getSchuldner();
			return String.format(KOSTENTRAGUNG_EINSEITIG,
					schuldner.parteiBezeichner(Beteiligter.NOMINATIV, schuldner.getLfdNr(), true));
		} else {
			KostenTragungsVerhaeltnis bkkl = elemente.prozessverhaeltnisse.get(0);
			KostenTragungsVerhaeltnis klbk = elemente.prozessverhaeltnisse.get(1);
			return String.format(KOSTENVERTEILUNG_FORMAT,
					bkkl.getSchuldner().parteiBezeichner(Beteiligter.NOMINATIV, bkkl.getSchuldner().getLfdNr(), true),
					bkkl.getKostenBruchteil() * 100,
					klbk.getSchuldner().parteiBezeichner(Beteiligter.NOMINATIV, klbk.getSchuldner().getLfdNr(), true),
					klbk.getKostenBruchteil() * 100);
		}
	}

	/**
	 * Die Methode erzeugt den Tenor zur vorläufigen Vollstreckbarkeit
	 * 
	 * @param elemente {@link VollstreckbarkeitsEntscheidungsElemente}
	 * @param liste    eine {@link VollstreckbarkeitsListe}
	 * @return einen {@link String}
	 */
	public String texteVollstreckbarkeit(VollstreckbarkeitsEntscheidungsElemente elemente,
			VollstreckbarkeitsListe liste) {
		String schuldner = JEWEILIGER_VOLLSTRECKUNGSSCHULDNER;
		String glaeubiger = JEWEILIGER_VOLLSTRECKUNGSGLAEUBIGER;
		if (liste.sindAlle708iVm713()) {
			return URTEIL_VORLAEUFIG_VOLLSTRECKBAR;
			// elemente.text =; return vollEntscheidung;
		} else if (liste.sindAlle709()) {
			return URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709;
			// ;elemente.text =return vollEntscheidung;
		} else if (liste.sindAlle708iVm711()) {
			/*
			 * Parteien einsetzen: Wenn die Liste der Vollstreckungsverhaeltnisse nur einen
			 * Eintrag hat, darf nur eine Partei vollstrecken. Da hilft die Liste der
			 * VollstreckungsVerhältnisse.
			 */
			if (1 == elemente.prozessverhaeltnisse.size()) {
				VollstreckungsVerhaeltnis verhaeltnis = elemente.prozessverhaeltnisse.get(0);
				schuldner = verhaeltnis.getSchuldner().parteiBezeichner(Beteiligter.DATIV,
						verhaeltnis.getSchuldner().getLfdNr(), true);
				glaeubiger = verhaeltnis.getGlaeubiger().parteiBezeichner(Beteiligter.NOMINATIV,
						verhaeltnis.getGlaeubiger().getLfdNr(), true);
			}
			/*
			 * Ansonsten dürfen beide vollstrecken (und es bleibt beim
			 * "jeweiligen Vollstreckungsschuldner-/gläubiger".
			 */
			return URTEIL_VORLAEUFIG_VOLLSTRECKBAR
					+ satzBeginn(String.format(URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711, schuldner, glaeubiger));
			// return vollEntscheidung; elemente.text =
		}
		/*
		 * Gemischter Vollstreckbarkeitstenor: die VollstreckungsVerhaeltnisse
		 * durchgehen. Dafür haben wir sie ja.
		 */
		StringBuffer tenor = new StringBuffer(URTEIL_VORLAEUFIG_VOLLSTRECKBAR);
		for (VollstreckungsVerhaeltnis verhaeltnis : elemente.prozessverhaeltnisse) {
			glaeubiger = verhaeltnis.getGlaeubiger().parteiBezeichner(Beteiligter.NOMINATIV,
					verhaeltnis.getGlaeubiger().getLfdNr(), true);
			schuldner = verhaeltnis.getSchuldner().parteiBezeichner(Beteiligter.DATIV,
					verhaeltnis.getSchuldner().getLfdNr(), true);
			if (verhaeltnis.getVollstreckbarkeit().isB709s1()) {
				tenor.append(String.format(URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709_KOMBI, schuldner));
			} else {
				tenor.append(satzBeginn(
						String.format(URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711, satzBeginn(schuldner), glaeubiger)));
			}
		}
//		elemente.text = 
		return tenor.toString();
	}

	/**
	 * Die Methode verfasst eine Streitwertfestsetzung und schreibt sie.
	 * 
	 * @param elemente {@link StreitwertEntscheidungsElemente}, in dem
	 *                 {@link StreitwertEntscheidungsElemente#streitwerte} gefüllt
	 *                 ist.
	 * @return ein {@link String} für in
	 *         {@link StreitwertEntscheidungsElemente#streitwerte#text}
	 */
	abstract public String texteStreitwert(StreitwertEntscheidungsElemente elemente);

	/**
	 * Die Methode hatStattHaben dient dazu, in einem String das Wort haben durch
	 * das Wort hat zu ersetzen, wenn nur einer die Kosten trägt. Das erkennt die
	 * Methode daran, dass weder ein Komma ", " noch ein "und" in dem Satz vorkommt.
	 * 
	 * @param satz ein String mit dem Satz, um den es geht.
	 * @return der gleiche Satz, unter Berücksichtigung des Singulars, wo es gerade
	 *         passt.
	 */
	public String hatStattHaben(String satz) {
		if (!satz.contains(TenorTexter.KOMMA) && !satz.contains(TenorTexter.UND) && satz.contains(TenorTexter.HABEN)) {
			final int wo = satz.indexOf(TenorTexter.HABEN);
			final String davor = satz.substring(0, wo);
			final String danach = satz.substring(wo + TenorTexter.HABEN.length());
			return davor + TenorTexter.HAT + danach;
		} else
			return satz;
	}

	/**
	 * Methode, um den ersten Buchstaben in einen Großbuchstaben umzuwandeln.
	 * 
	 * @param satz ein String mit dem Satz, um dessen Anfang es geht.
	 * @return den gleichen String, aber mit einem Großbuchstaben am Anfang.
	 */
	public String satzBeginn(String satz) {
		String zwischen = "";
		if (satz != null && satz.length() > 0)
			zwischen = satz.substring(0, 1).toUpperCase() + satz.substring(1);
		return zwischen;
	}

	/**
	 * Die Methode entfernt im übergebenen <code>satz</code> das letzte Komma,
	 * sofern dieser auf ein Komma endet.
	 * 
	 * @param satz eine Zeichenkette als String
	 * @return denselben String, ohne ein etwaiges Komma am Ende.
	 */
	public static String letztesKommaWeg(String satz) {
		String zwischen = satz;
		if (satz != null && satz.trim().endsWith(","))
			zwischen = satz.substring(0, satz.lastIndexOf(","));
		return zwischen;
	}

	/**
	 * Die Methode ersetzt das letzte Vorkommen von ", " durch ein " und ".
	 * 
	 * @param satz eine Zeichenkette als String
	 * @return
	 */
	public static String undStattKomma(String satz) {
		String zwischen = satz;
		int j = satz.lastIndexOf(TenorTexter.KOMMA);// (", ");
		if (j > 0 && j < satz.length()) {
			zwischen = satz.substring(0, j) + TenorTexter.UND + satz.substring(j + 2);
		}
		return zwischen;
	}

	/**
	 * Die Methode entscheidet, ob wird oder werden als Pluralform angemessen ist.
	 * 
	 * @param plural true, wenn die Pluralform verwendet werden soll
	 * @return "wird" oder "werden" als String
	 */
	public String getWirdWerden(boolean plural) {
		return plural ? TenorTexter.WERDEN : TenorTexter.WIRD;
	}

	/**
	 * Die Methode verändert die Satzstellung bei gesamtschuldnerischer Verurteilung
	 * 
	 * @param satz der Satz, der verbessert werden soll, als String
	 * @return derselbe Satz mit verbesserter Wortstellung
	 */
	public String verbessereGesamtschuldner(String satz) {
		return satz.replace(TenorTexter.GESAMTSCHULDNERISCH_FALSCH, TenorTexter.GESAMTSCHULDNERISCH_RICHTIG).replace(
				TenorTexter.GESAMTSCHULDNERISCH_DARUEBER_FALSCH, TenorTexter.GESAMTSCHULDNERISCH_DARUEBER_RICHTIG);
	}

	/**
	 * Hänge einem Array einen Wert an
	 * 
	 * Java, Mehr als eine Insel, Kapitel 3.8.19: Hänge zwei Arrays aneinander. Das
	 * ist ein gutes Beispiel für copyOf(), wenn das Zielfeld größer ist. Die Klasse
	 * Arrays bietet eine Reihe von copyOf()- bzw. copyOfRange()-Methoden, die
	 * gegenüber clone() den Vorteil haben, dass sie auch Bereichsangaben erlauben
	 * und das neue Feld größer machen können; im letzten Fall füllen die Methoden
	 * das Feld je nach Typ mit null, false oder 0.
	 * 
	 * @url "http://openbook.galileocomputing.de/javainsel/javainsel_03_008.html"
	 */
	public static Object[] appendArray(Object[] objekte, Object wert) {
		Object[] array = Arrays.copyOf(objekte, objekte.length + 1);
		System.arraycopy(new Object[] { wert }, 0, array, objekte.length, 1);
		return array;
	}

	/**
	 * Java, Mehr als eine Insel, Kapitel 3.8.19: Hänge zwei Arrays aneinander. Das
	 * ist ein gutes Beispiel für copyOf(), wenn das Zielfeld größer ist. Die Klasse
	 * Arrays bietet eine Reihe von copyOf()- bzw. copyOfRange()-Methoden, die
	 * gegenüber clone() den Vorteil haben, dass sie auch Bereichsangaben erlauben
	 * und das neue Feld größer machen können; im letzten Fall füllen die Methoden
	 * das Feld je nach Typ mit null, false oder 0.
	 * 
	 * @url "http://openbook.galileocomputing.de/javainsel/javainsel_03_008.html"
	 */
	public static Object[] concat(Object[] eins, Object[] zwei) {
		Object[] drei = Arrays.copyOf(eins, eins.length + zwei.length);
		System.arraycopy(zwei, 0, drei, eins.length, zwei.length);
		return drei;
	}

	/**
	 * @return gibt {@link #kostenKlaeger} als {@link GebuehrenVerzeichnis} zurück.
	 */
	public static /* GebuehrenTatbestand[] */ GebuehrenVerzeichnis getKostenKlaeger() {
		/*
		 * new GebuehrenTatbestand [ ] { KostenVerzeichnis . KV1210,
		 * VerguetungsVerzeichnis . VV3100, VerguetungsVerzeichnis . VV3104 };
		 */
		if (KOSTEN_KLAEGER.isEmpty()) {
			KOSTEN_KLAEGER.addAll(GebuehrenGesetzesSammlung.getGkg().getVerzeichnis().getAll("1210 KV"));
			KOSTEN_KLAEGER.addAll(GebuehrenGesetzesSammlung.getRvg().getVerzeichnis().getAll("3100 VV", "3104 VV"));
			Kostenrechner.getLogger().info("KostenKlaeger gesetzt!");
		}

		return KOSTEN_KLAEGER;
	}

	/**
	 * @return gibt {@link #kostenBeklagter} als {@link GebuehrenVerzeichnis}
	 *         zurück.
	 */
	public static /* GebuehrenTatbestand[] */ GebuehrenVerzeichnis getKostenBeklagter() {
		/*
		 * new GebuehrenTatbestand[] { VerguetungsVerzeichnis.VV3100,
		 * VerguetungsVerzeichnis.VV3104 };
		 */
		if (KOSTEN_BEKLAGTER.isEmpty()) {
			KOSTEN_BEKLAGTER.addAll(GebuehrenGesetzesSammlung.getRvg().getVerzeichnis().getAll("3100 VV", "3104 VV"));
			Kostenrechner.getLogger().info("KostenBeklagter gesetzt!");
		}
		return KOSTEN_BEKLAGTER;
	}

}
