/**
 * BaumbachFormel.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.logic.baumbach;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.TabulierendZeilen;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenPruefer;
import eu.gronos.kostenrechner.interfaces.Begruendend;
import eu.gronos.kostenrechner.interfaces.ParsendUndBauend;
import eu.gronos.kostenrechner.interfaces.TenorVorbereitend;
import eu.gronos.kostenrechner.interfaces.UnterContainerKlasse;
import eu.gronos.kostenrechner.logic.TenorTexter;
import eu.gronos.kostenrechner.logic.VollstreckbarkeitsHelfer;
import eu.gronos.kostenrechner.model.baumbach.BaumbachBeteiligtenListe;
import eu.gronos.kostenrechner.model.baumbach.BaumbachBeteiligter;
import eu.gronos.kostenrechner.model.baumbach.BaumbachGesamtschuldnerschaft;
import eu.gronos.kostenrechner.model.baumbach.BaumbachVollstreckbarkeit;
import eu.gronos.kostenrechner.model.baumbach.GerichtsKostenBeteiligter;
import eu.gronos.kostenrechner.model.baumbach.Streitgenossen;
import eu.gronos.kostenrechner.model.gebuehren.GebuehrenVerzeichnis;
import eu.gronos.kostenrechner.model.tenordaten.Allgemein;
import eu.gronos.kostenrechner.model.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.model.tenordaten.DoubleDataRows;
import eu.gronos.kostenrechner.model.tenordaten.HauptsacheEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.HauptsacheVerhaeltnis;
import eu.gronos.kostenrechner.model.tenordaten.KostenGrundEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.KostenTragungsVerhaeltnis;
import eu.gronos.kostenrechner.model.tenordaten.SonstigeEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.StreitwertEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.TenorDatenContainer;
import eu.gronos.kostenrechner.model.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.model.tenordaten.Vollstreckbarkeit;
import eu.gronos.kostenrechner.model.tenordaten.VollstreckbarkeitsEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.VollstreckbarkeitsListe;
import eu.gronos.kostenrechner.model.tenordaten.VollstreckungsVerhaeltnis;

/**
 * Der Tenor f&uuml;r <b>Teilunterliegen bei Streitgenossenschaft</b> kann nach
 * der Baumbach&prime;schen Kostenformel erzeugt werden.
 * 
 * @author Peter Schuster (setrok)
 * @date 29.04.2014
 * 
 */
public class BaumbachFormel implements TenorVorbereitend, Begruendend {

	private static final String BESCHREIBUNG = "Tenor bei Streitgenossen nach der sog. Baumbach'schen Formel (Hauptsachetenor, Kostentenor, Vollstreckbarkeitstenor sowie Streitwertbeschluss).";
	private static final String GRUENDE_91_92_100II = "Die Kostenentscheidung beruht auf §§ 91, 92, 100 Abs. 2 ZPO und der sogenannten Baumbach'schen Kostenformel,"
			+ " entsprechend der unterschiedlichen Beteiligung am Rechtsstreit: \n";
	private static final String KOPFZEILE_BEGINN = "Einzelangriffe / Verluste";
	private static final String ERGEBNIS_ZEILE = "Kostenquote hinsichtlich eines fiktiven Streitwerts von %,.2f EUR";
	private static final String VERLUST_GESAMT = "Verlust gesamt";
	final BaumbachBeteiligtenListe baumbachBeteiligtenListe;
	// private boolean b92ii;
	private boolean wirtschaftlichIdentisch;
	private double[][] kosten = null;
	private final StringBuilder gruende = new StringBuilder();
	private TabulierendZeilen zeilen;
	private VollstreckbarkeitsListe vollstreckbarkeit;
	private final TenorDatenContainer container;
	private final BaumbachTexter texter;// = new BaumbachTexter();
	private ParsendUndBauend<Streitgenossen> pruefer = new StreitgenossenPruefer(true);
	/**
	 * in der {@link UnterContainerKlasse} Allgemein stehen {@link Allgemein#b92ii}
	 * und {@link Allgemein#unwesentlich92ii.}
	 */
	private Allgemein allgemein;

	/**
	 * Konstruktor:
	 * 
	 * @param allgemein                {@link VerfahrensDatenContainer#allgemein}
	 * @param streitgenossen           {@link VerfahrensDatenContainer#streitgenossen}
	 * @param baumbachBeteiligtenListe ein Array BaumbachBeteiligter[] in der
	 *                                 Reihenfolge KLAEGER, DRITTWIDERBEKLAGTE
	 *                                 (falls vorhanden) und BEKLAGTE, mit
	 *                                 mindestens einem KLAEGER und einem BEKLAGTEn.
	 * @param b92iiZpo                 Soll § 92 II ZPO angewandt werden
	 *                                 (verhältnismäßig geringe Kostentragungen
	 *                                 werden unterschlagen)
	 * @param wirtschaftlichIdentisch  ist eine etwaige Widerklage wirtschaftlich
	 *                                 identisch mit der Hauptklageforderung?
	 * 
	 * @throws IllegalArgumentException wird geworfen, wenn eine der Vorgaben nicht
	 *                                  eingehalten wurde.
	 */
	public BaumbachFormel(VerfahrensDatenContainer verfahrensDaten) throws IllegalArgumentException {
		super();
		/*
		 * aus den drei Listen im Parameterobjekt eine BaumbachBeteiligtenListe bauen,
		 * in der Reihenfolge Kläger, Drittwiderbeklagte, Beklagte
		 */
		Streitgenossen streitgenossen = verfahrensDaten.streitgenossen;
		pruefer.parseEingabe(streitgenossen);
		baumbachBeteiligtenListe = new BaumbachBeteiligtenListe();
		baumbachBeteiligtenListe.addAll(streitgenossen.klaeger);
		/* Drittwiderbeklagte nur hinzufügen, wenn vorhanden */
		if (streitgenossen.drittwiderbeklagte != null && streitgenossen.drittwiderbeklagte.size() > 0)
			baumbachBeteiligtenListe.addAll(streitgenossen.drittwiderbeklagte);
		baumbachBeteiligtenListe.addAll(streitgenossen.beklagte);
		// this.b92ii = .b92ii;
		this.allgemein = verfahrensDaten.allgemein;
		this.wirtschaftlichIdentisch = streitgenossen.wirtschaftlichIdentisch;
		this.container = new TenorDatenContainer(verfahrensDaten);
		this.texter = new BaumbachTexter(baumbachBeteiligtenListe);
	}
	// Allgemein allgemein, Streitgenossen streitgenossen
	// allgemein, streitgenossen);
	// .b92iiZpo;
	/* Kläger hinzufügen (wenn vorhanden), sonst Fehler */
// if (streitgenossen.klaeger != null && streitgenossen.klaeger.size() > 0)
// else
// throw new IllegalArgumentException(TenorTexter.FEHLER_NIX_PARTEIEN);
	/* Beklagte hinzufügen, wenn vorhanden, sonst Fehler */
	// if (streitgenossen.beklagte != null && streitgenossen.beklagte.size() > 0)
	// else
	// throw new IllegalArgumentException(TenorTexter.FEHLER_NIX_PARTEIEN);
// if (baumbachBeteiligtenListe.zaehleAnWiderklageBeteiligte(Beteiligter.BEKLAGTE) > 0
// && (baumbachBeteiligtenListe.zaehleAnWiderklageBeteiligte(Beteiligter.KLAEGER)
// + baumbachBeteiligtenListe.zaehleAnWiderklageBeteiligte(Beteiligter.DRITTWIDERBEKLAGTE)) <= 0)
// throw new IllegalArgumentException(TenorTexter.FEHLER_NIX_WIDERBEKLAGTE);
// if (baumbachBeteiligtenListe.zaehleAnWiderklageBeteiligte(Beteiligter.BEKLAGTE) <= 0
// && (baumbachBeteiligtenListe.zaehleAnWiderklageBeteiligte(Beteiligter.KLAEGER)
// + baumbachBeteiligtenListe.zaehleAnWiderklageBeteiligte(Beteiligter.DRITTWIDERBEKLAGTE)) > 0)
// throw new IllegalArgumentException(TenorTexter.FEHLER_NIX_WIDERKLAEGER);
	// TODO (P) wenn ein BaumbachGesamtschuldner auf Bk-Seite
	// widerklagebeteiligt ist!
// if (baumbachBeteiligtenListe.zaehleKlaeger() > 1
// || baumbachBeteiligtenListe.zaehleAnWiderklageBeteiligte(Beteiligter.BEKLAGTE) > 1)
// throw new IllegalArgumentException(TenorTexter.FEHLER_NUR_EINER);
// if (!baumbachBeteiligtenListe.istReihenfolgeEingehalten())
// throw new IllegalArgumentException(TenorTexter.FEHLER_REIHENFOLGE_BETEILIGTE);

	/**
	 * Die Methode erzeugt den Hauptsachetenor, davon ausgehend, dass die
	 * Verurteilungen auf Zahlung lauten.
	 * 
	 * @return den Hauptsachetenor als {@link HauptsacheEntscheidungsElemente}, vor
	 *         allem den Tenor als String mit abschließendem Zeilenumbruch
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeHauptsacheEntscheidung()
	 */
	@Override
	public HauptsacheEntscheidungsElemente erzeugeHauptsacheEntscheidung() {
		HauptsacheEntscheidungsElemente hauptsache = new HauptsacheEntscheidungsElemente();
		hauptsache.prozessverhaeltnisse = baueHauptsacheVerhaeltnisse();
		texter.setImUebrigen(baueImUebrigen());
//		hauptsache.text = baueHauptsacheTenor(hauptsache.prozessverhaeltnisse, imUebrigen);
		hauptsache.text = texter.texteHauptsache(hauptsache);
		return hauptsache;
	}

	/**
	 * Gibt den ganzen Kostentenor.
	 * 
	 * @return den Kostentenor als {@link KostenGrundEntscheidungsElemente}, vor
	 *         allem den Text des Kostentenors mit abschließendem Zeilenumbruch
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeKostenEntscheidung()
	 */
	@Override
	public KostenGrundEntscheidungsElemente erzeugeKostenEntscheidung() {
		KostenGrundEntscheidungsElemente kostenEntscheidung = new KostenGrundEntscheidungsElemente();
		final double[][] quoten = getKosten();
		kostenEntscheidung.prozessverhaeltnisse = baueKostenTragungsVerhaeltnisse(quoten);

		texter.setQuoten(quoten);
		kostenEntscheidung.text = texter.texteKostenentscheidung(kostenEntscheidung);

		return kostenEntscheidung;
	}

	/**
	 * Die Methode erzeugt den Tenor zur vorläufigen Vollstreckbarkeit.
	 * 
	 * @return einen {@link VollstreckbarkeitsEntscheidungsElemente}
	 */
	@Override
	public VollstreckbarkeitsEntscheidungsElemente erzeugeVollstreckbarkeitsEntscheidung() {
		VollstreckbarkeitsEntscheidungsElemente vollEntscheidung = new VollstreckbarkeitsEntscheidungsElemente();
		if (vollstreckbarkeit == null)
			vollstreckbarkeit = berechneVollstreckbarkeit();
		erweitereGruende(vollstreckbarkeit.toString());
		vollEntscheidung.prozessverhaeltnisse = baueVollstreckungsVerhaeltnisse(vollstreckbarkeit);
		vollEntscheidung.text = texter.texteVollstreckbarkeit(vollEntscheidung, vollstreckbarkeit);
		return vollEntscheidung;
	}

	/**
	 * Die Methode erzeugt eine Streitwertfestsetzung, indem der Streitwert auf den
	 * höchsten Einzelstreitwert der baumbachBeteiligtenListe festgesetzt wird.
	 * 
	 * @return {@link StreitwertEntscheidungsElemente}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.Begruendend#erzeugeStreitwertEntscheidung()
	 */
	@Override
	public StreitwertEntscheidungsElemente erzeugeStreitwertEntscheidung() {
		StreitwertEntscheidungsElemente streitwert = new StreitwertEntscheidungsElemente();
		if (baumbachBeteiligtenListe.enthaeltAnWiderklageBeteiligte()) {
			double[] streitwerte = errechneStreitwert(true);
			// Bei einer Widerklage müssen beide Streitwerte ausgegeben werden
			for (double sw : streitwerte) {
				streitwert.streitwerte.add(new Long(new Double(sw).longValue()));
			}
		} else {
			// Ohne Widerklage muss nur ein Streitwert ausgegeben werden
			streitwert.streitwerte.add(new Long(
					new Double(baumbachBeteiligtenListe.getHoechstenStreitwert(Beteiligter.BEKLAGTE)).longValue()));
		}
		streitwert.text = texter.texteStreitwert(streitwert);
		return streitwert;
	}

	/**
	 * Die Methode erzeugt einen leeren sonstigen Tenor.
	 * 
	 * @return ein {@link SonstigeEntscheidungsElemente}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeSonstigeEntscheidung()
	 */
	@Override
	public SonstigeEntscheidungsElemente erzeugeSonstigeEntscheidung() {
		SonstigeEntscheidungsElemente sonstige = new SonstigeEntscheidungsElemente();
		sonstige.text = "";
		return sonstige;
	}

	/**
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#getBerechnungsTyp()
	 */
	@Override
	public String getBerechnungsTyp() {
		return BESCHREIBUNG;
	}

	/**
	 * Die Methode gibt die Begründung für den Kostentenor als String zurück.
	 * 
	 * @return den Text Gründe/Entscheidungsgründe als String
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.Begruendend#getGruende()
	 */
	@Override
	public String getGruende() {
		return gruende.toString();
	}

	/**
	 * Die Methode erweitert die bisherigen Gründe
	 * 
	 * @param text eine {@link CharSequence} mit den weiteren Gründen.
	 */
	public void erweitereGruende(CharSequence text) {
		gruende.append(text);
	}

	/**
	 * Die Methode dient dazu, den gesamten {@link TenorDatenContainer} zu erzeugen.
	 * 
	 * @return das Ergebnis von erzeugeHauptsachetenor() + erzeugeKostentenor() +
	 *         erzeugeStreitwertFestsetzung()
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeContainer()
	 */
	@Override
	public TenorDatenContainer erzeugeContainer() {
		container.berechnungsTyp = getBerechnungsTyp();
		container.hauptsacheEntscheidung = erzeugeHauptsacheEntscheidung();
		container.kostenEntscheidung = erzeugeKostenEntscheidung();
		container.vollstreckbarkeitsEntscheidung = erzeugeVollstreckbarkeitsEntscheidung();
		container.streitwertEntscheidung = erzeugeStreitwertEntscheidung();
		container.sonstigeEntscheidung = erzeugeSonstigeEntscheidung();
		container.begruendung = zeilen.toBegruendungsElemente(getGruende());
		return container;
	}

	/**
	 * Berechnet die Verteilung der Gerichtskosten
	 * 
	 * @return ein eindimensionales Array double[] mit einer Spalte für jeden
	 *         Beteiligten
	 */
	private double[] ermittleGerichtskostenQuoten() {
		double[] zwischen;
		double fiktiverStreitwert;
		zwischen = new double[baumbachBeteiligtenListe.size()];
		// fiktiven Streitwert berechnen
		fiktiverStreitwert = baumbachBeteiligtenListe.addiereStreitwerte();
		// Unterliegen des Klägers: Anderer fiktiver Streitwert nötig: nur die
		// Summe der Bk-Streitwerte!
		double klaegerUnterliegen = baumbachBeteiligtenListe.addiereAngreiferUnterliegen(Beteiligter.BEKLAGTE);
		// Bei Widerklage: zusätzlich noch das eigene Unterliegen des Klägers
		// hinsichtlich Widerklage
		if (baumbachBeteiligtenListe.isAnWiderklageBeteiligt(0))
			klaegerUnterliegen += baumbachBeteiligtenListe.getUnterliegenFor(0, false, false);
		zwischen[0] = (klaegerUnterliegen) * 100.0 / fiktiverStreitwert;
		// Wenn DWBk vorhanden: Unterliegen jedes DwBk berechnen
		if (baumbachBeteiligtenListe.zaehleDrittwiderbeklagte() > 0) {
			int[] dwbkBereich = baumbachBeteiligtenListe.minUndMax(Beteiligter.DRITTWIDERBEKLAGTE);
			for (int i = dwbkBereich[0]; i <= dwbkBereich[1]; i++) {
				double dwbkUnterliegen = 0.0;
				// Dwbk können BaumbachGesamtschuldner sein, deshalb:
				// beiGesamtschuldnernMultiplizieren = true
				dwbkUnterliegen += baumbachBeteiligtenListe.getUnterliegenFor(i, true, false);
				zwischen[i] = dwbkUnterliegen * 100.0 / fiktiverStreitwert;
			}
		}
		// Unterliegen jedes Beklagten
		int[] beklagtenBereich = baumbachBeteiligtenListe.minUndMax(Beteiligter.BEKLAGTE);
		for (int i = beklagtenBereich[0]; i <= beklagtenBereich[1]; i++) {
			// Erst einmal das eigene Unterliegen hinsichtlich der Klage
			double bkUnterliegen = baumbachBeteiligtenListe.getUnterliegenFor(i, true, false);
			zwischen[i] = bkUnterliegen * 100.0 / fiktiverStreitwert;
		}
		for (int i = beklagtenBereich[0]; i <= beklagtenBereich[1]; i++) {
			// Um Unterliegen bzgl. Widerklage bei jedem Widerbeklagten erhöhen,
			// sofern auch Widerkläger
			if (baumbachBeteiligtenListe.isEinzelpersonUndAnWiderklageBeteiligt(i)) {
				double bkUnterliegen = baumbachBeteiligtenListe.addiereAngreiferUnterliegen(Beteiligter.KLAEGER)
						+ baumbachBeteiligtenListe.addiereAngreiferUnterliegen(Beteiligter.DRITTWIDERBEKLAGTE);
				zwischen[i] += bkUnterliegen * 100.0 / fiktiverStreitwert;
			}
		}
		baueAngriffsTabelle(zwischen, fiktiverStreitwert);
		baueGruende();
		return zwischen;
	}

	/**
	 * Berechnet die Verteilung der außergerichtlichen Kosten des Klägers.
	 * 
	 * @return ein zweidimensionales Array double[][] mit einer Zeile je Kläger und
	 *         einer Spalte für jeden Beteiligten.
	 */
	private double[][] ermittleKlaegerkostenQuoten() {
		double[][] zwischen;
		double fiktiverStreitwert;
		// TODO Geht so nur für einen (!) Kläger
		final int kl = 0;
		zwischen = new double[1][baumbachBeteiligtenListe.size()];
		// fiktiven Streitwert berechnen
		// der fiktive Streitwert ist die Summe aller Bk-Streitwerte
		// sowie des kl.eigenen Sw
		fiktiverStreitwert = baumbachBeteiligtenListe.getStreitwertFor(kl)
				+ baumbachBeteiligtenListe.addiereStreitwerte(Beteiligter.BEKLAGTE);
		// Unterliegen des Klägers: Anderer fiktiver Streitwert nötig: nur die
		// Summe der Bk-Streitwerte!
		double klUnterliegen = baumbachBeteiligtenListe.addiereAngreiferUnterliegen(Beteiligter.BEKLAGTE);
		// Bei Widerklage muss hier noch das eigene Unterliegen hinzugerechnet
		// werden.
		if (baumbachBeteiligtenListe.isAnWiderklageBeteiligt(kl))
			klUnterliegen += baumbachBeteiligtenListe.getUnterliegenFor(kl, false, true);
		zwischen[kl][kl] = klUnterliegen * 100 / fiktiverStreitwert;
		int[] beklagtenBereich = baumbachBeteiligtenListe.minUndMax(Beteiligter.BEKLAGTE);
		for (int i = beklagtenBereich[0]; i <= beklagtenBereich[1]; i++) {
			// Inwieweit war der jeweilige Beklagte hinsichtlich
			// der Klage unterlegen?
			double bkUnterliegen = baumbachBeteiligtenListe.getUnterliegenFor(i, true, false);
			// Bei Widerklage muss hier noch mehr hinzugerechnet werden:
			// Inwieweit war der Beklagte (sofern Widerkläger und nicht
			// Gesamtschuldnervereinigung) mit seiner Widerklage unterlegen?
			if (baumbachBeteiligtenListe.isEinzelpersonUndAnWiderklageBeteiligt(i)
					&& baumbachBeteiligtenListe.isAnWiderklageBeteiligt(kl))
				bkUnterliegen += baumbachBeteiligtenListe.addiereKlaegerUnterliegenFor(kl, false, true);
			zwischen[kl][i] = (bkUnterliegen / fiktiverStreitwert) * 100.0;
		}
		return zwischen;
	}

	/**
	 * Berechnet die Verteilung der außergerichtlichen Kosten des
	 * Drittwiderbeklagten
	 * 
	 * @return ein zweidimensionales Array double[][] mit einer Zeile je
	 *         Drittwiderbeklagten und einer Spalte für jeden Beteiligten
	 */
	private double[][] ermittleDrittwiderbeklagtenkostenQuoten() {
		if (baumbachBeteiligtenListe.zaehleDrittwiderbeklagte() < 1)
			return null;
		double[][] zwischen = new double[baumbachBeteiligtenListe.zaehleDrittwiderbeklagte()][baumbachBeteiligtenListe
				.size()];
		final int[] dwbkBereich = baumbachBeteiligtenListe.minUndMax(Beteiligter.DRITTWIDERBEKLAGTE);
		final int[] bkBereich = baumbachBeteiligtenListe.minUndMax(Beteiligter.BEKLAGTE);
		for (int zeile = 0; zeile < baumbachBeteiligtenListe.zaehleDrittwiderbeklagte(); zeile++) {
			for (int beteiligter = 0; beteiligter < baumbachBeteiligtenListe.size(); beteiligter++) {
				// Der Dwbk, dem die zeile "gehört"
				if (beteiligter == zeile + dwbkBereich[0]
						&& !baumbachBeteiligtenListe.isGesamtschuldnerschaft(zeile + dwbkBereich[0]))
					zwischen[zeile][beteiligter] = baumbachBeteiligtenListe.getUnterliegenFor(beteiligter, false, true)
							* 100.0 / baumbachBeteiligtenListe.getStreitwertFor(beteiligter);
				// TODO: das klappt so nur, wenn es nur einen
				// widerklagenden Beklagten gibt.
				else if (beteiligter >= bkBereich[0] && baumbachBeteiligtenListe.isAnWiderklageBeteiligt(beteiligter)
						&& !baumbachBeteiligtenListe.isGesamtschuldnerschaft(zeile + dwbkBereich[0])) {
					// Wenn es die Spalte des widerklagenden Beklagten ist
					zwischen[zeile][beteiligter] = (baumbachBeteiligtenListe.getStreitwertFor(zeile + dwbkBereich[0])
							- baumbachBeteiligtenListe.getUnterliegenFor(zeile + dwbkBereich[0], false, true)) * 100.0
							/ baumbachBeteiligtenListe.getStreitwertFor(zeile + dwbkBereich[0]);
				} else
					// Rest mit 0.0 auffüllen
					zwischen[zeile][beteiligter] = 0.0;
			}
		}
		return zwischen;
	}

	/**
	 * Berechnet die Verteilung der außergerichtlichen Kosten der Beklagten
	 * 
	 * @return Array double[][] mit so vielen Zeilen wie Beklagte und der
	 *         Einfachheit halber (Einheitlichkeit) so vielen Spalten wie Parteien
	 *         insgesamt.
	 */
	private double[][] ermittleBeklagtenkostenQuoten() {
		double[][] zwischen = null;
		// Array muss so viele Zeilen wie Beklagte und der Einfachheit halber
		// (Einheitlichkeit) so viele Spalten wie Parteien insgesamt haben.
		zwischen = new double[baumbachBeteiligtenListe.zaehleBeklagte()][baumbachBeteiligtenListe.size()];
		final int[] bkBereich = baumbachBeteiligtenListe.minUndMax(Beteiligter.BEKLAGTE);
		final int[] klDwbkBereich = new int[] { baumbachBeteiligtenListe.minUndMax(Beteiligter.KLAEGER)[0],
				baumbachBeteiligtenListe.minUndMax(Beteiligter.DRITTWIDERBEKLAGTE)[1] };
		// zeilen-Schleife (i): für jeden Beklagten eine Zeile
		for (int zeile = 0; zeile < baumbachBeteiligtenListe.zaehleBeklagte(); zeile++) {
			// int betroffenerBeklagter: Index des betroffenen Beklagten
			final int betroffenerBeklagter = zeile + bkBereich[0];
			// Fiktiven Streitwert für den betroffenen Beklagten berechnen
			double fiktiverStreitwert = baumbachBeteiligtenListe.getStreitwertFor(betroffenerBeklagter);
			/*
			 * Das Unterliegen des Beklagten selbst setzt sich zusammen aus seinem
			 * Unterliegen hinsichtlich der Klage (und ggfls. der Wk) Zusätzliches
			 * Unterliegen wird einberechnet
			 */
			double bkUnterliegen = baumbachBeteiligtenListe.getUnterliegenFor(betroffenerBeklagter, false, true);
			/*
			 * Wenn er Widerkläger ist: fiktiven Streitwert um Streitwert von jedem
			 * widerbeklagten Kl./Dwbk erhöhen
			 */
			if (baumbachBeteiligtenListe.isAnWiderklageBeteiligt(betroffenerBeklagter))
				for (int klDwbk = klDwbkBereich[0]; klDwbk <= klDwbkBereich[1]; klDwbk++)
					if (baumbachBeteiligtenListe.isAnWiderklageBeteiligt(klDwbk))
						fiktiverStreitwert += baumbachBeteiligtenListe.getStreitwertFor(klDwbk);
			/*
			 * Spalten-Schleife int beteiligter: Für jede Partei eine Spalte
			 */
			for (int beteiligter = 0; beteiligter <= bkBereich[1]; beteiligter++) {
				/*
				 * Wenn der betroffeneBeklagte ein BaumbachGesamtschuldner ist, wird er
				 * übersprungen (sonst: Division durch 0)
				 */
				if (beteiligter >= klDwbkBereich[0] && beteiligter <= klDwbkBereich[1]
						&& !baumbachBeteiligtenListe.isGesamtschuldnerschaft(betroffenerBeklagter)) {
					/*
					 * Hier wird's bei Gesamtschuldnerischer Widerklage-Verurteilung kritisch: Bei
					 * Gesamtschuldnerschaft muss multipliziert werden (deshalb true), sonst trägt
					 * eine Kl/Dwbk-Gesamtschuldnerschaft zu wenig. Aber das eigene Unterliegen des
					 * Klägers darf nicht hinzugerechnet werden, wenn er selbst an der Reihe ist,
					 * sonst trägt er zuviel; die Kl/Dwbk-Gesamtschuldnerschaft kommt ja auch noch
					 * an die Reihe. Deshalb dort false.
					 */
					final double klUnterliegen = baumbachBeteiligtenListe.getUnterliegenFor(beteiligter, true, false);
					/*
					 * Kläger: Zusätzliches Unterliegen wird einberechnet. Hier braucht man auch das
					 * Unterliegen des Beklagten hinsichtlich einer gesamtschuldnerischen
					 * Verurteilung, weil die Bk-Gesamtschuldnerschaft mit den
					 * Bk-Außergerichtskosten nichts zu schaffen hat.
					 */
					if (baumbachBeteiligtenListe.getTypFor(beteiligter) == Beteiligter.KLAEGER)
						zwischen[zeile][beteiligter] = (baumbachBeteiligtenListe.getStreitwertFor(betroffenerBeklagter)
								- (baumbachBeteiligtenListe.getUnterliegenFor(betroffenerBeklagter, false, true))) * 100
								/ fiktiverStreitwert;
					else
						/*
						 * den Dwbk geht das Unterliegen des Bk hinsichtlich der Klage nichts an
						 */
						zwischen[zeile][beteiligter] = 0.0;
					/*
					 * Wenn Bk und Kl/Dwbk an Wk beteiligt sind, wird das bkUnterliegen
					 * heraufgesetzt...
					 * 
					 * (hier muss bei Gesamtschuldnerschaften multiplziert werden, sobald diese an
					 * der Reihe ist. Bei den einfachen Beteiligten darf aber nicht deren
					 * Unterliegen in der Gesamtschuldnerschaft hinzugerechnet werden (die kommen ja
					 * auch noch dran, da bkUnterliegen über die Spalten-Schleife hinweg fortlebt)
					 */
					if ((baumbachBeteiligtenListe.isAnWiderklageBeteiligt(betroffenerBeklagter)
							&& (baumbachBeteiligtenListe.isAnWiderklageBeteiligt(beteiligter)))) {
						bkUnterliegen += (baumbachBeteiligtenListe.getStreitwertFor(beteiligter)
								- baumbachBeteiligtenListe.getUnterliegenFor(beteiligter, true, false));
						/*
						 * ...und beim Kl./Dwbk wird das Unterliegen hinsichtlich der Wk hinzugerechnet
						 */
						zwischen[zeile][beteiligter] += (klUnterliegen * 100.0 / fiktiverStreitwert);
					}
				} else if (beteiligter == betroffenerBeklagter
						&& !baumbachBeteiligtenListe.isGesamtschuldnerschaft(betroffenerBeklagter)) {
					// Der betreffende Beklagte, dem die Zeile "gehört"
					zwischen[zeile][beteiligter] = bkUnterliegen;
					zwischen[zeile][beteiligter] *= 100.0;
					zwischen[zeile][beteiligter] /= fiktiverStreitwert;
				} else {
					/*
					 * Rest mit Nullen auffüllen (Beklagte müssen untereinander nix tragen)
					 */
					zwischen[zeile][beteiligter] = 0.0;
				}
			}
		}
		return zwischen;
	}

	/**
	 * Berechnet alle Kostenquoten, die gerichtlichen und die außergerichtlichen
	 * aller Beteiligten.
	 * 
	 * @return ein zweidimensionales Array double[][] mit einer Zeile für die
	 *         Gerichtskosten, und je einer weiteren Zeile für jeden Beteiligten. In
	 *         jeder Zeile hat jeder Beteiligte eine Spalte.
	 */
	private double[][] ermittleQuoten() {
		int beteiligte = baumbachBeteiligtenListe.size();
		double[][] zwischen = new double[beteiligte + 1][beteiligte];
		int zaehler = 0;
		zwischen[zaehler] = ermittleGerichtskostenQuoten();
		for (double[] quoten : ermittleKlaegerkostenQuoten()) {
			zaehler++;
			zwischen[zaehler] = quoten;
		}
		if (baumbachBeteiligtenListe.zaehleDrittwiderbeklagte() > 0)
			for (double[] quoten : ermittleDrittwiderbeklagtenkostenQuoten()) {
				zaehler++;
				zwischen[zaehler] = quoten;
			}
		for (double[] quoten : ermittleBeklagtenkostenQuoten()) {
			zaehler++;
			zwischen[zaehler] = quoten;
		}
		return zwischen;
	}

	/**
	 * Die Methode prüft, ob die Klage/Widerklage "im Übrigen" abgewiesen werden
	 * muss.
	 * 
	 * @return ein {@link BaumbachImUebrigen}
	 */
	private BaumbachImUebrigen baueImUebrigen() {
		// boolean[][] ganzZugesprochenOderAbgewiesen = new boolean[][]
		// { { false, true}, { false, true } };
		final BaumbachImUebrigen imUebrigen = new BaumbachImUebrigen(false, true, false, true);
		/*
		 * Wenn die Summe der Streitwerte so hoch ist wie die Summe des Unterliegens,
		 * dann hat der Kläger voll obsiegt.
		 */
		// [0][0]
		final double bkUnterliegen = baumbachBeteiligtenListe.addiereUnterliegen(Beteiligter.BEKLAGTE);
		imUebrigen.setKlageVollBegruendet(
				baumbachBeteiligtenListe.addiereStreitwerte(Beteiligter.BEKLAGTE) <= bkUnterliegen);
		/*
		 * Sobald eine Verurteilung vorliegt, wurde die Klage jedenfalls nicht mehr ganz
		 * abgewiesen.
		 */
		// [0][1] = false;
		if (bkUnterliegen > 0.0) {
			imUebrigen.setKlageGanzUnbegruendet(false);
		}
		if (!baumbachBeteiligtenListe.enthaeltAnWiderklageBeteiligte()) {
			return imUebrigen;
		}
		/*
		 * Wenn die Summe der Streitwerte so hoch ist wie die Summe des Unterliegens,
		 * dann hat der Widerkläger voll obsiegt.[1][0]
		 */
		final double wbkUnterliegen = baumbachBeteiligtenListe.addiereUnterliegen(Beteiligter.KLAEGER)
				+ baumbachBeteiligtenListe.addiereUnterliegen(Beteiligter.DRITTWIDERBEKLAGTE);
		imUebrigen.setWiderklageVollBegruendet((baumbachBeteiligtenListe.addiereStreitwerte(Beteiligter.KLAEGER)
				+ baumbachBeteiligtenListe.addiereStreitwerte(Beteiligter.DRITTWIDERBEKLAGTE)) <= wbkUnterliegen);
		/*
		 * Sobald eine Verurteilung vorliegt, wurde die Widerklage jedenfalls nicht mehr
		 * ganz abgewiesen.[1][1] = false;
		 */
		if (wbkUnterliegen > 0.0) {
			imUebrigen.setWiderklageGanzUnbegruendet(false);
		}
		return imUebrigen;
	}

	/**
	 * Die Methode listet alle Hauptsacheverhältnisse der Beteiligten untereinander
	 * auf
	 * 
	 * @return eine {@link List}&lt;{@link HauptsacheVerhaeltnis }>
	 */
	private List<HauptsacheVerhaeltnis> baueHauptsacheVerhaeltnisse() {
		List<HauptsacheVerhaeltnis> verhaeltnisse = new ArrayList<HauptsacheVerhaeltnis>();
		BaumbachBeteiligter kl = baumbachBeteiligtenListe
				.get(baumbachBeteiligtenListe.minUndMax(Beteiligter.KLAEGER)[0]);
		BaumbachBeteiligter wkl = null;
		// die Liste in "Tenorreihenfolge" durchgehen (Beklagte vor übrigen,
		// Gesamtschuldner vor einfachen)
		for (int index : baumbachBeteiligtenListe.beteiligteInTenorReihenfolge(false)) {
			BaumbachBeteiligter schuldner = baumbachBeteiligtenListe.get(index);
			BaumbachBeteiligter glaeubiger = null;
			// Wenn Typ Beklagter ist, ist der Gläubiger der Kläger, sonst der widerklagende
			// Beklagte
			// Da die Beklagten in der Liste vor den übrigen stehen (Tenorreihenfolge), muss
			// der Widerkläger
			// schon ermittelt sein, bevor es zum Kläger/den Widerbeklagten kommt
			if (schuldner.getTyp() == Beteiligter.BEKLAGTE) {
				glaeubiger = kl;
				if (schuldner.isAnWiderklageBeteiligt())
					wkl = schuldner;
			} else {
				glaeubiger = wkl;
			}
			if (schuldner.getUnterliegen() > 0.0) {
				HauptsacheVerhaeltnis verhaeltnis = new HauptsacheVerhaeltnis(glaeubiger, schuldner,
						baumbachBeteiligtenListe.hatDarueberHinaus(index), (long) schuldner.getUnterliegen());
				verhaeltnisse.add(verhaeltnis);
			}
		}
		return verhaeltnisse;
	}

	/**
	 * Die Methode listet alle Kostentragungsverhältnisse der Beteiligten
	 * untereinander auf.
	 * 
	 * @param quoten siehe {@link #ermittleQuoten()}
	 * @return eine {@link List}&lt;{@link KostenTragungsVerhaeltnis}>
	 */
	private List<KostenTragungsVerhaeltnis> baueKostenTragungsVerhaeltnisse(double[][] quoten) {
		List<KostenTragungsVerhaeltnis> verhaeltnisse = new ArrayList<KostenTragungsVerhaeltnis>(
				1 + baumbachBeteiligtenListe.size());
		for (int zeile = 0; zeile < quoten.length; zeile++) {
			Beteiligter glaeubiger;
			if (zeile == 0) {
				// die erste Zeile betrifft die Gerichtskosten
				glaeubiger = new GerichtsKostenBeteiligter();
			} else {
				glaeubiger = baumbachBeteiligtenListe.get(zeile - 1);
			}
			/*
			 * In Tenor-Reihenfolge bringen, d.h.: Die Spalten mit den
			 * BaumbachGesamtschuldner müssen an den Anfang der Liste ihres Beteiligtentyps
			 */
//			for (int spalte = 0; spalte < quoten[zeile].length; spalte++) {
			for (int spalte : baumbachBeteiligtenListe.beteiligteInTenorReihenfolge(true)) {
				BaumbachBeteiligter schuldner = baumbachBeteiligtenListe.get(spalte);
				double kostenBruchteil = quoten[zeile][spalte];
				if (kostenBruchteil > 0.0) {
					/*
					 * hier muss der Kostenbruchteil durch 100 geteilt werden, um den Anforderungen
					 * des TenorDatenContainers zu genügen.
					 */
					KostenTragungsVerhaeltnis verhaeltnis = new KostenTragungsVerhaeltnis(glaeubiger, schuldner,
							hatDarueberHinaus(spalte, quoten[zeile]), kostenBruchteil / 100.0);
					verhaeltnisse.add(verhaeltnis);
				}
			}
		}
		return verhaeltnisse;
	}

	/**
	 * Die Methode konvertiert die {@link #vollstreckbarkeit} in eine Liste der
	 * {@link VollstreckungsVerhaeltnis}se. Die Liste ist sortiert nach Maßgabe von
	 * {@link VollstreckungsVerhaeltnis#compareTo(eu.gronos.kostenrechner.model.tenordaten.ProzessVerhaeltnis)}.
	 * 
	 * @param liste eine {@link List}&lt;{@link Vollstreckbarkeit}&gt;
	 * @return eine {@link List}&lt;{@link VollstreckungsVerhaeltnis}&gt;
	 */
	private List<VollstreckungsVerhaeltnis> baueVollstreckungsVerhaeltnisse(VollstreckbarkeitsListe liste) {
		if (liste != null && !liste.isEmpty() && liste.sindAlle708iVm713()) {
			return new VollstreckbarkeitsHelfer(container.hauptsacheEntscheidung.prozessverhaeltnisse,
					container.kostenEntscheidung.prozessverhaeltnisse)
							.errechneAlleVollstreckungsVerhaeltnisse(liste.get(0));
		}
		List<VollstreckungsVerhaeltnis> verhaeltnisse = new ArrayList<VollstreckungsVerhaeltnis>();
		for (Vollstreckbarkeit vo : liste) {
			if (vo instanceof BaumbachVollstreckbarkeit) {
				BaumbachVollstreckbarkeit bvo = (BaumbachVollstreckbarkeit) vo;
				// Vor dem Hinzufügen müssen "leere" VollstreckbarkeitsObjekte und das Tragen
				// eigener Kosten herausgefiltert werden
				if (bvo.hatEintraege() && !bvo.getGlaeubiger().equals(bvo.getSchuldner())) {
					VollstreckungsVerhaeltnis verhaeltnis = new VollstreckungsVerhaeltnis(bvo);
					verhaeltnisse.add(verhaeltnis);
				}
			}
		}
		Collections.sort(verhaeltnisse);
		return verhaeltnisse;
	}

	/**
	 * Wenn eine der Quoten unter 10% ist, muss sie den verbleibenden sinnvoll
	 * "aufgeschlagen" werden:
	 * 
	 * § 92 Abs. 2 ZPO: Das Gericht kann der einen Partei die gesamten Prozesskosten
	 * auferlegen, wenn 1. die Zuvielforderung der anderen Partei verhältnismäßig
	 * geringfügig war und keine oder nur geringfügig höhere Kosten veranlasst hat
	 * oder 2. der Betrag der Forderung der anderen Partei von der Festsetzung durch
	 * richterliches Ermessen, von der Ermittlung durch Sachverständige oder von
	 * einer gegenseitigen Berechnung abhängig war.
	 * 
	 * @param quoten ein Array double[] mit den quoten im Urzustand
	 * @return ein Array double[] mit den quoten unter Berücksichtigung von § 92
	 *         Abs. 2 ZPO
	 */
	private double[] entferneUnwesentlichesUnterliegen(double[] quoten) {
		double[] bereinigteQuoten = new double[quoten.length];
		double summe = 0.0;
		// Erster Schritt: alle Quoten durchgehen
		for (int index = 0; index < bereinigteQuoten.length; index++) {
			// ... und Einträge unter 10% ("verhältnismäßig gering") nicht
			// übernehmen ...
			if (quoten[index] < (100.0 * allgemein.unwesentlich92ii))// 10.0
				bereinigteQuoten[index] = 0.0;
			else {
				// / ... andere hingegen aufsummieren
				bereinigteQuoten[index] = quoten[index];
				summe += quoten[index];
			}
		}
		// Wenn es keine Einzelquoten über 10% gab, dann das Array unversehrt
		// zurückgeben; ebenso, wenn die summe schon über 100% geht
		if (summe <= 0.0 || summe > 100.0)
			return quoten;
		// Zweiter Schritt: Sonst das Verhältnis der Summe zu 100% feststellen
		final double verhaeltnis = 100.0 / summe;
		// Dritter Schritt: Alle Quoten des neuen Arrays mit dem verhaeltnis
		// multiplizieren
		for (int index = 0; index < bereinigteQuoten.length; index++) {
			bereinigteQuoten[index] = bereinigteQuoten[index] * verhaeltnis;
		}
		return bereinigteQuoten;
	}

	/**
	 * Findet über die BaumbachBeteiligtenListe alle Gesamtschuldnerschaften, in
	 * denen der BaumbachBeteiligte mit dem Index beteiligter ist, dann gucken, ob
	 * sie > 0% tragen müssen
	 * 
	 * @param index   die Position in der BaumbachBeteiligtenListe
	 * @param spalten ein Array double[] mit den Quoten aller Beteiligten für eine
	 *                Zeile
	 * @return true, wenn der BaumbachBeteiligte unter index in einer
	 *         Gesamtschuldnerschaft ist, die eine Kostenquote > 0 tragen muss
	 */
	private boolean hatDarueberHinaus(int index, double[] spalten) {
		final ArrayList<Integer> darueberHinaus = findeDarueberHinaus(index, spalten);
		return darueberHinaus != null && darueberHinaus.size() > 0 && spalten != null;
	}

	/**
	 * Findet über die BaumbachBeteiligtenListe alle Gesamtschuldnerschaften, in
	 * denen der BaumbachBeteiligte mit dem Index beteiligter ist, dann gucken, ob
	 * sie > 0% tragen müssen
	 * 
	 * @param index   die Position in der BaumbachBeteiligtenListe
	 * @param spalten ein Array double[] mit den Quoten aller Beteiligten für eine
	 *                Zeile
	 * @return eine ArrayList mit den Indexen von allen Gesamtschuldnerschaften, die
	 *         eine Kostenquote > 0 tragen müssen, in denen der BaumbachBeteiligte
	 *         unter index in einer Gesamtschuldnerschaft verurteilt ist
	 */
	private ArrayList<Integer> findeDarueberHinaus(int index, double[] spalten) {
		ArrayList<Integer> liste = baumbachBeteiligtenListe.findeDarueberHinaus(index);
		ArrayList<Integer> darueberHinaus = new ArrayList<Integer>();
		if (liste != null && liste.size() > 0 && spalten != null)
			for (int i : liste) {
				if (spalten[i] > 0.0)
					darueberHinaus.add(new Integer(i));
			}
		return darueberHinaus;
	}

	/**
	 * Die Methode dient dazu, die gesamte mögliche Quote zurückzugeben. Es wird die
	 * zusätzliche Quote zurückgegeben, die sich aus den BaumbachGesamtschuldnern
	 * ergibt, die auch den referenzierten BaumbachBeteiligten repräsentieren.
	 * 
	 * {@link BaumbachBeteiligtenListe#getUnterliegenFor(int, boolean, boolean)}
	 * 
	 * @param index   die Position in der BaumbachBeteiligtenListe
	 * @param spalten ein Array double[] mit den Quoten aller Beteiligten für eine
	 *                Zeile
	 * @return die Summe als double
	 */
	private double berechneGanzeQuote(int index, double[] spalten) {
		/* Erst einmal trägt er, was er selbst trägt */
		double ganzeQuote = spalten[index];
		final ArrayList<Integer> darueberHinaus = findeDarueberHinaus(index, spalten);
		/* Und dann, wozu er gesamtschuldnerisch mit anderen verurteilt wurde */
		if (darueberHinaus != null && darueberHinaus.size() > 0) {
			for (Integer i : darueberHinaus)
				ganzeQuote += spalten[i];
		}
		return ganzeQuote;
	}

	/**
	 * Der Getter für {@link #kosten} gibt den Inhalt selbigen Felds zurück. Ist
	 * dieses noch <code>null</code>, so wird vorher noch {@link #ermittleQuoten()}
	 * aufgerufen, um es zu füllen und
	 * {@link #entferneUnwesentlichesUnterliegen(double[])
	 * entferneUnwesentlichesUnterliegen}, um ggfls. ein unwesentliches Unterliegen
	 * zu entfernen.
	 * 
	 * @return gibt {@link #kosten} als double[][] zurück
	 */
	double[][] getKosten() {
		if (kosten == null) {
			kosten = ermittleQuoten();
			if (allgemein.b92ii)
				for (int zeile = 0; zeile < kosten.length; zeile++) {
					/*
					 * Erst einmal die Quoten untersuchen und unwesentliches Unterliegen anderweitig
					 * verteilen (wenn ausgewählt)
					 */
					kosten[zeile] = entferneUnwesentlichesUnterliegen(kosten[zeile]);
				}

			if (Kostenrechner.debug) {
				StringBuilder sb = new StringBuilder();
				for (double d[] : kosten) {
					double summe = 0.0;
					for (double dd : d) {
						sb.append(String.format("%,.2f; ", dd));
						summe += dd;
					}
					sb.append(String.format("(Summe: %,.2f).%n", summe));
				}
				Kostenrechner.getLogger().info(sb.toString());
			}

		}
		return kosten;
	}

	/**
	 * Die Methode dient dazu, den Streitwert zu ermitteln, indem der Streitwert auf
	 * den höchsten Einzelstreitwert der baumbachBeteiligtenListe festgesetzt wird.
	 * 
	 * @param mitWiderklage wenn eine
	 *                      {@link BaumbachBeteiligtenListe#enthaeltAnWiderklageBeteiligte()
	 *                      Widerklage vorhanden} ist
	 * @return wenn mitWiderklage, dann ein Array double[3], bei dem 0 der
	 *         Streitwert insgesamt, 1 der Klagestreitwert und 2 der
	 *         Widerklagestreitwert sind; sonst double[1] nur mit dem Streitwert.
	 */
	private double[] errechneStreitwert(boolean mitWiderklage) {
		// Bei einer Widerklage müssen beide Streitwerte ermittelt werden
		if (mitWiderklage) {
			final double klSw = baumbachBeteiligtenListe.getHoechstenStreitwert(Beteiligter.BEKLAGTE);
			final double wkSw = baumbachBeteiligtenListe
					.getHoechstenStreitwert(Beteiligter.KLAEGER) > baumbachBeteiligtenListe
							.getHoechstenStreitwert(Beteiligter.DRITTWIDERBEKLAGTE)
									? baumbachBeteiligtenListe.getHoechstenStreitwert(Beteiligter.KLAEGER)
									: baumbachBeteiligtenListe.getHoechstenStreitwert(Beteiligter.DRITTWIDERBEKLAGTE);
			final double sw = !wirtschaftlichIdentisch ? klSw + wkSw : klSw > wkSw ? klSw : wkSw;
			return new double[] { sw, klSw, wkSw };
		} else {
			/* Ohne Widerklage muss nur ein Streitwert ermittelt werden */
			return new double[] { baumbachBeteiligtenListe.getHoechstenStreitwert(Beteiligter.BEKLAGTE) };
		}
	}

	/**
	 * Die Methode setzt die Entscheidungsgründe zurück (sofern nicht leer) und
	 * beginnt sie neu mit dem übergebenen Text.
	 * 
	 * @param text eine {@link CharSequence} mit dem Anfangssatz(-bestandteil) der
	 *             Gründe.
	 */
	private void starteGruende(CharSequence text) {
		if (gruende.length() > 0)
			gruende.delete(0, gruende.length());
		erweitereGruende(text);
	}

	/**
	 * Die Methode dient dazu, über {@link #MISSING()} die Gründe für die
	 * BaumbachFormel zu basteln.
	 */
	private void baueGruende() {
		starteGruende(GRUENDE_91_92_100II);
		zeilen.toStringBuilder(gruende);
	}

	/**
	 * Die Methode dient dazu, eine "Tabelle" (also eine ArrayList aus Arrays
	 * Object[]) zu erzeugen, das dann zur Begründung anhand der Einzelangriffe
	 * genutzt werden kann.
	 * 
	 * @param quoten             die Gerichtskostenquoten als double[][]
	 * @param fiktiverStreitwert ein double mit dem fiktiven Streitwert der
	 *                           Gerichtskosten
	 */
	private void baueAngriffsTabelle(double[] quoten, double fiktiverStreitwert) {
		/*
		 * Durch die Verwendung einer ArrayList muss die Anzahl der Zeilen nicht
		 * ermittelt werden, aber eine Liste der Angriffe muss über
		 * beteiligteWieVerklagt ermittelt werden.
		 */
		zeilen = new TabulierendZeilen();
		final ArrayList<Integer> beteiligtenIndicesZeilen = baumbachBeteiligtenListe.beteiligteWieVerklagt();
		/*
		 * Liste der Beteiligten/Gesamtschuldnerschaften über
		 * beteiligteInTenorReihenfolge für Zahl der Spalten ermitteln
		 */
		final ArrayList<Integer> beteiligtenIndicesSpalten = baumbachBeteiligtenListe
				.beteiligteInTenorReihenfolge(true);
		/* Hierin werden die Verluste der Beteiligten zum Summieren gespeichert */
		final double[] verluste = new double[beteiligtenIndicesSpalten.size()];
		for (int i = 0; i < verluste.length; i++)
			verluste[i] = 0.0;
		/* Kopfzeile und Ergebniszeile bauen und der ArrayList hinzufügen */
		final List<String> columnHeaders = new ArrayList<String>(beteiligtenIndicesSpalten.size() + 1);
		/*
		 * Kopfzeile mithilfe einer For-Schleife füllen; die Liste enthält eine Spalte
		 * mehr als beteiligteInTenorReihenfolge, weil zuerst noch KOPFZEILE_BEGINN
		 * eingefügt wird.
		 */
		columnHeaders.add(KOPFZEILE_BEGINN);
		for (Integer index : beteiligtenIndicesSpalten) {
			columnHeaders.add(baumbachBeteiligtenListe.parteiBezeichner(Beteiligter.GENITIV, index.intValue()));
		}
		zeilen.add(columnHeaders);
		/*
		 * ArrayList füllen: äußere Schleife: für jede Zeile eine ArrayList<Double>
		 * anlegen mit 1*String und 1..n*Double
		 */
		for (int zeile = 0; zeile < beteiligtenIndicesZeilen.size(); zeile++) {
			/*
			 * Die ArrayList<Double> mit den Spalten für die aktuelle Zeile
			 */
			DoubleDataRows doubles = new DoubleDataRows(beteiligtenIndicesSpalten.size());
			final int a = beteiligtenIndicesZeilen.get(zeile).intValue();
			final BaumbachBeteiligter bbtA = baumbachBeteiligtenListe.get(a);
			/* Die "Kopfspalte" links sagt Angriff wer gegen wen */
			final String rowHeader = baumbachBeteiligtenListe.benenneAngriffGegen(a);
			for (int spalte = 0; spalte < beteiligtenIndicesSpalten.size(); spalte++) {
				final int b = beteiligtenIndicesSpalten.get(spalte).intValue();
				final BaumbachBeteiligter bbtB = baumbachBeteiligtenListe.get(b);
				double wert = 0.0;
				/*
				 * Wert für innere Tabelle nach BeteiligtenTyp von a und b ermitteln
				 */
				if (a == b) {
					/*
					 * Wenn a und b identisch (dasselbe Element in der bbtl), dann bezieht sich das
					 * Unterliegen von b auf den Angriff gegen b, ergo: das Unterliegen von b ist
					 * hier richtig.
					 */
					wert = bbtB.getUnterliegen();
				} else if (bbtB.isGesamtschuldnerschaft() && baumbachBeteiligtenListe.getAlsGesamtschuldner(b).contains(
						bbtA.getTyp(), baumbachBeteiligtenListe.ermittleLaufendeNummerFuerBeteiligtenTyp(a))) {
					/*
					 * Wenn b eine gesamtschuldnerische Verurteilung ist und a (gegen den sich der
					 * Angriff richtet) darin enthalten ist, dann muss das Unterliegen der
					 * Gesamtschuldnerschaft in die Spalte.
					 */
					wert = bbtB.getUnterliegen();
				} else if (bbtA.getTyp() == Beteiligter.BEKLAGTE) {
					/*
					 * Wenn a Beklagter und b Kläger ist, dann Unterliegen berechnen
					 */
					if (bbtB.getTyp() == Beteiligter.KLAEGER)
						wert = bbtA.getStreitwert() - baumbachBeteiligtenListe.getUnterliegenFor(a, false, true);
				} else if ((bbtA.getTyp() == Beteiligter.KLAEGER && bbtA.isAnWiderklageBeteiligt())
						|| bbtA.getTyp() == Beteiligter.DRITTWIDERBEKLAGTE) {
					/*
					 * Wenn a Widerbeklagter ist und b Widerkläger, dann Unterliegen berechnen.
					 */
					if (bbtB.getTyp() == Beteiligter.BEKLAGTE && bbtB.isAnWiderklageBeteiligt())
						wert = bbtA.getStreitwert() - baumbachBeteiligtenListe.getUnterliegenFor(a, false, true);
				}
				/*
				 * Den ermittelten Wert in die Spalte schreiben und auf die Verluste addieren.
				 */
				doubles.add(new Double(wert));
				verluste[spalte] += wert;
			}
			zeilen.add(rowHeader, doubles);
		}
		/* Zeilen für Verlustsummen und Ergebnis erzeugen */
		final DoubleDataRows ergebnisDoubles = new DoubleDataRows(beteiligtenIndicesSpalten.size());
		final String ergebnisHeader = String.format(ERGEBNIS_ZEILE, fiktiverStreitwert);
		final DoubleDataRows verlusteDoubles = new DoubleDataRows(beteiligtenIndicesSpalten.size());
		final String verlusteHeader = VERLUST_GESAMT;
		/*
		 * Verlustsummen-Zeile und Ergebniszeile mithilfe einer For-Schleife füllen
		 */
		for (int spalte = 0; spalte < beteiligtenIndicesSpalten.size(); spalte++) {
			verlusteDoubles.add(new Double(verluste[spalte]));
			ergebnisDoubles.add(new Double(quoten[beteiligtenIndicesSpalten.get(spalte).intValue()]));
		}
		/* Verlustsummen-Zeile und Ergebniszeile hinzufügen */
		zeilen.add(verlusteHeader, verlusteDoubles);
		zeilen.add(ergebnisHeader, ergebnisDoubles);
	}

	/**
	 * Die Methode berechnet, inwieweit das Urteil vorläufig vollstreckbar ist.
	 * Vorläufige Vollstreckbarkeit: erst einmal gucken, ob die Hauptsache über die
	 * Schwelle von §709 kommt. Wenn nicht, dann bei den Kosten rechnen: Array
	 * double[][] durchgehen, Quote mit jeweiligen Kosten multiplizieren und für
	 * jede Partei zusammen rechnen. Die jeweils eigenen Kosten dabei auslassen.
	 * 
	 * @return eine {@link VollstreckbarkeitsListe}, deren 0. Element die
	 *         Unanfechtbarkeit speichert, die weiteren Elemente speichern die
	 *         vorläufige Vollstreckbarkeit für die
	 *         Einzelvollstreckungsverhältnisse.
	 */
	private VollstreckbarkeitsListe berechneVollstreckbarkeit() {
		VollstreckbarkeitsListe vollstreckbarkeit = new VollstreckbarkeitsListe();
		/* Unanfechtbarkeit prüfen und der Liste hinzufügen */
		vollstreckbarkeit.add(pruefeUnanfechtbarkeit());
		/* Nach festgestellter Unanfechtbarkeit war's das schon. */
		if (vollstreckbarkeit.sindAlle708iVm713())
			return vollstreckbarkeit;
		/*
		 * Einzelne Vollstreckungsverhältnisse durchgehen und
		 * BaumbachVollstreckbarkeitsObjekt der Liste hinzufügen; an quoten[][]
		 * entlanghangeln.
		 */
		double[][] quoten = getKosten();
		final long streitwert = (long) errechneStreitwert(baumbachBeteiligtenListe.enthaeltAnWiderklageBeteiligte())[0];
		for (int glaeubigerIndex = 0; glaeubigerIndex < baumbachBeteiligtenListe.size(); glaeubigerIndex++) {
			for (int schuldnerIndex = 0; schuldnerIndex < baumbachBeteiligtenListe.size(); schuldnerIndex++) {
				BaumbachBeteiligter glaeubiger = baumbachBeteiligtenListe.get(glaeubigerIndex);
				BaumbachBeteiligter schuldner = baumbachBeteiligtenListe.get(schuldnerIndex);
				if (glaeubigerIndex != schuldnerIndex && !(glaeubiger instanceof BaumbachGesamtschuldnerschaft)
						&& !(schuldner instanceof BaumbachGesamtschuldnerschaft)) {
					double hauptsache = 0;
					double vollstreckbareKosten = 0;
					if (glaeubiger.getTyp() == Beteiligter.KLAEGER) {
						/* Bei einem Kläger vorher die Gerichtskosten drauf KostenVerzeichnis.KV1210 */
						vollstreckbareKosten += berechneGanzeQuote(schuldnerIndex, quoten[0]) / 100.0
								* GebuehrenVerzeichnis.errechneGebuehr(TenorTexter.getKostenKlaeger().get(0),
										streitwert);
					}
					/*
					 * Die Anwaltskosten (KOSTEN_BEKLAGTER) immer - auch bei Kl und Dwbk
					 */
					vollstreckbareKosten += berechneGanzeQuote(schuldnerIndex, quoten[glaeubigerIndex + 1]) / 100.0
							* (long) (TenorTexter.getKostenBeklagter().errechneGebuehrenSumme(streitwert));
					// hauptsache berechnen, wenn es in der "Paarung" eine
					// Hauptsache gibt
					if ((Beteiligter.KLAEGER == glaeubiger.getTyp() && Beteiligter.BEKLAGTE == schuldner.getTyp())
							|| (Beteiligter.BEKLAGTE == glaeubiger.getTyp() && glaeubiger.isAnWiderklageBeteiligt()
									&& schuldner.isAnWiderklageBeteiligt() && (Beteiligter.KLAEGER == schuldner.getTyp()
											|| Beteiligter.DRITTWIDERBEKLAGTE == schuldner.getTyp()))) {
						hauptsache = baumbachBeteiligtenListe.getUnterliegenFor(schuldnerIndex, false, true);
					}
					vollstreckbarkeit.add(BaumbachVollstreckbarkeit.pruefeSicherheitsleistung((long) hauptsache,
							(long) vollstreckbareKosten, glaeubiger, schuldner));
				}
			}
		}
		return vollstreckbarkeit;
	}

	/**
	 * Die Methode prüft, ob das Urteil unanfechtbar ist (§§ 708, 713 ZPO).
	 * 
	 * @return ein {@link Vollstreckbarkeit}, das im Fall der Unanfechtbarkeit 708
	 *         und 713 auf <code>true</code> setzt. Ansonsten enthaelt es nichts.
	 */
	private Vollstreckbarkeit pruefeUnanfechtbarkeit() {
		/*
		 * Wenn der Streitwert schon nicht über 600 EUR geht, ist das Urteil
		 * unanfechtbar.
		 */
		if (errechneStreitwert(
				baumbachBeteiligtenListe.enthaeltAnWiderklageBeteiligte())[0] <= TenorTexter.BERUFUNGS_GRENZE) {
			return new Vollstreckbarkeit(true, false, false, false, true);
		} else if (errechneStreitwert(baumbachBeteiligtenListe.enthaeltAnWiderklageBeteiligte())[0] <= 2
				* TenorTexter.BERUFUNGS_GRENZE) {
			/*
			 * Beschwer jedes Einzelbeteiligten überprüfen, ob nicht über Berufungsgrenze
			 */
			for (int index = 0; index < baumbachBeteiligtenListe.size(); index++)
				if (!baumbachBeteiligtenListe.isGesamtschuldnerschaft(index)) {
					/* erstmal das "eigene" "ganze" Unterliegen */
					double unterliegen = baumbachBeteiligtenListe.getUnterliegenFor(index, false, true);
					/*
					 * Wenn der beteiligte ein "Angreifer" ist:
					 */
					/* Unterschleife beim Kläger ggü Bk. */
					if (baumbachBeteiligtenListe.getTypFor(index) == Beteiligter.KLAEGER) {
						double angreiferUnterliegen = 0.0;
						for (int angriff = 0; angriff < baumbachBeteiligtenListe.size(); angriff++) {
							if (baumbachBeteiligtenListe.getTypFor(angriff) == Beteiligter.BEKLAGTE
									&& !baumbachBeteiligtenListe.isGesamtschuldnerschaft(angriff)) {
								final double angriffsUnterliegen = baumbachBeteiligtenListe.getStreitwertFor(angriff)
										- baumbachBeteiligtenListe.getUnterliegenFor(angriff, false, true);
								/*
								 * Nach Vorhandensein von Gesamtschuldnerschaften unterscheiden
								 */
								if (baumbachBeteiligtenListe.enthaeltGesamtschuldner()
										&& angreiferUnterliegen < angriffsUnterliegen) {
									angreiferUnterliegen = angriffsUnterliegen;
								} else if (baumbachBeteiligtenListe.enthaeltGesamtschuldner()) {
									// nüscht
								} else {
									angreiferUnterliegen += angriffsUnterliegen;
								}
							}
						}
						unterliegen += angreiferUnterliegen;
					}
					/* Unterschleife beim Bk/Widerkläger ggü. Widerbeklagten */
					if (baumbachBeteiligtenListe.getTypFor(index) == Beteiligter.BEKLAGTE
							&& baumbachBeteiligtenListe.isAnWiderklageBeteiligt(index)) {
						double angreiferUnterliegen = 0.0;
						for (int angriff = 0; angriff < baumbachBeteiligtenListe.size(); angriff++) {
							if (!baumbachBeteiligtenListe.isGesamtschuldnerschaft(angriff)
									&& baumbachBeteiligtenListe.isAnWiderklageBeteiligt(angriff)
									&& (baumbachBeteiligtenListe.getTypFor(angriff) == Beteiligter.DRITTWIDERBEKLAGTE
											|| baumbachBeteiligtenListe.getTypFor(angriff) == Beteiligter.KLAEGER)) {
								final double angriffsUnterliegen = baumbachBeteiligtenListe.getStreitwertFor(angriff)
										- baumbachBeteiligtenListe.getUnterliegenFor(angriff, false, true);
								/*
								 * Nach Vorhandensein von Gesamtschuldnerschaften unterscheiden
								 */
								if (baumbachBeteiligtenListe.enthaeltGesamtschuldner()
										&& angreiferUnterliegen < angriffsUnterliegen) {
									angreiferUnterliegen = angriffsUnterliegen;
								} else if (baumbachBeteiligtenListe.enthaeltGesamtschuldner()) {
									// nüscht
								} else {
									angreiferUnterliegen += angriffsUnterliegen;
								}
							}
						}
						unterliegen += angreiferUnterliegen;
					}
					/*
					 * Sobald irgend etwas höher als die Berufungsgrenze ist, ist das Urteil
					 * anfechtbar.
					 */
					if (unterliegen > TenorTexter.BERUFUNGS_GRENZE)
						return new Vollstreckbarkeit();
				}
			/*
			 * Wenn wir hier angekommen sind (also aus der Schleife vorher nicht
			 * ausgestiegen sind), ist das Urteil unanfechtbar
			 */
			return new Vollstreckbarkeit(true, false, false, false, true);
		} else {
			return new Vollstreckbarkeit();
		}
	}

}
