/*
 * BaumbachTexter.java
 * eu.gronos.kostenrechner.logic.baumbach (Kostenrechner)
 */
package eu.gronos.kostenrechner.logic.baumbach;

import java.util.ArrayList;
import java.util.List;

import eu.gronos.kostenrechner.logic.TenorTexter;
import eu.gronos.kostenrechner.model.baumbach.BaumbachBeteiligtenListe;
import eu.gronos.kostenrechner.model.baumbach.BaumbachBeteiligter;
import eu.gronos.kostenrechner.model.baumbach.GerichtsKostenBeteiligter;
import eu.gronos.kostenrechner.model.baumbach.Streitgenossen;
import eu.gronos.kostenrechner.model.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.model.tenordaten.HauptsacheEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.HauptsacheVerhaeltnis;
import eu.gronos.kostenrechner.model.tenordaten.KostenGrundEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.KostenTragungsVerhaeltnis;
import eu.gronos.kostenrechner.model.tenordaten.StreitwertEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.VollstreckbarkeitsEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.VollstreckbarkeitsListe;
import eu.gronos.kostenrechner.model.tenordaten.VollstreckungsVerhaeltnis;

/**
 * Die Klasse verfasst die Tenortexte für {@link BaumbachFormel}
 *
 * @author Peter Schuster (setrok)
 * @date 7 Jun 2019
 *
 */
public class BaumbachTexter extends TenorTexter<Streitgenossen> {
	private BaumbachImUebrigen imUebrigen;
	private final BaumbachBeteiligtenListe baumbachBeteiligtenListe;
	private double[][] quoten;

	/**
	 * @param baumbachBeteiligtenListe eine {@link BaumbachBeteiligtenListe}
	 */
	public BaumbachTexter(BaumbachBeteiligtenListe baumbachBeteiligtenListe) {
		this.baumbachBeteiligtenListe = baumbachBeteiligtenListe;
	}

	/**
	 * Die Methode baut den Text für die {@link HauptsacheEntscheidungsElemente}
	 * 
	 * @param hauptsache {@link HauptsacheEntscheidungsElemente}
	 * 
	 * @see eu.gronos.kostenrechner.logic.TenorTexter#texteHauptsache(eu.gronos.kostenrechner.model.tenordaten.TenorDatenContainer.HauptsacheEntscheidungsElemente)
	 */
	@Override
	public String texteHauptsache(HauptsacheEntscheidungsElemente hauptsache) {
		StringBuffer tenor = new StringBuffer();
		/*
		 * Ermitteln, an welche Stelle des Tenors die Abweisung im Übrigen gehört.
		 * Jeweils an die Stelle, an der der letzte Beklagte (bzw. Widerklage) erwähnt
		 * wird; und wo das ist, das sagt mir findeGesamtschuldnerIndices
		 */
		boolean letzterBeklagter = false;
		/* die Prozessverhältnisse sind schon in der richtigen Reihenfolge */
		for (HauptsacheVerhaeltnis verhaeltnis : hauptsache.prozessverhaeltnisse) {
			/*
			 * Nach darüber hinaus gucken, um es an der richtigen Stelle in den Tenor zu
			 * stecken.
			 */
			String darueberHinaus;
			if (verhaeltnis.isDarueberHinaus()) {
				darueberHinaus = TenorTexter.DARUEBER_HINAUS;
			} else {
				darueberHinaus = "";
			}
			Beteiligter glaeubiger = verhaeltnis.getGlaeubiger();
			Beteiligter schuldner = verhaeltnis.getSchuldner();
			boolean einzigerSeinerArt = isEinzigerSeinerArt(schuldner);
			if (schuldner.getTyp() == Beteiligter.BEKLAGTE) {
				/* erst die Verurteilungen nach der Klage */
				tenor.append(
						super.verbessereGesamtschuldner(String.format("%s %s verurteilt, %,.2f EUR an %s zu zahlen.%n",
								satzBeginn(schuldner.parteiBezeichner(Beteiligter.NOMINATIV, schuldner.getLfdNr(),
										einzigerSeinerArt)),
								super.getWirdWerden(schuldner.isPlural()) + darueberHinaus,
								((BaumbachBeteiligter) schuldner).getUnterliegen(),
								glaeubiger.parteiBezeichner(Beteiligter.AKKUSATIV, glaeubiger.getLfdNr(), true))));
			} else {
				/* dann die Verurteilungen nach der Widerklage */
				if (!letzterBeklagter) {
					/*
					 * wenn letzterBeklagter noch false ist, dann ausgeben, ob die Klage
					 * ganz/imÜbrigen abgewiesen ist
					 */
					tenor.append(imUebrigen.toKlageAbweisungsString());
					letzterBeklagter = true;
				}
				boolean einzigerGlSeinerArt = isEinzigerSeinerArt(glaeubiger);
				tenor.append(super.verbessereGesamtschuldner(
						String.format("Auf die Widerklage %s %s verurteilt, %,.2f EUR an %s zu zahlen.%n",
								super.getWirdWerden(schuldner.isPlural()) + darueberHinaus,
								schuldner.parteiBezeichner(Beteiligter.NOMINATIV, schuldner.getLfdNr(),
										einzigerSeinerArt),
								((BaumbachBeteiligter) schuldner).getUnterliegen(), glaeubiger.parteiBezeichner(
										Beteiligter.AKKUSATIV, glaeubiger.getLfdNr(), einzigerGlSeinerArt))));
			}
		}
		/*
		 * Zusatz für Voll- oder Teilabweisung der Klage, falls letzterBeklagter immer
		 * noch false
		 */
		if (hauptsache.prozessverhaeltnisse.isEmpty() || !letzterBeklagter) {
			tenor.append(imUebrigen.toKlageAbweisungsString());
		}
		/* Zusatz für Voll- oder Teilabweisung der Widerklage */
		if (baumbachBeteiligtenListe.enthaeltAnWiderklageBeteiligte()) {
			tenor.append(imUebrigen.toWiderklageAbweisungsString());
		}
		return tenor.toString();
	}

	/**
	 * Die Methode erzeugt den Kostentenor aus den übergebenen
	 * {@link KostenGrundEntscheidungsElemente}n. Vorher muss
	 * {@link #setQuoten(double[][])} gesetzt sein; das Array wird noch gebraucht,
	 * um {@link #findeGleichlautendeQuoten(double[][])} nutzen zu können.
	 * 
	 * @param elemente die {@link KostenGrundEntscheidungsElemente}
	 * @return den Text des Kostentenors als {@link String} mit abschließendem
	 *         Zeilenumbruch
	 * 
	 * @see eu.gronos.kostenrechner.logic.TenorTexter#texteKostenentscheidung(eu.gronos.kostenrechner.model.tenordaten.TenorDatenContainer.KostenGrundEntscheidungsElemente)
	 */
	@Override
	public String texteKostenentscheidung(KostenGrundEntscheidungsElemente elemente) {
		if (quoten == null)
			throw new NullPointerException("Das Array quoten[][] darf nicht null sein!");
		StringBuffer tenor = new StringBuffer();
		StringBuffer satz = new StringBuffer();
		ArrayList<StringBuffer> kostenTenorSaetze = new ArrayList<StringBuffer>();
		/*
		 * Dann gucken, ob irgendwas genauso verteilt wird wie die Gerichtskosten.
		 */
		ArrayList<Integer> gleichlautendezeilen = findeGleichlautendeQuoten(quoten);
		boolean alleKostenGleich = (gleichlautendezeilen.size() == quoten.length - 1);
		BaumbachBeteiligtenListe gleiche = new BaumbachBeteiligtenListe();
		/*
		 * Die KostenGrundEntscheidungsElemente sind nach Gläubiger sortiert, deshalb
		 * muss man den Wechsel merken
		 */
		Beteiligter glaeubiger = null;

		for (int index = 0; index < elemente.prozessverhaeltnisse.size(); index++) {
			KostenTragungsVerhaeltnis verhaeltnis = elemente.prozessverhaeltnisse.get(index);
			/*
			 * Gläubigerwechsel abfragen. Bei einem Wechsel muss ich den vorigen Satz
			 * beenden und einen neuen Satzanfang einfügen.
			 */
			boolean gewechselt = !verhaeltnis.getGlaeubiger().equals(glaeubiger);
			glaeubiger = verhaeltnis.getGlaeubiger();
			/*
			 * Wenn der Gläubiger schon mit den Gerichtskosten ausgegeben wurde, dann
			 * überspringe ihn, zu den Kosten, die wie die Gerichtskosten verteilt werden,
			 * keine eigene Zeile ausgeben.
			 */
			if (gleiche.contains(glaeubiger))
				continue;
			/*
			 * Gerichtskosten ausgeben; dabei prüfen, ob die Kosten eines Gläubigers (z.B.
			 * des Klägers) genauso verteilt sind
			 */
			if (verhaeltnis.getGlaeubiger() instanceof GerichtsKostenBeteiligter) {
				if (gewechselt) {
					if (alleKostenGleich) {
						/* Wenn alle Kosten gleich verteilt sind: */
						satz.append(TenorTexter.DIE_KOSTEN_DES_RECHTSSTREITS_HABEN);
					} else if (gleichlautendezeilen == null || gleichlautendezeilen.isEmpty()) {
						/* Wenn nur die Gerichtskosten so verteilt sind: */
						satz.append(TenorTexter.DIE_GERICHTSKOSTEN_HABEN);
					} else {
						/*
						 * wenn etwas genauso verteilt ist wie die Gerichtskosten: anhängen und den
						 * Gläubiger der gleich verteilten Kosten merken.
						 */
						satz.append(TenorTexter.DIE_GERICHTSKOSTEN);
						for (Integer i : gleichlautendezeilen) {
							BaumbachBeteiligter gleicher = baumbachBeteiligtenListe.get(i.intValue() - 1);
							satz.append((TenorTexter.AUSSERGERICHTLICHE_KOSTEN_TEIL
									+ gleicher.parteiBezeichner(Beteiligter.GENITIV, gleicher.getLfdNr(),
											isEinzigerSeinerArt(gleicher) /* false */ )));
							gleiche.add(gleicher);
						}
						/*
						 * TODO komisch, wenn einer allein die Kosten trägt: mehrfaches "und" (immer
						 * noch?) - aber besser als nur Kommata
						 */
						satz = new StringBuffer(undStattKomma(satz.toString()));
						satz.append(" haben ");
					}
				}
			} else {
				if (gewechselt) {
					/*
					 * Wenn Gläubiger wechselt: Gesamt-Kostentenor ergänzen und neuen Satz beginnen.
					 */
					kostenTenorSaetze.add(satz);
					satz = new StringBuffer(String.format(TenorTexter.DIE_AUSSERG_KOSTEN, glaeubiger.parteiBezeichner(
							Beteiligter.GENITIV, glaeubiger.getLfdNr(), isEinzigerSeinerArt(glaeubiger) /* false */ )));
				}
			}
			String darueberHinaus = "";
			if (verhaeltnis.isDarueberHinaus()) {
				darueberHinaus = TenorTexter.DARUEBER_HINAUS;
			}
			/*
			 * Wenn eine Partei 100% trägt, nur den Bezeichner ausgeben, sonst auch die
			 * Prozente; umgekehrte Abfrage wegen Double-Ungenauigkeiten.
			 */
			if (verhaeltnis.getKostenBruchteil() < 1.0) {
				satz.append(String.format(TenorTexter.PARTEI_ZU_PROZENT,
						verhaeltnis.getSchuldner().parteiBezeichner(Beteiligter.NOMINATIV,
								verhaeltnis.getSchuldner().getLfdNr(),
								isEinzigerSeinerArt(verhaeltnis.getSchuldner())/* false */ ) + darueberHinaus,
						(100.0 * verhaeltnis.getKostenBruchteil())));
			} else {
				satz.append(verhaeltnis.getSchuldner().parteiBezeichner(Beteiligter.NOMINATIV,
						verhaeltnis.getSchuldner().getLfdNr(),
						isEinzigerSeinerArt(verhaeltnis.getSchuldner()) /* false */ ));
			}
		}
		/*
		 * Zum Schluss nochmal: Gesamt-Kostentenor ergänzen.
		 */
		kostenTenorSaetze.add(satz);
		for (StringBuffer sb : kostenTenorSaetze) {
			// tenor.append();//kostenTenorSaetze[zeile]
			tenor.append(super.hatStattHaben(undStattKomma(letztesKommaWeg(sb.toString()))));
			tenor.append(TenorTexter.ZU_TRAGEN);
		}
		if (!alleKostenGleich)
			tenor.append(TenorTexter.KOSTENTRAGUNG_SELBST);
		return tenor.toString();
	}

	@Override
	public String texteVollstreckbarkeit(VollstreckbarkeitsEntscheidungsElemente elemente,
			VollstreckbarkeitsListe liste) {
		String schuldner = TenorTexter.JEWEILIGER_VOLLSTRECKUNGSSCHULDNER;
		String glaeubiger = TenorTexter.JEWEILIGER_VOLLSTRECKUNGSGLAEUBIGER;

		if (liste.sindAlle708iVm713()) {
			return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR;
		} else if (liste.sindAlle709()) {
			return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709;
		} else if (liste.sindAlle708iVm711()) {
			if (1 == zaehle708iVm711(elemente.prozessverhaeltnisse)) {
				// liste.werIst708iVm711().size()
				/*
				 * Wenn es nur ein solches Verhältnis gibt, dann die Parteien konkret benennen.
				 */
				Beteiligter schuldnerBeteiligter = elemente.prozessverhaeltnisse.get(0).getSchuldner();
				Beteiligter glaeubigerBeteiligter = elemente.prozessverhaeltnisse.get(0).getGlaeubiger();
				schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.DATIV, schuldnerBeteiligter.getLfdNr(),
						isEinzigerSeinerArt(schuldnerBeteiligter));
				glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.NOMINATIV,
						glaeubigerBeteiligter.getLfdNr(), isEinzigerSeinerArt(glaeubigerBeteiligter));
			}
			/* Sonst bleibt's beim "jeweiligen" Gläubiger/Schuldner */
			return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR + String
					.format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711, satzBeginn(schuldner), glaeubiger);
		}
		StringBuffer tenor = new StringBuffer(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR);
		/*
		 * Bei gemischtem Tenor erst die 709-er, dann die 708/711-er Verhältnisse
		 * ausgeben. Dafür zwei StringBuffer füllen.
		 */
		StringBuffer tenor709 = new StringBuffer();
		StringBuffer tenor708 = new StringBuffer();
		final int size709 = zaehle709(elemente.prozessverhaeltnisse);// liste.werIst709().size();
		final int size708u711 = zaehle708iVm711(elemente.prozessverhaeltnisse);// liste.werIst708iVm711().size();
		if (size709 > 1) {
			tenor709 = new StringBuffer("Hinsichtlich des jeweiligen Vollstreckungsverhältnisses ");
		}
		if (size708u711 > 1) {
			tenor708 = new StringBuffer("Hinsichtlich des jeweiligen Vollstreckungsverhältnisses ");
		}

		/*
		 * Alle VollstreckungsVerhaeltnisse durchgehen. Jeweils nach
		 * Vollstreckbarkeitsart trennen. Innerhalb der Vollstreckbarkeitsart gucken, ob
		 * es das einzige der Art ist.
		 */
		for (VollstreckungsVerhaeltnis verhaeltnis : elemente.prozessverhaeltnisse) {
			Beteiligter schuldnerBeteiligter = verhaeltnis.getSchuldner();
			Beteiligter glaeubigerBeteiligter = verhaeltnis.getGlaeubiger();
			if (verhaeltnis.getVollstreckbarkeit().isB709s1()) {
				/* Gegen Sicherheitsleistung. */
				if (size709 == 1) {
					schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.DATIV,
							schuldnerBeteiligter.getLfdNr(), isEinzigerSeinerArt(schuldnerBeteiligter));
					glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.AKKUSATIV,
							glaeubigerBeteiligter.getLfdNr(), isEinzigerSeinerArt(glaeubigerBeteiligter));
					tenor709 = new StringBuffer(String.format("Für %s gegenüber %s", glaeubiger, schuldner));
				} else {
					glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.GENITIV,
							glaeubigerBeteiligter.getLfdNr(), isEinzigerSeinerArt(glaeubigerBeteiligter));
					schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.AKKUSATIV,
							schuldnerBeteiligter.getLfdNr(), isEinzigerSeinerArt(schuldnerBeteiligter));
					tenor709.append(String.format("%s gegen %s, ", glaeubiger, schuldner));
				}
			} else if (verhaeltnis.getVollstreckbarkeit().isB711()) {
				/* Abwendungsbefugnis */
				if (size708u711 == 1) {
					schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.DATIV,
							schuldnerBeteiligter.getLfdNr(), isEinzigerSeinerArt(schuldnerBeteiligter));
					glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.NOMINATIV,
							glaeubigerBeteiligter.getLfdNr(), isEinzigerSeinerArt(glaeubigerBeteiligter));
					tenor708 = new StringBuffer(String.format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711,
							satzBeginn(schuldner), glaeubiger));
				} else {
					glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.GENITIV,
							glaeubigerBeteiligter.getLfdNr(), isEinzigerSeinerArt(glaeubigerBeteiligter));
					schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.AKKUSATIV,
							schuldnerBeteiligter.getLfdNr(), isEinzigerSeinerArt(schuldnerBeteiligter));
					tenor708.append(String.format("%s gegen %s, ", glaeubiger, schuldner));
				}
			}
		}

		/*
		 * Erst die §709-Vollstreckbarkeitsverhältnisse anhängen – hier muss immer der
		 * Abschlus dran, aber ggfs. noch Kommata weg!
		 */
		if (size709 > 0) {
			if (1 == size709) {
				tenor.append(tenor709);
			} else {
				tenor.append(TenorTexter.undStattKomma(TenorTexter.letztesKommaWeg(tenor709.toString())));
			}
			tenor.append(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709_EINZELN);
		}

		/* Dann die §708-Vollstreckbarkeitsverhältnisse */
		if (size708u711 == 1) {
			/* Bei einem ist der Abschluss schon dran */
			tenor.append(tenor708);
		} else if (size708u711 > 1) {
			/* wenn es mehrere gibt, muss noch ein Abschluss dran. */
			tenor.append(String.format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711_WERIST,
					TenorTexter.undStattKomma(TenorTexter.letztesKommaWeg(tenor708.toString())),
					TenorTexter.JEWEILIGER_VOLLSTRECKUNGSSCHULDNER, TenorTexter.JEWEILIGER_VOLLSTRECKUNGSGLAEUBIGER));
		}
		return tenor.toString();
	}

	/**
	 * Die Methode erzeugt eine Streitwertfestsetzung, einen formatierten String
	 * nach dem Muster "Der Streitwert wird auf %,.2f EUR festgesetzt.%n", bei einer
	 * Widerklage eine Zusammensetzung aus Gesamtstreitwert, Klagewert und
	 * Widerklagewert.
	 * 
	 * @param elemente {@link StreitwertEntscheidungsElemente}
	 * 
	 * @see eu.gronos.kostenrechner.logic.TenorTexter#texteStreitwert(eu.gronos.kostenrechner.model.tenordaten.TenorDatenContainer.StreitwertEntscheidungsElemente)
	 */
	@Override
	public String texteStreitwert(StreitwertEntscheidungsElemente elemente) {
		if (baumbachBeteiligtenListe.enthaeltAnWiderklageBeteiligte()) {
			// Bei einer Widerklage müssen beide Streitwerte ausgegeben werden
			return String.format(TenorTexter.STREITWERT_KLAGE_WIDERKLAGE, (double) elemente.streitwerte.get(0),
					(double) elemente.streitwerte.get(1), (double) elemente.streitwerte.get(2));
		} else {
			// Ohne Widerklage muss nur ein Streitwert ausgegeben werden
			return String.format(TenorTexter.STREITWERT_KLAGE, (double) elemente.streitwerte.get(0));
		}
	}

	/**
	 * @param quoten d. {@link #quoten}, d. gesetzt werden soll als
	 *               {@link double[][]}.
	 */
	public void setQuoten(double[][] quoten) {
		this.quoten = quoten;
	}

	/**
	 * @param imUebrigen d. {@link #imUebrigen}, d. gesetzt werden soll als
	 *                   {@link BaumbachImUebrigen}.
	 */
	public void setImUebrigen(BaumbachImUebrigen imUebrigen) {
		this.imUebrigen = imUebrigen;
	}

	/**
	 * Die Methode ermittelt, ob der Beteiligte der einzige für seinen
	 * Beteiligten-Typ ist. Gesamtschuldner zählen nicht.
	 * 
	 * @param beteiligter ein {@link Beteiligter}
	 * 
	 * @return false, wenn es noch andere Beteiligte vom selben Typ gibt, die nicht
	 *         BaumbachGesamtschuldner sind; sonst true.
	 */
	private boolean isEinzigerSeinerArt(Beteiligter beteiligter) {
		return baumbachBeteiligtenListe.findeGesamtschuldnerIndices(beteiligter.getTyp(), false).size() < 2;
	}

	/**
	 * Findet alle Kostenverteilungen von außergerichtlichen Kosten, die mit der
	 * Verteilung der Gerichtskosten völlig identisch sind (z.B. werden ohne
	 * Widerklage die Gerichtskosten und die außergerichtlichen Kosten des Klägers
	 * oft gleich verteilt oder eine Partei muss z.B. alles tragen).
	 * 
	 * @param quoten ein Array double[][] mit allen Kostenquoten
	 * @return eine ArrayList<Integer> mit allen Zeilen, die identisch sind (dürfte
	 *         meist aber nur eine Zeile sein).
	 */
	private ArrayList<Integer> findeGleichlautendeQuoten(double[][] quoten) {
		ArrayList<Integer> liste = new ArrayList<Integer>();
		if (quoten == null || quoten.length < 1)
			return null;
		for (int zeile = 0; zeile < quoten.length; zeile++) {
			if (zeile > 0 && istGleichlautendeQuote(quoten[0], quoten[zeile]))
				liste.add(new Integer(zeile));
		}
		return liste;
	}

	/**
	 * Prüft, ob die beiden double[]-Arrays identisch sind, um zu gucken, ob die
	 * außergerichtlichen Kosten eines bestimmten Beteiligten genauso verteilt sind
	 * wie die Gerichtskosten.
	 * 
	 * @param gerichtskostenQuoten double[]-Array mit der Verteilung der
	 *                             Gerichtskosten
	 * @param andereQuoten         double[]-Array mit der Verteilung der
	 *                             außergerichtlichen Kosten eines Beteiligten
	 * @return true, wenn identisch
	 */
	private boolean istGleichlautendeQuote(double[] gerichtskostenQuoten, double[] andereQuoten) {
		boolean zwischen = true;
		for (int spalte = 0; spalte < andereQuoten.length; spalte++) {
			/*
			 * Muss auf zwei Nachkommastellen runden, sonst erkennt er Identität nicht immer
			 * wegen Double-Rundungsfehlern.
			 */
			if (Math.round(gerichtskostenQuoten[spalte] * 100) != Math.round(andereQuoten[spalte] * 100))
				zwischen = false;
		}
		return zwischen;
	}

	private int zaehle708iVm711(List<VollstreckungsVerhaeltnis> liste) {
		int size = 0;
		for (VollstreckungsVerhaeltnis verhaeltnis : liste) {
			if (verhaeltnis.getVollstreckbarkeit().isB711())
				size++;
		}
		return size;
	}
	
	private int zaehle709(List<VollstreckungsVerhaeltnis> liste) {
		int size = 0;
		for (VollstreckungsVerhaeltnis verhaeltnis : liste) {
			if (verhaeltnis.getVollstreckbarkeit().isB709s1())
				size++;
		}
		return size;
	}
	
	// private String texte(VollstreckbarkeitsEntscheidungsElemente elemente,
	// VollstreckbarkeitsListe liste) {
	// String schuldner = TenorTexter.JEWEILIGER_VOLLSTRECKUNGSSCHULDNER;
	// String glaeubiger = TenorTexter.JEWEILIGER_VOLLSTRECKUNGSGLAEUBIGER;
	//
	// if (liste.sindAlle708iVm713()) {
	// return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR;
	// } else if (liste.sindAlle709()) {
	// return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709;
	// } else if (liste.sindAlle708iVm711()) {
	// ArrayList<Integer> werIst708iVm711 = liste.werIst708iVm711();
	// /*
	// * Wenn es nur ein solches Verhältnis gibt, dann die Parteien richtig
	// ausgeben.
	// */
	// if (werIst708iVm711.size() == 1) {
	// Beteiligter schuldnerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst708iVm711.get(0)))
	// .getSchuldner();
	// Beteiligter glaeubigerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst708iVm711.get(0)))
	// .getGlaeubiger();
	// schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.DATIV,
	// schuldnerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(schuldnerBeteiligter));
	// glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.NOMINATIV,
	// glaeubigerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(glaeubigerBeteiligter));
	// }
	// /* Sonst bleibt's beim "jeweiligen" Gläubiger/Schuldner */
	// return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR + String
	// .format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711,
	// satzBeginn(schuldner), glaeubiger);
	// }
	// StringBuffer tenor = new
	// StringBuffer(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR);
	// /*
	// * Bei gemischtem Tenor erst die 709-er, dann die 708/711-er Verhältnisse
	// * ausgeben.
	// */
	// ArrayList<Integer> werIst709 = liste.werIst709();
	// if (werIst709.size() == 1) {
	// Beteiligter schuldnerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst709.get(0))).getSchuldner();
	// Beteiligter glaeubigerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst709.get(0)))
	// .getGlaeubiger();
	// schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.DATIV,
	// schuldnerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(schuldnerBeteiligter));
	// glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.AKKUSATIV,
	// glaeubigerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(glaeubigerBeteiligter));
	// tenor.append(String.format("Für %s gegenüber %s", glaeubiger, schuldner)
	// + TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709_EINZELN);
	// } else if (!werIst709.isEmpty()) {
	// tenor.append(liste.benenneWerIst(werIst709, baumbachBeteiligtenListe)
	// + TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709_EINZELN);
	// }
	// ArrayList<Integer> werIst708iVm711 = liste.werIst708iVm711();
	// if (werIst708iVm711.size() == 1) {
	// Beteiligter schuldnerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst708iVm711.get(0)))
	// .getSchuldner();
	// Beteiligter glaeubigerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst708iVm711.get(0)))
	// .getGlaeubiger();
	// schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.DATIV,
	// schuldnerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(schuldnerBeteiligter));
	// glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.NOMINATIV,
	// glaeubigerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(glaeubigerBeteiligter));
	// tenor.append(String.format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711,
	// satzBeginn(schuldner),
	// glaeubiger));
	// } else if (!werIst708iVm711.isEmpty()) {
	// tenor.append(String.format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711_WERIST,
	// liste.benenneWerIst(werIst708iVm711, baumbachBeteiligtenListe),
	// TenorTexter.JEWEILIGER_VOLLSTRECKUNGSSCHULDNER,
	// TenorTexter.JEWEILIGER_VOLLSTRECKUNGSGLAEUBIGER));
	// }
	// tenor.append(texte(elemente, liste));
	// return tenor.toString();
	// }
	//
	// /**
	// * Die Methode dient dazu, alle Vollstreckungsverhältnisse einer von {@link
	// * #werIst708iVm711()} oder {@link #werIst709()} zurückgegebenen {@link
	// * ArrayList} in einen lesbaren String umzuwandeln.
	// *
	// * @param werIst eine {@link ArrayList}
	// *
	// * @param baumbachBeteiligtenListe anhand der Liste wird überprüft, ob
	// Gläubiger
	// * bzw. Schuldner der einzige seiner Art sind.
	// *
	// * @return ein String: Hinsichtlich des jeweiligen Vollstreckungsverhältnisses
	// * des &lt;jeweiligen Gläubigers&gt; gegen den &lt;jeweiligen Schuldner&gt;,
	// * (usw.)
	// */
	// public String benenneWerIst(ArrayList<Integer> werIst,
	// BaumbachBeteiligtenListe baumbachBeteiligtenListe) {
	// if (werIst == null || werIst.isEmpty())
	// return "";
	// StringBuffer benannt = new StringBuffer("Hinsichtlich des jeweiligen
	// Vollstreckungsverhältnisses ");
	// for (int index : werIst) {
	// if (get(index) instanceof BaumbachVollstreckbarkeit)
	// benannt.append(
	// ((BaumbachVollstreckbarkeit)
	// get(index)).benenneParteien(baumbachBeteiligtenListe) + ", ");
	// }
	// return
	// TenorTexter.undStattKomma(TenorTexter.letztesKommaWeg(benannt.toString()));
	// }

	// private String texte(VollstreckbarkeitsEntscheidungsElemente elemente,
	// VollstreckbarkeitsListe liste) {
	// String schuldner = TenorTexter.JEWEILIGER_VOLLSTRECKUNGSSCHULDNER;
	// String glaeubiger = TenorTexter.JEWEILIGER_VOLLSTRECKUNGSGLAEUBIGER;
	//
	// if (liste.sindAlle708iVm713()) {
	// return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR;
	// } else if (liste.sindAlle709()) {
	// return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709;
	// } else if (liste.sindAlle708iVm711()) {
	// ArrayList<Integer> werIst708iVm711 = liste.werIst708iVm711();
	// /*
	// * Wenn es nur ein solches Verhältnis gibt, dann die Parteien richtig
	// ausgeben.
	// */
	// if (werIst708iVm711.size() == 1) {
	// Beteiligter schuldnerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst708iVm711.get(0)))
	// .getSchuldner();
	// Beteiligter glaeubigerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst708iVm711.get(0)))
	// .getGlaeubiger();
	// schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.DATIV,
	// schuldnerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(schuldnerBeteiligter));
	// glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.NOMINATIV,
	// glaeubigerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(glaeubigerBeteiligter));
	// }
	// /* Sonst bleibt's beim "jeweiligen" Gläubiger/Schuldner */
	// return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR + String
	// .format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711,
	// satzBeginn(schuldner), glaeubiger);
	// }
	// StringBuffer tenor = new
	// StringBuffer(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR);
	// /*
	// * Bei gemischtem Tenor erst die 709-er, dann die 708/711-er Verhältnisse
	// * ausgeben.
	// */
	// ArrayList<Integer> werIst709 = liste.werIst709();
	// if (werIst709.size() == 1) {
	// Beteiligter schuldnerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst709.get(0))).getSchuldner();
	// Beteiligter glaeubigerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst709.get(0)))
	// .getGlaeubiger();
	// schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.DATIV,
	// schuldnerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(schuldnerBeteiligter));
	// glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.AKKUSATIV,
	// glaeubigerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(glaeubigerBeteiligter));
	// tenor.append(String.format("Für %s gegenüber %s", glaeubiger, schuldner)
	// + TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709_EINZELN);
	// } else if (!werIst709.isEmpty()) {
	// tenor.append(liste.benenneWerIst(werIst709, baumbachBeteiligtenListe)
	// + TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709_EINZELN);
	// }
	// ArrayList<Integer> werIst708iVm711 = liste.werIst708iVm711();
	// if (werIst708iVm711.size() == 1) {
	// Beteiligter schuldnerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst708iVm711.get(0)))
	// .getSchuldner();
	// Beteiligter glaeubigerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst708iVm711.get(0)))
	// .getGlaeubiger();
	// schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.DATIV,
	// schuldnerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(schuldnerBeteiligter));
	// glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.NOMINATIV,
	// glaeubigerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(glaeubigerBeteiligter));
	// tenor.append(String.format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711,
	// satzBeginn(schuldner),
	// glaeubiger));
	// } else if (!werIst708iVm711.isEmpty()) {
	// tenor.append(String.format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711_WERIST,
	// liste.benenneWerIst(werIst708iVm711, baumbachBeteiligtenListe),
	// TenorTexter.JEWEILIGER_VOLLSTRECKUNGSSCHULDNER,
	// TenorTexter.JEWEILIGER_VOLLSTRECKUNGSGLAEUBIGER));
	// }
	// tenor.append(texte(elemente, liste));
	// return tenor.toString();
	// }
	//
	// /**
	// * Die Methode dient dazu, alle Vollstreckungsverhältnisse einer von {@link
	// * #werIst708iVm711()} oder {@link #werIst709()} zurückgegebenen {@link
	// * ArrayList} in einen lesbaren String umzuwandeln.
	// *
	// * @param werIst eine {@link ArrayList}
	// *
	// * @param baumbachBeteiligtenListe anhand der Liste wird überprüft, ob
	// Gläubiger
	// * bzw. Schuldner der einzige seiner Art sind.
	// *
	// * @return ein String: Hinsichtlich des jeweiligen Vollstreckungsverhältnisses
	// * des &lt;jeweiligen Gläubigers&gt; gegen den &lt;jeweiligen Schuldner&gt;,
	// * (usw.)
	// */
	// public String benenneWerIst(ArrayList<Integer> werIst,
	// BaumbachBeteiligtenListe baumbachBeteiligtenListe) {
	// if (werIst == null || werIst.isEmpty())
	// return "";
	// StringBuffer benannt = new StringBuffer("Hinsichtlich des jeweiligen
	// Vollstreckungsverhältnisses ");
	// for (int index : werIst) {
	// if (get(index) instanceof BaumbachVollstreckbarkeit)
	// benannt.append(
	// ((BaumbachVollstreckbarkeit)
	// get(index)).benenneParteien(baumbachBeteiligtenListe) + ", ");
	// }
	// return
	// TenorTexter.undStattKomma(TenorTexter.letztesKommaWeg(benannt.toString()));
	// }


	// private String texte(VollstreckbarkeitsEntscheidungsElemente elemente,
	// VollstreckbarkeitsListe liste) {
	// String schuldner = TenorTexter.JEWEILIGER_VOLLSTRECKUNGSSCHULDNER;
	// String glaeubiger = TenorTexter.JEWEILIGER_VOLLSTRECKUNGSGLAEUBIGER;
	//
	// if (liste.sindAlle708iVm713()) {
	// return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR;
	// } else if (liste.sindAlle709()) {
	// return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709;
	// } else if (liste.sindAlle708iVm711()) {
	// ArrayList<Integer> werIst708iVm711 = liste.werIst708iVm711();
	// /*
	// * Wenn es nur ein solches Verhältnis gibt, dann die Parteien richtig
	// ausgeben.
	// */
	// if (werIst708iVm711.size() == 1) {
	// Beteiligter schuldnerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst708iVm711.get(0)))
	// .getSchuldner();
	// Beteiligter glaeubigerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst708iVm711.get(0)))
	// .getGlaeubiger();
	// schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.DATIV,
	// schuldnerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(schuldnerBeteiligter));
	// glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.NOMINATIV,
	// glaeubigerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(glaeubigerBeteiligter));
	// }
	// /* Sonst bleibt's beim "jeweiligen" Gläubiger/Schuldner */
	// return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR + String
	// .format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711,
	// satzBeginn(schuldner), glaeubiger);
	// }
	// StringBuffer tenor = new
	// StringBuffer(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR);
	// /*
	// * Bei gemischtem Tenor erst die 709-er, dann die 708/711-er Verhältnisse
	// * ausgeben.
	// */
	// ArrayList<Integer> werIst709 = liste.werIst709();
	// if (werIst709.size() == 1) {
	// Beteiligter schuldnerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst709.get(0))).getSchuldner();
	// Beteiligter glaeubigerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst709.get(0)))
	// .getGlaeubiger();
	// schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.DATIV,
	// schuldnerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(schuldnerBeteiligter));
	// glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.AKKUSATIV,
	// glaeubigerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(glaeubigerBeteiligter));
	// tenor.append(String.format("Für %s gegenüber %s", glaeubiger, schuldner)
	// + TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709_EINZELN);
	// } else if (!werIst709.isEmpty()) {
	// tenor.append(liste.benenneWerIst(werIst709, baumbachBeteiligtenListe)
	// + TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709_EINZELN);
	// }
	// ArrayList<Integer> werIst708iVm711 = liste.werIst708iVm711();
	// if (werIst708iVm711.size() == 1) {
	// Beteiligter schuldnerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst708iVm711.get(0)))
	// .getSchuldner();
	// Beteiligter glaeubigerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst708iVm711.get(0)))
	// .getGlaeubiger();
	// schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.DATIV,
	// schuldnerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(schuldnerBeteiligter));
	// glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.NOMINATIV,
	// glaeubigerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(glaeubigerBeteiligter));
	// tenor.append(String.format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711,
	// satzBeginn(schuldner),
	// glaeubiger));
	// } else if (!werIst708iVm711.isEmpty()) {
	// tenor.append(String.format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711_WERIST,
	// liste.benenneWerIst(werIst708iVm711, baumbachBeteiligtenListe),
	// TenorTexter.JEWEILIGER_VOLLSTRECKUNGSSCHULDNER,
	// TenorTexter.JEWEILIGER_VOLLSTRECKUNGSGLAEUBIGER));
	// }
	// tenor.append(texte(elemente, liste));
	// return tenor.toString();
	// }
	//
	// /**
	// * Die Methode dient dazu, alle Vollstreckungsverhältnisse einer von {@link
	// * #werIst708iVm711()} oder {@link #werIst709()} zurückgegebenen {@link
	// * ArrayList} in einen lesbaren String umzuwandeln.
	// *
	// * @param werIst eine {@link ArrayList}
	// *
	// * @param baumbachBeteiligtenListe anhand der Liste wird überprüft, ob
	// Gläubiger
	// * bzw. Schuldner der einzige seiner Art sind.
	// *
	// * @return ein String: Hinsichtlich des jeweiligen Vollstreckungsverhältnisses
	// * des &lt;jeweiligen Gläubigers&gt; gegen den &lt;jeweiligen Schuldner&gt;,
	// * (usw.)
	// */
	// public String benenneWerIst(ArrayList<Integer> werIst,
	// BaumbachBeteiligtenListe baumbachBeteiligtenListe) {
	// if (werIst == null || werIst.isEmpty())
	// return "";
	// StringBuffer benannt = new StringBuffer("Hinsichtlich des jeweiligen
	// Vollstreckungsverhältnisses ");
	// for (int index : werIst) {
	// if (get(index) instanceof BaumbachVollstreckbarkeit)
	// benannt.append(
	// ((BaumbachVollstreckbarkeit)
	// get(index)).benenneParteien(baumbachBeteiligtenListe) + ", ");
	// }
	// return
	// TenorTexter.undStattKomma(TenorTexter.letztesKommaWeg(benannt.toString()));
	// }

	// private String texte(VollstreckbarkeitsEntscheidungsElemente elemente,
	// VollstreckbarkeitsListe liste) {
	// String schuldner = TenorTexter.JEWEILIGER_VOLLSTRECKUNGSSCHULDNER;
	// String glaeubiger = TenorTexter.JEWEILIGER_VOLLSTRECKUNGSGLAEUBIGER;
	//
	// if (liste.sindAlle708iVm713()) {
	// return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR;
	// } else if (liste.sindAlle709()) {
	// return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709;
	// } else if (liste.sindAlle708iVm711()) {
	// ArrayList<Integer> werIst708iVm711 = liste.werIst708iVm711();
	// /*
	// * Wenn es nur ein solches Verhältnis gibt, dann die Parteien richtig
	// ausgeben.
	// */
	// if (werIst708iVm711.size() == 1) {
	// Beteiligter schuldnerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst708iVm711.get(0)))
	// .getSchuldner();
	// Beteiligter glaeubigerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst708iVm711.get(0)))
	// .getGlaeubiger();
	// schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.DATIV,
	// schuldnerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(schuldnerBeteiligter));
	// glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.NOMINATIV,
	// glaeubigerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(glaeubigerBeteiligter));
	// }
	// /* Sonst bleibt's beim "jeweiligen" Gläubiger/Schuldner */
	// return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR + String
	// .format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711,
	// satzBeginn(schuldner), glaeubiger);
	// }
	// StringBuffer tenor = new
	// StringBuffer(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR);
	// /*
	// * Bei gemischtem Tenor erst die 709-er, dann die 708/711-er Verhältnisse
	// * ausgeben.
	// */
	// ArrayList<Integer> werIst709 = liste.werIst709();
	// if (werIst709.size() == 1) {
	// Beteiligter schuldnerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst709.get(0))).getSchuldner();
	// Beteiligter glaeubigerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst709.get(0)))
	// .getGlaeubiger();
	// schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.DATIV,
	// schuldnerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(schuldnerBeteiligter));
	// glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.AKKUSATIV,
	// glaeubigerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(glaeubigerBeteiligter));
	// tenor.append(String.format("Für %s gegenüber %s", glaeubiger, schuldner)
	// + TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709_EINZELN);
	// } else if (!werIst709.isEmpty()) {
	// tenor.append(liste.benenneWerIst(werIst709, baumbachBeteiligtenListe)
	// + TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709_EINZELN);
	// }
	// ArrayList<Integer> werIst708iVm711 = liste.werIst708iVm711();
	// if (werIst708iVm711.size() == 1) {
	// Beteiligter schuldnerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst708iVm711.get(0)))
	// .getSchuldner();
	// Beteiligter glaeubigerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst708iVm711.get(0)))
	// .getGlaeubiger();
	// schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.DATIV,
	// schuldnerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(schuldnerBeteiligter));
	// glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.NOMINATIV,
	// glaeubigerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(glaeubigerBeteiligter));
	// tenor.append(String.format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711,
	// satzBeginn(schuldner),
	// glaeubiger));
	// } else if (!werIst708iVm711.isEmpty()) {
	// tenor.append(String.format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711_WERIST,
	// liste.benenneWerIst(werIst708iVm711, baumbachBeteiligtenListe),
	// TenorTexter.JEWEILIGER_VOLLSTRECKUNGSSCHULDNER,
	// TenorTexter.JEWEILIGER_VOLLSTRECKUNGSGLAEUBIGER));
	// }
	// tenor.append(texte(elemente, liste));
	// return tenor.toString();
	// }
	//
	// /**
	// * Die Methode dient dazu, alle Vollstreckungsverhältnisse einer von {@link
	// * #werIst708iVm711()} oder {@link #werIst709()} zurückgegebenen {@link
	// * ArrayList} in einen lesbaren String umzuwandeln.
	// *
	// * @param werIst eine {@link ArrayList}
	// *
	// * @param baumbachBeteiligtenListe anhand der Liste wird überprüft, ob
	// Gläubiger
	// * bzw. Schuldner der einzige seiner Art sind.
	// *
	// * @return ein String: Hinsichtlich des jeweiligen Vollstreckungsverhältnisses
	// * des &lt;jeweiligen Gläubigers&gt; gegen den &lt;jeweiligen Schuldner&gt;,
	// * (usw.)
	// */
	// public String benenneWerIst(ArrayList<Integer> werIst,
	// BaumbachBeteiligtenListe baumbachBeteiligtenListe) {
	// if (werIst == null || werIst.isEmpty())
	// return "";
	// StringBuffer benannt = new StringBuffer("Hinsichtlich des jeweiligen
	// Vollstreckungsverhältnisses ");
	// for (int index : werIst) {
	// if (get(index) instanceof BaumbachVollstreckbarkeit)
	// benannt.append(
	// ((BaumbachVollstreckbarkeit)
	// get(index)).benenneParteien(baumbachBeteiligtenListe) + ", ");
	// }
	// return
	// TenorTexter.undStattKomma(TenorTexter.letztesKommaWeg(benannt.toString()));
	// }


	/*
	 * Findet über die BaumbachBeteiligtenListe alle Gesamtschuldnerschaften, in
	 * denen der BaumbachBeteiligte mit dem Index beteiligter ist, dann gucken, ob
	 * sie > 0% tragen müssen
	 * 
	 * @todo TODO hier wohl überflüssig
	 * 
	 * @param index die Position in der BaumbachBeteiligtenListe
	 * 
	 * @param spalten ein Array double[] mit den Quoten aller Beteiligten für eine
	 * Zeile
	 * 
	 * @return true, wenn der BaumbachBeteiligte unter index in einer
	 * Gesamtschuldnerschaft ist, die eine Kostenquote > 0 tragen muss private
	 * boolean hatDarueberHinaus(int index, double[] spalten) { ArrayList<Integer>
	 * liste = baumbachBeteiligtenListe.findeDarueberHinaus(index);
	 * ArrayList<Integer> darueberHinaus = new ArrayList<Integer>(); if (liste !=
	 * null && liste.size() > 0 && spalten != null) for (int i : liste) { if
	 * (spalten[i] > 0.0) darueberHinaus.add(new Integer(i)); } return
	 * darueberHinaus != null && darueberHinaus.size() > 0 && spalten != null; }
	 */

//	public String test(KostenGrundEntscheidungsElemente elemente) {
//		StringBuffer kostenTenor = new StringBuffer();
//		String[] kostenTenorSaetze = new String[quoten.length];
//		/*
//		 * Dann gucken, ob irgendwas genauso verteilt wird wie die Gerichtskosten.
//		 */
//		ArrayList<Integer> gleichlautendezeilen = findeGleichlautendeQuoten(quoten);
//		for (int zeile = 0; zeile < quoten.length; zeile++) {
//			/*
//			 * Einleitung basteln: Die erste Zeile (0) ist für die Gerichtskosten, die
//			 * weiteren je für einen Beteiligten, außer für BaumbachGesamtschuldner, die
//			 * bekommen einen null-String
//			 */
//			if (zeile == 0) {
//				if (gleichlautendezeilen == null || gleichlautendezeilen.isEmpty())
//					kostenTenorSaetze[zeile] = TenorTexter.DIE_GERICHTSKOSTEN_HABEN;
//				else if (gleichlautendezeilen.size() == quoten.length - 1)
//					/* Wenn alle Kosten gleich verteilt sind: */
//					kostenTenorSaetze[zeile] = TenorTexter.DIE_KOSTEN_DES_RECHTSSTREITS_HABEN;
//				else {
//					/*
//					 * wenn etwas genauso verteilt ist wie die Gerichtskosten: anhängen
//					 */
//					kostenTenorSaetze[zeile] = TenorTexter.DIE_GERICHTSKOSTEN;
//					for (Integer i : gleichlautendezeilen)
//						kostenTenorSaetze[zeile] += (TenorTexter.AUSSERGERICHTLICHE_KOSTEN_TEIL
//								+ baumbachBeteiligtenListe.parteiBezeichner(Beteiligter.GENITIV, i.intValue() - 1));
//					/*
//					 * TODO komisch, wenn einer allein die Kosten trägt: mehrfaches "und" - aber
//					 * besser als nur Kommata
//					 */
//					kostenTenorSaetze[zeile] = TenorTexter.undStattKomma(kostenTenorSaetze[zeile]) + " haben ";
//				}
//			} else if (!baumbachBeteiligtenListe.isGesamtschuldnerschaft(zeile - 1)
//					/*
//					 * zu den Kosten, die wie die Gerichtskosten verteilt werden, keine eigene Zeile
//					 * ausgeben.
//					 */
//					&& (gleichlautendezeilen == null || !gleichlautendezeilen.contains(zeile)))
//				kostenTenorSaetze[zeile] = String.format(TenorTexter.DIE_AUSSERG_KOSTEN,
//						baumbachBeteiligtenListe.parteiBezeichner(Beteiligter.GENITIV, zeile - 1));
//			else
//				kostenTenorSaetze[zeile] = null;
//			// Die Spalten mit den BaumbachGesamtschuldner müssen an den
//			// Anfang der Liste ihres Beteiligtentyps
//			for (int spalte : baumbachBeteiligtenListe.beteiligteInTenorReihenfolge(true))
//				if (kostenTenorSaetze[zeile] != null && quoten[zeile][spalte] > 0.0 && quoten[zeile][spalte] < 100.0) {
//					// "darüber hinaus"
//					String darueberHinaus = hatDarueberHinaus(spalte, quoten[zeile]) ? TenorTexter.DARUEBER_HINAUS : "";
//					kostenTenorSaetze[zeile] += String.format(TenorTexter.PARTEI_ZU_PROZENT,
//							baumbachBeteiligtenListe.parteiBezeichner(Beteiligter.NOMINATIV, spalte) + darueberHinaus,
//							quoten[zeile][spalte]);
//				} else if (kostenTenorSaetze[zeile] != null && quoten[zeile][spalte] == 100.0) {
//					/*
//					 * Wenn eine Partei 100% trägt, nur den Bezeichner ausgeben.
//					 */
//					kostenTenorSaetze[zeile] += baumbachBeteiligtenListe.parteiBezeichner(Beteiligter.NOMINATIV,
//							spalte);
//				}
//			// Gesamt-Kostentenor ergänzen
//			if (kostenTenorSaetze[zeile] != null) {
//				kostenTenor.append(super.hatStattHaben(
//						TenorTexter.undStattKomma(TenorTexter.letztesKommaWeg(kostenTenorSaetze[zeile]))));
//				kostenTenor.append(TenorTexter.ZU_TRAGEN);
//			}
//		}
//		if (!kostenTenor.toString().contains(TenorTexter.DIE_KOSTEN_DES_RECHTSSTREITS_HABEN) && !kostenTenor.toString()
//				.contains(super.hatStattHaben(TenorTexter.DIE_KOSTEN_DES_RECHTSSTREITS_HABEN)))
//			kostenTenor.append(TenorTexter.KOSTENTRAGUNG_SELBST);
//		return kostenTenor.toString();
//	}

	// private String texte(VollstreckbarkeitsEntscheidungsElemente elemente,
	// VollstreckbarkeitsListe liste) {
	// String schuldner = TenorTexter.JEWEILIGER_VOLLSTRECKUNGSSCHULDNER;
	// String glaeubiger = TenorTexter.JEWEILIGER_VOLLSTRECKUNGSGLAEUBIGER;
	//
	// if (liste.sindAlle708iVm713()) {
	// return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR;
	// } else if (liste.sindAlle709()) {
	// return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709;
	// } else if (liste.sindAlle708iVm711()) {
	// ArrayList<Integer> werIst708iVm711 = liste.werIst708iVm711();
	// /*
	// * Wenn es nur ein solches Verhältnis gibt, dann die Parteien richtig
	// ausgeben.
	// */
	// if (werIst708iVm711.size() == 1) {
	// Beteiligter schuldnerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst708iVm711.get(0)))
	// .getSchuldner();
	// Beteiligter glaeubigerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst708iVm711.get(0)))
	// .getGlaeubiger();
	// schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.DATIV,
	// schuldnerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(schuldnerBeteiligter));
	// glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.NOMINATIV,
	// glaeubigerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(glaeubigerBeteiligter));
	// }
	// /* Sonst bleibt's beim "jeweiligen" Gläubiger/Schuldner */
	// return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR + String
	// .format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711,
	// satzBeginn(schuldner), glaeubiger);
	// }
	// StringBuffer tenor = new
	// StringBuffer(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR);
	// /*
	// * Bei gemischtem Tenor erst die 709-er, dann die 708/711-er Verhältnisse
	// * ausgeben.
	// */
	// ArrayList<Integer> werIst709 = liste.werIst709();
	// if (werIst709.size() == 1) {
	// Beteiligter schuldnerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst709.get(0))).getSchuldner();
	// Beteiligter glaeubigerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst709.get(0)))
	// .getGlaeubiger();
	// schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.DATIV,
	// schuldnerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(schuldnerBeteiligter));
	// glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.AKKUSATIV,
	// glaeubigerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(glaeubigerBeteiligter));
	// tenor.append(String.format("Für %s gegenüber %s", glaeubiger, schuldner)
	// + TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709_EINZELN);
	// } else if (!werIst709.isEmpty()) {
	// tenor.append(liste.benenneWerIst(werIst709, baumbachBeteiligtenListe)
	// + TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709_EINZELN);
	// }
	// ArrayList<Integer> werIst708iVm711 = liste.werIst708iVm711();
	// if (werIst708iVm711.size() == 1) {
	// Beteiligter schuldnerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst708iVm711.get(0)))
	// .getSchuldner();
	// Beteiligter glaeubigerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst708iVm711.get(0)))
	// .getGlaeubiger();
	// schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.DATIV,
	// schuldnerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(schuldnerBeteiligter));
	// glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.NOMINATIV,
	// glaeubigerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(glaeubigerBeteiligter));
	// tenor.append(String.format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711,
	// satzBeginn(schuldner),
	// glaeubiger));
	// } else if (!werIst708iVm711.isEmpty()) {
	// tenor.append(String.format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711_WERIST,
	// liste.benenneWerIst(werIst708iVm711, baumbachBeteiligtenListe),
	// TenorTexter.JEWEILIGER_VOLLSTRECKUNGSSCHULDNER,
	// TenorTexter.JEWEILIGER_VOLLSTRECKUNGSGLAEUBIGER));
	// }
	// tenor.append(texte(elemente, liste));
	// return tenor.toString();
	// }
	//
	// /**
	// * Die Methode dient dazu, alle Vollstreckungsverhältnisse einer von {@link
	// * #werIst708iVm711()} oder {@link #werIst709()} zurückgegebenen {@link
	// * ArrayList} in einen lesbaren String umzuwandeln.
	// *
	// * @param werIst eine {@link ArrayList}
	// *
	// * @param baumbachBeteiligtenListe anhand der Liste wird überprüft, ob
	// Gläubiger
	// * bzw. Schuldner der einzige seiner Art sind.
	// *
	// * @return ein String: Hinsichtlich des jeweiligen Vollstreckungsverhältnisses
	// * des &lt;jeweiligen Gläubigers&gt; gegen den &lt;jeweiligen Schuldner&gt;,
	// * (usw.)
	// */
	// public String benenneWerIst(ArrayList<Integer> werIst,
	// BaumbachBeteiligtenListe baumbachBeteiligtenListe) {
	// if (werIst == null || werIst.isEmpty())
	// return "";
	// StringBuffer benannt = new StringBuffer("Hinsichtlich des jeweiligen
	// Vollstreckungsverhältnisses ");
	// for (int index : werIst) {
	// if (get(index) instanceof BaumbachVollstreckbarkeit)
	// benannt.append(
	// ((BaumbachVollstreckbarkeit)
	// get(index)).benenneParteien(baumbachBeteiligtenListe) + ", ");
	// }
	// return
	// TenorTexter.undStattKomma(TenorTexter.letztesKommaWeg(benannt.toString()));
	// }

	// private String texte(VollstreckbarkeitsEntscheidungsElemente elemente,
	// VollstreckbarkeitsListe liste) {
	// String schuldner = TenorTexter.JEWEILIGER_VOLLSTRECKUNGSSCHULDNER;
	// String glaeubiger = TenorTexter.JEWEILIGER_VOLLSTRECKUNGSGLAEUBIGER;
	//
	// if (liste.sindAlle708iVm713()) {
	// return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR;
	// } else if (liste.sindAlle709()) {
	// return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709;
	// } else if (liste.sindAlle708iVm711()) {
	// ArrayList<Integer> werIst708iVm711 = liste.werIst708iVm711();
	// /*
	// * Wenn es nur ein solches Verhältnis gibt, dann die Parteien richtig
	// ausgeben.
	// */
	// if (werIst708iVm711.size() == 1) {
	// Beteiligter schuldnerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst708iVm711.get(0)))
	// .getSchuldner();
	// Beteiligter glaeubigerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst708iVm711.get(0)))
	// .getGlaeubiger();
	// schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.DATIV,
	// schuldnerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(schuldnerBeteiligter));
	// glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.NOMINATIV,
	// glaeubigerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(glaeubigerBeteiligter));
	// }
	// /* Sonst bleibt's beim "jeweiligen" Gläubiger/Schuldner */
	// return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR + String
	// .format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711,
	// satzBeginn(schuldner), glaeubiger);
	// }
	// StringBuffer tenor = new
	// StringBuffer(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR);
	// /*
	// * Bei gemischtem Tenor erst die 709-er, dann die 708/711-er Verhältnisse
	// * ausgeben.
	// */
	// ArrayList<Integer> werIst709 = liste.werIst709();
	// if (werIst709.size() == 1) {
	// Beteiligter schuldnerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst709.get(0))).getSchuldner();
	// Beteiligter glaeubigerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst709.get(0)))
	// .getGlaeubiger();
	// schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.DATIV,
	// schuldnerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(schuldnerBeteiligter));
	// glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.AKKUSATIV,
	// glaeubigerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(glaeubigerBeteiligter));
	// tenor.append(String.format("Für %s gegenüber %s", glaeubiger, schuldner)
	// + TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709_EINZELN);
	// } else if (!werIst709.isEmpty()) {
	// tenor.append(liste.benenneWerIst(werIst709, baumbachBeteiligtenListe)
	// + TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709_EINZELN);
	// }
	// ArrayList<Integer> werIst708iVm711 = liste.werIst708iVm711();
	// if (werIst708iVm711.size() == 1) {
	// Beteiligter schuldnerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst708iVm711.get(0)))
	// .getSchuldner();
	// Beteiligter glaeubigerBeteiligter = ((BaumbachVollstreckbarkeit)
	// liste.get(werIst708iVm711.get(0)))
	// .getGlaeubiger();
	// schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.DATIV,
	// schuldnerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(schuldnerBeteiligter));
	// glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.NOMINATIV,
	// glaeubigerBeteiligter.getLfdNr(),
	// isEinzigerSeinerArt(glaeubigerBeteiligter));
	// tenor.append(String.format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711,
	// satzBeginn(schuldner),
	// glaeubiger));
	// } else if (!werIst708iVm711.isEmpty()) {
	// tenor.append(String.format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711_WERIST,
	// liste.benenneWerIst(werIst708iVm711, baumbachBeteiligtenListe),
	// TenorTexter.JEWEILIGER_VOLLSTRECKUNGSSCHULDNER,
	// TenorTexter.JEWEILIGER_VOLLSTRECKUNGSGLAEUBIGER));
	// }
	// tenor.append(texte(elemente, liste));
	// return tenor.toString();
	// }
	//
	// /**
	// * Die Methode dient dazu, alle Vollstreckungsverhältnisse einer von {@link
	// * #werIst708iVm711()} oder {@link #werIst709()} zurückgegebenen {@link
	// * ArrayList} in einen lesbaren String umzuwandeln.
	// *
	// * @param werIst eine {@link ArrayList}
	// *
	// * @param baumbachBeteiligtenListe anhand der Liste wird überprüft, ob
	// Gläubiger
	// * bzw. Schuldner der einzige seiner Art sind.
	// *
	// * @return ein String: Hinsichtlich des jeweiligen Vollstreckungsverhältnisses
	// * des &lt;jeweiligen Gläubigers&gt; gegen den &lt;jeweiligen Schuldner&gt;,
	// * (usw.)
	// */
	// public String benenneWerIst(ArrayList<Integer> werIst,
	// BaumbachBeteiligtenListe baumbachBeteiligtenListe) {
	// if (werIst == null || werIst.isEmpty())
	// return "";
	// StringBuffer benannt = new StringBuffer("Hinsichtlich des jeweiligen
	// Vollstreckungsverhältnisses ");
	// for (int index : werIst) {
	// if (get(index) instanceof BaumbachVollstreckbarkeit)
	// benannt.append(
	// ((BaumbachVollstreckbarkeit)
	// get(index)).benenneParteien(baumbachBeteiligtenListe) + ", ");
	// }
	// return
	// TenorTexter.undStattKomma(TenorTexter.letztesKommaWeg(benannt.toString()));
	// }

//	private String texte(VollstreckbarkeitsEntscheidungsElemente elemente, VollstreckbarkeitsListe liste) {
//		String schuldner = TenorTexter.JEWEILIGER_VOLLSTRECKUNGSSCHULDNER;
//		String glaeubiger = TenorTexter.JEWEILIGER_VOLLSTRECKUNGSGLAEUBIGER;
//
//		if (liste.sindAlle708iVm713()) {
//			return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR;
//		} else if (liste.sindAlle709()) {
//			return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709;
//		} else if (liste.sindAlle708iVm711()) {
//			ArrayList<Integer> werIst708iVm711 = liste.werIst708iVm711();
//			/*
//			 * Wenn es nur ein solches Verhältnis gibt, dann die Parteien richtig ausgeben.
//			 */
//			if (werIst708iVm711.size() == 1) {
//				Beteiligter schuldnerBeteiligter = ((BaumbachVollstreckbarkeit) liste.get(werIst708iVm711.get(0)))
//						.getSchuldner();
//				Beteiligter glaeubigerBeteiligter = ((BaumbachVollstreckbarkeit) liste.get(werIst708iVm711.get(0)))
//						.getGlaeubiger();
//				schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.DATIV, schuldnerBeteiligter.getLfdNr(),
//						isEinzigerSeinerArt(schuldnerBeteiligter));
//				glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.NOMINATIV,
//						glaeubigerBeteiligter.getLfdNr(), isEinzigerSeinerArt(glaeubigerBeteiligter));
//			}
//			/* Sonst bleibt's beim "jeweiligen" Gläubiger/Schuldner */
//			return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR + String
//					.format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711, satzBeginn(schuldner), glaeubiger);
//		}
//		StringBuffer tenor = new StringBuffer(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR);
//		/*
//		 * Bei gemischtem Tenor erst die 709-er, dann die 708/711-er Verhältnisse
//		 * ausgeben.
//		 */
//		ArrayList<Integer> werIst709 = liste.werIst709();
//		if (werIst709.size() == 1) {
//			Beteiligter schuldnerBeteiligter = ((BaumbachVollstreckbarkeit) liste.get(werIst709.get(0))).getSchuldner();
//			Beteiligter glaeubigerBeteiligter = ((BaumbachVollstreckbarkeit) liste.get(werIst709.get(0)))
//					.getGlaeubiger();
//			schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.DATIV, schuldnerBeteiligter.getLfdNr(),
//					isEinzigerSeinerArt(schuldnerBeteiligter));
//			glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.AKKUSATIV, glaeubigerBeteiligter.getLfdNr(),
//					isEinzigerSeinerArt(glaeubigerBeteiligter));
//			tenor.append(String.format("Für %s gegenüber %s", glaeubiger, schuldner)
//					+ TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709_EINZELN);
//		} else if (!werIst709.isEmpty()) {
//			tenor.append(liste.benenneWerIst(werIst709, baumbachBeteiligtenListe)
//					+ TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709_EINZELN);
//		}
//		ArrayList<Integer> werIst708iVm711 = liste.werIst708iVm711();
//		if (werIst708iVm711.size() == 1) {
//			Beteiligter schuldnerBeteiligter = ((BaumbachVollstreckbarkeit) liste.get(werIst708iVm711.get(0)))
//					.getSchuldner();
//			Beteiligter glaeubigerBeteiligter = ((BaumbachVollstreckbarkeit) liste.get(werIst708iVm711.get(0)))
//					.getGlaeubiger();
//			schuldner = schuldnerBeteiligter.parteiBezeichner(Beteiligter.DATIV, schuldnerBeteiligter.getLfdNr(),
//					isEinzigerSeinerArt(schuldnerBeteiligter));
//			glaeubiger = glaeubigerBeteiligter.parteiBezeichner(Beteiligter.NOMINATIV, glaeubigerBeteiligter.getLfdNr(),
//					isEinzigerSeinerArt(glaeubigerBeteiligter));
//			tenor.append(String.format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711, satzBeginn(schuldner),
//					glaeubiger));
//		} else if (!werIst708iVm711.isEmpty()) {
//			tenor.append(String.format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711_WERIST,
//					liste.benenneWerIst(werIst708iVm711, baumbachBeteiligtenListe),
//					TenorTexter.JEWEILIGER_VOLLSTRECKUNGSSCHULDNER, TenorTexter.JEWEILIGER_VOLLSTRECKUNGSGLAEUBIGER));
//		}
//		tenor.append(texte(elemente, liste));
//		return tenor.toString();
//	}
//
//	/**
//	 * Die Methode dient dazu, alle Vollstreckungsverhältnisse einer von {@link
//	 * #werIst708iVm711()} oder {@link #werIst709()} zurückgegebenen {@link
//	 * ArrayList} in einen lesbaren String umzuwandeln.
//	 * 
//	 * @param werIst eine {@link ArrayList}
//	 * 
//	 * @param baumbachBeteiligtenListe anhand der Liste wird überprüft, ob Gläubiger
//	 * bzw. Schuldner der einzige seiner Art sind.
//	 * 
//	 * @return ein String: Hinsichtlich des jeweiligen Vollstreckungsverhältnisses
//	 * des &lt;jeweiligen Gläubigers&gt; gegen den &lt;jeweiligen Schuldner&gt;,
//	 * (usw.)
//	 */ 
//	public String benenneWerIst(ArrayList<Integer> werIst, BaumbachBeteiligtenListe baumbachBeteiligtenListe) {
//		if (werIst == null || werIst.isEmpty())
//			return "";
//		StringBuffer benannt = new StringBuffer("Hinsichtlich des jeweiligen Vollstreckungsverhältnisses ");
//		for (int index : werIst) {
//			if (get(index) instanceof BaumbachVollstreckbarkeit)
//				benannt.append(
//						((BaumbachVollstreckbarkeit) get(index)).benenneParteien(baumbachBeteiligtenListe) + ", ");
//		}
//		return TenorTexter.undStattKomma(TenorTexter.letztesKommaWeg(benannt.toString()));
//	}


}
