/**
 * AngriffsVerteidigungsmittelStaffelung.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.logic.forderungen;

import java.util.ArrayList;
import java.util.Arrays;

import eu.gronos.kostenrechner.controller.TabulierendZeilen;
import eu.gronos.kostenrechner.controller.forderungen.ForderungsStaffelungPruefer;
import eu.gronos.kostenrechner.interfaces.Begruendend;
import eu.gronos.kostenrechner.interfaces.ParsendUndBauend;
import eu.gronos.kostenrechner.interfaces.TenorVorbereitend;
import eu.gronos.kostenrechner.interfaces.UnterContainerKlasse;
import eu.gronos.kostenrechner.logic.TenorTexter;
import eu.gronos.kostenrechner.logic.VollstreckbarkeitsHelfer;
import eu.gronos.kostenrechner.model.forderungen.Angriffserfolg;
import eu.gronos.kostenrechner.model.forderungen.AufrechnungsForderung;
import eu.gronos.kostenrechner.model.forderungen.Forderung;
import eu.gronos.kostenrechner.model.forderungen.ForderungsStaffelung;
import eu.gronos.kostenrechner.model.forderungen.KlageForderung;
import eu.gronos.kostenrechner.model.tenordaten.Allgemein;
import eu.gronos.kostenrechner.model.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.model.tenordaten.DoubleDataRows;
import eu.gronos.kostenrechner.model.tenordaten.HauptsacheEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.HauptsacheVerhaeltnis;
import eu.gronos.kostenrechner.model.tenordaten.KostenGrundEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.KostenTragungsVerhaeltnis;
import eu.gronos.kostenrechner.model.tenordaten.SonstigeEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.StreitwertEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.TenorDatenContainer;
import eu.gronos.kostenrechner.model.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.model.tenordaten.Vollstreckbarkeit;
import eu.gronos.kostenrechner.model.tenordaten.VollstreckbarkeitsEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.VollstreckbarkeitsListe;

/**
 * Eine Klasse zum Berechnen von Streitwert und von Kostenverteilung bei
 * Staffelung von Klageforderungen und Hilfsaufrechnungen
 * 
 * @author Peter Felix Schuster (SchusterP2)
 * @date 01.08.2014
 */
public class AngriffsVerteidigungsmittelStaffelung implements TenorVorbereitend, Begruendend {
	private final TenorDatenContainer container;

	private static final String[] COLUMN_HEADERS = new String[] { "Angriffs-/Verteidigungsmittel",
			"Höhe der Forderung (EUR)", "Erhöht den Streitwert um (EUR)", "Davon begründet (EUR)",
			"(verbleibende) Klageforderung" };
	private static final String GRUENDE_START_ANGRIFFS_VERTEIDIGUNGS_MITTEL = "Denn das Verhältnis von Obsiegen und Unterliegen bemisst sich anhand folgender Angriffs- und Verteidigungsmittel:\n";
	private static final String BESCHREIBUNG = "Tenor bei Klage/Hilfsaufrechnung (Hauptsachetenor, Kostentenor, Vollstreckbarkeitstenor und Streitwertbeschluss)";
	/**
	 * Der Beteiligter mit dem Namen klaeger speichert Genus/Numerus für die
	 * Tenorerzeugung
	 */
	private final Beteiligter klaeger;
	/**
	 * Der Beteiligter mit dem Namen beklagter speichert Genus/Numerus für die
	 * Tenorerzeugung
	 */
	private final Beteiligter beklagter;
	private Angriffserfolg hauptsache = null;
	private double[] kosten = null;
	/**
	 * in der {@link UnterContainerKlasse} Allgemein stehen {@link Allgemein#b92ii}
	 * und {@link Allgemein#unwesentlich92ii} sowie den
	 * {@link Allgemein#streitwert}.
	 */
	private Allgemein allgemein;
	/**
	 * Die {@link UnterContainerKlasse} des Typs {@link ForderungsStaffelung}
	 * enthält die {@link ArrayList}&lt;{@link Forderung}&gt; mit allen Forderungen,
	 * nach denen zu berechnen ist.
	 */
	private ForderungsStaffelung staffelung;
	private StringBuilder gruende = new StringBuilder();
	private TabulierendZeilen zeilen;
	private VollstreckbarkeitsListe vollstreckbarkeitsListe;
	private final AngriffsTexter texter = new AngriffsTexter();
	private final ParsendUndBauend<ForderungsStaffelung> pruefer = new ForderungsStaffelungPruefer(true);

	/**
	 * Konstruktor:
	 * 
	 * 
	 * @param forderungsStaffelung ein Parameterobjekt {@link ForderungsStaffelung},
	 *                             das beinhaltet:
	 *                             {@link ForderungsStaffelung.forderungen} eine
	 *                             ArrayList<Forderung> mit allen Forderungen, nach
	 *                             denen zu berechnen ist.
	 *                             {@link ForderungsStaffelung#parteien} ein
	 *                             Beteiligter klaeger und ein Beteiligter
	 *                             beklagter, in dem Genus/Numerus für die
	 *                             Tenorerzeugung gespeichert sind
	 * @param allgemein            ein Parameterobjekt {@link Allgemein} mit b92ii =
	 *                             § 92 Abs. 2 ZPO berücksichtigen (d.h.
	 *                             unwesentlichesUnterliegenUebergehen) und ein
	 *                             Unterliegen < 10% wie kein Unterliegen behandeln
	 * @throws IllegalArgumentException Wenn forderungen null oder leer ist; oder
	 *                                  wenn die Reihenfolge (erst Klageforderungen,
	 *                                  dann Aufrechnungsforderungen) nicht
	 *                                  eingehalten ist.
	 * 
	 */
	public AngriffsVerteidigungsmittelStaffelung(VerfahrensDatenContainer verfahrensDaten)
			throws IllegalArgumentException {
		this.staffelung = verfahrensDaten.staffelung;
		pruefer.parseEingabe(staffelung);
		setzeForderungenZurueck(staffelung.forderungen);
		this.klaeger = staffelung.parteien.get(0);
		this.beklagter = staffelung.parteien.get(1);
		this.allgemein = verfahrensDaten.allgemein;
		this.container = new TenorDatenContainer(verfahrensDaten);
		allgemein.streitwert = -1.0;
	}

	/**
	 * Die Methode soll einen Hauptsachetenor erzeugen.
	 * 
	 * @return {@link HauptsacheEntscheidungsElemente}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeHauptsacheEntscheidung()
	 */
	@Override
	public HauptsacheEntscheidungsElemente erzeugeHauptsacheEntscheidung() {
		HauptsacheEntscheidungsElemente entscheidung = new HauptsacheEntscheidungsElemente();
		/* Hauptsache nur berechnen, wenn nicht schon geschehen */
		if (hauptsache == null)
			hauptsache = berechneHauptsache();
		entscheidung.prozessverhaeltnisse.clear();
		entscheidung.prozessverhaeltnisse
				.add(new HauptsacheVerhaeltnis(klaeger, beklagter, false, (long) hauptsache.getErfolg()));// hauptsache[1]
		texter.setGanz(hauptsache.getVolleForderung());
		entscheidung.text = texter.texteHauptsache(entscheidung);
		return entscheidung;
	}

	/**
	 * Die Methode soll einen Kostentenor erzeugen.
	 * 
	 * @return einen String
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeKostenEntscheidung()
	 */
	@Override
	public KostenGrundEntscheidungsElemente erzeugeKostenEntscheidung() {
		KostenGrundEntscheidungsElemente kostenEntscheidung = new KostenGrundEntscheidungsElemente();
		kostenEntscheidung.prozessverhaeltnisse.clear();
		/*
		 * Kostenquoten nur berechnen, wenn nicht schon geschehen (vor erster Berechnung
		 * sind sie null).
		 */
		if (kosten == null)
			kosten = berechneKostenverteilung();
		if ((kosten[0] <= 0 && !allgemein.b92ii) || (kosten[0] < allgemein.unwesentlich92ii && allgemein.b92ii)) {
			kostenEntscheidung.prozessverhaeltnisse.add(new KostenTragungsVerhaeltnis(klaeger, beklagter, false, 1.0));
		} else if ((kosten[1] <= 0 && !allgemein.b92ii)
				|| (kosten[1] < allgemein.unwesentlich92ii && allgemein.b92ii)) {
			kostenEntscheidung.prozessverhaeltnisse.add(new KostenTragungsVerhaeltnis(beklagter, klaeger, false, 1.0));
		} else {
			kostenEntscheidung.prozessverhaeltnisse
					.add(new KostenTragungsVerhaeltnis(beklagter, klaeger, false, kosten[0]));
			kostenEntscheidung.prozessverhaeltnisse
					.add(new KostenTragungsVerhaeltnis(klaeger, beklagter, false, kosten[1]));
		}
		if (kosten[0] <= 0 || kosten[1] <= 0)
			starteGruende(TenorTexter.KOSTENENTSCHEIDUNG_91);
		else if (allgemein.b92ii
				&& (kosten[0] <= allgemein.unwesentlich92ii || kosten[1] <= allgemein.unwesentlich92ii))
			starteGruende(TenorTexter.KOSTENENTSCHEIDUNG_91_92_II);
		else
			starteGruende(TenorTexter.KOSTENENTSCHEIDUNG_92_II);
		erweitereGruende(GRUENDE_START_ANGRIFFS_VERTEIDIGUNGS_MITTEL);
		zeilen.toStringBuilder(gruende);
		kostenEntscheidung.text = texter.texteKostenentscheidung(kostenEntscheidung);
		return kostenEntscheidung;
	}

	/**
	 * Die Methode erzeugt den Tenor zur vorläufigen Vollstreckbarkeit und die
	 * entsprechenden
	 * {@link VollstreckbarkeitsEntscheidungsElemente#prozessverhaeltnisse}
	 * 
	 * @return einen {@link VollstreckbarkeitsEntscheidungsElemente}
	 */
	@Override
	public VollstreckbarkeitsEntscheidungsElemente erzeugeVollstreckbarkeitsEntscheidung() {
		VollstreckbarkeitsEntscheidungsElemente vollEntscheidung = new VollstreckbarkeitsEntscheidungsElemente();

		if (vollstreckbarkeitsListe == null)
			vollstreckbarkeitsListe = berechneVollstreckbarkeit();
		erweitereGruende(vollstreckbarkeitsListe.toString());
		VollstreckbarkeitsHelfer helfer = new VollstreckbarkeitsHelfer(
				container.hauptsacheEntscheidung.prozessverhaeltnisse,
				container.kostenEntscheidung.prozessverhaeltnisse);
		vollEntscheidung.prozessverhaeltnisse = helfer.toVollstreckbarkeitsVerhaeltnisse(klaeger, beklagter,
				vollstreckbarkeitsListe);

		vollEntscheidung.text = texter.texteVollstreckbarkeit(vollEntscheidung, vollstreckbarkeitsListe);
		return vollEntscheidung;
	}

	/**
	 * Die Methode soll eine Streitwertfestsetzung erzeugen.
	 * 
	 * @return {@link StreitwertEntscheidungsElemente}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeStreitwertEntscheidung()
	 */
	@Override
	public StreitwertEntscheidungsElemente erzeugeStreitwertEntscheidung() {
		StreitwertEntscheidungsElemente swe = new StreitwertEntscheidungsElemente();
		/*
		 * Streitwert nur berechnen, wenn nicht schon geschehen (vor erster Berechnung
		 * ist er < 0).
		 */
		if (allgemein.streitwert < 0)
			allgemein.streitwert = berechneStreitwert();
		swe.streitwerte.add(new Double(allgemein.streitwert).longValue());
		swe.text = texter.texteStreitwert(swe);
		return swe;
	}

	/**
	 * Die Methode erzeugt einen leeren sonstigen Tenor.
	 * 
	 * @return ein {@link SonstigeEntscheidungsElemente}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeSonstigeEntscheidung()
	 */
	@Override
	public SonstigeEntscheidungsElemente erzeugeSonstigeEntscheidung() {
		SonstigeEntscheidungsElemente sonstige = new SonstigeEntscheidungsElemente();
		sonstige.text = "";
		return sonstige;
	}

	/**
	 * Die Methode gibt die Begründung für den Kostentenor als String zurück.
	 * 
	 * @return den Text Gründe/Entscheidungsgründe als String
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.Begruendend#getGruende()
	 */
	@Override
	public String getGruende() {
		return this.gruende.toString();
	}

	/**
	 * Die Methode dient dazu, den gesamten {@link TenorDatenContainer} zu erzeugen.
	 * 
	 * @return das Ergebnis von erzeugeHauptsachetenor() + erzeugeKostentenor() +
	 *         erzeugeStreitwertFestsetzung()
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeContainer()
	 */
	@Override
	public TenorDatenContainer erzeugeContainer() {
		container.berechnungsTyp = getBerechnungsTyp();
		container.hauptsacheEntscheidung = erzeugeHauptsacheEntscheidung();
		container.kostenEntscheidung = erzeugeKostenEntscheidung();
		container.vollstreckbarkeitsEntscheidung = erzeugeVollstreckbarkeitsEntscheidung();
		container.streitwertEntscheidung = erzeugeStreitwertEntscheidung();
		container.sonstigeEntscheidung = erzeugeSonstigeEntscheidung();
		container.begruendung = zeilen.toBegruendungsElemente(getGruende());
		return container;
	}/* .text */

	/**
	 * Die Methode soll eine Beschreibung für die Art des Tenor zurückgeben, damit
	 * andere Klassen etwa das Dokument danach benennen können.
	 * 
	 * @return
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#getBerechnungsTyp()
	 */
	@Override
	public String getBerechnungsTyp() {
		return BESCHREIBUNG;
	}

	/**
	 * Berechnet, inwieweit sich verbliebene Klageforderung und der Wert der
	 * Aufrechnung decken, indem es den kleineren von zwei Beträgen zurückliefert.
	 * Setzt dabei auch direkt das Feld effektiverWert, so dass eine Berechnung
	 * ausreicht und sich das Ergebnis danach auch mit
	 * {@link AufrechnungsForderung#getEffektiverWert() getEffektiverWert()}
	 * erfragen lässt.
	 * 
	 * @param verbliebeneKlageforderung was von der Klageforderung übrig blieb
	 * @param aufrechnung               eine {@link AufrechnungsForderung}
	 * 
	 * @return den kleineren der Werte
	 * @see Forderung#berechneSchnittmenge(double, double)
	 */
	public static double berechneEffektivenWert(AufrechnungsForderung aufrechnung, double verbliebeneKlageforderung) {
		aufrechnung.setEffektiverWert(berechneSchnittmenge(verbliebeneKlageforderung, aufrechnung.getWert()));
		return aufrechnung.getEffektiverWert();
	}

	/**
	 * Berechnet, inwieweit sich verbliebene Klageforderung und der Erfolg der
	 * Aufrechnung decken, indem es den kleineren von zwei Beträgen zurückliefert.
	 * Setzt dabei auch direkt das Feld effektiverErfolg, so dass eine Berechnung
	 * ausreicht und sich das Ergebnis danach auch mit
	 * {@link AufrechnungsForderung#getEffektiverErfolg() getEffektiverErfolg()}
	 * erfragen lässt.
	 * 
	 * @param verbliebeneKlageforderung was von der Klageforderung übrig blieb
	 * @param aufrechnung               eine {@link AufrechnungsForderung}
	 * 
	 * @return den kleineren der Werte
	 * @see Forderung#berechneSchnittmenge(double, double)
	 */
	public static double berechneEffektivenErfolg(AufrechnungsForderung aufrechnung, double verbliebeneKlageforderung) {
		aufrechnung.setEffektiverErfolg(berechneSchnittmenge(verbliebeneKlageforderung, aufrechnung.getErfolg()));
		return aufrechnung.getEffektiverErfolg();
	}

	/**
	 * Die Methode dient dazu, bei allen {@link AufrechnungsForderung
	 * AufrechnungsForderungen} der <code>liste</code>
	 * {@link AufrechnungsForderung#setzeEffektivenErfolgZurueck()} und
	 * {@link AufrechnungsForderung#setzeEffektivenWertZurueck()} durchzuführen. Das
	 * muss bei Initialiiserung der Berechnung durchgeführt werden. Wenn seit der
	 * letzten Berechnung Werte verändert wurden, kommt es sonst zu Fehlern.
	 * 
	 * @param liste eine {@link ArrayList}&lt;{@link Forderung}&gt;
	 */
	private void setzeForderungenZurueck(ArrayList<Forderung> liste) {
		for (Forderung forderung : liste)
			if (forderung instanceof AufrechnungsForderung) {
				AufrechnungsForderung aufrechnung = (AufrechnungsForderung) forderung;
				aufrechnung.setzeEffektivenErfolgZurueck();
				aufrechnung.setzeEffektivenWertZurueck();
			}
		;
	}

	/**
	 * Die Methode berechnet das Obsiegen des Klägers in der Hauptsache und
	 * ermittelt dabei auch die vollständige Klageforderung.
	 * 
	 * @return ein Array double[], bei dem index 0 die komplette Klage ist und index
	 *         1 die verbleibende Klageforderung für den Tenor.
	 */
	private Angriffserfolg berechneHauptsache() {// double[]
		double verbleibend = 0;// erfolg
		double volleForderung = 0;
		for (Forderung f : staffelung.forderungen) {
			if (f instanceof KlageForderung) {
				/* Das Obsiegen bzgl. Klageforderungen wird zusammengerechnet */
				verbleibend += f.getErfolg();
				/* Auch der Wert, sofern nicht wirtschaftlich identisch */
				if (!((KlageForderung) f).isWirtschaftlichIdentisch())
					volleForderung += f.getWert();
			} else if (f instanceof AufrechnungsForderung) {
				/* Aufrechnungen werden abgezogen, soweit noch etwas übrig ist. */
				AufrechnungsForderung af = (AufrechnungsForderung) f;
				/*
				 * Dabei wird abgefragt, ob der effektive Erfolg schon berechnet wurde (dann ist
				 * getEffektiverErfolg nicht mehr < 0)
				 */
				verbleibend -= af.getEffektiverErfolg() < 0 ? berechneEffektivenErfolg(af, verbleibend)
						: af.getEffektiverErfolg();
			}
		}
		return new Angriffserfolg(volleForderung, verbleibend);
	}

	/**
	 * Die Methode berechnet die Kostenverteilung, damit das Ergebnis von
	 * {@link AngriffsVerteidigungsmittelStaffelung#erzeugeKostenEntscheidung()
	 * erzeugeKostentenor()} ausgelesen werden kann.
	 * 
	 * @return ein Array double[] mit der Quote des Klägers und des Beklagten,
	 *         jeweils zwischen 0.0 und 1.0
	 * 
	 * @see AngriffsVerteidigungsmittelStaffelung#erzeugeKostenEntscheidung()
	 */
	private double[] berechneKostenverteilung() {
		/*
		 * Streitwert nur berechnen, wenn nicht schon geschehen (vor erster Berechnung
		 * ist er < 0).
		 */
		if (allgemein.streitwert < 0)
			allgemein.streitwert = berechneStreitwert();
		double klaegerunterliegen = 0.0;
		double beklagtenunterliegen = 0.0;
		double verbleibend = 0.0;
		int aufrechnungen = 0;
		/*
		 * Liste der Zeilen der Begründungstabelle anlegen und mit der Kopfzeile
		 * beginnen
		 */
		zeilen = new TabulierendZeilen();
		zeilen.add(new ArrayList<String>(Arrays.asList(COLUMN_HEADERS)));
		for (Forderung f : staffelung.forderungen) {
			DoubleDataRows doubles = new DoubleDataRows(4);
			String beschreibung;
			if (f instanceof KlageForderung) {
				beschreibung = f.getBeschreibung();
				/* Das Obsiegen bzgl. Klageforderungen wird zusammengerechnet */
				verbleibend += f.getErfolg();
				/*
				 * Das Unterliegen bemisst sich hier allein nach Wert und Erfolg.
				 */
				klaegerunterliegen += f.getWert() - f.getErfolg();
				beklagtenunterliegen += f.getErfolg();
				/*
				 * Bei den Hauptklageforderungen muss einiges doppelt in die Begründungstabelle
				 * ausgegeben werden.
				 */
				doubles.add(new Double(f.getWert()));
				doubles.add(new Double(f.getWert()));
				doubles.add(new Double(f.getErfolg()));
				doubles.add(new Double(verbleibend));
				zeilen.add(beschreibung, doubles);
			} else if (f instanceof AufrechnungsForderung) {
				aufrechnungen++;
				beschreibung = String.format("%d. %s", aufrechnungen, f.getBeschreibung());
				/* Aufrechnungen werden abgezogen, soweit noch etwas übrig ist. */
				AufrechnungsForderung af = (AufrechnungsForderung) f;
				/*
				 * Das Unterliegen bemisst sich hier nach effektivem Wert und effektivem Erfolg.
				 * Dabei wird abgefragt, ob der effektive Erfolg schon berechnet wurde (dann ist
				 * getEffektiverErfolg nicht mehr < 0)
				 */
				double effektiverErfolg = af.getEffektiverErfolg() < 0 ? berechneEffektivenErfolg(af, verbleibend)
						: af.getEffektiverErfolg();
				double effektiverWert = af.getEffektiverWert() < 0 ? berechneEffektivenWert(af, verbleibend)
						: af.getEffektiverWert();
				verbleibend -= effektiverErfolg;
				klaegerunterliegen += effektiverErfolg;
				beklagtenunterliegen += effektiverWert - effektiverErfolg;
				/*
				 * Bei den Aufrechnungen wird alles in der Begründungstabelle gebraucht
				 */
				doubles.add(new Double(f.getWert()));
				doubles.add(new Double(af.getEffektiverWert()));
				doubles.add(new Double(af.getEffektiverErfolg()));
				doubles.add(new Double(verbleibend));
				zeilen.add(beschreibung, doubles);
			}
		}
		/* Eine Ergebniszeile gibt es hier nicht */
		return new double[] { klaegerunterliegen / allgemein.streitwert, beklagtenunterliegen / allgemein.streitwert };
	}

	/**
	 * Die Methode berechnet, inwieweit das Urteil vorläufig vollstreckbar ist.
	 * 
	 * @return eine {@link VollstreckbarkeitsListe}, deren 0. Element die
	 *         Unanfechtbarkeit speichert, wenn nicht unanfechtbar das 1. Element
	 *         das Verhältnis kl -> bk und das 2. Element das Verhältnis bk -> kl.
	 * 
	 */
	private VollstreckbarkeitsListe berechneVollstreckbarkeit() {
		VollstreckbarkeitsListe vollstreckbarkeitsListe = new VollstreckbarkeitsListe();
		Vollstreckbarkeit unanfechtbarkeit = pruefeUnanfechtbarkeit();
		vollstreckbarkeitsListe.add(unanfechtbarkeit);
		if (unanfechtbarkeit.isB713())
			return vollstreckbarkeitsListe;
		/*
		 * Kostenquoten nur berechnen, wenn nicht schon geschehen (vor erster Berechnung
		 * sind sie null).
		 */
		if (kosten == null)
			kosten = berechneKostenverteilung();
		/*
		 * Beim Kläger hinsichtlich Hauptsache einsetzen, was von seiner
		 * Hauptklageforderung begründet ist (hauptsache[1]) und hinsichtlich der Kosten
		 * die üblichen Klägerkosten, soweit sie der Beklagte tragen muss (kosten[1])
		 */
		vollstreckbarkeitsListe.add(Vollstreckbarkeit.pruefeSicherheitsleistung((long) hauptsache.getErfolg(), // hauptsache[1],
				(long) (TenorTexter.getKostenKlaeger().errechneGebuehrenSumme((long) allgemein.streitwert)
						* kosten[1])));
		/*
		 * Beim Kläger hinsichtlich Hauptsache 0 einsetzen und hinsichtlich der Kosten
		 * die üblichen Klägerkosten, soweit sie der Kläger tragen muss (kosten[0])
		 */
		vollstreckbarkeitsListe.add(Vollstreckbarkeit.pruefeSicherheitsleistung(0L,
				(long) (TenorTexter.getKostenBeklagter().errechneGebuehrenSumme((long) allgemein.streitwert)
						* kosten[0])));
		return vollstreckbarkeitsListe;
	}

	/**
	 * Die Methode prüft, ob das Urteil unanfechtbar ist (§§ 708, 713 ZPO).
	 * 
	 * @return ein {@link Vollstreckbarkeit}, das im Fall der Unanfechtbarkeit 708
	 *         und 713 auf <code>true</code> setzt. Ansonsten enthaelt es nichts.
	 */
	private Vollstreckbarkeit pruefeUnanfechtbarkeit() {
		Vollstreckbarkeit unanfechtbarkeit = new Vollstreckbarkeit();
		/*
		 * Streitwert nur berechnen, wenn nicht schon geschehen (vor erster Berechnung
		 * ist er < 0).
		 */
		if (allgemein.streitwert < 0)
			allgemein.streitwert = berechneStreitwert();
		if (allgemein.streitwert <= TenorTexter.BERUFUNGS_GRENZE) {
			unanfechtbarkeit = new Vollstreckbarkeit(true, false, false, false, true);
		} else if (allgemein.streitwert <= TenorTexter.BERUFUNGS_GRENZE * 2) {
			/* Hauptsache nur berechnen, wenn nicht schon geschehen */
			if (hauptsache == null)
				hauptsache = berechneHauptsache();
			/*
			 * Wie ist der Kläger beschwert? Das, was von seiner Klage nicht durchgedrungen
			 * ist.
			 */
			double klaeger = hauptsache.getVolleForderung() - hauptsache.getErfolg();// hauptsache[0] - hauptsache[1];
			/* Wie ist der Beklagte beschwert: Das wozu er verurteilt wurde ... */
			double beklagter = hauptsache.getErfolg();// hauptsache[1];
			/*
			 * ... und das, was er an Aufrechnungsforderungen "verbraten" hat, ob mit oder
			 * ohne Erfolg.
			 */
			for (Forderung f : staffelung.forderungen) {
				if (f instanceof AufrechnungsForderung)
					beklagter += ((AufrechnungsForderung) f).getEffektiverWert();
			}
			if (klaeger <= TenorTexter.BERUFUNGS_GRENZE && beklagter <= TenorTexter.BERUFUNGS_GRENZE) {
				unanfechtbarkeit = new Vollstreckbarkeit(true, false, false, false, true);
			}
		}
		return unanfechtbarkeit;
	}

	/**
	 * Die Methode berechnet den Streitwert.
	 * 
	 * @return den Streitwert als double
	 */
	private double berechneStreitwert() {
		double wert = 0.0;
		double verbleibend = 0.0;
		for (Forderung f : staffelung.forderungen) {
			if (f instanceof KlageForderung) {
				/* Das Obsiegen bzgl. Klageforderungen wird zusammengerechnet */
				verbleibend += f.getErfolg();
				/* Auch der Wert, sofern nicht wirtschaftlich identisch */
				if (!((KlageForderung) f).isWirtschaftlichIdentisch())
					wert += f.getWert();
			} else if (f instanceof AufrechnungsForderung) {
				/* Aufrechnungen werden abgezogen, soweit noch etwas übrig ist. */
				AufrechnungsForderung af = (AufrechnungsForderung) f;
				/*
				 * Dabei wird abgefragt, ob der effektive Wert und der effektive Erfolg schon
				 * berechnet wurden (dann sind getEffektiverWert bzw. getEffektiverErfolg nicht
				 * mehr < 0)
				 */
				wert += af.getEffektiverWert() < 0 ? berechneEffektivenWert(af, verbleibend) : af.getEffektiverWert();
				verbleibend -= af.getEffektiverErfolg() < 0 ? berechneEffektivenErfolg(af, verbleibend)
						: af.getEffektiverErfolg();
			}
		}
		return wert;
	}

	/**
	 * Die Methode setzt die Entscheidungsgründe zurück (sofern nicht leer) und
	 * beginnt sie neu mit dem übergebenen Text.
	 * 
	 * @param text eine {@link CharSequence} mit dem Anfangssatz(-bestandteil) der
	 *             Gründe.
	 */
	private void starteGruende(CharSequence text) {
		if (gruende.length() > 0)
			gruende.delete(0, gruende.length());
		erweitereGruende(text);
	}

	/**
	 * Die Methode erweitert die bisherigen Gründe
	 * 
	 * @param text eine {@link CharSequence} mit den weiteren Gründen.
	 */
	private void erweitereGruende(CharSequence text) {
		gruende.append(text);
	}

	/**
	 * Berechnet, inwieweit sich zwei Beträge decken, indem es den kleineren von
	 * zwei Beträgen zurückliefert.
	 * 
	 * @param verbliebeneKlageforderung was von der Klageforderung übrig blieb
	 * @param andere                    der andere Wert, z.B. den Wert oder den
	 *                                  Erfolg
	 * @return das kleinere von beiden oder 0, falls einer der Parameter negativ
	 *         ist.
	 */
	private static double berechneSchnittmenge(double verbliebeneKlageforderung, double andere) {
		if (verbliebeneKlageforderung < 0 || andere < 0)
			return 0.0;
		else
			return verbliebeneKlageforderung < andere ? verbliebeneKlageforderung : andere;
	}
}
// ForderungsStaffelung forderungsStaffelung = verfahrensDaten.staffelung;
// this.forderungen = forderungsStaffelung.forderungen;
// private boolean unwesentlichesUnterliegenUebergehen;
// private double unwesentlich92ii;
// private ArrayList<Object[]> zeilenListe;
// private int[] range = new int[2];
// Allgemein allgemein, ForderungsStaffelung forderungsStaffelung allgemein,
// forderungsStaffelung
// if (forderungen == null || forderungen.size() < 1)
// throw new IllegalArgumentException(TenorTexter.FEHLER_FORDERUNG_LEER);
// if (!isReihenfolgeEingehalten(forderungen))
// throw new
// IllegalArgumentException(TenorTexter.FEHLER_REIHENFOLGE_FORDERUNGEN);
// klaeger;
// .beklagter;
// unwesentlichesUnterliegenUebergehen;
// this.unwesentlichesUnterliegenUebergehen = verfahrensDaten.allgemein.b92ii;
// this.unwesentlich92ii = verfahrensDaten.allgemein.unwesentlich92ii;
// double ganz = hauptsache[0];// hauptsache[0]
// double erfolg = hauptsache[1];
// // /**
// * Die Methode dient dazu, die Reihenfolge der Forderungen zu prüfen!
// * 
// * @param forderungen die zu prüfende ArrayList<Forderung>
// * @return true, wenn die Reihenfolge eingehalten ist: Erste muss Klageforderung
// *         sein, dann alle übrigen Klageforderungen, dann Aufrechnungen (und
// *         keine Klageforderungen mehr)
// */
// private boolean isReihenfolgeEingehalten(ArrayList<Forderung> forderungen) {
// boolean antwort = true;
// boolean umstiegAufAufrechnung = false;
// if (!(forderungen.get(0) instanceof KlageForderung))
// antwort = false;
// for (Forderung forderung : forderungen) {
// if (forderung instanceof AufrechnungsForderung)
// umstiegAufAufrechnung = true;
// if (umstiegAufAufrechnung && forderung instanceof KlageForderung)
// antwort = false;
// }
// return antwort;
// }
// new double[] { volleForderung, verbleibend };
// String festsetzung;
//double[]

// private final ArrayList<Forderung> forderungen;
// private double streitwert = -1.0;