/**
 * GebuehrenBerechnungsAufstellung.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.logic.gebuehren;

import java.util.ArrayList;
import java.util.Arrays;

import eu.gronos.kostenrechner.controller.TabulierendZeilen;
import eu.gronos.kostenrechner.controller.gebuehren.GebuehrenBerechnungPruefer;
import eu.gronos.kostenrechner.interfaces.Begruendend;
import eu.gronos.kostenrechner.interfaces.ParsendUndBauend;
import eu.gronos.kostenrechner.interfaces.TenorVorbereitend;
import eu.gronos.kostenrechner.logic.TenorTexter;
import eu.gronos.kostenrechner.model.gebuehren.AnwaltsGebuehrenTabelle;
import eu.gronos.kostenrechner.model.gebuehren.GebuehrenAnrechnungsTatbestand;
import eu.gronos.kostenrechner.model.gebuehren.GebuehrenBerechnung;
import eu.gronos.kostenrechner.model.gebuehren.GebuehrenErhoehungsTatbestand;
import eu.gronos.kostenrechner.model.gebuehren.GebuehrenSatzTatbestand;
import eu.gronos.kostenrechner.model.gebuehren.GebuehrenTatbestand;
import eu.gronos.kostenrechner.model.gebuehren.GebuehrenVerzeichnis;
import eu.gronos.kostenrechner.model.gebuehren.GerichtsGebuehrenTabelle;
import eu.gronos.kostenrechner.model.gebuehren.MehrfachPauschalTatbestand;
import eu.gronos.kostenrechner.model.gebuehren.MehrwertsteuerTatbestand;
import eu.gronos.kostenrechner.model.tenordaten.DoubleDataRows;
import eu.gronos.kostenrechner.model.tenordaten.HauptsacheEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.KostenGrundEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.SonstigeEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.StreitwertEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.TenorDatenContainer;
import eu.gronos.kostenrechner.model.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.model.tenordaten.VollstreckbarkeitsEntscheidungsElemente;

/**
 * Klasse zur Berechnung von Gebühren
 *
 * @author Peter Schuster (setrok)
 * @date 24.08.2014
 *
 */
public class GebuehrenBerechnungsAufstellung implements TenorVorbereitend, Begruendend /* Tenorierend */ {

	private static final String BESCHREIBUNG = "Berechnung von Gebühren";
	private static final String[] COLUMN_HEADERS = new String[] { "Gebührentatbestand", "Satz", "Prozent (%)",
			"Gebührenbetrag (EUR)" };
	private final ArrayList<GebuehrenTatbestand> values;
	private final long streitwert;
	private final TenorDatenContainer container;
	private TabulierendZeilen zeilen;
	private StringBuilder gruende;
	private ParsendUndBauend<GebuehrenBerechnung> pruefer = new GebuehrenBerechnungPruefer(true);

	/**
	 * Der einzige Konstruktor.
	 * 
	 * @param gebuehrenBerechnung ein Parameterobjekt {@link GebuehrenBerechnung}
	 *                            mit den GebuehrenSatzTatbeständen,
	 *                            AuslagenTatbeständen und
	 *                            MehrwertsteuerTatbeständen; darf nicht null sein,
	 *                            und ein long mit dem Streitwert, muss
	 *                            größer/gleich 0 sein.
	 * @throws NullPointerException     wenn values null ist
	 * @throws IllegalArgumentException wenn streitwert kleiner 0 ist
	 */
	public GebuehrenBerechnungsAufstellung(VerfahrensDatenContainer verfahrensDaten)
			throws NullPointerException, IllegalArgumentException {
		super();
		GebuehrenBerechnung gebuehrenBerechnung = verfahrensDaten.gebuehrenBerechnung;
		pruefer.parseEingabe(gebuehrenBerechnung);
		/*
		 * Arrays.asList generiert eine starre Liste, die nicht erweitert werden kann;
		 * deshalb sicherheitshalber noch addAll hinterher.
		 */
		this.values = new ArrayList<GebuehrenTatbestand>();
		values.addAll(gebuehrenBerechnung.gebuehren);

		this.streitwert = gebuehrenBerechnung.streitwert;
		container = new TenorDatenContainer(verfahrensDaten);
	}

	/**
	 * Die Methode wandelt über {@link StringBuilder#toString()} die Gründe in einen
	 * {@link String} um, hier eine Aufstellung mit den Einzelwerten der Gebühren.
	 * 
	 * @return ein {@link String}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.Begruendend#getGruende()
	 */
	@Override
	public String getGruende() {
		return gruende.toString();
	}

	/**
	 * Die Methode gibt einen leeren Hauptsachetenor zurück
	 * 
	 * @return ""
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeHauptsacheEntscheidung()
	 */
	@Override
	public HauptsacheEntscheidungsElemente erzeugeHauptsacheEntscheidung() {
		HauptsacheEntscheidungsElemente hauptsache = new HauptsacheEntscheidungsElemente();
		hauptsache.text = "";
		return hauptsache;
	}

	/**
	 * Die Methode gibt einen leeren Kostentenor zurück
	 * 
	 * @return ""
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeKostenEntscheidung()
	 */
	@Override
	public KostenGrundEntscheidungsElemente erzeugeKostenEntscheidung() {
		KostenGrundEntscheidungsElemente kostenEntscheidung = new KostenGrundEntscheidungsElemente();
		kostenEntscheidung.text = "";
		return kostenEntscheidung;
	}

	/**
	 * Die Methode gibt einen leeren Vollstreckbarkeitstenor zurück
	 * 
	 * @return ""
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeVollstreckbarkeitsEntscheidung()
	 */
	@Override
	public VollstreckbarkeitsEntscheidungsElemente erzeugeVollstreckbarkeitsEntscheidung() {
		VollstreckbarkeitsEntscheidungsElemente vollEntscheidung = new VollstreckbarkeitsEntscheidungsElemente();
		vollEntscheidung.text = "";
		return vollEntscheidung;
	}

	/**
	 * Die Methode gibt statt einer wirklichen Streitwertentscheidung die
	 * Feststellung zurück, dass der Streitwert mal auf den übergebenen Betrag
	 * festgesetzt wurde. Da Hauptsacheentscheidung, Kostengrundentscheidung und
	 * Vollstreckbarkeitsentscheidung leer sind, ist das der erste Satz des
	 * Gesamttenors.
	 * 
	 * @return
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeStreitwertEntscheidung()
	 */
	@Override
	public StreitwertEntscheidungsElemente erzeugeStreitwertEntscheidung() {
		StreitwertEntscheidungsElemente swe = new StreitwertEntscheidungsElemente();
		swe.text = String.format(TenorTexter.STREITWERT_FESTGESETZT_VERGANGENHEIT, (double) streitwert);
		swe.streitwerte.add(streitwert);
		return swe;
	}

	/**
	 * Die Methode gibt als {@link SonstigeEntscheidungsElemente} eine
	 * Gebührenaufstellung zurück.
	 * 
	 * @return {@link SonstigeEntscheidungsElemente}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeSonstigeEntscheidung()
	 */
	@Override
	public SonstigeEntscheidungsElemente erzeugeSonstigeEntscheidung() {
		SonstigeEntscheidungsElemente sonstige = new SonstigeEntscheidungsElemente();
		sonstige.text = berrechneGebuehrenAufstellung();
		return sonstige;
	}

	/**
	 * Die Methode dient dazu, den gesamten {@link TenorDatenContainer} zu erzeugen.
	 * 
	 * @return einen befüllten {@link TenorDatenContainer}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeContainer()
	 */
	@Override
	public TenorDatenContainer erzeugeContainer() {
		container.berechnungsTyp = getBerechnungsTyp();
		container.hauptsacheEntscheidung = erzeugeHauptsacheEntscheidung();
		container.kostenEntscheidung = erzeugeKostenEntscheidung();
		container.vollstreckbarkeitsEntscheidung = erzeugeVollstreckbarkeitsEntscheidung();
		container.streitwertEntscheidung = erzeugeStreitwertEntscheidung();
		container.sonstigeEntscheidung = erzeugeSonstigeEntscheidung();
		container.begruendung = zeilen.toBegruendungsElemente(getGruende());
		return container;
	}

	/**
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#getBerechnungsTyp()
	 */
	@Override
	public String getBerechnungsTyp() {
		return BESCHREIBUNG;
	}

	/**
	 * Die Methode soll die Summe aller Gebühren der Tabelle liefern.
	 * 
	 * @return einen String mit der Aufstellung aller GebührenTatbestände und ihres
	 *         Betrags für diesen Streitwert
	 */
	private String berrechneGebuehrenAufstellung() {
		StringBuffer aufstellung = new StringBuffer("Die Gebühren berechnen sich wie folgt:\n");
		final String zeilenFormat = "%s \t %s \t %,.2f EUR;%n";
		double summe = 0;
		double rvgsumme = 0;
		zeilen = new TabulierendZeilen();
		zeilen.add(Arrays.asList(COLUMN_HEADERS));
		for (GebuehrenTatbestand gt : values) {
			final DoubleDataRows doubles = new DoubleDataRows(3);
			/*
			 * errechneGebuehr ignoriert MehrwertsteuerTatbestände; diese muss deshalb extra
			 * gerechnet werden.
			 */
			if (gt instanceof MehrwertsteuerTatbestand) {
				/* Bei Mehrwertsteuer ist das erste Double-Feld (der Gebührensatz) leer (0.0) */
				doubles.add(new Double(0.0));
				final MehrwertsteuerTatbestand mwst = (MehrwertsteuerTatbestand) gt;
				final double einzelposition = mwst.errechneSteuer(rvgsumme);
				final String bemerkung = String.format("(%,.1f%%)", mwst.getSteuerSatz() * 100.0);
				/* Ins zweite Double-Feld kommt die Prozentangabe */
				doubles.add(new Double(mwst.getSteuerSatz() * 100.0));
				summe += einzelposition;
				aufstellung.append(String.format(zeilenFormat, gt.langBezeichnung(), bemerkung, einzelposition));
				/* Ins dritte Feld (EUR) kommt die Einzelposition */
				doubles.add(new Double(einzelposition));
				/*
				 * Wenn die Gebühren isoliert werden, die RVG-Summe wieder auf 0 setzen.
				 */
				if (mwst.isIsoliert())
					rvgsumme = 0.0;
				/* Und die Zeile in die Tabelle nehmen */
				zeilen.add(gt.langBezeichnung(), doubles);//getBezeichnung()
			} else {
				/* summe um die Gebühr erhöhen */
				final double einzelposition = GebuehrenVerzeichnis.errechneGebuehr(gt, streitwert);
				String ergaenzung = "";
				String bemerkung = "";
				if (gt instanceof MehrfachPauschalTatbestand)
					ergaenzung = String.format(" x %d %s", ((MehrfachPauschalTatbestand) gt).getAnzahl(),
							((MehrfachPauschalTatbestand) gt).getEinheit());
				if (gt instanceof GebuehrenErhoehungsTatbestand)
					ergaenzung = String.format(" x %d", ((GebuehrenErhoehungsTatbestand) gt).getAnzahl());
				if (gt instanceof GebuehrenSatzTatbestand)
					bemerkung = String.format("(%,.1f)", ((GebuehrenSatzTatbestand) gt).getSatz());
				if (gt instanceof GebuehrenAnrechnungsTatbestand)
					bemerkung = String.format("(%,.2f)", ((GebuehrenSatzTatbestand) gt).getSatz());
				if (gt instanceof GebuehrenSatzTatbestand) {
					/* Bei allem, was einen Satz hat, kommt der in die erste Double-Spalte */
					doubles.add(new Double(((GebuehrenSatzTatbestand) gt).getSatz()));
				} else {
					/* Sonst kommt da 0.0 rein */
					doubles.add(new Double(0.0));
				}
				/* die zweite Spalte (%) ist auf jeden Fall 0.0 */
				doubles.add(new Double(0.0));
				summe += einzelposition;
				doubles.add(new Double(einzelposition));
				aufstellung.append(
						String.format(zeilenFormat, gt.langBezeichnung() + ergaenzung, bemerkung, einzelposition));
				/*
				 * rvgsumme nur erhöhen, wenn RVG-Gebühr (wird für MwSt gebraucht).
				 */
				if (gt.getGebuehrenKlasse() == AnwaltsGebuehrenTabelle.class)
					rvgsumme += GebuehrenVerzeichnis.errechneGebuehr(gt, streitwert);
				/* Und die Zeile in die Tabelle nehmen */
				zeilen.add(gt.langBezeichnung() + ergaenzung, doubles);
			}
		}
		final DoubleDataRows doublesumme = new DoubleDataRows(3);
		doublesumme.add(new Double(0.0));
		doublesumme.add(new Double(0.0));
		doublesumme.add(new Double(summe));
		zeilen.add("Gesamt", doublesumme);
		aufstellung.append(String.format(zeilenFormat, "", "Gesamt", summe));
		baueGruende();
		return aufstellung.toString();
	}

	/**
	 * Die Methode dient dazu, die Gründe zusammen zu setzen.
	 */
	private void baueGruende() {
		gruende = new StringBuilder();
		final GebuehrenSatzTatbestand vv = new GebuehrenSatzTatbestand("", "", 1.0, AnwaltsGebuehrenTabelle.class);
		final GebuehrenSatzTatbestand kv = new GebuehrenSatzTatbestand("", "", 1.0, GerichtsGebuehrenTabelle.class);
		gruende.append(String.format(
				"Bei einem Streitwert von %,.2f EUR  beträgt eine Gerichtsgebühr %,.2f EUR und eine Anwaltsgebühr %,.2f EUR. %nAufstellung: %n",
				(double) streitwert, GebuehrenVerzeichnis.errechneGebuehr(kv, streitwert),
				GebuehrenVerzeichnis.errechneGebuehr(vv, streitwert)));
		zeilen.toStringBuilder(gruende);
	}
}
// gebuehrenBerechnung
// if (gebuehrenBerechnung.gebuehren != null) {
// /*
// * Arrays.asList generiert eine starre Liste, die nicht erweitert werden kann;
// * deshalb sicherheitshalber noch addAll hinterher.
// */
// this.values = new ArrayList<GebuehrenTatbestand>();
// // values.addAll(Arrays.asList(gebuehrenBerechnung.gebuehren));
// values.addAll(gebuehrenBerechnung.gebuehren);
//} else
// throw new NullPointerException("Die Liste der Tatbestände darf nicht leer (null)");
// if (streitwert < 0)
// throw new IllegalArgumentException("Der Streitwert muss größer/gleich 0 sein");
