/**
 * MehrkostenMethode.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.logic.gebuehren;

import java.util.Arrays;
import java.util.List;

import eu.gronos.kostenrechner.controller.TabulierendZeilen;
import eu.gronos.kostenrechner.model.gebuehren.GebuehrenTatbestand;
import eu.gronos.kostenrechner.model.gebuehren.GebuehrenVerzeichnis;
import eu.gronos.kostenrechner.model.gebuehren.Teilklageruecknahme;
import eu.gronos.kostenrechner.model.tenordaten.Allgemein;
import eu.gronos.kostenrechner.model.tenordaten.DoubleDataRows;
import eu.gronos.kostenrechner.model.tenordaten.VerfahrensDatenContainer;

/**
 * MehrkostenMethode berechnet das Unterliegen des Klägers dadurch, dass die
 * Mehrkosten, die auf den zurückgenommenen Teil entfallen, errechnet und diese
 * in Relation zu den tatsächlichen entstandenen Kosten des Rechtsstreits
 * gesetzt werden (sog. Mehrkostenmethode, Anders/Gehle, A-198, dies. Antrag und
 * Entscheidung im Zivilprozess, Teil B Rn. 404, 410 ff.).
 * 
 * @author Peter Schuster (setrok)
 * @date 25.04.2014
 * 
 */
public class MehrkostenMethode extends StreitwertReduktion {

	private static final String[] COLUMN_HEADERS = new String[] { "Gebührentatbestand",
			"Tatsächlicher Streitwert (EUR)", "Tatsächliche Gebührenhöhe (EUR)", "Reduzierter Streitwert (EUR)",
			"Fiktive Gebührenhöhe (EUR)", "Mehrkosten (EUR)", "Differenz (%)" };
	private static final String BESCHREIBUNG = "Tenor nach der sog. Mehrkostenmethode (Hauptsachetenor, Kostentenor, Vollstreckbarkeitstenor und Streitwertbeschluss).";
	private static final String WEITER_GRUENDE = "der sogenannten Mehrkostenmethode.\n";
	private static final String GRUENDE_BERECHNUNG_MEHRKOSTEN = "Danach ist bei einer teilweisen Klagerücknahme die Kostenquote dadurch zu ermitteln, dass die durch den zurückgenommenen Teil entstandenen Mehrkosten in Relation zu den tatsächlich entstandenen Kosten des Rechtsstreits zu setzen sind. ";
	private static final String GRUENDE_EINLEITUNG_TABELLE = "Im einzelnen errechnen sich diese Punkte wie folgt:\n";
	private static final String GRUENDE_VERBLIEBENER_GEGENSTAND = "Da die Klage hinsichtlich des verbliebenen Streitgegenstands nicht vollständig erfolgreich war";
	private static final String GRUENDE_VERBLIEBENER_GEGENSTAND_WERTVOLLER = " und der Wert des verbliebenen Streitwerts die angefallenen Kosten des Rechtsstreits übersteigt, ist für den zurückgenommenen Teil das Kosteninteresse anzusetzen. ";
	private static final String GRUENDE_VERBLIEBENER_GEGENSTAND_SONST = ", ist auch das Unterliegen diesbezüglich zu berücksichtigen. ";
	/**
	 * Soll für den zurückgenommenen Teil nur auf das Kosteninteresse abgestellt
	 * werden?
	 */
	private boolean aufKosteninteresseAbstellen = false;

	/**
	 * Der Konstruktor ruft den Konstruktor der Oberklasse auf.
	 * 
	 * 
	 * @param teilklageruecknahme Parameterojbkekt vom Typ
	 *                            {@link Teilklageruecknahme}, das enthält die
	 *                            {@link Teilklageruecknahme#parteien} klaeger
	 *                            Informationen zur Klägerseite als Beteiligter
	 *                            (wird zur Tenorierung benötigt) und beklagter
	 *                            Informationen zur Beklagten als Beteiligter (wird
	 *                            zur Tenorierung benötigt), die Arrays
	 *                            gebuehrenVorher und gebuehrenNachher, (Arrays
	 *                            GebuehrenTatbestand[] mit allen schon vor bzw.
	 *                            erst nach der "Streitwertreduktion"
	 *                            (Teilklagerücknahme bzw -erledigung) entstandenen
	 *                            Gebühren, die
	 *                            {@link Teilklageruecknahme#streitwerteUndObsiegen}
	 *                            eine ArrayList<Long> mit streitwerteUndObsiegen[0]
	 *                            als dem alten (höheren) Streitwert,
	 *                            streitwerteUndObsiegen[1] als dem neuen,
	 *                            niedrigeren Streitwert und die
	 *                            streitwerteUndObsiegen[2], die Angabe, in welcher
	 *                            Höhe der Kläger letzlich obsiegt hat.
	 * 
	 * @param allgemein           Paremeterobjekt {@link Allgemein}, das enthält:
	 *                            b92ii Soll § 92 II ZPO angewandt werden
	 *                            (verhältnismäßig geringe Kostentragungen werden
	 *                            unterschlagen)
	 * @throws IllegalArgumentException wird geworfen, wenn eine der beiden
	 *                                  folgenden Plausibilitäten verletzt ist: a)
	 *                                  der frühere Streitwert muss größer als der
	 *                                  spätere (oder gleich) sein, b) der spätere
	 *                                  Streitwert muss größer als die Verurteilung
	 *                                  (oder gleich) sein.
	 * 
	 * @see eu.gronos.kostenrechner.logic.gebuehren.StreitwertReduktion#StreitwertReduktion(Allgemein,
	 *      Teilklageruecknahme)
	 * 
	 */
	public MehrkostenMethode(VerfahrensDatenContainer vefahrensDaten) throws IllegalArgumentException {
		super(vefahrensDaten);
	}
	// Allgemein allgemein, Teilklageruecknahme teilklageruecknahme)
	// allgemein, teilklageruecknahme);

	/**
	 * @see eu.gronos.kostenrechner.interfaces.Begruendend#getBerechnungsTyp()
	 */
	@Override
	public String getBerechnungsTyp() {
		return BESCHREIBUNG;
	}

	/**
	 * Die Methode berechneUnterliegensQuote dient dazu, die Unterliegensquote nach
	 * Maßgabe der dem Konstruktor übergebenen Werte zu berechnen. Berücksichtigt
	 * jetzt auch, dass der Kläger nach Teilklagerücknahme noch mit einem weiteren
	 * Teil unterliegen kann...
	 * 
	 * @return einen double zwischen 0.0 und 1.0, der in Prozent umgerechnet das
	 *         Unterliegen ausdrückt.
	 * 
	 */
	@Override
	protected double berechneUnterliegensQuote() {
		double summeTatsaechlicheEUR = 0.0;
		double summeFiktiveEUR = 0.0;
		double mehrkostenEUR = 0.0;
		/*
		 * Eigentlich wäre so eine Tabelle (Schneider, zit. n. Anders/Gehle, Rn. A-198)
		 * ein Array Object[][], aber eine ArrayList ist schöner, weil man vorher die
		 * Anzahl der Zeilen nicht wissen muss. Die Spaltenanzahl ist aber klar: 7.
		 */
		zeilen = new TabulierendZeilen();
		zeilen.add(Arrays.asList(COLUMN_HEADERS));
		/*
		 * Äußere Zähler-Schleife: Anhand des Arrays streitwerte eine Schleife
		 * durchlaufen, für alles vorher, dann alles nachher. De facto wird die Schleife
		 * genau zweimal durchlaufen, aber zweimal denselben Code zu schreiben, ist
		 * doof. :)
		 */
		for (int i = 0; i < streitwerte.length; i++) {
			List<GebuehrenTatbestand> tatbestaende = i == 0 ? vorher : nachher;
			if (tatbestaende != null) {
				/* Alle Gebührentatbestände für diesen Streitwert durchgehen */
				for (GebuehrenTatbestand gt : tatbestaende) {
					final DoubleDataRows doubles = new DoubleDataRows(6);
					/*
					 * Welche Gebühr ist tatsächlich angefallen? Die tatsächlichen Kosten beinhalten
					 * die vor der Reduktion angefallenen Gebühren nach dem alten, höheren
					 * Streitwert und die danach entstandenen Gebühren nach dem neuen, niedrigeren
					 * Streitwert. Ausrechnen und zur Summe hinzurechnen.
					 */
					doubles.add(new Double(streitwerte[i]));
					double gebuehrTatsaechlich = GebuehrenVerzeichnis.errechneGebuehr(gt, streitwerte[i]);
					doubles.add(new Double(gebuehrTatsaechlich));
					summeTatsaechlicheEUR += gebuehrTatsaechlich;
					/*
					 * Welche Gebühr wäre bei dem niedrigeren Streitwert angefallen? Die Kosten, die
					 * bei anfänglichem Einklagen nur des verbleibenden Betrags entstandenen wären
					 * (fiktive). Das sind die vor der Reduktion angefallenen Gebühren und die
					 * danach entstandenen, beide nach dem niedrigeren Streitwert, also
					 * streitwerte[1]. Ausrechnen und zur Summe hinzurechnen.
					 */
					doubles.add(new Double(streitwerte[1]));
					double gebuehrFiktiv = GebuehrenVerzeichnis.errechneGebuehr(gt, streitwerte[1]);
					doubles.add(new Double(gebuehrFiktiv));
					summeFiktiveEUR += gebuehrFiktiv;
					/*
					 * Die Differenz hat der Kläger durch seine anfängliche Gier verursacht, also
					 * die Mehrkosten für diese Gebühr. Diese dann auch zur Summe der Mehrkosten
					 * hinzurechnen.
					 */
					double differenzEUR = gebuehrTatsaechlich - gebuehrFiktiv;
					mehrkostenEUR += differenzEUR;
					doubles.add(new Double(differenzEUR));
					double differenzQuote = differenzEUR / gebuehrTatsaechlich;
					doubles.add(new Double(differenzQuote * 100.0 /* Prozent */));
					/* Und die Zeile in die Tabelle nehmen */
					zeilen.add(gt.langBezeichnung(), doubles);//toString()
				}
			}
		}
		/*
		 * Zum Schluss noch eine Zeile für den verbliebenen Streitgegenstand. Falls von
		 * dem reduzierten Streitwert dann nicht alles zugesprochen wurde, ist der
		 * Verlust noch einmal höher...
		 */
		double unterliegensQuote;
		if (verurteilung < streitwerte[1]) {
			double anteileAnRechtsstreit = super.berechneAnteileAnRechtsstreit()[0];
			/*
			 * Böse Trickserei: Wenn der Kläger gar nicht gewonnen hat oder der verbliebene
			 * Streitgegenstand wertvoller ist als die tatsächlich angefallenen Kosten, soll
			 * sich die Verteilung im wesentlich nach dem verbliebenen Streitgegenstand
			 * richten. Das wird dadurch erreicht, dass die Mehrkosten den tatsächlichen
			 * Kosten gleichgesetzt werden. So zählt der zurückgenommene Anteil soviel wie
			 * seine Kosten.
			 */
			aufKosteninteresseAbstellen = verurteilung == 0L || streitwerte[1] > summeTatsaechlicheEUR;
			String zwischenUeberschrift;
			String verbliebeneUeberschrift;
			double faktor;
			if (aufKosteninteresseAbstellen) {
				/*
				 * Wenn wir aufs Kosteninteresse abstellen müssen, muss das auch in der Tabelle
				 * stehen); dafür wird nicht nach Anteilen gewichtet (faktor = 1.0)
				 */
				mehrkostenEUR = summeTatsaechlicheEUR;
				zwischenUeberschrift = "Zwischenergebnis (Mehrkosten = Kosteninteresse)";
				faktor = 1.0;
				verbliebeneUeberschrift = "Verbliebener Streitgegenstand";
			} else {
				/*
				 * Wenn wir nicht aufs Kosteninteresse abstellen, dann müssen die Zwischensumme
				 * und der verbliebene Streitgegenstand gemäß der Anteile am Rechtsstreit
				 * gewichtet werden. Das Gewichten geht über die Multiplikation mit
				 * anteileAnRechtsstreit...
				 */
				mehrkostenEUR *= anteileAnRechtsstreit;
				summeTatsaechlicheEUR *= anteileAnRechtsstreit;
				summeFiktiveEUR *= anteileAnRechtsstreit;
				zwischenUeberschrift = String.format("Zwischensumme (gewichtet zu %,.2f%%)",
						anteileAnRechtsstreit * 100);
				/*
				 * ... und hinsichtlich des verbliebenen Streitgegenstands mit faktor.
				 */
				faktor = super.berechneAnteileAnRechtsstreit()[1];
				verbliebeneUeberschrift = String.format("Verbliebener Streitgegenstand (gewichtet zu %,.2f%%)",
						faktor * 100);
			}
			/*
			 * Damit die ausgegebene Tabelle (halbwegs) nachvollziehbar bleibt, brauchen wir
			 * eine Zwischensummen-Zeile
			 */
			unterliegensQuote = mehrkostenEUR / summeTatsaechlicheEUR;
			final DoubleDataRows zwischenWerte = new DoubleDataRows(6);
			zwischenWerte.add(new Double(streitwerte[0]));
			zwischenWerte.add(new Double(summeTatsaechlicheEUR));
			zwischenWerte.add(new Double(streitwerte[1]));
			zwischenWerte.add(new Double(summeFiktiveEUR));
			zwischenWerte.add(new Double(mehrkostenEUR));
			zwischenWerte.add(new Double(unterliegensQuote * 100.0 /* Prozent */));
			zeilen.add(zwischenUeberschrift, zwischenWerte);
			/*
			 * Dann auch die Werte für den verbliebenen Streitgegenstand den Summen
			 * hinzurechnen. faktor = s.o., entweder 1.0 oder gemäß Anteil am Rechtsstreit
			 */
			summeTatsaechlicheEUR += ((double) streitwerte[1]) * faktor;
			summeFiktiveEUR += (double) verurteilung * faktor;
			double restUnterliegen = (double) (streitwerte[1] - verurteilung);
			mehrkostenEUR += restUnterliegen * faktor;
			double restQuote = restUnterliegen / (double) streitwerte[1];
			/* Und die Zeile erzeugen */
			final DoubleDataRows verbliebeneWerte = new DoubleDataRows(6);
			verbliebeneWerte.add(new Double(streitwerte[1]));
			verbliebeneWerte.add(new Double(streitwerte[1]));
			verbliebeneWerte.add(new Double(verurteilung));
			verbliebeneWerte.add(new Double(verurteilung));
			verbliebeneWerte.add(new Double(restUnterliegen));
			verbliebeneWerte.add(new Double(restQuote * 100.0 /* Prozent */));
			zeilen.add(verbliebeneUeberschrift, verbliebeneWerte);
		}
		/* Und die Fußzeile muss hinzu... */
		unterliegensQuote = mehrkostenEUR / summeTatsaechlicheEUR;
		final DoubleDataRows gesamtWerte = new DoubleDataRows(6);
		gesamtWerte.add(new Double(streitwerte[0]));
		gesamtWerte.add(new Double(summeTatsaechlicheEUR));
		gesamtWerte.add(new Double(streitwerte[1]));
		gesamtWerte.add(new Double(summeFiktiveEUR));
		gesamtWerte.add(new Double(mehrkostenEUR));
		gesamtWerte.add(new Double(unterliegensQuote * 100.0 /* Prozent */));
		zeilen.add("Gesamtsumme", gesamtWerte);
		baueGruende(unterliegensQuote);
		return unterliegensQuote;
	}

	/**
	 * Die Methode dient dazu, die Gründe zusammen zu setzen.
	 * 
	 * @param unterliegensQuote ein double zwischen 0.0 und 1.0, der in Prozent
	 *                          umgerechnet das Unterliegen ausdrückt.
	 */
	private void baueGruende(double unterliegensQuote) {
		if (unterliegensQuote == 0.0 || unterliegensQuote == 1.0) {
			super.starteGruende(GRUENDE_EINLEITUNG_91);
			super.erweitereGruende("\n");// TODO
		} else {
			super.starteGruende(GRUENDE_EINLEITUNG_92_269 + WEITER_GRUENDE);
			super.erweitereGruende(GRUENDE_BERECHNUNG_MEHRKOSTEN);
			if (streitwerte[1] > verurteilung) {
				if (aufKosteninteresseAbstellen)
					super.erweitereGruende(
							GRUENDE_VERBLIEBENER_GEGENSTAND + GRUENDE_VERBLIEBENER_GEGENSTAND_WERTVOLLER);
				else
					super.erweitereGruende(GRUENDE_VERBLIEBENER_GEGENSTAND + GRUENDE_VERBLIEBENER_GEGENSTAND_SONST);
			}
			super.erweitereGruende(GRUENDE_EINLEITUNG_TABELLE);
		}
		zeilen.toStringBuilder(gruende);
	}

}