/**
 * QuotenMethode.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.logic.gebuehren;

import java.util.Arrays;
import java.util.List;

import eu.gronos.kostenrechner.controller.TabulierendZeilen;
import eu.gronos.kostenrechner.model.gebuehren.GebuehrenTatbestand;
import eu.gronos.kostenrechner.model.gebuehren.GebuehrenVerzeichnis;
import eu.gronos.kostenrechner.model.gebuehren.Teilklageruecknahme;
import eu.gronos.kostenrechner.model.tenordaten.Allgemein;
import eu.gronos.kostenrechner.model.tenordaten.DoubleDataRows;
import eu.gronos.kostenrechner.model.tenordaten.VerfahrensDatenContainer;

/**
 * QuotenMethode berechnet das Unterliegen des Klägers nach Teilklagerücknahme
 * oder Teilerledigung analog § 92 Abs. 1 ZPO ähnlich einem teilweisen
 * Unterliegen. Nach Schneider (Kosten S. 94, 197 ff.; MüKo § 269 Rn 57) ist für
 * jede Gebühr eine gesonderte Quote zu ermitteln, da einige Gebühren nur nach
 * dem verringerten Streitwert anfallen (Anders/Gehle, A-198).
 * 
 * @author Peter Schuster (setrok)
 * @date 27.04.2014
 */
public class QuotenMethode extends StreitwertReduktion {
	private static final String[] COLUMN_HEADERS = new String[] { "Gebührentatbestand", "Streitwert (EUR)",
			"Gebührenbetrag (EUR)", "Verlustquote Kl. (%)", "Unterliegen Kl. (EUR)" };

	private static final String BESCHREIBUNG = "Tenor nach der sog. Quotenmethode (Hauptsachetenor, Kostentenor, Vollstreckbarkeitstenor und Streitwertbeschluss)";

	private static final String WEITER_GRUENDE = "der sogenannten Quotenmethode.\n";

	/**
	 * Der Konstruktor ruft den Konstruktor der Oberklasse auf.
	 * 
	 * 
	 * @param teilklageruecknahme Parameterojbkekt vom Typ
	 *                            {@link Teilklageruecknahme}, das enthält die
	 *                            {@link Teilklageruecknahme#parteien} klaeger
	 *                            Informationen zur Klägerseite als Beteiligter
	 *                            (wird zur Tenorierung benötigt) und beklagter
	 *                            Informationen zur Beklagten als Beteiligter (wird
	 *                            zur Tenorierung benötigt), die Arrays
	 *                            gebuehrenVorher und gebuehrenNachher, (Arrays
	 *                            GebuehrenTatbestand[] mit allen schon vor bzw.
	 *                            erst nach der "Streitwertreduktion"
	 *                            (Teilklagerücknahme bzw -erledigung) entstandenen
	 *                            Gebühren, die
	 *                            {@link Teilklageruecknahme#streitwerteUndObsiegen}
	 *                            eine ArrayList<Long> mit streitwerteUndObsiegen[0]
	 *                            als dem alten (höheren) Streitwert,
	 *                            streitwerteUndObsiegen[1] als dem neuen,
	 *                            niedrigeren Streitwert und die
	 *                            streitwerteUndObsiegen[2], die Angabe, in welcher
	 *                            Höhe der Kläger letzlich obsiegt hat.
	 * 
	 * @param allgemein           Paremeterobjekt {@link Allgemein}, das enthält:
	 *                            b92ii Soll § 92 II ZPO angewandt werden
	 *                            (verhältnismäßig geringe Kostentragungen werden
	 *                            unterschlagen)
	 * @throws IllegalArgumentException wird geworfen, wenn eine der beiden
	 *                                  folgenden Plausibilitäten verletzt ist: a)
	 *                                  der frühere Streitwert muss größer als der
	 *                                  spätere (oder gleich) sein, b) der spätere
	 *                                  Streitwert muss größer als die Verurteilung
	 *                                  (oder gleich) sein.
	 * 
	 * @see eu.gronos.kostenrechner.logic.gebuehren.StreitwertReduktion#StreitwertReduktion(Allgemein,
	 *      Teilklageruecknahme)
	 * 
	 */
	public QuotenMethode(VerfahrensDatenContainer verfahrensDaten) throws IllegalArgumentException {
		super(verfahrensDaten);
	}
	// Allgemein allgemein, Teilklageruecknahme teilklageruecknahme) //allgemein,
	// teilklageruecknahme);

	/**
	 * @see eu.gronos.kostenrechner.interfaces.Begruendend#getBerechnungsTyp()
	 */
	@Override
	public String getBerechnungsTyp() {
		return BESCHREIBUNG;
	}

	/**
	 * Die Methode berechneUnterliegensQuote dient dazu, die Unterliegensquote nach
	 * Maßgabe der dem Konstruktor übergebenen Werte zu berechnen.
	 * 
	 * @return einen double zwischen 0.0 und 1.0, der in Prozent umgerechnet das
	 *         Unterliegen ausdrückt.
	 * 
	 * @see eu.gronos.kostenrechner.logic.gebuehren.StreitwertReduktion#berechneUnterliegensQuote()
	 */
	@Override
	protected double berechneUnterliegensQuote() {
		double summeUnterliegenEUR = 0.0;
		double summeBetraegeEUR = 0.0;
		/*
		 * Eigentlich wäre so eine Tabelle (Schneider, zit. n. Anders/Gehle, Rn. A-198)
		 * ein Array Object[][], aber eine ArrayList ist schöner, weil man vorher die
		 * Anzahl der Zeilen nicht wissen muss. Die Spaltenanzahl ist aber klar: 5.
		 */
		zeilen = new TabulierendZeilen();
		zeilen.add(Arrays.asList(COLUMN_HEADERS));
		/*
		 * Äußere Zähler-Schleife: Anhand des Arrays streitwerte eine Schleife
		 * durchlaufen, für alles vorher, dann alles nachher. De facto wird die Schleife
		 * genau zweimal durchlaufen, aber zweimal denselben Code zu schreiben, ist
		 * doof. :)
		 */
		for (int i = 0; i < streitwerte.length; i++) {
			List<GebuehrenTatbestand> tatbestaende = i == 0 ? vorher : nachher;
			/*
			 * Die Verlustquote (Bruch zwischen 0.0 und 1.0) muss immer daran gemessen
			 * werden, wie hoch der Kläger letztlich obsiegt, steht also im Nenner des
			 * Bruchs. Bei der vorher-Liste steht der (höhere) Streitwert vor der Reduktion
			 * im Zähler, bei der nachher-Liste der (niedrigere) Streitwert nach der
			 * Reduktion.
			 */
			double verlustQuote = 1.0 - ((double) verurteilung / (double) streitwerte[i]);
			if (tatbestaende != null) {
				/*
				 * Innere Schleife: alle GebührenTB des jeweiligen Arrays GebuehrenTatbestand[]
				 * durchgehen,
				 */
				for (GebuehrenTatbestand gt : tatbestaende) {
					final DoubleDataRows doubles = new DoubleDataRows(4);
					doubles.add(new Double(streitwerte[i]));
					/*
					 * Betrag der Gebühr berechnen und dem Summenfeld hinzu addieren.
					 */
					double betragGebuehr = GebuehrenVerzeichnis.errechneGebuehr(gt, streitwerte[i]);
					summeBetraegeEUR += betragGebuehr;
					doubles.add(new Double(betragGebuehr));
					/*
					 * Das Unterliegen des Klägers für diesen Gebührentatbestand errechnet sich aus
					 * diesem Betrag im Verhältnis zur Quote dieser Zeile. Auch hier zum Summenfeld
					 * addieren.
					 */
					double betragVerlust = verlustQuote * betragGebuehr;
					doubles.add(new Double(verlustQuote * 100.0 /* Prozent */));
					doubles.add(new Double(betragVerlust));
					summeUnterliegenEUR += betragVerlust;
					/* Und die Zeile in die Tabelle nehmen */
					zeilen.add(gt.langBezeichnung(), doubles);//toString()
				}
			}
		}
		/*
		 * Keine Prozentangabe, sondern ein Bruch zwischen 0.0 und 1.0: Summe der
		 * Verlustbeträge durch Summe aller Gebühren
		 */
		final DoubleDataRows gesamtWerte = new DoubleDataRows(4);
		gesamtWerte.add(new Double(0));
		gesamtWerte.add(new Double(summeBetraegeEUR));
		double unterliegensQuote = summeUnterliegenEUR / summeBetraegeEUR;
		gesamtWerte.add(new Double(unterliegensQuote * 100.0 /* Prozent */));
		gesamtWerte.add(new Double(summeUnterliegenEUR));
		/* Ergebniszeile in die Tabelle */
		zeilen.add("Gesamt:", gesamtWerte);
		baueGruende(unterliegensQuote);
		return unterliegensQuote;
	}

	/**
	 * Die Methode dient dazu, die Gründe zusammen zu setzen.
	 * 
	 * @param unterliegensQuote ein double zwischen 0.0 und 1.0, der in Prozent
	 *                          umgerechnet das Unterliegen ausdrückt.
	 */
	private void baueGruende(double unterliegensQuote) {
		if (unterliegensQuote == 0.0 || unterliegensQuote == 1.0) {
			super.starteGruende(GRUENDE_EINLEITUNG_91);
			super.erweitereGruende(
					"Denn auch unter Berücksichtigung der quotalen Beteiligung der Parteien an allen Gebühren ergibt sich ein eindeutiges Ergebnis, wie sich aus folgender Tabelle ergibt:\n");
		} else {
			super.starteGruende(GRUENDE_EINLEITUNG_92_269 + WEITER_GRUENDE);
			super.erweitereGruende(
					"Denn an den einzelnen Gebühren sind die Parteien mit verschiedenen Quoten beteiligt, je nachdem, ob die einzelne Gebühr vor oder nach der Reduktion angefallen ist, wie sich aus der folgenden Tabelle ergibt:\n");
		}
		zeilen.toStringBuilder(gruende);
	}

}