/**
 * TeilruecknahmeTeilerledigung.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.logic.gebuehren;

import java.util.List;

import eu.gronos.kostenrechner.controller.TabulierendZeilen;
import eu.gronos.kostenrechner.controller.gebuehren.TeilklageruecknahmePruefer;
import eu.gronos.kostenrechner.interfaces.Begruendend;
import eu.gronos.kostenrechner.interfaces.ParsendUndBauend;
import eu.gronos.kostenrechner.interfaces.TenorVorbereitend;
import eu.gronos.kostenrechner.interfaces.UnterContainerKlasse;
import eu.gronos.kostenrechner.logic.TenorTexter;
import eu.gronos.kostenrechner.logic.VollstreckbarkeitsHelfer;
import eu.gronos.kostenrechner.model.gebuehren.GebuehrenTatbestand;
import eu.gronos.kostenrechner.model.gebuehren.Teilklageruecknahme;
import eu.gronos.kostenrechner.model.tenordaten.Allgemein;
import eu.gronos.kostenrechner.model.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.model.tenordaten.HauptsacheEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.HauptsacheVerhaeltnis;
import eu.gronos.kostenrechner.model.tenordaten.KostenGrundEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.KostenTragungsVerhaeltnis;
import eu.gronos.kostenrechner.model.tenordaten.SonstigeEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.StreitwertEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.TenorDatenContainer;
import eu.gronos.kostenrechner.model.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.model.tenordaten.Vollstreckbarkeit;
import eu.gronos.kostenrechner.model.tenordaten.VollstreckbarkeitsEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.VollstreckbarkeitsListe;

/**
 * Oberklasse für alle Klasse, die Teilklageerledigung oder Teilklagerücknahme
 * behandeln (bisher Mehrkostenmethode und Quotenmethode). Sie enthält alle
 * gemeinsam benötigten Methoden.
 * 
 * @author Peter Schuster (setrok)
 * @date 16.06.2014
 * 
 */
public abstract class StreitwertReduktion implements TenorVorbereitend, Begruendend {

	static protected final String GRUENDE_EINLEITUNG_91 = "Die Kostenentscheidung beruht auf § 91 ZPO.";
	static protected final String GRUENDE_EINLEITUNG_92 = "Die Kostenentscheidung beruht auf §§ 91, 92 ZPO und ";
	static protected final String GRUENDE_EINLEITUNG_92_269 = "Die Kostenentscheidung beruht auf §§ 91, 92, 269 Abs. 3 ZPO und ";

	protected final List<GebuehrenTatbestand> vorher;
	protected final List<GebuehrenTatbestand> nachher;
	protected final long[] streitwerte;
//	public static final GebuehrenTabelle RVG = new AnwaltsGebuehrenTabelle();
//	public static final GebuehrenTabelle GKG = new GerichtsGebuehrenTabelle();
	protected final long verurteilung;
	final Beteiligter klaeger;
	final Beteiligter beklagter;
	protected final StringBuilder gruende;
	protected TabulierendZeilen zeilen;
	/**
	 * Der quote speichert das Unterliegen des Klägers als double zwischen 0.0 und
	 * 1.0. Vor Initialisierung ist der Wert -1.0.
	 */
	private double quote = -1.0;
	private VollstreckbarkeitsListe vollstreckbarkeitsListe;
	// private final boolean b92ii;
	private final TenorDatenContainer container;
	private ParsendUndBauend<Teilklageruecknahme> pruefer = new TeilklageruecknahmePruefer(true);

	/**
	 * Die Methode berechneUnterliegensQuote dient dazu, die Unterliegensquote nach
	 * Maßgabe der dem Konstruktor übergebenen Werte zu berechnen.
	 * 
	 * @return einen double zwischen 0.0 und 1.0, der in Prozent umgerechnet das
	 *         Unterliegen ausdrückt.
	 */
	protected abstract double berechneUnterliegensQuote();

	private final StreitwertReduktionTexter texter = new StreitwertReduktionTexter();
	/**
	 * in der {@link UnterContainerKlasse} Allgemein stehen {@link Allgemein#b92ii}
	 * und {@link Allgemein#unwesentlich92ii.}
	 */
	private Allgemein allgemein;

	/**
	 * Konstruktor:
	 * 
	 * @param teilklageruecknahme Parameterojbkekt vom Typ
	 *                            {@link Teilklageruecknahme}, das enthält die
	 *                            {@link Teilklageruecknahme#parteien} klaeger
	 *                            Informationen zur Klägerseite als Beteiligter
	 *                            (wird zur Tenorierung benötigt) und beklagter
	 *                            Informationen zur Beklagten als Beteiligter (wird
	 *                            zur Tenorierung benötigt), die Arrays
	 *                            gebuehrenVorher und gebuehrenNachher, (Arrays
	 *                            GebuehrenTatbestand[] mit allen schon vor bzw.
	 *                            erst nach der "Streitwertreduktion"
	 *                            (Teilklagerücknahme bzw -erledigung) entstandenen
	 *                            Gebühren, die
	 *                            {@link Teilklageruecknahme#streitwerteUndObsiegen}
	 *                            ein Array long[3] mit streitwerteUndObsiegen[0]
	 *                            als dem alten (höheren) Streitwert,
	 *                            streitwerteUndObsiegen[1] als dem neuen,
	 *                            niedrigeren Streitwert und die
	 *                            streitwerteUndObsiegen[2], die Angabe, in welcher
	 *                            Höhe der Kläger letzlich obsiegt hat.
	 * 
	 * @param allgemein           Paremeterobjekt {@link Allgemein}, das enthält:
	 *                            b92ii Soll § 92 II ZPO angewandt werden
	 *                            (verhältnismäßig geringe Kostentragungen werden
	 *                            unterschlagen)
	 * @throws IllegalArgumentException wird geworfen, wenn eine der beiden
	 *                                  folgenden Plausibilitäten verletzt ist: a)
	 *                                  der frühere Streitwert muss größer als der
	 *                                  spätere (oder gleich) sein, b) der spätere
	 *                                  Streitwert muss größer als die Verurteilung
	 *                                  (oder gleich) sein.
	 */
	public StreitwertReduktion(VerfahrensDatenContainer verfahrensDaten) throws IllegalArgumentException {
		super();
		Teilklageruecknahme teilklageruecknahme = verfahrensDaten.teilklageruecknahme;
		this.streitwerte = new long[] { teilklageruecknahme.streitwerteUndObsiegen.get(0),
				teilklageruecknahme.streitwerteUndObsiegen.get(1) };
		this.verurteilung = teilklageruecknahme.streitwerteUndObsiegen.get(2);
		// Ausgegliedert in ParsendUndBauend
		pruefer.parseEingabe(teilklageruecknahme);
		this.klaeger = teilklageruecknahme.parteien.get(0);
		this.beklagter = teilklageruecknahme.parteien.get(1);
		this.vorher = teilklageruecknahme.vorher;
		this.nachher = teilklageruecknahme.nachher;
		// this.b92ii = .b92ii;
		this.allgemein = verfahrensDaten.allgemein;
		gruende = new StringBuilder();
		container = new TenorDatenContainer(verfahrensDaten);
	}
	// Allgemein allgemein, Teilklageruecknahme teilklageruecknahme)
	// klaeger;
	// beklagter;
	// allgemein, teilklageruecknahme);
	// container.allgemein = allgemein;
	// container.teilklageruecknahme = teilklageruecknahme;
// if (streitwerte == null || streitwerte[1] > streitwerte[0])
// throw new IllegalArgumentException(TenorTexter.FEHLER_REIHENFOLGE_STREITWERTE);
// if (verurteilung > streitwerte[1])
// throw new IllegalArgumentException(TenorTexter.FEHLER_REIHENFOLGE_VERURTEILUNG);

	/**
	 * Die Methode erzeugeHauptsachetenor dient dazu, einen einfachen
	 * Hauptsachetenor zu erzeugen.
	 * 
	 * @return ein {@link HauptsacheEntscheidungsElemente}-Objekt mit dem
	 *         Hauptsachetenor als String mit abschließendem Zeilenumbruch.
	 */
	public HauptsacheEntscheidungsElemente erzeugeHauptsacheEntscheidung() {
		HauptsacheEntscheidungsElemente hauptsache = new HauptsacheEntscheidungsElemente();
		hauptsache.prozessverhaeltnisse.clear();
		hauptsache.prozessverhaeltnisse.add(new HauptsacheVerhaeltnis(klaeger, beklagter, false, verurteilung));
		// Ausgegliedert in TenorTexter
		texter.setStreitwerte(streitwerte);
		hauptsache.text = texter.texteHauptsache(hauptsache);
		return hauptsache;
	}

	/**
	 * Die Methode erzeugeKostentenor dient dazu, einen Kostentenor zu erzeugen. Die
	 * Angaben zu den Parteien werden aus den Konstruktor-Parametern
	 * <code>klaeger</code> und <code>beklagter</code> entnommen, die Quote aus der
	 * Methode {@link berechneUnterliegensQuote()} der abgeleiteten Klassen.
	 * 
	 * @return den Tenortext als String im Format "Die Kosten haben %s zu %,.2f%%
	 *         und %s zu %,.2f%% zu tragen.%n"
	 */
	public KostenGrundEntscheidungsElemente erzeugeKostenEntscheidung() {
		KostenGrundEntscheidungsElemente kostenEntscheidung = new KostenGrundEntscheidungsElemente();
		if (getQuote() == 0.0) {
			kostenEntscheidung.prozessverhaeltnisse.add(new KostenTragungsVerhaeltnis(klaeger, beklagter, false, 1.0));
		} else if (getQuote() == 1.0) {
			kostenEntscheidung.prozessverhaeltnisse.add(new KostenTragungsVerhaeltnis(beklagter, klaeger, false, 1.0));
		} else {
			kostenEntscheidung.prozessverhaeltnisse
					.add(new KostenTragungsVerhaeltnis(beklagter, klaeger, false, getQuote()));
			kostenEntscheidung.prozessverhaeltnisse
					.add(new KostenTragungsVerhaeltnis(klaeger, beklagter, false, (1.0 - getQuote())));
		}
		kostenEntscheidung.text = texter.texteKostenentscheidung(kostenEntscheidung);
		return kostenEntscheidung;
	}

	/**
	 * Die Methode erzeugt den Tenor zur vorläufigen Vollstreckbarkeit
	 * 
	 * @return einen String
	 */
	@Override
	public VollstreckbarkeitsEntscheidungsElemente erzeugeVollstreckbarkeitsEntscheidung() {
		VollstreckbarkeitsEntscheidungsElemente vollEntscheidung = new VollstreckbarkeitsEntscheidungsElemente();

		if (vollstreckbarkeitsListe == null)
			vollstreckbarkeitsListe = berechneVollstreckbarkeit();
		erweitereGruende(vollstreckbarkeitsListe.toString());
		VollstreckbarkeitsHelfer helfer = new VollstreckbarkeitsHelfer(
				container.hauptsacheEntscheidung.prozessverhaeltnisse,
				container.kostenEntscheidung.prozessverhaeltnisse);
		vollEntscheidung.prozessverhaeltnisse = helfer.toVollstreckbarkeitsVerhaeltnisse(klaeger, beklagter,
				vollstreckbarkeitsListe);

		vollEntscheidung.text = texter.texteVollstreckbarkeit(vollEntscheidung, vollstreckbarkeitsListe);
		return vollEntscheidung;
	}

	/**
	 * Die Methode erzeugt eine Streitwertfestsetzung unter Berücksichtigung des
	 * früheren und des späteren Streitwerts. Das Datum der Reduktion muss man noch
	 * von Hand nachtragen.
	 * 
	 * @return einen String im Format "Der Streitwert wird auf %,.2f EUR bis zum
	 *         ..., danach auf %,.2f EUR festgesetzt.%n"
	 */
	public StreitwertEntscheidungsElemente erzeugeStreitwertEntscheidung() {
		StreitwertEntscheidungsElemente elemente = container.streitwertEntscheidung;
		elemente.streitwerte.add(streitwerte[0]);
		elemente.streitwerte.add(streitwerte[1]);
		elemente.text = texter.texteStreitwert(elemente);
		return elemente;
	}

	/**
	 * Die Methode erzeugt einen leeren sonstigen Tenor.
	 * 
	 * @return ein {@link SonstigeEntscheidungsElemente}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeSonstigeEntscheidung()
	 */
	@Override
	public SonstigeEntscheidungsElemente erzeugeSonstigeEntscheidung() {
		SonstigeEntscheidungsElemente sonstige = new SonstigeEntscheidungsElemente();
		sonstige.text = "";
		return sonstige;
	}

	/**
	 * Die Methode dient dazu, den gesamten {@link TenorDatenContainer} zu erzeugen.
	 * 
	 * @return
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeContainer()
	 */
	@Override
	public TenorDatenContainer erzeugeContainer() {
		container.berechnungsTyp = getBerechnungsTyp();
		container.hauptsacheEntscheidung = erzeugeHauptsacheEntscheidung();
		container.kostenEntscheidung = erzeugeKostenEntscheidung();
		container.vollstreckbarkeitsEntscheidung = erzeugeVollstreckbarkeitsEntscheidung();
		container.streitwertEntscheidung = erzeugeStreitwertEntscheidung();
		container.sonstigeEntscheidung = erzeugeSonstigeEntscheidung();
		container.begruendung = zeilen.toBegruendungsElemente(getGruende());
		return container;
	}

	/**
	 * @return die Entscheidungsgründe als String
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.Begruendend#getGruende()
	 */
	@Override
	public String getGruende() {
		return gruende.toString();
	}

	/**
	 * @return vorher gibt den vorher als {@link List} zurück
	 */
	List<GebuehrenTatbestand> getVorher() {
		return vorher;
	}

	/**
	 * Die Methode gibt nachher als {@link List} zurück
	 * 
	 * @return
	 */
	List<GebuehrenTatbestand> getNachher() {
		return nachher;
	}

	/**
	 * Getter für {@link #quote}. Wenn der Wert noch nicht gesetzt wurde (dann ist
	 * er < 0.0), wird er über {@link #berechneUnterliegensQuote()} berechnet. Je
	 * nach {@link #b92ii} wird {@link #entferneUnwesentlichesUnterliegen(double)}
	 * aufgerufen.
	 * 
	 * @return gibt {@link #quote} als double zurück.
	 */
	double getQuote() {
		if (quote < 0.0) {
			quote = berechneUnterliegensQuote();
			if (isB92ii())
				quote = entferneUnwesentlichesUnterliegen(quote);
		}
		return quote;
	}

	/**
	 * @return gibt {@link #b92ii} als boolean zurück, also ob § 92 II ZPO angewandt
	 *         werden soll (verhältnismäßig geringe Kostentragungen werden dann
	 *         unterschlagen).
	 */
	boolean isB92ii() {
		return allgemein.b92ii;
	}

	/**
	 * @return streitwerte gibt den streitwerte als long[] zurück.
	 */
	long[] getStreitwerte() {
		return streitwerte;
	}

	/**
	 * Die Methode dient dazu, &sect;&nbsp;92 Abs.&nbsp;2 ZPO anzuwenden, also eine
	 * verhältnismäßig geringe Kostentragung zu unterschlagen
	 * 
	 * @param quote die Ausgangs-{@link #getQuote() Quote}
	 * @return die bereinigte {@link #getQuote() Quote}
	 */
	private double entferneUnwesentlichesUnterliegen(double quote) {
		if (quote > 0.0 && quote < allgemein.unwesentlich92ii)// 0.1
			return 0.0;
		else if ((1.0 - quote) < allgemein.unwesentlich92ii)// quote > 0.9
			return 1.0;
		else
			return quote;
	}

	/**
	 * Die Methode berechneAnteileAnRechtsstreit dient dazu, die Anteile des
	 * zurückgenommenen und des verbleibenden Teils am ursprünglichen Streitwert und
	 * damit dem Gesamtrechtsstreits zu ermitteln.
	 * 
	 * @return einen array double[2], mit double-Instanzen zwischen 0.0 und 1.0, bei
	 *         dem double[0] dem Anteil des zurückgenommen Teils am früheren
	 *         Gesamtstreitwert entspricht und double[1] dem verbleibenden Teil am
	 *         früheren Gesamtstreitwert.
	 */
	protected double[] berechneAnteileAnRechtsstreit() {
		double[] zwischen = new double[] { 1.0, 0.0 };
		zwischen[0] = (double) (streitwerte[0] - streitwerte[1]) / (double) streitwerte[0];
		zwischen[1] = (double) streitwerte[1] / (double) streitwerte[0];
		return zwischen;
	}

	/**
	 * Die Methode berechneLetztenVerlust dient dazu, die Verlustquote bzgl. des
	 * niedrigeren Streitwerts nach der Reduktion zu berechnen.
	 * 
	 * @return einen Wert zwischen 0.0 und 1.0, der in Prozente umgerechnet werden
	 *         kann und der Quote entspricht.
	 */
	protected double berechneLetzteVerlustQuote() {
		double zwischen = (double) (streitwerte[1] - verurteilung) / streitwerte[1];
		return zwischen;
	}

	/**
	 * Die Methode berechnet, inwieweit das Urteil vorläufig vollstreckbar ist.
	 * 
	 * @return eine {@link VollstreckbarkeitsListe}, deren 0. Element die
	 *         Unanfechtbarkeit speichert, wenn nicht unanfechtbar das 1. Element
	 *         das Verhältnis kl -> bk und das 2. Element das Verhältnis bk -> kl.
	 */
	private VollstreckbarkeitsListe berechneVollstreckbarkeit() {
		VollstreckbarkeitsListe vollstreckbarkeitsListe = new VollstreckbarkeitsListe();
		vollstreckbarkeitsListe.add(pruefeUnanfechtbarkeit());
		/* Nach festgestellter Unanfechtbarkeit war's das schon. */
		if (vollstreckbarkeitsListe.sindAlle708iVm713()) {
			return vollstreckbarkeitsListe;
		}
		/*
		 * Beim Verhältnis Kl ./. Bk hinsichtlich der Hauptsache einsetzen, was der
		 * Kläger letzlich obsiegt. Hinsichtlich der Kosten die Kosten des Klägers (nach
		 * Streitwert vor Reduktion) einsetzen, soweit sie der Beklagte tragen muss.
		 */
		Vollstreckbarkeit voKl = Vollstreckbarkeit.pruefeSicherheitsleistung(verurteilung,
				(long) (TenorTexter.getKostenKlaeger().errechneGebuehrenSumme(streitwerte[0]) * (1.0 - getQuote())));
		vollstreckbarkeitsListe.add(voKl);
		/*
		 * Beim Verhältnis Bk ./. Kl hinsichtlich der Hauptsache 0L einsetzen,
		 * hinsichtlich der Kosten die des Bk einsetzen (Streitwert vor Reduktion),
		 * soweit sie der Kl tragen muss.
		 */
		Vollstreckbarkeit voBk = Vollstreckbarkeit.pruefeSicherheitsleistung(0L,
				(long) (TenorTexter.getKostenBeklagter().errechneGebuehrenSumme(streitwerte[0]) * getQuote()));
		vollstreckbarkeitsListe.add(voBk);
		return vollstreckbarkeitsListe;
	}

	/**
	 * Die Methode prüft, ob das Urteil unanfechtbar ist (§§ 708, 713 ZPO).
	 * 
	 * @return ein {@link Vollstreckbarkeit}, das im Fall der Unanfechtbarkeit 708
	 *         und 713 auf <code>true</code> setzt. Ansonsten enthaelt es nichts.
	 */
	private Vollstreckbarkeit pruefeUnanfechtbarkeit() {
		/*
		 * Erst einmal das VollstreckbarkeitsObjekt "leer" initialisieren; wenn's nicht
		 * unanfechtbar.
		 */
		Vollstreckbarkeit unanfechtbarkeit = new Vollstreckbarkeit();
		/*
		 * Wenn schon der Streitwert vor Reduktion nicht über 600 EUR geht, ist das
		 * Urteil unanfechtbar.
		 */
		if (streitwerte[0] <= TenorTexter.BERUFUNGS_GRENZE) {
			unanfechtbarkeit = new Vollstreckbarkeit(true, false, false, false, true);
		} else if (streitwerte[0] <= TenorTexter.BERUFUNGS_GRENZE * 2) {
			/*
			 * Wie ist der Kläger beschwert? Das, was von seiner Klage nicht durchgedrungen
			 * ist: Differenz zwischen Streitwert vor Reduktion und letztlich obsiegt. Wie
			 * ist der Beklagte beschwert: Das wozu er verurteilt wurde ... Nur wenn beides
			 * <= 600 ist, dann ist das Urteil unanfechtbar.
			 */
			if (streitwerte[0] - verurteilung <= TenorTexter.BERUFUNGS_GRENZE
					&& verurteilung <= TenorTexter.BERUFUNGS_GRENZE) {
				unanfechtbarkeit = new Vollstreckbarkeit(true, false, false, false, true);
			}
		}
		return unanfechtbarkeit;
	}

	/**
	 * Die Methode setzt die Entscheidungsgründe zurück (sofern nicht leer) und
	 * beginnt sie neu mit dem übergebenen Text.
	 * 
	 * @param text eine {@link CharSequence} mit dem Anfangssatz(-bestandteil) der
	 *             Gründe.
	 */
	protected void starteGruende(CharSequence text) {
		if (gruende.length() > 0)
			gruende.delete(0, gruende.length());
		erweitereGruende(text);
	}

	/**
	 * Die Methode erweitert die bisherigen Gründe
	 * 
	 * @param text eine {@link CharSequence} mit den weiteren Gründen.
	 */
	void erweitereGruende(CharSequence text) {
		gruende.append(text);
	}

}