/*
 * StreitwertReduktionTexter.java
 * eu.gronos.kostenrechner.logic.gebuehren (Kostenrechner)
 */
package eu.gronos.kostenrechner.logic.gebuehren;

import eu.gronos.kostenrechner.logic.TenorTexter;
import eu.gronos.kostenrechner.model.gebuehren.Teilklageruecknahme;
import eu.gronos.kostenrechner.model.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.model.tenordaten.HauptsacheEntscheidungsElemente;
import eu.gronos.kostenrechner.model.tenordaten.HauptsacheVerhaeltnis;
import eu.gronos.kostenrechner.model.tenordaten.StreitwertEntscheidungsElemente;

/**
 * Die Klasse verfasst die Tenortexte für {@link StreitwertReduktion}
 *
 * @author Peter Schuster (setrok)
 * @date 3 Jun 2019
 *
 */
public class StreitwertReduktionTexter extends TenorTexter<Teilklageruecknahme> {
	private long[] streitwerte;

	public StreitwertReduktionTexter() {
		super();
	}

	@Override
	public String texteHauptsache(HauptsacheEntscheidungsElemente hauptsache) {
		HauptsacheVerhaeltnis verhaeltnis = hauptsache.prozessverhaeltnisse.get(0);
		if (verhaeltnis.getVerurteilung() > 0)
			return satzBeginn(String.format(TenorTexter.VERURTEILUNG_ZAHLUNG_AN_GGF_TEILABWEISUNG_FORMAT,
					verhaeltnis.getSchuldner().parteiBezeichner(Beteiligter.NOMINATIV,
							verhaeltnis.getSchuldner().getLfdNr(), true),
					getWirdWerden(verhaeltnis.getSchuldner().isPlural()),
					verhaeltnis.getGlaeubiger().parteiBezeichner(Beteiligter.AKKUSATIV,
							verhaeltnis.getGlaeubiger().getLfdNr(), true),
					(double) verhaeltnis.getVerurteilung(),
					verhaeltnis.getVerurteilung() < streitwerte[1] ? TenorTexter.KLAGEABWEISUNG_IM_UEBRIGEN_LEERSCHRITT
							: ""));
		else
			return TenorTexter.KLAGEABWEISUNG_ZEILE;
	}

	/**
	 * Die Methode verfasst eine Streitwertfestsetzung unter Berücksichtigung des
	 * früheren und des späteren Streitwerts. Das Datum der Reduktion muss man noch
	 * von Hand nachtragen.
	 * 
	 * @param elemente {@link StreitwertEntscheidungsElemente}, in dem
	 *                 {@link StreitwertEntscheidungsElemente#streitwerte} gefüllt
	 *                 ist.
	 * @return einen String im Format "Der Streitwert wird auf %,.2f EUR bis zum
	 *         ..., danach auf %,.2f EUR festgesetzt.%n" für
	 *         {@link StreitwertEntscheidungsElemente#streitwerte#text}
	 * @see eu.gronos.kostenrechner.logic.TenorTexter#texteStreitwert(eu.gronos.kostenrechner.model.tenordaten.TenorDatenContainer.StreitwertEntscheidungsElemente)
	 */
	@Override
	public String texteStreitwert(StreitwertEntscheidungsElemente elemente) {
		return String.format(TenorTexter.STREITWERTFESTSETZUNG_FORMAT, (double) elemente.streitwerte.get(0),
				(double) elemente.streitwerte.get(1));
	}

	public void setStreitwerte(long[] streitwerte) {
		this.streitwerte = streitwerte;
	}
}
