/**
 * BaumbachBeteiligte.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.model.baumbach;

import java.util.ArrayList;
import java.util.Collection;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.logic.TenorTexter;
import eu.gronos.kostenrechner.model.tenordaten.Beteiligter;

/**
 * ArrayList-Ableger für BaumbachBeteiligter mit Methoden aus BaumbachQuote
 * (alle Beklagten geben, Streitwert-Addition usw): Die Klasse
 * BaumbachBeteiligte erweitert die Klasse ArrayList<BaumbachBeteiligter> um
 * Funktionen zur Streitwert-Addition, heraussuchen und Zählen der Beteiligten
 * eines bestimmten Beteiligtentyps usw.
 * 
 * @author Peter Schuster (setrok)
 * @date 30.05.2014
 * 
 * @todo TODO ich brauche noch einen Konverter der "herkömmlichen" Arrays
 *       BaumbachBeteiligter[], die durch eine Methode
 *       gesamtschuldnerischerSockelBetrag usw eine BaumbachBeteiligtenListe
 *       mitsamt den BaumbachGesamtschuldnern erstellt.
 * 
 */
public class BaumbachBeteiligtenListe extends ArrayList<BaumbachBeteiligter> {

	/**
	 * Der long mit dem Namen serialVersionUID ist irgendwie wichtig
	 */
	private static final long serialVersionUID = -8096730827452814426L;
	private static final String ANGRIFFS_NAME = "%s ./. %s auf %,.2f EUR";

	/**
	 * Der Konstruktor ohne Parameter kann nur für interne Zwecke aufgerufen werden.
	 * 
	 */
	public BaumbachBeteiligtenListe() {
		super();
	}

	/**
	 * Der Konstruktion mit einem Parameter intialisiert die Klasse mit den
	 * Beteiligten dieser Collection. Er kann dazu benutzt werden, nur die
	 * Beteiligten eines einzigen Beteiligtentyps aufzunehmen (z.B. nur Kläger) oder
	 * alle Beteiligten aller Typen aufzunehmen.
	 * 
	 * @param c eine Collection<BaumbachBeteiligter>
	 * @throws IllegalArgumentException wenn die Reihenfolge Kläger,
	 *                                  Drittwiderbeklagte, Beklagte nicht
	 *                                  eingehalten ist
	 */
	public BaumbachBeteiligtenListe(Collection<? extends BaumbachBeteiligter> c) throws IllegalArgumentException {
		super(c);
		if (!istReihenfolgeEingehalten())
			throw new IllegalArgumentException(
					"Die Collection der BaumbachBeteiligten muss die Beteiligten bzgl. des Typs in der Reihenfolge KLAEGER, [DRITTWIDERBEKLAGTE,] BEKLAGTE enthalten!");
	}

	/**
	 * Der Konstruktor mit zwei Parametern initialisiert die Klasse mit den
	 * Beteiligten der ersten Collection und fügt danach die Beteiligten der zweiten
	 * Collection hinzu.
	 * 
	 * @param kl die Collection<BaumbachBeteiligter> mit den Klägern
	 * @param bk die Collection<BaumbachBeteiligter> mit den Beklagten
	 * @throws IllegalArgumentException wenn die Reihenfolge Kläger,
	 *                                  Drittwiderbeklagte, Beklagte nicht
	 *                                  eingehalten ist
	 */
	public BaumbachBeteiligtenListe(Collection<? extends BaumbachBeteiligter> kl,
			Collection<? extends BaumbachBeteiligter> bk) throws IllegalArgumentException {
		super(kl);
		addAll(bk);
		if (!istReihenfolgeEingehalten())
			throw new IllegalArgumentException(
					"Die Collection der BaumbachBeteiligten muss die Beteiligten bzgl. des Typs in der Reihenfolge KLAEGER, [DRITTWIDERBEKLAGTE,] BEKLAGTE enthalten!");
	}

	/**
	 * Der Konstruktor mit zwei Parametern initialisiert die Klasse mit den
	 * Beteiligten der ersten Collection und fügt danach die Beteiligten der beiden
	 * anderen Collections hinzu.
	 * 
	 * @param kl   die Collection<BaumbachBeteiligter> mit den Klägern
	 * @param dwbk die Collection<BaumbachBeteiligter> mit den Drittwiderbeklagten
	 * @param bk   die Collection<BaumbachBeteiligter> mit den Beklagten
	 * @throws IllegalArgumentException wenn die Reihenfolge Kläger,
	 *                                  Drittwiderbeklagte, Beklagte nicht
	 *                                  eingehalten ist
	 */
	public BaumbachBeteiligtenListe(Collection<? extends BaumbachBeteiligter> kl,
			Collection<? extends BaumbachBeteiligter> dwbk, Collection<? extends BaumbachBeteiligter> bk)
			throws IllegalArgumentException {
		super(kl);
		addAll(dwbk);
		addAll(bk);
		if (!istReihenfolgeEingehalten())
			throw new IllegalArgumentException(
					"Die Collection der BaumbachBeteiligten muss die Beteiligten bzgl. des Typs in der Reihenfolge KLAEGER, [DRITTWIDERBEKLAGTE,] BEKLAGTE enthalten!");
	}

	/**
	 * Die Methode gibt einen BaumbachBeteiligten der BaumbachBeteiligtenListe als
	 * BaumbachGesamtschuldner zurück.
	 * 
	 * @param index
	 * @return
	 */
	public BaumbachGesamtschuldnerschaft getAlsGesamtschuldner(int index) {
		return (BaumbachGesamtschuldnerschaft) super.get(index);
	}

	/**
	 * Die Methode addiereStreitwerte dient dazu, die Streitwerte aller
	 * BaumbachBeteiligten (ohne die Gesamtschuldner) der BaumbachBeteiligtenListe
	 * zu summieren.
	 * 
	 * @return die Summe der Streitwerte als double
	 */
	public double addiereStreitwerte() {
		double zwischen = 0.0;
		for (BaumbachBeteiligter bb : this)
			if (!bb.isGesamtschuldnerschaft())
				zwischen += bb.getStreitwert();
		return zwischen;
	}

	/**
	 * Die Methode addiereStreitwerte dient dazu, die Streitwerte aller
	 * BaumbachBeteiligten eines betimmten beteiligtenTyp (ohne die Gesamtschuldner)
	 * der BaumbachBeteiligtenListe zu summieren.
	 * 
	 * @param beteiligtenTyp eine der Konstanten BaumbachBeteiligter.KLAEGER,
	 *                       BaumbachBeteiligter.DRITTWIDERBEKLAGTE oder
	 *                       BaumbachBeteiligter.BEKLAGTE.
	 * @return die Summe der Streitwerte als double
	 */
	public double addiereStreitwerte(int beteiligtenTyp) {
		double zwischen = 0.0;
		for (BaumbachBeteiligter bb : this)
			if (bb.getTyp() == beteiligtenTyp && !bb.isGesamtschuldnerschaft())
				zwischen += bb.getStreitwert();
		return zwischen;
	}

	/**
	 * Die Methode addiereStreitwerte dient dazu, die Verurteilungsbeträge aller
	 * BaumbachBeteiligten eines betimmten beteiligtenTyp der
	 * BaumbachBeteiligtenListe zu summieren. Bei einer
	 * {@link BaumbachGesamtschuldnerschaft} multipliziert die Methode den Betrag
	 * mit der Anzahl der Gesamtschuldner.
	 * 
	 * @param beteiligtenTyp eine der Konstanten BaumbachBeteiligter.KLAEGER,
	 *                       BaumbachBeteiligter.DRITTWIDERBEKLAGTE oder
	 *                       BaumbachBeteiligter.BEKLAGTE.
	 * @return die Summe der Verurteilungsbeträge als double
	 */
	public double addiereUnterliegen(int beteiligtenTyp) {
		double zwischen = 0.0;
		// TODO passt das für BaumbachGesamtschuldner?
		for (BaumbachBeteiligter bb : this)
			if (bb.getTyp() == beteiligtenTyp)
				if (bb.isGesamtschuldnerschaft())
					zwischen += (bb.getUnterliegen() * ((BaumbachGesamtschuldnerschaft) bb).getAnzahl());
				else
					zwischen += bb.getUnterliegen();
		// zwischen += (bb.isGesamtschuldner() ? (bb.getUnterliegen()
		// * ((BaumbachGesamtschuldner) bb).getAnzahl() ) : bb
		// .getUnterliegen());
		return zwischen;
	}

	/**
	 * Die Methode ermittelt das Unterliegen des Angreifers, indem von den
	 * Streitwerten der Gegenseite eines bestimmten Typs
	 * {@link Beteiligter#getTyp()} das Unterliegen desselben Typs abgezogen wird.
	 * 
	 * @param typGegenseite {@link Beteiligter#getTyp()}
	 * @return
	 */
	public double addiereAngreiferUnterliegen(int typGegenseite) {
		return addiereStreitwerte(typGegenseite) - addiereUnterliegen(typGegenseite);
	}

	/**
	 * Die Methode zieht das klägerische Unterliegen vom klägerischen Streitwert ab,
	 * um das Angriffsunterliegen des widerklagenden Beklagten zu ermitteln.
	 * 
	 * @param kl
	 * @param beiGesamtschuldnernMultiplizieren                             Bei
	 *                                                                      Gesamtschuldnern
	 *                                                                      wird das
	 *                                                                      Unterliegen
	 *                                                                      mit der
	 *                                                                      Anzahl
	 *                                                                      der
	 *                                                                      repräsentierten
	 *                                                                      BaumbachBeteiligten
	 *                                                                      (getAnzahl)
	 *                                                                      multipliziert.
	 * @param zusaetzlichesGesamtschuldnerischesUnterliegenBeruecksichtigen es wird
	 *                                                                      das
	 *                                                                      zusätzliche
	 *                                                                      Unterliegen
	 *                                                                      zurückgegeben,
	 *                                                                      das sich
	 *                                                                      aus den
	 *                                                                      BaumbachGesamtschuldnern
	 *                                                                      ergibt,
	 *                                                                      die auch
	 *                                                                      den
	 *                                                                      referenzierten
	 *                                                                      BaumbachBeteiligten
	 *                                                                      repräsentieren
	 * @return
	 */
	public double addiereKlaegerUnterliegenFor(final int kl, boolean beiGesamtschuldnernMultiplizieren,
			boolean zusaetzlichesGesamtschuldnerischesUnterliegenBeruecksichtigen) {
		return getStreitwertFor(kl) - getUnterliegenFor(kl, beiGesamtschuldnernMultiplizieren,
				zusaetzlichesGesamtschuldnerischesUnterliegenBeruecksichtigen);
	}

	/**
	 * Die Methode zaehleBeteiligteFuerTyp dient dazu, alle Beteiligte eines Typs
	 * der BaumbachBeteiligtenListe zu zählen.
	 * 
	 * @param beteiligtenTyp eine der Konstanten BaumbachBeteiligter.KLAEGER,
	 *                       BaumbachBeteiligter.DRITTWIDERBEKLAGTE oder
	 *                       BaumbachBeteiligter.BEKLAGTE.
	 * 
	 * @return die Anzahl dieser Beteiligten als int
	 */
	public int zaehleBeteiligteFuerTyp(int beteiligtenTyp) {
		int zwischen = 0;
		for (Beteiligter bb : this)
			if (bb.getTyp() == beteiligtenTyp)
				zwischen++;
		return zwischen;
	}

	/**
	 * Die Methode zaehleKlaeger dient dazu, alle Kläger der
	 * BaumbachBeteiligtenListe zu zählen
	 * 
	 * @return die Anzahl der Kläger als int
	 */
	public int zaehleKlaeger() {
		int zwischen = zaehleBeteiligteFuerTyp(Beteiligter.KLAEGER);
		return zwischen;
	}

	/**
	 * Die Methode zaehleBeklagte dient dazu, alle Beklagten der
	 * BaumbachBeteiligtenListe zu zählen
	 * 
	 * @return die Anzahl der Beklagten als int
	 */
	public int zaehleBeklagte() {
		int zwischen = zaehleBeteiligteFuerTyp(Beteiligter.BEKLAGTE);
		return zwischen;
	}

	/**
	 * Die Methode zaehleDrittwiderbeklagte dient dazu, alle Drittwiderbeklagten der
	 * BaumbachBeteiligtenListe zu zählen
	 * 
	 * @return die Anzahl der Drittwiderbeklagten als int
	 */
	public int zaehleDrittwiderbeklagte() {
		int zwischen = zaehleBeteiligteFuerTyp(Beteiligter.DRITTWIDERBEKLAGTE);
		return zwischen;
	}

	/**
	 * Die Methode zählt alle an einer Widerklage beteiligte BaumbachBeteiligten der
	 * BaumbachBeteiligtenListe eines bestimmten Typs.
	 * 
	 * @param beteiligtenTyp eine der Konstanten Beteiligter.KLAEGER,
	 *                       Beteiligter.DRITTWIDERBEKLAGTE oder
	 *                       Beteiligter.BEKLAGTE oder -1, wenn für alle Typen
	 *                       gezählt werden soll.
	 * @return die Anzahl
	 */
	public int zaehleAnWiderklageBeteiligte(int beteiligtenTyp) {
		int anzahl = 0;
		for (BaumbachBeteiligter bbt : this)
			if (bbt.isAnWiderklageBeteiligt() && (beteiligtenTyp < 0 || beteiligtenTyp == bbt.getTyp()))
				anzahl++;
		return anzahl;
	}

	/**
	 * Die Methode zählt alle an einer Widerklage beteiligte BaumbachBeteiligten der
	 * BaumbachBeteiligtenListe.
	 * 
	 * @return die Anzahl
	 */
	public int zaehleAnWiderklageBeteiligte() {
		return zaehleAnWiderklageBeteiligte(-1);
	}

	/**
	 * Die Methode istReihenfolgeEingehalten prüft, ob die Beteiligten in der
	 * Reihenfolge Kläger, Drittwiderbeklagte, Beklagte in der Collection sind.
	 * 
	 * @return true, wenn die Reihenfolge eingehalten ist, sonst false.
	 */
	public boolean istReihenfolgeEingehalten() {
		boolean zwischen = true;
		int highest = 0;
		for (Beteiligter bb : this)
			if (bb.getTyp() > highest)
				highest = bb.getTyp();
			else if (bb.getTyp() < highest)
				zwischen = false;
		return zwischen;
	}

	/**
	 * Die Methode enthaeltAnWiderklageBeteiligte ermittelt, ob BaumbachBeteiligte
	 * der BaumbachBeteiligtenListe an einer Widerklage beteiligt sind, ne.
	 * 
	 * @return true, wenn mindestens ein BaumbachBeteiligter gefunden wird.
	 */
	public boolean enthaeltAnWiderklageBeteiligte() {
		// boolean zwischen = false;
		// for (BaumbachBeteiligter bbt : this)
		// if (bbt.isAnWiderklageBeteiligt())
		// zwischen = true;
		// return zwischen;
		return zaehleAnWiderklageBeteiligte() > 0;
	}

	/**
	 * Die Methode prüft, ob BaumbachGesamtschuldner in der BaumbachBeteiligtenListe
	 * sind
	 * 
	 * @return true, wenn die Liste mindestens einen BaumbachGesamtschuldner enthält
	 */
	public boolean enthaeltGesamtschuldner() {
		return zaehleGesamtschuldner() > 0;
	}

	/**
	 * Die Methode minUndMax dient dazu, die Indizes des ersten und des Letzten
	 * Vertreters eines Beteiligtentyps zu erfragen.
	 * 
	 * @param beteiligtenTyp eine der Konstanten Beteiligter.KLAEGER,
	 *                       Beteiligter.DRITTWIDERBEKLAGTE oder
	 *                       Beteiligter.BEKLAGTE.
	 * @return ein Array int[2], bei dem int[0] der erste und int[1] der letzte
	 *         Index des Beteiligtentyps in der BaumbachBeteiligtenListe ist oder
	 *         null, wenn der Typ nicht bekannt ist.
	 */
	public int[] minUndMax(int beteiligtenTyp) {
		int[] minMax = new int[2];
		int[] parteien = new int[] { zaehleKlaeger(), zaehleDrittwiderbeklagte(), zaehleBeklagte() };
		switch (beteiligtenTyp) {
		case Beteiligter.KLAEGER:
			minMax[0] = 0;
			minMax[1] = parteien[Beteiligter.KLAEGER] - 1;
			break;
		case Beteiligter.DRITTWIDERBEKLAGTE:
			minMax[0] = parteien[Beteiligter.KLAEGER];
			minMax[1] = parteien[Beteiligter.KLAEGER] + parteien[Beteiligter.DRITTWIDERBEKLAGTE] - 1;
			break;
		case Beteiligter.BEKLAGTE:
			minMax[0] = parteien[Beteiligter.KLAEGER] + parteien[Beteiligter.DRITTWIDERBEKLAGTE];
			minMax[1] = parteien[Beteiligter.KLAEGER] + parteien[Beteiligter.DRITTWIDERBEKLAGTE]
					+ parteien[Beteiligter.BEKLAGTE] - 1;
			break;
		default:
			minMax = null;
		}
		return minMax;
	}

	/**
	 * Die Methode parteiBezeichner dient dazu, den ParteiBezeichner eines
	 * bestimmten Mitglieds der Liste zurückzugeben
	 * 
	 * @param casus eine der Konstanten BaumbachBeteiligter.NOMINATIV,
	 *              BaumbachBeteiligter.GENITIV, BaumbachBeteiligter.DATIV,
	 *              BaumbachBeteiligter.AKKUSATIV
	 * @param index die Nummer bezogen auf die vollständige BaumbachBeteiligtenListe
	 *              (wie bei get(int))
	 * @return
	 */
	public String parteiBezeichner(int casus, int index) {
		// laufende Nummer bestimmen
		int laufendeNummer = ermittleLaufendeNummerFuerBeteiligtenTyp(index);
		// ist der Beteiligte der einzige seiner Art?
		boolean einzigerSeinerArt = isEinzigerSeinerArt(index);
		// (findeGesamtschuldnerIndices(getTyp(index), false)).size() <= 1;
		// zaehleBeteiligteFuerTyp(getTyp(index)) <= 1;
		// Methode parteiBezeichner des Beteiligten aufrufen
		return get(index).parteiBezeichner(casus, laufendeNummer, einzigerSeinerArt);
	}

	/**
	 * Die Methode ermittelt die laufende Nummer eines Beteiligten der
	 * BaumbachBeteiligtenListe für seinen jeweiligen Beteiligtentyp.
	 * 
	 * @param index die Nummer bezogen auf die vollständige BaumbachBeteiligtenListe
	 *              (wie bei get(int))
	 * @return die Nummer bezogen die BaumbachBeteiligten der
	 *         BaumbachBeteiligtenListe eines bestimmten Typs (z.B. 1 für den
	 *         Beklagten zu 1) als int
	 */
	public int ermittleLaufendeNummerFuerBeteiligtenTyp(int index) {
		return index + 1 - minUndMax(getTypFor(index))[0];
	}

	/**
	 * Die Methode getGenusNumerus dient zum Durchgriff auf die
	 * getGenusNumerus()-Methode des Beteiligten bei <code>index</code>.
	 * 
	 * @param index die Nummer bezogen auf die vollständige BaumbachBeteiligtenListe
	 *              (wie bei get(int))
	 * @return den GenusNumerus als int
	 */
	public int getGenusNumerusFor(int index) {
		return get(index).getGenusNumerus();
	}

	/**
	 * Die Methode getTyp dient zum Durchgriff auf die getTyp()-Methode des
	 * Beteiligten bei <code>index</code>
	 * 
	 * @param index die Nummer bezogen auf die vollständige BaumbachBeteiligtenListe
	 *              (wie bei get(int))
	 * @return den beteiligtenTyp als int
	 */
	public int getTypFor(int index) {
		return get(index).getTyp();
	}

	/**
	 * Die Methode dient zum Durchgriff auf die getStreitwert()-Methode des
	 * Beteiligten bei <code>index</code>
	 * 
	 * @param index die Nummer bezogen auf die vollständige BaumbachBeteiligtenListe
	 *              (wie bei get(int))
	 * @return den Streitwert als double
	 */
	public double getStreitwertFor(int index) {
		return get(index).getStreitwert();
	}

	/**
	 * Die Methode getUnterliegen dient dazu, das Unterliegen für den
	 * BaumbachBeteiligten bei <code>index</code> zurückzugeben.
	 * 
	 * @param index                                                         die
	 *                                                                      Nummer
	 *                                                                      bezogen
	 *                                                                      auf die
	 *                                                                      vollständige
	 *                                                                      BaumbachBeteiligtenListe
	 *                                                                      (wie bei
	 *                                                                      get(int))
	 * @param beiGesamtschuldnernMultiplizieren                             Bei
	 *                                                                      Gesamtschuldnern
	 *                                                                      wird das
	 *                                                                      Unterliegen
	 *                                                                      mit der
	 *                                                                      Anzahl
	 *                                                                      der
	 *                                                                      repräsentierten
	 *                                                                      BaumbachBeteiligten
	 *                                                                      (getAnzahl)
	 *                                                                      multipliziert.
	 * @param zusaetzlichesGesamtschuldnerischesUnterliegenBeruecksichtigen es wird
	 *                                                                      das
	 *                                                                      zusätzliche
	 *                                                                      Unterliegen
	 *                                                                      zurückgegeben,
	 *                                                                      das sich
	 *                                                                      aus den
	 *                                                                      BaumbachGesamtschuldnern
	 *                                                                      ergibt,
	 *                                                                      die auch
	 *                                                                      den
	 *                                                                      referenzierten
	 *                                                                      BaumbachBeteiligten
	 *                                                                      repräsentieren
	 * @return das Unterliegen als double, samt den gewünschten hinzugerechneten
	 *         Werten.
	 */
	public double getUnterliegenFor(int index, boolean beiGesamtschuldnernMultiplizieren,
			boolean zusaetzlichesGesamtschuldnerischesUnterliegenBeruecksichtigen) {
		double unterliegen = beiGesamtschuldnernMultiplizieren ? getGanzesUnterliegen(index)
				: get(index).getUnterliegen();
		if (zusaetzlichesGesamtschuldnerischesUnterliegenBeruecksichtigen)
			unterliegen += getZusaetzlichesUnterliegen(index);
		return unterliegen;
	}

	/**
	 * Die Methode isGesamtschuldner ermittelt, ob der BaumbachBeteiligte bei
	 * <code>index</code> ein BaumbachGesamtschuldner ist.
	 * 
	 * @param index die Position der BaumbachBeteiligtenListe
	 * @return true, wenn der Beteiligte an der Position ein BaumbachGesamtschuldner
	 *         ist.
	 */
	public boolean isGesamtschuldnerschaft(int index) {
		return get(index).isGesamtschuldnerschaft();
	}

	/**
	 * Die Methode ermittelt, ob der BaumbachBeteiligte bei <code>index</code> an
	 * einer Widerklage beteiligt ist.
	 * 
	 * @param index die Position der BaumbachBeteiligtenListe
	 * @return true, wenn der Beteiligte an der Position an der Widerklage beteiligt
	 *         ist.
	 */
	public boolean isAnWiderklageBeteiligt(int index) {
		return get(index).isAnWiderklageBeteiligt();
	}

	/**
	 * Die Methode ermittelt, ob der BaumbachBeteiligte bei <code>index</code> an
	 * einer Widerklage beteiligt und keine Gesamtschuldnerschaft ist.
	 * 
	 * @param index die Position der BaumbachBeteiligtenListe
	 * @return
	 */
	public boolean isEinzelpersonUndAnWiderklageBeteiligt(int index) {
		return isAnWiderklageBeteiligt(index) && !isGesamtschuldnerschaft(index);
	}

	/**
	 * Die Methode ermittelt für einen Beteiligtentyp den höchsten Streitwert, der
	 * in der BaumbachBeteiligtenListe vertreten ist.
	 * 
	 * @param beteiligtenTyp eine der Konstanten Beteiligter.KLAEGER,
	 *                       Beteiligter.BEKLAGTE und
	 *                       Beteiligter.DRITTWIDERBEKLAGTE.
	 * 
	 * @return den gefundenen Streitwert als double
	 */
	public double getHoechstenStreitwert(int beteiligtenTyp) {
		double hoechster = 0.0;
		for (BaumbachBeteiligter bbt : this)
			if (bbt.getTyp() == beteiligtenTyp && bbt.getStreitwert() > 0.0 && bbt.getStreitwert() > hoechster)
				hoechster = bbt.getStreitwert();
		return hoechster;
	}

	/**
	 * Zur Tenorerstellung: wo befinden sich die Gesamtschuldnersammlungen?
	 * 
	 * @param beteiligtenTyp      eine der Konstanten Beteiligter.DRITTWIDERBEKLAGTE
	 *                            bzw. Beteiligter.BEKLAGTE
	 * @param gesamtschuldnerisch sollen Gesamtschuldnersammlungen gefunden werden
	 *                            (true) oder die übrigen?
	 * @return eine ArrayList<Integer> mit den indices der
	 *         Gesamtschuldnersammlungne, bezogen auf die BaumbachBeteiligtenListe.
	 */
	public ArrayList<Integer> findeGesamtschuldnerIndices(int beteiligtenTyp, boolean gesamtschuldnerisch) {
		// TODO die zaehleGesamtschuldner-Methoden hiervon ableiten
		ArrayList<Integer> indices = new ArrayList<Integer>();
		for (int index = 0; index < size(); index++)
			if (get(index).isGesamtschuldnerschaft() == gesamtschuldnerisch && getTypFor(index) == beteiligtenTyp)
				indices.add(index);
		return indices;
	}

	/**
	 * Gesamtschuldner zuerst, Klagetenor vor Widerklagetenor. Die Spalten mit den
	 * BaumbachGesamtschuldner müssen an den Anfang der Liste ihres Beteiligtentyps
	 * 
	 * @param klaegerVorBeklagten falls nicht Klagetenor vor Widerklagetenor
	 * @return eine {@link ArrayList}&lt;{@link Integer}&gt;, die auf die
	 *         {@link BaumbachBeteiligter BaumbachBeteiligten} der Liste in der
	 *         gewünschten Reihenfolge verweisen.
	 */
	public ArrayList<Integer> beteiligteInTenorReihenfolge(boolean klaegerVorBeklagten) {
		ArrayList<Integer> indices = new ArrayList<Integer>();
		/*
		 * Erst einmal die abzuarbeitende Liste der Beteiligtentypen anlegen für die
		 * äußere Schleife; diese wird von der äußeren Schleife verarbeitet und ist vom
		 * Parameter klaegerVorBeklagten abhängig.
		 */
		int[] typen = klaegerVorBeklagten
				? new int[] { Beteiligter.KLAEGER, Beteiligter.DRITTWIDERBEKLAGTE, Beteiligter.BEKLAGTE }
				: new int[] { Beteiligter.BEKLAGTE, Beteiligter.KLAEGER, Beteiligter.DRITTWIDERBEKLAGTE };
		for (int typ : typen) {
			/*
			 * Die innere Schleife führt findeGesamtschuldnerIndices einmal mit true und
			 * einmal mit false aus.
			 */
			for (boolean gesamtschuldnerisch : new boolean[] { true, false }) {
				ArrayList<Integer> zwischen = null;
				int t = typ;
				if (gesamtschuldnerisch) {
					/*
					 * DRITTWIDEBEKLAGTE gesamtschuldnerisch soll vorgezogen werden, wenn eigetnlich
					 * gesamtschuldnerische Kläger "dran" wären und muss dafür übersprungen werden,
					 * wenn sie eigentlich "dran" wäre.
					 */
					if (typ == Beteiligter.KLAEGER)
						t = Beteiligter.DRITTWIDERBEKLAGTE;
					else if (typ == Beteiligter.DRITTWIDERBEKLAGTE)
						continue;
				}
				zwischen = findeGesamtschuldnerIndices(t, gesamtschuldnerisch);
				if (zwischen.size() > 0)
					indices.addAll(zwischen);
			}
		}
		// if (klaegerVorBeklagten) {
		// ArrayList<Integer> zwischen = null; zwischen =
		// findeGesamtschuldnerIndices( Beteiligter.DRITTWIDERBEKLAGTE, true);
		// if (zwischen.size() > 0) indices.addAll(zwischen);
		// zwischen = findeGesamtschuldnerIndices(Beteiligter.KLAEGER, false);
		// if (zwischen.size() > 0) indices.addAll(zwischen);
		// zwischen = findeGesamtschuldnerIndices(
		// Beteiligter.DRITTWIDERBEKLAGTE, false);
		// if (zwischen.size() > 0) indices.addAll(zwischen);
		// zwischen = findeGesamtschuldnerIndices(Beteiligter.BEKLAGTE, true);
		// if (zwischen.size() > 0) indices.addAll(zwischen);
		// zwischen = findeGesamtschuldnerIndices(Beteiligter.BEKLAGTE, false);
		// if (zwischen.size() > 0) indices.addAll(zwischen);
		// } else {
		// ArrayList<Integer> zwischen = null;
		// zwischen = findeGesamtschuldnerIndices(Beteiligter.BEKLAGTE, true);
		// if (zwischen.size() > 0) indices.addAll(zwischen);
		// zwischen = findeGesamtschuldnerIndices(Beteiligter.BEKLAGTE, false);
		// if (zwischen.size() > 0) indices.addAll(zwischen);
		// zwischen = findeGesamtschuldnerIndices(
		// Beteiligter.DRITTWIDERBEKLAGTE, true);
		// if (zwischen.size() > 0) indices.addAll(zwischen);
		// zwischen = findeGesamtschuldnerIndices(Beteiligter.KLAEGER, false);
		// if (zwischen.size() > 0) indices.addAll(zwischen);
		// zwischen = findeGesamtschuldnerIndices(
		// Beteiligter.DRITTWIDERBEKLAGTE, false);
		// if (zwischen.size() > 0) indices.addAll(zwischen);
		// }
		return indices;
	}

	/**
	 * Die Methode ermittelt, welche Beteiligte in Anspruch genommen wurden.
	 * 
	 * @return eine ArrayList<Integer> mit den indices aller einfachen Beteiligten
	 *         (d.h. keine Gesamtschuldnerschaften), die verklagt wurden, in der
	 *         Reihenfolge BEKLAGTE, KLAEGER, DRITTWIDERBEKLAGTE
	 */
	public ArrayList<Integer> beteiligteWieVerklagt() {
		ArrayList<Integer> indices = new ArrayList<Integer>();
		/*
		 * Reihenfolge der Liste muss BEKLAGTE, KLAEGER, DRITTWIDERBEKLAGTE sein.
		 */
		for (int typ : new int[] { Beteiligter.BEKLAGTE, Beteiligter.KLAEGER, Beteiligter.DRITTWIDERBEKLAGTE }) {
			for (int index = 0; index < size(); index++)
				/* Hier sollen nur einfache Beteiligte aufgeführt werden. */
				if (getTypFor(index) == typ && !isGesamtschuldnerschaft(index)) {
					/* BEKLAGTE und DRITTWIDERBEKLAGTE immer aufnehmen */
					if (typ == Beteiligter.BEKLAGTE || typ == Beteiligter.DRITTWIDERBEKLAGTE)
						indices.add(new Integer(index));
					/* KLAEGER nur aufnehmen, wenn widerbeklagt */
					if (typ == Beteiligter.KLAEGER && isAnWiderklageBeteiligt(index))
						indices.add(new Integer(index));
				}
		}
		return indices;
	}

	/**
	 * 
	 * Die Methode ermittelt, ob ein "einfacher" BaumbachBeteiligter auch
	 * gesamtschuldnerisch verurteilt wurde. Das macht dann sein "normales"
	 * Unterliegen zu einem "darüber hinaus"
	 * 
	 * @param index die Position in der BaumbachBeteiligtenListe
	 * 
	 * @return true, wenn der Beteiligte auch in einer Gesamtschuldnerschaft ist.
	 */
	public boolean hatDarueberHinaus(int index) {
		ArrayList<Integer> liste = findeDarueberHinaus(index);
		return liste != null && liste.size() > 0;
	}

	/**
	 * Die Methode findet für jeden BaumbachBeteiligten alle
	 * Gesamtschuldnerschaften, in denen er vertreten ist.
	 * 
	 * @param index die Position in der BaumbachBeteiligtenListe
	 * @return eine ArrayList<Integer> mit den Indexen von allen
	 *         Gesamtschuldnerschaften
	 */
	public ArrayList<Integer> findeDarueberHinaus(int index) {
		ArrayList<Integer> liste = new ArrayList<Integer>();
		if (get(index) instanceof BaumbachGesamtschuldnerschaft) {
			/*
			 * Gesamtschuldnerschaften: "true", wenn es eine andere
			 * BaumbachGesamtschuldnerschaft mit mehr indices.length gibt, in der alle
			 * indices der BaumbachGesamtschuldnerschaft unter index enthalten sind.
			 */
			for (int i : findeGesamtschuldnerIndices(get(index).getTyp(), true)) {
				BaumbachGesamtschuldnerschaft derGrosse = getAlsGesamtschuldner(i);
				BaumbachGesamtschuldnerschaft derKleine = getAlsGesamtschuldner(index);
				if (derGrosse.getAufzaehlung().length > derKleine.getAufzaehlung().length) {
					boolean alleEnthalten = true;
					/*
					 * jedes Element von derKleine durchgehen, ob es auch in derGrosse enthalten ist
					 */
					for (int mitglied : derKleine.getAufzaehlung()) {
						BaumbachBeteiligtenListe mitglieder = derKleine.getBaumbachBeteiligtenListe();
						if (!derGrosse.contains(mitglieder.getTypFor(mitglied),
								mitglieder.ermittleLaufendeNummerFuerBeteiligtenTyp(mitglied)))
							alleEnthalten = false;
					}
					if (alleEnthalten)
						liste.add(new Integer(i));
				}
			}
		} else {
			/* Einfacher BaumbachBeteiligter */
			int obertyp = get(index).getTyp();
			/*
			 * Wenn der Typ ein Kläger ist, dann ist der dazugehörige Typ für
			 * gesamtschuldnerische Verurteilungen der Drittwiderbeklagte
			 */
			if (obertyp == Beteiligter.KLAEGER)
				obertyp = Beteiligter.DRITTWIDERBEKLAGTE;
			for (int i : findeGesamtschuldnerIndices(obertyp, true))
				if (((BaumbachGesamtschuldnerschaft) get(i)).contains(get(index).getTyp(),
						ermittleLaufendeNummerFuerBeteiligtenTyp(index)))
					liste.add(new Integer(i));
		}
		return liste;
	}

	/**
	 * Die Methode geht die einfachen BaumbachBeteiligten eines Typs durch und
	 * guckt, inwiefern sie übereinstimmend verurteilt worden sind. In dieser Höhe
	 * erzeugt sie eine BaumbachGesamtschuldnerschaft und zieht sie von den
	 * Beteiligten im Unterliegen ab.
	 * 
	 * @param beteiligtenTyp
	 * @return die erzeugten BaumbachGesamtschuldnerschaft oder null, wenn keine
	 *         Sockelquote > 0 gefunden wurde.
	 */
	public BaumbachGesamtschuldnerschaft extrahiereGesamtschuldnerschaft(int beteiligtenTyp) {
		ArrayList<Integer> eingangsListe;
		ArrayList<Integer> ausgabeListe = new ArrayList<Integer>();
		int genusNumerus;
		int maenner = 0;
		int frauen = 0;
		double sockelbetrag = 0;
		boolean widerklageDabei = false;
		int[] indices;

		// Wenn typ == KLAEGER, immer auch DRITTWIDERBEKLAGTE
		// berücksichtigen und umgekehrt!
		int typ = (beteiligtenTyp == Beteiligter.KLAEGER) ? typ = Beteiligter.DRITTWIDERBEKLAGTE : beteiligtenTyp;
		if (typ == Beteiligter.DRITTWIDERBEKLAGTE) {
			eingangsListe = findeGesamtschuldnerIndices(Beteiligter.KLAEGER, false);
			eingangsListe.addAll(findeGesamtschuldnerIndices(Beteiligter.DRITTWIDERBEKLAGTE, false));
		} else
			eingangsListe = findeGesamtschuldnerIndices(beteiligtenTyp, false);

		// Gesamtschuldnerischen "Sockelbetrag" (also den Betrag, den alle
		// mindestens erreichen, sofern überhaupt unterlegen) finden (Schleife
		// 1)
		for (int i : eingangsListe) {
			final double zwischen = get(i).getUnterliegen();
			if (sockelbetrag <= 0)
				sockelbetrag = zwischen;
			else if (zwischen > 0 && zwischen < sockelbetrag)
				sockelbetrag = zwischen;
		}
		// Wenn nach der Schleife der Sockelbetrag immer noch 0 ist, gibt's wohl
		// nicht mehr zu finden (führt sonst zu Endlosschleife).
		if (sockelbetrag <= 0)
			return null;

		// Alle einfachen (!) BaumbachBeteiligten des Typs finden, die
		// den Sockelbetrag mindestens erreichen und die indices in einer
		// ArrayList<Integer> speichern - muss leider zweite Schleife sein
		for (int i : eingangsListe) {
			final BaumbachBeteiligter bb = get(i);
			if (bb.getUnterliegen() >= sockelbetrag) {
				ausgabeListe.add(i);
				// Dabei auch Geschlecht ermitteln
				if (bb.getGenusNumerus() == Beteiligter.MAENNLICH
						|| bb.getGenusNumerus() == Beteiligter.MAENNLICH + Beteiligter.PLURAL)
					maenner++;
				else
					frauen++;
				// widerklageDabei ist bei BEKLAGTE immer false, ansonsten
				// abhängig davon ob Widerbeklagte dabei sind:
				if (beteiligtenTyp != Beteiligter.BEKLAGTE && bb.isAnWiderklageBeteiligt())
					widerklageDabei = true;
			}
		}

		// Wenn nicht mehr als einer gefunden wird: return null;
		// Gesamtschuldnerschaft erfordert mindestens zwei Mitspieler
		if (ausgabeListe.size() < 2)
			return null;
		Kostenrechner.getLogger().info(String.format("Size: %d.", ausgabeListe.size()));
		// System.out.println("Size: " + ausgabeListe.size());

		// Dann von allen "einfachen" BaumbachBeteiligten in ausgangsListe vom
		// unterliegen den sockelbetrag abziehen - Dritte Schleife (darf nicht
		// in der zweiten Schleife erfolgen, denn da wissen wir noch nicht, ob
		// die ausgangsListe letztlich mindestens zwei Elemente hat)
		for (int i : ausgabeListe)
			get(i).setUnterliegen(get(i).getUnterliegen() - sockelbetrag);

		// indices füllen; toArray klappt nicht, weil Integer[] != int[]
		indices = new int[ausgabeListe.size()];
		for (int i = 0; i < ausgabeListe.size(); i++)
			indices[i] = ausgabeListe.get(i);

		// Zählen, ob mehr Männer oder Frauen; bei Patt: Männlich
		genusNumerus = Beteiligter.PLURAL + (frauen > maenner ? Beteiligter.WEIBLICH : Beteiligter.MAENNLICH);

		// Aus diesen Beteiligten einen neuen BaumbachGesamtschuldner
		// basteln mit typ = Dwbk oder Bk (je nach Parameter beteiligtenTyp),
		// sockelbetrag als unterliegen, indices als int[] und
		// "this" als BaumbachBeteiligtenListe
		return new BaumbachGesamtschuldnerschaft(typ, genusNumerus, sockelbetrag, widerklageDabei, indices, this);
	}

	/**
	 * Die Methode benennt den Angriff gegen den Beteiligten mit dem
	 * <code>index</code>, indem es die {@link #parteiBezeichner(int, int)} und den
	 * Streitwert ausgibt.
	 * 
	 * @param index
	 * @return ein {@link String}
	 */
	public String benenneAngriffGegen(int index) {
		return String.format(ANGRIFFS_NAME, parteiBezeichner(Beteiligter.NOMINATIV, findeAngreiferVon(index)),
				parteiBezeichner(Beteiligter.AKKUSATIV, index), getStreitwertFor(index));
	}

	/**
	 * Die Methode zaehleRichtigeBeteiligte zählt alle "richtigen"
	 * BaumbachBeteiligten der Liste (also die nicht "nur" Gesamtschuldner sind).
	 * 
	 * @return die Anzahl der BaumbachBeteiligten der BaumbachBeteiligtenListe, die
	 *         keine BaumbachGesamtschuldner sind.
	 */
	int zaehleRichtigeBeteiligte() {
		return size() - zaehleGesamtschuldner();
	}

	/**
	 * Die Methode zaehleGesamtschuldner zählt alle Gesamtschuldner der
	 * BaumbachBeteiligtenListe.
	 * 
	 * @return die Anzahl der BaumbachGesamtschuldner der BaumbachBeteiligtenListe
	 */
	int zaehleGesamtschuldner() {
		int anzahl = 0;
		for (BaumbachBeteiligter bbt : this)
			if (bbt.isGesamtschuldnerschaft())
				anzahl++;
		return anzahl;
	}

	/**
	 * Die Methode alleBeteiligtenEinesTyps dient dazu, eine neue
	 * BaumbachBeteiligtenListe nur mit den Beteiligten eines Typs
	 * <code>beteiligtenTyp</code> zurückzugeben.
	 * 
	 * @param beteiligtenTyp eine der Konstanten BaumbachBeteiligter.KLAEGER,
	 *                       BaumbachBeteiligter.DRITTWIDERBEKLAGTE oder
	 *                       BaumbachBeteiligter.BEKLAGTE.
	 * @return eine neue Instanz BaumbachBeteiligte mit allen Beteiligten dieses
	 *         Typs, die auch leer sein kann.
	 */
	BaumbachBeteiligtenListe alleBeteiligtenEinesTyps(int beteiligtenTyp) {
		// TODO hier wird noch eine Methode gebraucht, die nach
		// Pseudo-/Gesamtschuldnern trennt
		if (zaehleBeteiligteFuerTyp(beteiligtenTyp) <= 0)
			return new BaumbachBeteiligtenListe();
		int[] fromTo = minUndMax(beteiligtenTyp);
		return new BaumbachBeteiligtenListe(subList(fromTo[0], fromTo[1]));
	}

	/**
	 * Die Methode parteiBezeichnerListe dient dazu, eine Liste der repräsentierten
	 * BaumbachBeteiligten zu erzeugen.
	 * 
	 * @param casus   eine der Konstanten NOMINATIV, GENITIV, DATIV, AKKUSATIV
	 * @param indexes ein Array int[] mit den indexes der BaumbachBeteiligtenListe
	 * @return einen String mit einem passenden Bezeichner etwa "der Kläger", "die
	 *         Kläger", "der Beklagte zu 1.)".
	 */
	String parteiBezeichnerListe(int casus, int indexes[]) {
		String zwischen = "";
		int typ = -1;
		for (int index : indexes) {
			// Wenn der beteiligtenTyp zwischenzeitlich wechselt, den Bezeichner
			// ausgeben ...
			if (getTypFor(index) != typ)
				zwischen += Beteiligter.parteiBezeichner(getTypFor(index),
						getGenusNumerusFor(index) + (!isEinzigerSeinerArt(index) ? Beteiligter.PLURAL : 0), -1, casus,
						true);
			// ... und den typ erhöhen
			typ = getTypFor(index);
			// Laufende Nummer, bezogen auf den Typ berechnen...
			// int laufendeNummer = index - minUndMax(typ)[0] + 1;
			if (!isEinzigerSeinerArt(index))
				zwischen += Beteiligter.fuegeLaufendeNummerHinzu(ermittleLaufendeNummerFuerBeteiligtenTyp(index));
			zwischen += ", ";
		}
		return TenorTexter.undStattKomma(TenorTexter.letztesKommaWeg(zwischen.replace("  ", " ")));
	}

	/**
	 * Die Methode ermittelt, ob der {@link Beteiligter} bei dem angegebenen
	 * <code>index</code> der einzige für seinen Beteiligten-Typ ist.
	 * Gesamtschuldner zählen nicht.
	 * 
	 * @param index die Position in der BaumbachBeteiligtenListe
	 * @return false, wenn es noch andere Beteiligte vom selben Typ gibt, die nicht
	 *         BaumbachGesamtschuldner sind; sonst true.
	 */
	boolean isEinzigerSeinerArt(int index) {
		return (findeGesamtschuldnerIndices(getTypFor(index), false)).size() < 2;
	}

	/**
	 * Die Methode ermittelt, ob der angegebene {@link Beteiligter} in der Liste und
	 * der einzige für seinen Beteiligten-Typ ist. Gesamtschuldner zählen nicht.
	 * 
	 * @param beteiligter der {@link Beteiligter}
	 * @return <code>true</code>, wenn es der einzige Nicht-Gesamtschuldner seines
	 *         {@link Beteiligter#getTyp()}s ist.
	 */
	boolean isEinzigerSeinerArt(Beteiligter beteiligter) {
		return contains(beteiligter) && (findeGesamtschuldnerIndices(beteiligter.getTyp(), false)).size() < 2;
	}

	/**
	 * Die Methode kurzBezeichnerListe dient dazu, eine Liste der repräsentierten
	 * BaumbachBeteiligten mittels deren kurzBezeichner zu erzeugen.
	 * 
	 * @param casus   eine der Konstanten NOMINATIV, GENITIV, DATIV, AKKUSATIV
	 * @param indexes ein Array int[] mit den indexes der BaumbachBeteiligtenListe
	 * @return einen String mit einem passenden Bezeichner etwa Kläger-1, Beklagte-2
	 * 
	 * @see eu.gronos.kostenrechner#Beteiligter#kurzBezeichner(int)
	 */
	String kurzBezeichnerListe(int casus, int[] indexes) {
		String liste = "";
		for (int index : indexes)
			liste += get(index).kurzBezeichner(casus) + ((zaehleBeteiligteFuerTyp(getTypFor(index)) > 1
					? "-" + ermittleLaufendeNummerFuerBeteiligtenTyp(index)
					: "")) + ", ";
		return TenorTexter.letztesKommaWeg(liste);
	}

	/**
	 * Die Methode getGanzesUnterliegen dient dazu, das Unterliegen für den
	 * BaumbachBeteiligten bei <code>index</code> zurückzugeben. Die Methode
	 * unterscheidet nach BaumbachGesamtschuldnern und anderen BaumbachBeteiligten.
	 * Bei Gesamtschuldnern wird das Unterliegen mit der Anzahl der repräsentierten
	 * BaumbachBeteiligten (getAnzahl) multipliziert. So passt es dann zum fiktiven
	 * Streitwert.
	 * 
	 * @param index die Nummer bezogen auf die vollständige BaumbachBeteiligtenListe
	 *              (wie bei get(int))
	 * @return das vollständige Unterliegen als double
	 */
	private double getGanzesUnterliegen(int index) {
		double unterliegen = get(index).getUnterliegen();
		// Bei Gesamtschuldnern muss das Unterliegen vervielfältigt werden
		if (get(index).isGesamtschuldnerschaft()) {
			unterliegen = unterliegen * ((double) getAlsGesamtschuldner(index).getAnzahl());
		}
		return unterliegen;
	}

	/**
	 * Die Methode getZusaetzlichesUnterliegen dient dazu, für einen
	 * BaumbachBeteiligten (der nicht Instanz von BaumbachGesamtschuldner ist) bei
	 * <code>index</code>, das zusätzliche Unterliegen zurückzugeben, das sich aus
	 * den BaumbachGesamtschuldnern ergibt, die auch den referenzierten
	 * BaumbachBeteiligten repräsentieren...
	 * 
	 * @param index die Nummer bezogen auf die vollständige BaumbachBeteiligtenListe
	 *              (wie bei get(int))
	 * @return das zusätzliche Unterliegen als double
	 */
	private double getZusaetzlichesUnterliegen(int index) {
		double unterliegen = 0.0;
		final BaumbachBeteiligter beteiligter = get(index);
		if (beteiligter.isGesamtschuldnerschaft())
			return unterliegen;
		// Gesamte BaumbachBeteiligtenListe durchgehen
		for (BaumbachBeteiligter bbt : this)
			// Wenn ein BaumbachGesamtschuldner auch den BaumbachBeteiligten bei
			// index enthält...
			if (bbt.isGesamtschuldnerschaft() && ((BaumbachGesamtschuldnerschaft) bbt).contains(beteiligter.getTyp(),
					ermittleLaufendeNummerFuerBeteiligtenTyp(index)))
				// ...dann muss dessen Unterliegen dazu gerechnet werden
				unterliegen += bbt.getUnterliegen();
		return unterliegen;
	}

	/**
	 * Die Methode ermittelt den Angreifer zu einem Beteiligten
	 * 
	 * @param index bezüglich der Liste
	 * @return den index des Angreifers oder -1, wenn keiner ausfindig zu machen
	 *         ist.
	 */
	int findeAngreiferVon(int index) {
		int ergebnis = -1;
		switch (getTypFor(index)) {
		case Beteiligter.KLAEGER:
			if (!isAnWiderklageBeteiligt(index))
				return -1;
			for (int i = 0; i < size(); i++)
				if (getTypFor(i) == Beteiligter.BEKLAGTE && isAnWiderklageBeteiligt(i))
					ergebnis = i;
			return ergebnis;
		case Beteiligter.DRITTWIDERBEKLAGTE:
			for (int i = 0; i < size(); i++)
				if (getTypFor(i) == Beteiligter.BEKLAGTE && isAnWiderklageBeteiligt(i))
					ergebnis = i;
			return ergebnis;
		case Beteiligter.BEKLAGTE:
			for (int i = 0; i < size(); i++)
				if (getTypFor(i) == Beteiligter.KLAEGER)
					ergebnis = i;
			return ergebnis;
		default:
			return ergebnis;
		}
	}
}
