/**
 * BaumbachBeteiligter.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.model.baumbach;

import javax.xml.bind.JAXB;
import javax.xml.bind.annotation.XmlAttribute;

import eu.gronos.kostenrechner.model.tenordaten.Beteiligter;

/**
 * Eine Klasse, abgeleitet von Beteiligter, die zusätzlich den Streitwert, das
 * Unterliegen und die Beteiligung an einer Widerklage speichert.
 * 
 * @author Peter Schuster (setrok)
 * @date: 27.04.2014
 * 
 * @todo TODO equals(), hash()
 * 
 */
public class BaumbachBeteiligter extends Beteiligter {
	public static final String UND_WK_WBK = "und %s";
	public static final String PARTEI_UND_WK_WBK = "%s und %s";
	public final static String[] WIDERBEKLAGTEN_BEUGUNG = new String[] { "Widerbeklagter", "Widerbeklagte" };
	public final static String[] WIDERKLAEGER_BEUGUNG = new String[] { "Widerkläger", "Widerklägerin" };

	private static final long serialVersionUID = -3484780882204159173L;
	/**
	 * Der double mit dem Namen streitwert speichert den Streitwert der gegen (!)
	 * den Beteiligten geltend gemachten Forderung(en), nicht der von ihm geltend
	 * gemachten.
	 */
	private double streitwert;
	/**
	 * Der double mit dem Namen unterliegen speichert das Unterliegen des jeweiligen
	 * Beteiligten hinsichtlich der gegen ihn (!) geltend gemachten Forderung, bei
	 * einem Beklagten also hinsichtlich der Hauptklage, bei einem
	 * Kläger/Drittwiderbeklagten hinsichtlich der Widerklage. Nicht gemeint ist das
	 * Unterliegen hinsichtlich der selbst von dem Beteiligten gegen andere geltend
	 * gemachten Forderungen.
	 */
	private double unterliegen;
	// final
	/**
	 * Der boolean mit dem Namen anWiderklageBeteiligt speichert, ob der
	 * BaumbachBeteiligte an der Widerklage beteiligt ist (der Beklagte also
	 * zugleich Widerkläger, ein Kläger zugleich Widerbeklagter ist).
	 */
	private boolean anWiderklageBeteiligt;

	/**
	 * Konstruktor:
	 * 
	 * @param typ                   eine der Konstanten Beteiligter.KLAEGER,
	 *                              Beteiligter.DRITTWIDERBEKLAGTE,
	 *                              Beteiligter.BEKLAGTE
	 * @param genusNumerus          eine der Konstanten MAENNLICH oder WEIBLICH
	 * @param streitwert            Streitwert bzgl. des Angriffs gegen die
	 *                              jeweilige Partei
	 * @param unterliegen           das Unterliegen des Beteiligten bzgl. des
	 *                              Angriffs gegen die jeweilige Partei (aus Sicht
	 *                              der Partei)
	 * @param anWiderklageBeteiligt ist die jeweilige Partei an der Widerklage
	 *                              beteiligt?
	 */
	public BaumbachBeteiligter(int typ, int genusNumerus, double streitwert, double unterliegen,
			boolean anWiderklageBeteiligt) {
		super(typ, genusNumerus);
		this.streitwert = streitwert;
		this.unterliegen = unterliegen;
		this.anWiderklageBeteiligt = anWiderklageBeteiligt;
	}

	/**
	 * Konstruktor, der statt typ und genusNumerus eine Instanz von Beteiligter
	 * nimmt und ausliest. Dieser Konstruktor richtet eine Instanz mit denselben
	 * Werten für typ und genusNumerus wie der übergegebene Beteiligte ein (soll die
	 * Arbeit bei abgeleiteten Klassen und mit CellEditor erleichtern).
	 * 
	 * @param beteiligter           ein Beteiligter, von dem typ und genusNumerus
	 *                              übernommen werden.
	 * @param streitwert            Streitwert bzgl. des Angriffs gegen die
	 *                              jeweilige Partei
	 * @param unterliegen           das Unterliegen des Beteiligten bzgl. des
	 *                              Angriffs gegen die jeweilige Partei (aus Sicht
	 *                              der Partei)
	 * @param anWiderklageBeteiligt ist die jeweilige Partei an der Widerklage
	 *                              beteiligt?
	 */
	public BaumbachBeteiligter(Beteiligter beteiligter, double streitwert, double unterliegen,
			boolean anWiderklageBeteiligt) {
		super(beteiligter);
		this.streitwert = streitwert;
		this.unterliegen = unterliegen;
		this.anWiderklageBeteiligt = anWiderklageBeteiligt;
	}

	/**
	 * Konstruktor ohne Parameter für {@link JAXB}
	 * 
	 */
	public BaumbachBeteiligter() {
		super();
	}

	/**
	 * @return streitwert gibt den streitwert als double zurück, also den Streitwert
	 *         bzgl. des Angriffs gegen die jeweilige Partei.
	 */
	@XmlAttribute(name = "streitwert")
	public double getStreitwert() {
		return streitwert;
	}

	/**
	 * @param streitwert the streitwert to set
	 */
	public void setStreitwert(double streitwert) {
		/*
		 * Bei nicht-widerbeklagten Klägern keine Änderung zulassen TODO Muss woanders
		 * geprüft werden, sonst Fehler! if (getTyp() == KLAEGER &&
		 * !isAnWiderklageBeteiligt()) return; if (streitwert < 0) return;
		 */
		this.streitwert = streitwert;
	}

	/**
	 * @return unterliegen gibt den unterliegen als double zurück, also das
	 *         Unterliegen des Beteiligten bzgl. des Angriffs gegen die jeweilige
	 *         Partei (aus Sicht der Partei)
	 */
	@XmlAttribute(name = "unterliegen")
	public double getUnterliegen() {
		return unterliegen;
	}

	/**
	 * @param unterliegen d. unterliegen, d. gesetzt werden soll
	 */
	public void setUnterliegen(double unterliegen) {
		this.unterliegen = unterliegen;
	}

	/**
	 * @return anWiderklageBeteiligt gibt den anWiderklageBeteiligt als boolean
	 *         zurück, also ob die jeweilige Partei an der Widerklage beteiligt ist.
	 */
	@XmlAttribute(name = "widerklage")
	public boolean isAnWiderklageBeteiligt() {
		return anWiderklageBeteiligt;
	}

	/**
	 * @param anWiderklageBeteiligt the anWiderklageBeteiligt to set
	 */
	public void setAnWiderklageBeteiligt(boolean anWiderklageBeteiligt) {
		if (getTyp() == DRITTWIDERBEKLAGTE)
			this.anWiderklageBeteiligt = true;
		else
			this.anWiderklageBeteiligt = anWiderklageBeteiligt;
	}

	/**
	 * @return gibt true zurück, wenn es sich um einen BaumbachGesamtschuldner
	 *         handelt (hier also immer false)
	 */
	public boolean isGesamtschuldnerschaft() {
		return false;
	}

	/**
	 * Die Methode erzeugt eine String-Repräsentation des Beteiligten durch Aufruf
	 * der Methode {@link parteiBezeichner(int)} im Nominativ mit zusätzlicher
	 * laufender Nummer.
	 * 
	 * @return einen Parteibezeichner
	 * 
	 * @see eu.gronos.kostenrechner.model.tenordaten.Beteiligter#toString()
	 */
	public String toString() {
		return parteiBezeichner(NOMINATIV, this.getLfdNr(), false);
	}
}
