/**
 * BaumbachGesamtschuldner.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.model.baumbach;

import javax.xml.bind.JAXB;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlElementWrapper;
import javax.xml.bind.annotation.XmlTransient;

import eu.gronos.kostenrechner.model.tenordaten.Beteiligter;

/**
 * Eine von BaumbachBeteiligter abgeleitete Klasse, um gesamtschuldnerische
 * Verurteilungen zu speichern.
 * 
 * @author Peter Schuster (setrok)
 * @date 01.06.2014
 * 
 */
@XmlAccessorType(value = XmlAccessType.FIELD)
public class BaumbachGesamtschuldnerschaft extends BaumbachBeteiligter {

	private static final long serialVersionUID = -323033831927041650L;
	private static final String GESAMTSCHULDNERISCH = " gesamtschuldnerisch";
	@XmlTransient
	private BaumbachBeteiligtenListe baumbachBeteiligtenListe;
	@XmlElementWrapper(name = "aufzaehlung")
	@XmlElement(name = "position")
	private/* final */int[] aufzaehlung;

	/**
	 * Erstellt eine Instanz einer gesamtschuldnerischen Verurteilung mit den
	 * folgenden Parametern. Das Feld
	 * {@link BaumbachBeteiligter#getStreitwert() streitwert} wird immer auf 0.0
	 * gesetzt, damit es bei der Berechnung des fiktiven Streitwerts nicht zu
	 * Fehlern kommt.
	 * 
	 * @param typ
	 *            eine der Konstanten Beteiligter.KLAEGER,
	 *            Beteiligter.DRITTWIDERBEKLAGTE, Beteiligter.BEKLAGTE
	 * @param genusNumerus
	 *            eine der Konstanten MAENNLICH oder WEIBLICH
	 * @param unterliegen
	 *            das Unterliegen des Beteiligten bzgl. des Angriffs gegen die
	 *            jeweilige Partei (aus Sicht der Partei)
	 * @param anWiderklageBeteiligt
	 *            ist die jeweilige Partei an der Widerklage beteiligt?
	 * @param aufzaehlung
	 *            welche Beteiligten sind hier Gesamtschuldner, als Array int[]
	 * @param baumbachBeteiligtenListe
	 *            die BaumbachBeteiligtenListe, auf die sich die Nummerierung in
	 *            <code>aufzaehlung</code> bezieht.
	 */
	public BaumbachGesamtschuldnerschaft(int typ, int genusNumerus,
			double unterliegen, boolean anWiderklageBeteiligt,
			int[] aufzaehlung, BaumbachBeteiligtenListe baumbachBeteiligtenListe) {
		super(typ, genusNumerus, 0, unterliegen, anWiderklageBeteiligt);
		this.baumbachBeteiligtenListe = baumbachBeteiligtenListe;
		this.aufzaehlung = aufzaehlung;
	}

	/**
	 * Konstruktor ohne Parameter für {@link JAXB}
	 * 
	 */
	public BaumbachGesamtschuldnerschaft() {
		super();
	}

	/**
	 * Die Methode gibt den Streitwert als double zurück.
	 * 
	 * @return immer 0.0 als double, damit es bei der Berechnung des fiktiven
	 *         Streitwerts nicht zu Fehlern kommt.
	 * 
	 * @see eu.gronos.kostenrechner.model.baumbach.BaumbachBeteiligter#getStreitwert()
	 */
	@Override
	@XmlTransient
	public double getStreitwert() {
		return 0.0;
	}

	/**
	 * Die Methode gibt Zugriff auf die BaumbachBeteiligtenListe, um die
	 * Referenzen in {@link BaumbachGesamtschuldnerschaft#getAufzaehlung()
	 * aufzaehlung} auflösen zu können.
	 * 
	 * @return gibt die baumbachBeteiligtenListe als BaumbachBeteiligtenListe
	 *         zurück.
	 */
	public BaumbachBeteiligtenListe getBaumbachBeteiligtenListe() {
		return baumbachBeteiligtenListe;
	}

	/**
	 * @param baumbachBeteiligtenListe
	 *            d. {@link #baumbachBeteiligtenListe}, um die Referenzen in
	 *            {@link BaumbachGesamtschuldnerschaft#getAufzaehlung()
	 *            aufzaehlung} auflösen zu können
	 */
	public void setBaumbachBeteiligtenListe(
			BaumbachBeteiligtenListe baumbachBeteiligtenListe) {
		if (baumbachBeteiligtenListe != null)
			this.baumbachBeteiligtenListe = baumbachBeteiligtenListe;
	}

	/**
	 * Der Getter gibt Zugriff auf die aufzaehlung der von der
	 * gesamtschuldnerischen Verurteilung betroffenen
	 * {@link BaumbachBeteiligter BaumbachBeteiligten}.
	 * 
	 * @return gibt die aufzaehlung (der BaumbachBeteiligten) als int[] zurück.
	 */
	public int[] getAufzaehlung() {
		return aufzaehlung;
	}

	/**
	 * *
	 * 
	 * @param aufzaehlung
	 *            die aufzaehlung (der BaumbachBeteiligten) als int[]
	 */
	public void setAufzaehlung(int[] aufzaehlung) {
		if (aufzaehlung != null)
			this.aufzaehlung = aufzaehlung;
	}

	/**
	 * Die Methode dient dazu, die Anzahl der Gesamtschuldner festzustellen
	 * 
	 * @return die Anzahl der repräsentierten Gesamtschuldner
	 */
	int getAnzahl() {
		return getAufzaehlung().length;
	}

	/**
	 * @return gibt true zurück, wenn es sich um einen BaumbachGesamtschuldner
	 *         handelt (hier also immer true).
	 */
	public boolean isGesamtschuldnerschaft() {
		return true;
	}

	/**
	 * Die Methode erzeugt einen passenden Bezeichner zu dem repräsentierten
	 * BaumbachGesamtschuldner.
	 * 
	 * @param casus
	 *            eine der Konstanten NOMINATIV, GENITIV, DATIV, AKKUSATIV
	 * @return einen String mit einem passenden Bezeichner etwa "der Kläger",
	 *         "die Kläger", "der Beklagte zu 1.)".
	 * 
	 * @see eu.gronos.kostenrechner.model.baumbach.BaumbachBeteiligter#parteiBezeichner(int,
	 *      int, boolean)
	 */
	private String parteiBezeichner(int casus) {
		// String zwischen = "";
		// int typ = -1;
		// for (int i : aufzaehlung) {
		// if (baumbachBeteiligtenListe.get(i).getTyp() != typ)
		// zwischen += super.parteiBezeichner(baumbachBeteiligtenListe
		// .get(i).getTyp(), baumbachBeteiligtenListe.get(i)
		// .getGenusNumerus() + PLURAL, -1, casus, true);
		// typ = baumbachBeteiligtenListe.get(i).getTyp();
		// int laufendeNummer = i - baumbachBeteiligtenListe.minUndMax(typ)[0]
		// + 1;
		// zwischen += fuegeLaufendeNummerHinzu(laufendeNummer) + ", ";
		// }
		// } return zwischen;
		return baumbachBeteiligtenListe.parteiBezeichnerListe(casus,
				aufzaehlung) + GESAMTSCHULDNERISCH;
	}

	/**
	 * Die Methode parteiBezeichner dient dazu, einen passenden Bezeichner zu
	 * dem repräsentierten BaumbachGesamtschuldner zu erzeugen.
	 * 
	 * @param casus
	 *            eine der Konstanten NOMINATIV, GENITIV, DATIV, AKKUSATIV
	 * @param laufendeNummer
	 *            dient der Kompatibilität zur Oberklasse
	 * @param einzigerSeinerArt
	 *            dient der Kompatibilität zur Oberklasse
	 * @return einen String mit einem passenden Bezeichner etwa
	 *         "die Beklagten zu 1.), 3.), 5.) gesamtschuldnerisch".
	 * 
	 * @see eu.gronos.kostenrechner.model.baumbach.BaumbachBeteiligter#parteiBezeichner(int,
	 *      int, boolean)
	 */
	public String parteiBezeichner(int casus, int laufendeNummer,
			boolean einzigerSeinerArt) {
		return parteiBezeichner(casus);
	}

	/**
	 * Die Methode kurzBezeichner dient dazu, einen parteiBezeichner ohne
	 * bestimmten Artikel zurückzugeben.
	 * 
	 * @param casus
	 *            eine der Konstanten NOMINATIV, GENITIV, DATIV, AKKUSATIV
	 * @return einen String mit einem passenden Bezeichner etwa Kläger-1,
	 *         Beklagte-2
	 * 
	 * @see eu.gronos.kostenrechner.model.tenordaten.Beteiligter#kurzBezeichner(int)
	 * @see eu.gronos.kostenrechner.model.baumbach.BaumbachBeteiligtenListe#kurzBezeichnerListe
	 *      (int,int[])
	 */
	public String kurzBezeichner(int casus) {
		return baumbachBeteiligtenListe.kurzBezeichnerListe(casus,
				getAufzaehlung()) + GESAMTSCHULDNERISCH;
	}

	/**
	 * Die Methode ermittelt, ob der entsprechende Beteiligte in der Aufzählung
	 * enthalten ist.
	 * 
	 * @param beteiligtenTyp
	 *            eine der Konstanten BaumbachBeteiligter.KLAEGER,
	 *            BaumbachBeteiligter.DRITTWIDERBEKLAGTE,
	 *            BaumbachBeteiligter.BEKLAGTE
	 * @param laufendeNummer
	 *            die laufende Nummer bezogen auf diesen beteiligtenTyp, bezogen
	 *            auf die BaumbachBeteiligtenListe baumbachBeteiligtenListe
	 * @return true, wenn dieser Beteiligte mit repräsentiert wird.
	 */
	public boolean contains(int beteiligtenTyp, int laufendeNummer) {
		// Standardwert ist false, der erhalten bleibt, wenn die Schleife keinen
		// Treffer liefert
		boolean zwischen = false;
		// Die Schleife muss die repräsentierten Beteiligten durchgehen
		for (int i : aufzaehlung) {
			// Die Schleife muss bei jedem vergleichen, ob beteiligtenTyp und
			// laufende Nummer übereinstimmen
			if (baumbachBeteiligtenListe
					.ermittleLaufendeNummerFuerBeteiligtenTyp(i) == laufendeNummer
					&& baumbachBeteiligtenListe.get(i).getTyp() == beteiligtenTyp)
				zwischen = true;
			// TODO: muss noch mit equals() prüfen, ob identisch
		}
		return zwischen;
	}

	/**
	 * Die Methode stellt fest, ob <code>laufendeNummer</code> den Typ
	 * <code>beteiligtenTyp</code> oder dazugehörig hat. D.h. sie behandelt
	 * KLAEGER und DRITTWIDERBEKLAGTE gleich.
	 * 
	 * @param beteiligtenTyp
	 *            eine der Konstanten {@link Beteiligter#KLAEGER},
	 *            {@link Beteiligter#BEKLAGTE} oder
	 *            {@link Beteiligter#DRITTWIDERBEKLAGTE}
	 * @param laufendeNummer
	 *            die {@link Beteiligter#lfdNr}
	 * @return true, wenn <code>laufendeNummer</code> den Typ
	 *         <code>beteiligtenTyp</code> oder dazugehörig hat
	 */
	boolean istGleicherTyp(int beteiligtenTyp, int laufendeNummer) {
		return baumbachBeteiligtenListe.get(laufendeNummer).getTyp() == beteiligtenTyp
				|| (baumbachBeteiligtenListe.get(laufendeNummer).getTyp() == Beteiligter.KLAEGER && beteiligtenTyp == Beteiligter.DRITTWIDERBEKLAGTE)
				|| (baumbachBeteiligtenListe.get(laufendeNummer).getTyp() == Beteiligter.DRITTWIDERBEKLAGTE && beteiligtenTyp == Beteiligter.KLAEGER);
	}

	/**
	 * Die Methode gibt zurück, ob die Instanz ein oder mehrere "echte" Personen
	 * repräsentiert.
	 * 
	 * @return true, wenn aufzaehlung mehrere Elemente enthält
	 * 
	 * @see eu.gronos.kostenrechner.model.tenordaten.Beteiligter#isPlural()
	 */
	@Override
	public boolean isPlural() {
		return aufzaehlung != null && aufzaehlung.length > 1;
	}

	// private boolean istEinheitlicherTyp() {
	// boolean zwischen = true;
	// int typ = -1;
	// for (int i : aufzaehlung)
	// if (typ > -1 && bbt.get(i).getTyp() != typ) {
	// zwischen = false;
	// typ = bbt.get(i).getTyp();
	// } else
	// typ = bbt.get(i).getTyp();
	// return zwischen;
	// }
}
