/**
 * BaumbachVollstreckbarkeitsObjekt.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.model.baumbach;

import eu.gronos.kostenrechner.logic.baumbach.BaumbachFormel;
import eu.gronos.kostenrechner.model.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.model.tenordaten.Vollstreckbarkeit;

/**
 * Von {@link Vollstreckbarkeit} abgeleitete Klasse für Verwendung in
 * {@link BaumbachFormel}, die auch die Felder glaeubiger und schuldner vom Typ
 * {@link Beteiligter} enthält.
 * 
 * @author Peter Felix Schuster (SchusterP2)
 * @date 12.11.2014
 */
public class BaumbachVollstreckbarkeit extends Vollstreckbarkeit {

	/**
	 * glaeubiger speichert den Vollstreckungsgläubiger als {@link Beteiligter}
	 */
	private final Beteiligter glaeubiger;
	/**
	 * schuldner speichert den Vollstreckungsschuldner als {@link Beteiligter}
	 */
	private final Beteiligter schuldner;

	/**
	 * Dieser Konstruktor klont das übergebene {@link Vollstreckbarkeit} und setzt
	 * die finalen Felder {@link #glaeubiger} und {@link #schuldner}.
	 * 
	 * @param vo         das zu klonende {@link Vollstreckbarkeit}
	 * @param glaeubiger {@link #glaeubiger}
	 * @param schuldner  {@link #schuldner}
	 */
	public BaumbachVollstreckbarkeit(Vollstreckbarkeit vo, Beteiligter glaeubiger, Beteiligter schuldner)
			throws IllegalArgumentException {
		this(vo.isB708(), vo.isB709s1(), vo.isB709s2(), vo.isB711(), vo.isB713(), glaeubiger, schuldner);
	}

	/**
	 * Initialisiert das {@link Vollstreckbarkeit} direkt mit den übergegebenen
	 * Parametern.
	 * 
	 * @param b708       {@link Vollstreckbarkeit#b708 b708}
	 * @param b709s1     {@link Vollstreckbarkeit#b709s1 b709s1}
	 * @param b709s2     {@link Vollstreckbarkeit#b709s2 b709s2}
	 * @param b711       {@link Vollstreckbarkeit#b711 b711}
	 * @param b713       {@link Vollstreckbarkeit#b713 b713}
	 * @param glaeubiger {@link #glaeubiger}
	 * @param schuldner  {@link #schuldner}
	 */
	BaumbachVollstreckbarkeit(boolean b708, boolean b709s1, boolean b709s2, boolean b711, boolean b713,
			Beteiligter glaeubiger, Beteiligter schuldner) throws IllegalArgumentException {
		super(b708, b709s1, b709s2, b711, b713);
		this.glaeubiger = glaeubiger;
		this.schuldner = schuldner;
	}

	/**
	 * Dieser Konstruktor setzt nur die finalen Felder {@link #glaeubiger} und
	 * {@link #schuldner}.
	 * 
	 * @param glaeubiger {@link #glaeubiger}
	 * @param schuldner  {@link #schuldner}
	 */
	BaumbachVollstreckbarkeit(Beteiligter glaeubiger, Beteiligter schuldner) {
		super();
		this.glaeubiger = glaeubiger;
		this.schuldner = schuldner;
	}

	/**
	 * @return the glaeubiger
	 */
	public Beteiligter getGlaeubiger() {
		return glaeubiger;
	}

	/**
	 * @return the schuldner
	 */
	public Beteiligter getSchuldner() {
		return schuldner;
	}

	/**
	 *
	 * @return die ausgewählten Vollstreckbarkeitsvorschriften und
	 *         Vollstreckungsbeteiligten als Klartext
	 * 
	 * @see eu.gronos.kostenrechner.model.tenordaten.Vollstreckbarkeit#toString()
	 */
	@Override
	public String toString() {
		return super.toString() + " ("
				+ glaeubiger.parteiBezeichner(Beteiligter.NOMINATIV, glaeubiger.getLfdNr(), false) + " ./. "
				+ schuldner.parteiBezeichner(Beteiligter.NOMINATIV, glaeubiger.getLfdNr(), false) + ")";
	}

	/**
	 * Die Methode dient dazu, das Vollstreckungsverhältnis zu benennen.
	 * 
	 * @param baumbachBeteiligtenListe anhand der Liste wird überprüft, ob Gläubiger
	 *                                 bzw. Schuldner der einzige seiner Art sind.
	 * 
	 * @return einen String nach dem Format "des &lt;Gläubigers&gt gegen den
	 *         &lt;Schuldner&gt;"
	 */
	public String benenneParteien(BaumbachBeteiligtenListe baumbachBeteiligtenListe) {
//		baumbachBeteiligtenListe.zaehleBeteiligteFuerTyp(schuldner.getTyp()) < 2;
//		return zaehleBeteiligteFuerTyp(beteiligter.getTyp()) < 2;
		final boolean glaeubigerEinziger = baumbachBeteiligtenListe.isEinzigerSeinerArt(glaeubiger);
		final boolean schuldnerEinziger = baumbachBeteiligtenListe.isEinzigerSeinerArt(schuldner);
		return glaeubiger.parteiBezeichner(Beteiligter.GENITIV, glaeubiger.getLfdNr(), glaeubigerEinziger) + " gegen "
				+ schuldner.parteiBezeichner(Beteiligter.AKKUSATIV, schuldner.getLfdNr(), schuldnerEinziger);
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = super.hashCode();
		result = prime * result + ((glaeubiger == null) ? 0 : glaeubiger.hashCode());
		result = prime * result + ((schuldner == null) ? 0 : schuldner.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (!super.equals(obj)) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		BaumbachVollstreckbarkeit other = (BaumbachVollstreckbarkeit) obj;
		if (schuldner == null) {
			if (other.schuldner != null) {
				return false;
			}
		} else if (!schuldner.equals(other.schuldner)) {
			return false;
		}
		if (glaeubiger == null) {
			if (other.glaeubiger != null) {
				return false;
			}
		} else if (!glaeubiger.equals(other.glaeubiger)) {
			return false;
		}
		return true;
	}

	/**
	 * Die Methode vergleicht das andere {@link Vollstreckbarkeit} mit diesem. Und
	 * zwar nach den Kriterien von
	 * {@link Vollstreckbarkeit#compareTo(Vollstreckbarkeit)}, bei Gleichheit nach
	 * der Reihenfolge der Schuldner, bei nochmaliger Gleichheit nach Reihenfolge
	 * der Gläubiger.
	 * 
	 * @param other das zu vergleichende Objekt.
	 * @return a negative integer, zero, or a positive integer as this object is
	 *         less than, equal to, or greater than the specified object.
	 * 
	 * @see eu.gronos.kostenrechner.model.tenordaten.Vollstreckbarkeit#compareTo(eu.gronos.kostenrechner.model.tenordaten.Vollstreckbarkeit)
	 */
	@Override
	public int compareTo(Vollstreckbarkeit other) {
		if (equals(other))
			return 0;
		if (super.compareTo(other) != 0)
			return super.compareTo(other);
		/*
		 * Hier erst nach Reihenfolge der Vollstreckungsschuldner, dann der Gläubiger
		 * vergleichen
		 */
		if (other instanceof BaumbachVollstreckbarkeit) {
			final BaumbachVollstreckbarkeit bvOther = (BaumbachVollstreckbarkeit) other;
			if (getSchuldner().compareTo(bvOther.getSchuldner()) != 0)
				return getSchuldner().compareTo(bvOther.getSchuldner());
			return getGlaeubiger().compareTo(bvOther.getGlaeubiger());
		}
		return 0;
	}

	/**
	 * Die Methode prüft, ob gegen Sicherheitsleistung oder mit Abwendungsbefugnis
	 * vollstreckbar ist. Wenn beides mal 0 übergeben wird, liefert sie ein leeres
	 * Objekt. Sozusagen eine Fabrikmethode mit Prüfung.
	 * 
	 * @param hauptsache die Vollstreckbarkeit hinsichtlich der Hauptsache im
	 *                   konkreten Vollstreckungsverhältnis.
	 * @param kosten     die Vollstreckbarkeit hinsichtlich der Kosten im konkreten
	 *                   Vollstreckungsverhältnis.
	 * @param glaeubiger {@link #glaeubiger}
	 * @param schuldner  {@link #schuldner}
	 * @return ein neues {@link BaumbachVollstreckbarkeit}
	 * @throws IllegalArgumentException sollte eigentlich nicht kommen.
	 * @see Vollstreckbarkeit#pruefeSicherheitsleistung(long, long)
	 * @see BaumbachVollstreckbarkeit#BaumbachVollstreckbarkeitsObjekt(Vollstreckbarkeit,
	 *      Beteiligter, Beteiligter)
	 */
	public static BaumbachVollstreckbarkeit pruefeSicherheitsleistung(long hauptsache, long kosten,
			Beteiligter glaeubiger, Beteiligter schuldner) throws IllegalArgumentException {
		return new BaumbachVollstreckbarkeit(Vollstreckbarkeit.pruefeSicherheitsleistung(hauptsache, kosten),
				glaeubiger, schuldner);
	}
}
