/**
 * AufrechnungsForderung.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.model.forderungen;

import javax.xml.bind.JAXB;
import javax.xml.bind.annotation.XmlTransient;

/**
 * Eine Klasse für Hilfsaufrechnungen
 * 
 * @author Peter Felix Schuster (SchusterP2)
 * @date: 31.07.2014
 */
public class AufrechnungsForderung extends Forderung {

	private static final long serialVersionUID = 4494763842223765977L;
	private static final String BESCHREIBUNG = "Hilfsaufrechnung";
	private double effektiverWert = -1;
	private double effektiverErfolg = -1;

	/**
	 * Erzeugt eine neue Instanz einer AufrechnungsForderung (d.h.
	 * Hilfsaufrechnung). Die Felder effektiverWert und effektiverErfolg werden
	 * zunächst mit -1 initialisiert.
	 * 
	 * @param gegenforderung ob es sich um eine Forderung des Klägers gegen den
	 *                       Beklagten (false) oder eine Forderung des Beklagten
	 *                       gegen den Kläger handelt
	 * @param wert           die Höhe der Forderung, also ihren Wert.
	 * @param erfolg         inwieweit die Forderung Erfolg hatte.
	 * @throws IllegalArgumentException wenn der Erfolg höher als der Wert ist
	 */
	public AufrechnungsForderung(boolean gegenforderung, double wert, double erfolg) throws IllegalArgumentException {
		super(gegenforderung, wert, erfolg);
	}

	/**
	 * Der Standardkonstruktor erzeugt eine neue Instanz einer AufrechnungsForderung
	 * (d.h. Hilfsaufrechnung). Die Felder effektiverWert und effektiverErfolg
	 * werden zunächst mit -1 initialisiert. Gegenforderung wird als Standard auf
	 * true gesetzt.
	 * 
	 * @param wert   die Höhe der Forderung, also ihren Wert.
	 * @param erfolg inwieweit die Forderung Erfolg hatte.
	 * @throws IllegalArgumentException wenn der Erfolg höher als der Wert ist
	 */
	public AufrechnungsForderung(double wert, double erfolg) throws IllegalArgumentException {
		this(true, wert, erfolg);
	}

	/**
	 * Konstruktor ohne Parameter für {@link JAXB}
	 * 
	 */
	public AufrechnungsForderung() {
		super();
		super.setGegenforderung(true);
	}

	/**
	 * Gibt eine Beschreibung der Forderungsart zurück
	 * 
	 * @return "Hilfsaufrechnung"
	 * 
	 * @see eu.gronos.kostenrechner.model.forderungen.Forderung#getBeschreibung()
	 */
	@Override
	public String getBeschreibung() {
		return BESCHREIBUNG;
	}

	/**
	 * @return den Wert der Aufrechnungsforderung, soweit er sich mit der
	 *         verbliebenen Klageforderung deckt ("Schnittmenge" = effektiver Wert).
	 */
	@XmlTransient
	public double getEffektiverWert() {
		return effektiverWert;
	}

	/**
	 * @param effektiverWert the effektiverWert to set
	 */
	public void setEffektiverWert(double effektiverWert) {
		this.effektiverWert = effektiverWert;
	}

	/**
	 * Diese Methode dient dazu, den effektiven Wert zurückzusetzen. Wenn seit der
	 * letzten Berechnung Werte verändert wurden, kommt es sonst zu Fehlern.
	 * 
	 */
	public void setzeEffektivenWertZurueck() {
		setEffektiverWert(-1);
	}

	/**
	 * @return den Erfolg der Aufrechnungsforderung, soweit er sich mit der
	 *         verbliebenen Klageforderung deckt ("Schnittmenge" = effektiver Erfolg).
	 */
	@XmlTransient
	public double getEffektiverErfolg() {
		return effektiverErfolg;
	}

	/**
	 * @param effektiverErfolg the effektiverErfolg to set
	 */
	public void setEffektiverErfolg(double effektiverErfolg) {
		this.effektiverErfolg = effektiverErfolg;
	}

	/**
	 * Die Methode dient dazu, den effektiven Erfolg zurück zu setzen. Wenn seit der
	 * letzten Berechnung Werte verändert wurden, kommt es sonst zu Fehlern.
	 * 
	 */
	public void setzeEffektivenErfolgZurueck() {
		setEffektiverErfolg(-1);
	}
}
