/**
 * Forderung.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.model.forderungen;

import java.io.Serializable;

import javax.xml.bind.JAXB;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlType;

/**
 * Grundklasse für Forderungen.
 * 
 * @author Peter Felix Schuster (SchusterP2)
 * @date: 30.07.2014
 *
 */
@XmlType(propOrder = { "wert", "erfolg" })
public abstract class Forderung implements Serializable {
	private static final long serialVersionUID = 5194917413163124727L;
	private boolean gegenforderung;
	private double wert;
	private double erfolg;

	/**
	 * Konstruktor:
	 * 
	 * @param gegenforderung ob es sich um eine Forderung des Klägers gegen den
	 *                       Beklagten (false) oder eine Forderung des Beklagten
	 *                       gegen den Kläger handelt
	 * @param wert           die Höhe der Forderung, also ihren Wert.
	 * @param erfolg         inwieweit die Forderung Erfolg hatte.
	 */
	Forderung(boolean gegenforderung, double wert, double erfolg) throws IllegalArgumentException {
		this.gegenforderung = gegenforderung;
		this.wert = wert;
		setErfolg(erfolg);
	}

	/**
	 * Konstruktor ohne Parameter für {@link JAXB}
	 */
	public Forderung() {
		super();
	}

	/**
	 * @return <code>boolean</code>, ob es sich um eine Forderung des Klägers gegen
	 *         den Beklagten (<code>false</code>) oder eine Forderung des Beklagten
	 *         gegen den Kläger handelt (<code>true</code>).
	 */
	@XmlAttribute(name = "gegenforderung")
	public boolean isGegenforderung() {
		return gegenforderung;
	}

	/**
	 * @param gegenforderung the gegenforderung to set
	 */
	public void setGegenforderung(boolean gegenforderung) {
		this.gegenforderung = gegenforderung;
	}

	/**
	 * @return die Höhe der Forderung, also ihren Wert.
	 */
	@XmlAttribute(name = "wert")
	public double getWert() {
		return wert;
	}

	/**
	 * @param wert the wert to set
	 */
	public void setWert(double wert) {
		this.wert = wert;
	}

	/**
	 * @return inwieweit die Forderung Erfolg hatte.
	 */
	@XmlAttribute(name = "erfolg")
	public double getErfolg() {
		return erfolg;
	}

	/**
	 * @param erfolg the erfolg to set
	 */
	public void setErfolg(double erfolg) {
		this.erfolg = erfolg;
	}
	// throws IllegalArgumentException
	// if (erfolg > wert)
	// throw new IllegalArgumentException(
	// "Eine Forderung kann nicht erfolgreicher sein als ihr Wert (ne ultra
	// petita)!");
	// else

	/**
	 * Eine zu überschreibende Funktion für String-Ausgaben
	 * 
	 * @return eine interne Beschreibung der Forderung für ableitende Klassen.
	 */
	public abstract String getBeschreibung();

	/**
	 * toString:
	 * 
	 * @return
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		String isGegenforderung = gegenforderung ? "Bk ./. Kl" : "Kl ./. Bk";
		return String.format("Forderung [beschreibung=%s (%s), wert=%s, erfolg=%s]", getBeschreibung(),
				isGegenforderung, wert, erfolg);
	}

}
