/**
 * GebuehrenTatbestand.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.model.gebuehren;

import javax.xml.bind.JAXB;
import javax.xml.bind.annotation.XmlAttribute;

/**
 * Eine Klasse, um GebührenTatbestände zu speichern, entweder aus dem VV RVG
 * oder dem KV GKG. Zusätzlich zu dem abstrakten GebuehrenTatbestand gibt es
 * noch einen Gebührensatz.
 * 
 * @author Peter Schuster (setrok)
 * @date 25.04.2014
 * 
 */
public class GebuehrenSatzTatbestand extends GebuehrenTatbestand {

	private static final long serialVersionUID = -3368857385467838790L;
	public static final String ART = "Gebührensatz";
	/**
	 * satz speichert den Gebührenfaktor als double, also den Faktor, mit dem eine
	 * 1.0-Gebühr multipliziert werden muss, z.B. 1.0 oder 1.2.
	 */
	private/* final */double satz;

	/**
	 * Erstellt eine Instanz der Klasse mit den angegebenen Werten.
	 * 
	 * @param vorschrift      Die gesetzliche Vorschrift, nach der sich der
	 *                        GebuehrenTatbestand richtet.
	 * @param bezeichnung     Die Bezeichnung des Gebührentatbestands nach dem
	 *                        jeweiligen Verzeichnis als String, z.B. "Nr. 1100 KV"
	 * @param satz            Der Gebührenfaktor als double, z.B. 1.0 oder 1.2
	 * @param gebuehrenKlasse Klasse der Gebühr, also den Verweis auf die Klasse
	 *                        eines Ablegers von GebuehrenTabelle, also entweder
	 *                        GerichtsGebuehrenTabelle.class oder
	 *                        AnwaltsGebuehrenTabelle.class
	 * 
	 */
	public GebuehrenSatzTatbestand(String vorschrift, String bezeichnung, double satz,
			Class<? extends GebuehrenTabelle> gebuehrenKlasse) {
		super(vorschrift, bezeichnung, gebuehrenKlasse);
		this.satz = satz;
	}

	/**
	 * Konstruktor ohne Parameter für {@link JAXB}
	 * 
	 */
	public GebuehrenSatzTatbestand() {
		super();
	}

	/**
	 * Der Getter für {@link #satz satz}
	 * 
	 * @return Der Gebührenfaktor als double, z.B. 1.0 oder 1.2
	 */
	@XmlAttribute(name = "satz")
	public double getSatz() {
		return satz;
	}

	/**
	 * 
	 * @param satz der Gebührenfaktor als double, z.B. 1.0 oder 1.2
	 */
	public void setSatz(double satz) {
//		if (satz >= 0.0 || this instanceof GebuehrenAnrechnungsTatbestand)
		this.satz = satz;
	}

	/**
	 * Returns a string representation of the object. In general, the toString
	 * method returns a string that "textually represents" this object. The result
	 * should be a concise but informative representation that is easy for a person
	 * to read. It is recommended that all subclasses override this method.
	 * 
	 * Die toString() Methode der Klasse GebuehrenTatbestand liefert einen String
	 * mit der Bezeichnung des Gebührentatbestand und dem Satz in Klammern.
	 * 
	 * @return einen String nach dem Format "Text (Satz)", z.B. "Verfahrensgebühr,
	 *         Ziffer xxxx VVRVG (1,2)"
	 * 
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		return String.format("GebuehrenSatzTatbestand [vorschrift=%s, bezeichnung=%s, gebuehrenKlasse=%s, satz=%s]",
				getVorschrift(), getBezeichnung(), getGebuehrenKlasse(), satz);
	}
	// @Override
//	public String tonString() {
//		return String.format("%s (%,.2f)", getBezeichnung(), satz);
//	}

	/**
	 * Die Methode baut einen {@link Object#hashCode()}
	 * 
	 * @return einen <code>int</code>, der sich aus
	 *         {@link GebuehrenTatbestand#hashCode()} sowie {@link #getSatz()}
	 *         berechnet. berechnet.
	 * 
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = super.hashCode();
		long temp;
		temp = Double.doubleToLongBits(satz);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		return result;
	}

	/**
	 * Die Methode vergleicht anhand von {@link #getSatz()} sowie
	 * {@link GebuehrenTatbestand#equals(Object)} die Gleichheit
	 * 
	 * @param obj das zu vergleichende {@link Object}
	 * @return <code>true</code>, wenn die Kriterien übereinstimmen, sonst
	 *         <code>false</code>
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (!super.equals(obj)) {
			return false;
		}
		if (!(obj instanceof GebuehrenSatzTatbestand)) {
			return false;
		}
		GebuehrenSatzTatbestand other = (GebuehrenSatzTatbestand) obj;
		if (Double.doubleToLongBits(satz) != Double.doubleToLongBits(other.satz)) {
			return false;
		}
		if (getBezeichnung() == null) {
			if (other.getBezeichnung() != null) {
				return false;
			}
		} else if (!getBezeichnung().equals(other.getBezeichnung())) {
			return false;
		}
		if (getGebuehrenKlasse() == null) {
			if (other.getGebuehrenKlasse() != null) {
				return false;
			}
		} else if (!getGebuehrenKlasse().equals(other.getGebuehrenKlasse())) {
			return false;
		}
		return true;// super.equals(obj);
	}

}
