/**
 * GebuehrenRechner.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.model.gebuehren;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Calendar;

import javax.xml.bind.JAXB;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlElementWrapper;
import javax.xml.bind.annotation.XmlElements;
import javax.xml.bind.annotation.XmlRootElement;

/**
 * Die Klasse dient als Oberklasse für eine Gebührenberechnung von
 * Gebührenordnungen, die sich nach Streit- bzw. Gegenstandstandswert richten.
 * 
 * Um nunmehr eine Speicherung als XML zu ermöglichen, stellt diese Klasse nur
 * noch die Methoden der Berechnung zur Verfügung. Die eigentlichen Daten
 * speichert eine {@link ArrayList}&lt;{@link GebuehrenZeile}&gt; mit
 * {@link GebuehrenGrundZeile} als erster und weiteren {@link GebuehrenZeile
 * GebührenZeilen}, die als Parameterobjekt genutzt wird.
 * 
 * Maßgeblich für Gebührenordnung, die sich nach Streit- bzw.
 * Gegenstandstandswert richten, ist die Wertgebührenvorschrift der jeweiligen
 * Gebührenordnung (z.B. § 13 RVG, § 34 GKG). Gebührenordnungen sind so
 * aufgebaut, dass die 1,0-Gebühr bis zum einem bestimmten untersten
 * Gegenstands-/Streitwert (<code>sprung</code>) einen bestimmten Sockelbetrag (
 * <code>hoehe</code>) beträgt &mdash; dafür gibt's eine
 * {@link GebuehrenGrundZeile}. Diese Gebühr erhöht sich mit jeder weiteren
 * {@link GebuehrenZeile} dann Gegenstandswert bis <code>grenze</code> Euro, für
 * jeden angefangenen Betrag von weiteren <code>sprung</code> Euro, um
 * <code>hoehe</code> Euro.
 * 
 * Darf nicht mehr <code>abstract</code> sein, damit sie mit {@link JAXB}
 * initialisiert werden kann.
 * 
 * @author Peter Schuster (setrok)
 * @date 01.01.2015
 *
 */
@XmlRootElement(name = "Wertgebuehren", namespace = "http://www.kostentenor.de/gebuehren")
public/* abstract */class GebuehrenTabelle {

	private ArrayList<GebuehrenZeile> tabelle = new ArrayList<GebuehrenZeile>();

	// public GebuehrenTatbestand[] gebuehren = new GebuehrenTatbestand[0];// = new
	// GebuehrenVerzeichnis();//ArrayList<GebuehrenTatbestand>();
	private GebuehrenVerzeichnis verzeichnis;

	/**
	 * Der Konstruktor der abstrakten Oberklasse nimmt die zur Berechnung nötige
	 * {@link ArrayList}&lt;{@link GebuehrenZeile}&gt; entgegen
	 * 
	 * @param tabelle eine {@link ArrayList}&lt;{@link GebuehrenZeile}&gt;
	 * 
	 *                private GebuehrenTabelle(ArrayList<GebuehrenZeile> tabelle) {
	 *                setTabelle(tabelle); }
	 */

	/**
	 * Konstruktor ohne Parameter für {@link JAXB}
	 * 
	 */
	public GebuehrenTabelle() {
		super();
	}

	/**
	 * Die {@link ArrayList}&lt;{@link GebuehrenZeile}&gt; enthält alle Werte der
	 * jeweiligen Gebührenordnung, nämlich als Sockel vom Typ
	 * {@link GebuehrenGrundZeile} den Gegenstands-/Streitwert (
	 * <code>grundSprung</code>) bis zu einen bestimmten Sockelbetrag (
	 * <code>grundHoehe</code>) sowie alle weiteren {@link GebuehrenZeile
	 * GebuehrenZeilen}.
	 * 
	 * @return gibt eine {@link ArrayList}&lt; {@link GebuehrenZeile}&gt; zurück.
	 */
	@XmlElements({ @XmlElement(name = "betraegtDieGebuehr", type = GebuehrenGrundZeile.class),
			@XmlElement(name = "gebuehrErhoehtSichBei", type = GebuehrenZeile.class) })
	public ArrayList<GebuehrenZeile> getTabelle() {
		return tabelle;
	}

	/**
	 * @param tabelle d. {@link #tabelle}, d. gesetzt werden soll als
	 *                {@link ArrayList}&lt;{@link GebuehrenZeile}&gt;
	 * @throws IllegalArgumentException wenn die Reihenfolge nicht eingehalten ist.
	 */
	public void setTabelle(ArrayList<GebuehrenZeile> tabelle) throws IllegalArgumentException {
		if (istReihenfolgeEingehalten(tabelle))
			this.tabelle = tabelle;
		else
			throw new IllegalArgumentException(
					"Die erste Zeile muss eine GebuehrenGrundZeile sein. Keine der weiteren Zeilen darf eine GebuehrenGrundZeile sein.");
	}

	/**
	 * @return gibt {@link #verzeichnis} als {@link GebuehrenVerzeichnis} zurück.
	 */
	@XmlElements({ @XmlElement(name = "anrechnung", type = GebuehrenAnrechnungsTatbestand.class),
			@XmlElement(name = "wertrahmen", type = GebuehrenRahmenTatbestand.class),
			@XmlElement(name = "gebuehrenSatz", type = GebuehrenSatzTatbestand.class),
			@XmlElement(name = "mehrpauschale", type = MehrfachPauschalTatbestand.class),
			@XmlElement(name = "auslagen", type = AuslagenTatbestand.class),
			@XmlElement(name = "mehrpauschale", type = MehrfachPauschalTatbestand.class),
			@XmlElement(name = "pauschale", type = PauschalTatbestand.class),
			@XmlElement(name = "steuer", type = MehrwertsteuerTatbestand.class),
			@XmlElement(name = "erhoehungsSatz", type = GebuehrenErhoehungsTatbestand.class) })
	@XmlElementWrapper(name = "verzeichnis")
	public GebuehrenVerzeichnis getVerzeichnis() {
		return verzeichnis;
	}

	/**
	 * @param verzeichnis d. {@link #verzeichnis}, d. gesetzt werden soll als
	 *                    {@link GebuehrenVerzeichnis}.
	 */
	public void setVerzeichnis(GebuehrenVerzeichnis verzeichnis) {
		this.verzeichnis = verzeichnis;
	}

	/**
	 * Gibt eine 1,0 Gebühr nach dem übergebenen <code>streitwert</code> zurück.
	 * 
	 * - Startwert nehmen
	 * 
	 * - Beginn äußere Schleife: für jede Zeile aus (Grenze, Sprung, Höhe)
	 * 
	 * - innere Schleife: addiere solange Zeile.Sprung auf die Zähler-Variable und
	 * Zeile.Höhe auf die Zwischen-Variable, bis der nächste Schritt Zeile.Grenze
	 * oder den Streitwert überschreiten würde. Ende innere Schleife
	 * 
	 * - Abbruch, wenn Zähler + Zeile.Höhe > Streitwert
	 * 
	 * @param streitwert Der Streitwert als Ganzzahl (long).
	 * @return Gibt eine 1,0 Gebühr nach dem übergebenen <code>streitwert</code>
	 *         zurück (als double).
	 */
	public double errechneGebuehr(long streitwert) {
		long zwischenGebuehr = 0L;
		long zwischenStreitWert = 0L;
		/* Beginn äußere Schleife: für jede Zeile aus (Grenze, Sprung, Höhe) */
		for (GebuehrenZeile zeile : getTabelle()) {
			/* Startwert nehmen (35 EUR) */
			if (zeile instanceof GebuehrenGrundZeile) {
				GebuehrenGrundZeile grund = (GebuehrenGrundZeile) zeile;
				// long zwischenGebuehr = getTabelle().getGrundHoehe();
				zwischenGebuehr = grund.getHoehe();
				// long zwischenStreitWert = getTabelle().getGrundSprung();
				zwischenStreitWert = grund.getSprung();
			}
			/*
			 * innere Schleife: addiere solange Zeile.Sprung auf die Zähler-Variable und
			 * Zeile.Höhe auf die Zwischen-Variable, bis der nächste Schritt Zeile.Grenze
			 * oder den Streitwert überschreiten würde. Ende innere Schleife
			 */
			else
				while (zwischenStreitWert <= zeile.getGrenze()
						&& ((zwischenStreitWert + zeile.getHoehe()) <= zeile.getGrenze())
						&& (zwischenStreitWert + zeile.getHoehe()) <= streitwert) {
					zwischenGebuehr += zeile.getHoehe();
					zwischenStreitWert += zeile.getSprung();
				}
			/* Abbruch, wenn Zähler + Zeile.Höhe > Streitwert */
			if ((zwischenStreitWert + zeile.getHoehe()) >= streitwert)
				break;
		}
		return (double) zwischenGebuehr;
	}

	/**
	 * Gibt eine 1,0 * <code>faktor</code> Gebühr nach dem übergebenen
	 * <code>streitwert</code> zurück.
	 * 
	 * @param streitwert der Streitwert als Ganzzahl (long)
	 * @param faktor     der Faktor nach Kostenverzeichnis bzw. Gebührenverzeichnis
	 *                   (z.B. 1,2 oder 3,0)
	 * @return gibt die entsprechende Gebühr als double zurück.
	 */
	public double errechneGebuehr(long streitwert, double faktor) {
		return errechneGebuehr(streitwert) * faktor;
	}

	/**
	 * Die Methode baut eine ArrayList&lt;Long&gt; mit allen Streitwerten, die einen
	 * Sprung darstellen.
	 * 
	 * @return die erstellte ArrayList&lt;Long&gt;
	 */
	public ArrayList<Long> errechneStreitwertListe() {
		ArrayList<Long> liste = new ArrayList<Long>();
		long streitwert = 0L;
		// war hier falsch: liste.add(streitwert);
		for (GebuehrenZeile zeile : getTabelle()) {
			/* Startwert nehmen (35 EUR) */
			if (zeile instanceof GebuehrenGrundZeile) {
				GebuehrenGrundZeile grund = (GebuehrenGrundZeile) zeile;
				streitwert = grund.getSprung();
				// Die erste Zeile muss schon rein:
				liste.add(streitwert);
			} else
				while (streitwert <= zeile.getGrenze() && (streitwert + zeile.getHoehe()) <= zeile.getGrenze()) {
					streitwert += zeile.getSprung();
					liste.add(streitwert);
				}
		}
		return liste;
	}

	/**
	 * Die Methode dient dazu, eine {@link GebuehrenTabelle} über
	 * {@link JAXB#unmarshal(InputStream, Class)} aus einem {@link InputStream} zu
	 * erlangen und aus dieser dann die {@link #getTabelle() Tabelle} zu entnehmen.
	 * Die Methode dient dazu, eine {@link GebuehrenTabelle} mit den GebuehrenZeilen
	 * der jeweiligen Wertgebührenordnung (z.B. § 13 RVG, § 34 GKG) zu
	 * initialisieren.
	 * 
	 * Diese sind so aufgebaut, dass die 1,0-Gebühr bis zum einem bestimmten
	 * untersten Gegenstands-/Streitwert einen bestimmten Sockelbetrag beträgt (das
	 * wird mit dem Konstruktor initialisiert,
	 * {@link eu.gronos.kostenrechner.GebuehrenRechnerAlt#GebuehrenTabelle(long,long, Calendar,Calendar)}
	 * ). Diese Gebühr erhöht sich dann Gegenstandswert bis <code>grenze</code>
	 * Euro, für jeden angefangenen Betrag von weiteren <code>sprung</code> Euro, um
	 * <code>hoehe</code> Euro
	 * 
	 * @param stream ein {@link InputStream} von einer XML-Datei für {@link JAXB}
	 * @return eine {@link GebuehrenTabelle}, aus der man mit {@link #getTabelle()}
	 *         eine gefüllte {@link ArrayList}&lt;{@link GebuehrenZeile}&gt; und mit
	 *         {@link #getVerzeichnis()} die {@link GebuehrenTatbestand}e
	 * 
	 * @throws NullPointerException wenn <code>stream</code> == <code>null</code>
	 *                              ist.
	 * 
	 * @see eu.gronos.kostenrechner.model.gebuehren.GebuehrenZeile#GebuehrenZeile(long,long,long)
	 */
	public /*protected*/ GebuehrenTabelle readFromStream(InputStream stream) throws NullPointerException {
		if (stream == null)
			throw new NullPointerException("Der InputStream für die angegebene Resource darf nicht null sein.");
		final GebuehrenTabelle tab = JAXB.unmarshal(stream, GebuehrenTabelle.class);
		return tab;
	}

	/**
	 * Die Methode dient dazu, eine {@link GebuehrenTabelle} über
	 * {@link JAXB#unmarshal(InputStream, Class)} aus einem {@link InputStream} zu
	 * erlangen und aus dieser dann die {@link #getTabelle() Tabelle} zu entnehmen.
	 * Die Methode dient dazu, eine {@link GebuehrenTabelle} mit den GebuehrenZeilen
	 * der jeweiligen Wertgebührenordnung (z.B. § 13 RVG, § 34 GKG) zu
	 * initialisieren.
	 * 
	 * Diese sind so aufgebaut, dass die 1,0-Gebühr bis zum einem bestimmten
	 * untersten Gegenstands-/Streitwert einen bestimmten Sockelbetrag beträgt (das
	 * wird mit dem Konstruktor initialisiert,
	 * {@link eu.gronos.kostenrechner.GebuehrenRechnerAlt#GebuehrenTabelle(long,long, Calendar,Calendar)}
	 * ). Diese Gebühr erhöht sich dann Gegenstandswert bis <code>grenze</code>
	 * Euro, für jeden angefangenen Betrag von weiteren <code>sprung</code> Euro, um
	 * <code>hoehe</code> Euro
	 *
	 * 
	 * @param stream ein {@link InputStream} von einer XML-Datei für {@link JAXB}
	 * @return eine gefüllte {@link ArrayList}&lt;{@link GebuehrenZeile}&gt;
	 *
	 * @throws NullPointerException wenn <code>stream</code> == <code>null</code>
	 *                              ist.
	 * 
	 * @see eu.gronos.kostenrechner.model.gebuehren.GebuehrenZeile#GebuehrenZeile(long,long,long)
	 *
	 */
	protected ArrayList<GebuehrenZeile> readTabelleFromStream(InputStream stream) throws NullPointerException {
		final GebuehrenTabelle tab = readFromStream(stream);
		return tab.getTabelle();
	}

	/**
	 * Die Methode dient dazu, zu prüfen, ob die erste Zeile eine
	 * {@link GebuehrenGrundZeile} ist, wie es sein muss. Keine der weiteren Zeilen
	 * darf eine {@link GebuehrenGrundZeile} sein.
	 * 
	 * @param tabelle die zu prüfende {@link ArrayList}&lt;{@link GebuehrenZeile}
	 *                &gt;
	 * @return <code>true</code>, wenn eingehalten, sonst <code>false</code>. Auch
	 *         dann <code>false</code>, wenn die {@link ArrayList}&lt;
	 *         {@link GebuehrenZeile}&gt; <code>null</code> oder zu klein ist.
	 */
	private boolean istReihenfolgeEingehalten(ArrayList<GebuehrenZeile> tabelle) {
		if (tabelle == null || tabelle.size() < 2)
			return false;
		/* Die erste Zeile muss eine GebuehrenGrundZeile sein */
		if (!(tabelle.get(0) instanceof GebuehrenGrundZeile))
			return false;
		/* Keine der weiteren Zeilen darf eine GebuehrenGrundZeile sein */
		for (int index = 1; index < tabelle.size(); index++)
			if (tabelle.get(index) instanceof GebuehrenGrundZeile)
				return false;
		return true;
	}

	/*
	 * TODO private void test(File file) throws JAXBException, PropertyException {
	 * File datei = new File(file.getAbsolutePath()+"_gebtab.xml"); JAXBContext
	 * context = JAXBContext.newInstance(GebuehrenTabelle.class); Marshaller
	 * marshaller = context.createMarshaller();
	 * marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, Boolean.TRUE);
	 * 
	 * marshaller.marshal(new GerichtsGebuehrenTabelle(), datei); }
	 */

	/*
	 * 
	 * Die Methode mergeAllGebuehrenTatbestaende dient dazu, zwei Arrays
	 * GebuehrenTatbestand[] (z.B. gkg und rvg) zu verbinden.
	 * 
	 * @param erste das erste Array GebuehrenTatbestand[]
	 * 
	 * @param zweite das zweite Array GebuehrenTatbestand[]
	 * 
	 * @return ein verbundenes Array private static GebuehrenTatbestand[]
	 * mergeAllGebuehrenTatbestaende(GebuehrenTatbestand[] erste,
	 * GebuehrenTatbestand[] zweite) { GebuehrenTatbestand[] zwischen =
	 * Arrays.copyOf(erste, erste.length + zweite.length); System.arraycopy(zweite,
	 * 0, zwischen, erste.length, zweite.length); return zwischen; }
	 */
}
