/*
 * GebuehrenVerzeichnis.java
 * eu.gronos.kostenrechner.model.gebuehren (Kostenrechner)
 */
package eu.gronos.kostenrechner.model.gebuehren;

import java.util.ArrayList;

import javax.xml.bind.annotation.XmlAttribute;

/**
 * Oberklasse zu den Gebührenverzeichnissen {@link KostenVerzeichnis} und
 * {@link VerguetungsVerzeichnis}
 *
 * @author Peter Schuster (setrok)
 * @date 23.06.2019
 *
 */
public class GebuehrenVerzeichnis extends ArrayList<GebuehrenTatbestand> {

	private static final long serialVersionUID = 6564487211842926610L;
	private String title = new String();

	public GebuehrenVerzeichnis() {
		super();
	}

	/**
	 * Die Methode ist ein modifizierter {@link #get(int)}-Getter
	 * 
	 * @param vorschriften Auflistung der
	 * @return alle Elemente, die {@link GebuehrenTatbestand#getVorschrift()}
	 *         enthalten
	 * @throws IllegalArgumentException, wenn eine der vorschriften nicht gefunden
	 *                                   wird
	 */
	public GebuehrenVerzeichnis getAll(String... vorschriften) throws IllegalArgumentException {
		GebuehrenVerzeichnis verzeichnis = new GebuehrenVerzeichnis();
		boolean gefunden = false;
		for (String vorschrift : vorschriften) {
			gefunden = false;
			for (GebuehrenTatbestand gt : this) {
				if (vorschrift != null && gt != null && vorschrift.equals(gt.getVorschrift())) {
					gefunden = true;
					verzeichnis.add(gt);
				}
			}
			if (!gefunden) {
				throw new IllegalArgumentException(
						String.format("Gebührentatbestand nicht im Verzeichnis: %s!", vorschrift));
			}
		}

		return verzeichnis;
	}

	/**
	 * Die Methode fügt nur die Elemente aus dem Verzeichnis hinzu, die den
	 * Suchbegriff {@link String} enthalten. Groß-/Kleinschreibung wird ignoriert.
	 * 
	 * @param term ein {@link String} mit dem Suchbegriff
	 * @return {@link GebuehrenVerzeichnis}
	 */
	public GebuehrenVerzeichnis filter(String term) {
		GebuehrenVerzeichnis verzeichnis = new GebuehrenVerzeichnis();
		for (GebuehrenTatbestand gt : this) {
			if (gt != null && gt.toString().toLowerCase().contains(term.toLowerCase())) {
				verzeichnis.add(gt);
			}
		}
		return verzeichnis;
	}

	/**
	 * Die Methode dient dazu, die passende Gebühr oder Auslage zu berechnen.
	 * 
	 * @param gt         einen GebuehrenSatzTatbestand oder AuslagenTatbestand,
	 *                   andere Ableger von GebuehrenTatbestand werden ignoriert.
	 * @param streitwert Streitwert (wird bei Auslagen ignoriert)
	 * @return die Höhe der Gebühr (dann errechnet nach 1,0 Gebühr für den
	 *         streitwert, multipliziert mit dem Gebührensatz nach gt) bzw. Auslage
	 */
	public static double errechneGebuehr(GebuehrenTatbestand gt, long streitwert) {
		if (gt != null && gt instanceof AuslagenTatbestand)
			return ((AuslagenTatbestand) gt).getBetrag();
		else if (gt != null && gt instanceof PauschalTatbestand)
			return ((PauschalTatbestand) gt).getBetrag();
		else if (gt != null && gt instanceof GebuehrenSatzTatbestand)
			return GebuehrenVerzeichnis.findeGebuehrenTabelle(gt).errechneGebuehr(streitwert,
					((GebuehrenSatzTatbestand) gt).getSatz());
		else
			return 0.0;
	}

	/**
	 * Die Methode soll die Summe aller Gebühren liefern. Mehrwertsteuer wird
	 * ignoriert.
	 * 
	 * @param streitwert ein long mit dem Streitwert
	 * @return ein double mit der Summe der Gebühren für diesen Streitwert
	 */
	public double errechneGebuehrenSumme(long streitwert) {
		double summe = 0;
		for (GebuehrenTatbestand gt : this) {
			/* summe um die Gebühr erhöhen */
			summe += GebuehrenVerzeichnis.errechneGebuehr(gt, streitwert);
		}
		return summe;
	}

	/**
	 * Die Methode soll die Summe aller Gebühren der Tabelle liefern und auch die
	 * Umsatzsteuer ausrechnen, wenn sie im {@link GebuehrenVerzeichnis} vorkommen.
	 * 
	 * @param streitwert ein long mit dem Streitwert
	 * @return ein double mit der Summe der Gebühren für diesen Streitwert
	 */
	public double errechneGebuehrenSummeMitSteuer(long streitwert) {
		double summe = 0;
		double rvgsumme = 0;
		for (GebuehrenTatbestand gt : this) {
			/*
			 * errechneGebuehr ignoriert MehrwertsteuerTatbestände; diese muss deshalb extra
			 * gerechnet werden.
			 */
			if (gt instanceof MehrwertsteuerTatbestand) {
				MehrwertsteuerTatbestand mwst = (MehrwertsteuerTatbestand) gt;
				summe += mwst.errechneSteuer(rvgsumme);
				/*
				 * Wenn die Gebühren isoliert werden, die RVG-Summe wieder auf 0 setzen.
				 */
				if (mwst.isIsoliert())
					rvgsumme = 0.0;
			} else {
				/* summe um die Gebühr erhöhen */
				summe += GebuehrenVerzeichnis.errechneGebuehr(gt, streitwert);
				/*
				 * rvgsumme nur erhöhen, wenn RVG-Gebühr (wird für MwSt gebraucht).
				 */
				if (gt.getGebuehrenKlasse() == AnwaltsGebuehrenTabelle.class)
					rvgsumme += GebuehrenVerzeichnis.errechneGebuehr(gt, streitwert);
			}
		}
		return summe;
	}

	/**
	 * @return gibt {@link #title} als {@link String} zurück.
	 */
	@XmlAttribute(name = "title")
	public String getTitle() {
		return title;
	}

	/**
	 * @param title d. {@link #title}, d. gesetzt werden soll als {@link String}.
	 */
	public void setTitle(String title) {
		this.title = title;
	}

	/**
	 * Die Methode findeGebuehrenTabelle findet heraus, ob ein Gebührentatbestand
	 * zum RVG oder GKG gehört und gibt eine Instanz dieser Tabelle zurück.
	 * 
	 * @param gt der GebuehrenTatbestand
	 * @return die zum GebuehrenTatbestand gehörende GebuehrenTabelle
	 */
	private static GebuehrenTabelle findeGebuehrenTabelle(GebuehrenTatbestand gt) {
		if (AnwaltsGebuehrenTabelle.class.equals(gt.getGebuehrenKlasse()))
			return GebuehrenGesetzesSammlung.getRvg();
		else if (GerichtsGebuehrenTabelle.class.equals(gt.getGebuehrenKlasse()))
			return GebuehrenGesetzesSammlung.getGkg();
		return null;
	}

	/*
	 * @param tatbestaende ein {@link GebuehrenVerzeichnis} mit den angefallenen
	 * Gebührentatbeständen. public static double
	 * errechneGebuehrenSumme(GebuehrenVerzeichnis tatbestaende, long streitwert) {
	 * for (GebuehrenTatbestand gt : tatbestaende) {
	 * 
	 * }
	 */
}