/*
 * MehrfachPauschalTatbestand.java
 * eu.gronos.kostenrechner.model.gebuehren (Kostenrechner)
 */
package eu.gronos.kostenrechner.model.gebuehren;

import javax.xml.bind.JAXB;
import javax.xml.bind.annotation.XmlAttribute;

import eu.gronos.kostenrechner.interfaces.GebuehrenMitAnzahlLinear;

/**
 * Von {@link GebuehrenTatbestand} und {@link PauschalTatbestand} abgeleitete
 * Klasse, für Auslagen, die von der Anzahl bestimmter Geschehen abhängt, etwa
 * eine Kilometerpauschale, die von der Anzahl gefahrener Kilometer abhängt.
 *
 * @author Peter Schuster (setrok)
 * @date 02.10.2019
 *
 */
public class MehrfachPauschalTatbestand extends PauschalTatbestand
		implements GebuehrenMitAnzahlLinear<Double>, Cloneable {

	private static final long serialVersionUID = -785436174722370248L;
	public static final String ART = "mehrfache Pauschale";
	private int anzahl;
	private double jeweils;
	private String einheit;
	/**
	 * berechnet: der {@link boolean} wird auf <code>true</code> gesetzt, wenn
	 * {@link #getBetrag()} einmal {@link #berechne()}t wurde, wieder auf
	 * <code>false</code> gesetzt, wenn sich {@link #getAnzahl()} oder
	 * {@link #getJeweils()} ändern.
	 */
	private boolean berechnet;

	/**
	 * Dieser Konstruktor (ohne die Höhe der Auslagen) kann benutzt werden, wenn die
	 * Höhe variabel, also nicht immer dieselbe ist (etwa bei Kosten der
	 * Beweisaufnahme). Eine Abfrage der Auslagen liefert so lange -1.0, bis ein
	 * anderer Wert mit {@link #setBetrag(double)} festgesetzt wird.
	 * 
	 * @param vorschrift      Die gesetzliche Vorschrift, nach der sich der
	 *                        GebuehrenTatbestand richtet.
	 * @param bezeichnung     Die Bezeichnung des Gebührentatbestands nach dem
	 *                        jeweiligen Verzeichnis als String, z.B. "Nr. 1100 KV"
	 * @param gebuehrenKlasse Klasse der Gebühr, also den Verweis auf die Klasse
	 *                        eines Ablegers von GebuehrenTabelle, also entweder
	 *                        GerichtsGebuehrenTabelle.class oder
	 *                        AnwaltsGebuehrenTabelle.class
	 * 
	 * @see eu.gronos.kostenrechner.model.gebuehren.calculation.gebuehren.eu.gronos.kostenrechner.controller.GebuehrenSatzTatbestand.GebuehrenTatbestand#Constructor(String,double,Class)
	 */
	public MehrfachPauschalTatbestand(String vorschrift, String bezeichnung,
			Class<? extends GebuehrenTabelle> gebuehrenKlasse) {
		super(vorschrift, bezeichnung, gebuehrenKlasse);
		berechnet = false;
	}

	/**
	 * Dieser Konstruktor (mit Höhe der Auslagen) kann benutzt werden, wenn die Höhe
	 * der Auslage bei Initialisieren bereits feststeht, z.B. weil es eigentlich
	 * eine Gebühr mit fester (streitwertunabhängiger) Höhe ist, z.B. die
	 * Aktenversandpauschale.
	 * 
	 * @param vorschrift      Die gesetzliche Vorschrift, nach der sich der
	 *                        GebuehrenTatbestand richtet.
	 * @param bezeichnung     Die Bezeichnung des Gebührentatbestands nach dem
	 *                        jeweiligen Verzeichnis als String, z.B. "Nr. 1100 KV"
	 * @param anzahl          Anzahl der abgegoltenen Einheiten, z.B.
	 *                        <code>30</code>, wenn 30km gefahren sind.
	 * @param einheitsPreis   die Höhe der jeweiligen Auslagen oder der
	 *                        streitwertunabhängigen Gebühr/Pauschale, oder ein
	 *                        negativer Wert (-1.0), wenn diese noch nicht feststeht
	 *                        und später gesetzt werden soll.
	 * @param einheit         Bezeichnung der Einheit, also z.B. &quot;km&quot;
	 * @param gebuehrenKlasse Klasse der Gebühr, also den Verweis auf die Klasse
	 *                        eines Ablegers von GebuehrenTabelle, also entweder
	 *                        GerichtsGebuehrenTabelle.class oder
	 *                        AnwaltsGebuehrenTabelle.class
	 */
	public MehrfachPauschalTatbestand(String vorschrift, String bezeichnung, int anzahl, double einheitsPreis,
			String einheit, Class<? extends GebuehrenTabelle> gebuehrenKlasse) {
		super(vorschrift, bezeichnung, gebuehrenKlasse);
		setAnzahl(anzahl);
		setJeweils(einheitsPreis);
		setEinheit(einheit);
		berechne();
	}

	/**
	 * Konstruktor ohne Parameter für {@link JAXB}
	 * 
	 */
	public MehrfachPauschalTatbestand() {
		super();
		berechnet = false;
	}

	/**
	 * Die Methode überschreibt {@link PauschalTatbestand#getBetrag()} und errechnet
	 * den Betrag, falls noch nicht geschehen (oder verändert).
	 * 
	 * @return den errechneten {@link PauschalTatbestand#getBetrag()}
	 * 
	 * @see eu.gronos.kostenrechner.model.gebuehren.PauschalTatbestand#getBetrag()
	 */
	@Override
	public double getBetrag() {
		if (!berechnet)
			berechne();
		return super.getBetrag();
	}

	/**
	 * @return gibt {@link #anzahl} als {@link int} zurück, also die Anzahl der
	 *         abgegoltenen Einheiten, z.B. <code>30</code>, wenn 30km gefahren
	 *         sind.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.GebuehrenMitAnzahlLinear#getAnzahl()
	 */
	@Override
	@XmlAttribute(name = "anzahl")
	public int getAnzahl() {
		return anzahl;
	}

	/**
	 * @param anzahl d. {@link #anzahl}, d. gesetzt werden soll als {@link int}.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.GebuehrenMitAnzahlLinear#setAnzahl(int)
	 */
	@Override
	public void setAnzahl(int anzahl) {
		this.anzahl = anzahl;
		berechnet = false;
	}

	/**
	 * @return gibt {@link #jeweils} als {@link Double} zurück, also die Höhe der
	 *         jeweiligen Auslagen oder der streitwertunabhängigen Gebühr/Pauschale,
	 *         oder ein negativer Wert (-1.0), wenn diese noch nicht feststeht und
	 *         später gesetzt werden soll.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.GebuehrenMitAnzahlLinear#getJeweils()
	 */
	@Override
	@XmlAttribute(name = "je")
	public Double getJeweils() {
		return jeweils;
	}

	/**
	 * @param jeweils d. {@link #jeweils}, d. gesetzt werden soll als
	 *                {@link Double}.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.GebuehrenMitAnzahlLinear#setJeweils(Object)
	 */
	@Override
	public void setJeweils(Double jeweils) {
		this.jeweils = jeweils;
		berechnet = false;
	}

	/**
	 * @return gibt {@link #einheit} als {@link String} zurück, also die Bezeichnung
	 *         der Einheit, also z.B. &quot;km&quot;
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.GebuehrenMitAnzahlLinear#getEinheit()
	 */
	@Override
	@XmlAttribute(name = "einheit")
	public String getEinheit() {
		return einheit;
	}

	/**
	 * @param einheit d. {@link #einheit}, d. gesetzt werden soll als
	 *                {@link String}, also die Bezeichnung der Einheit, also z.B.
	 *                &quot;km&quot;
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.GebuehrenMitAnzahlLinear#setEinheit(java.lang.String)
	 */
	@Override
	public void setEinheit(String einheit) {
		this.einheit = einheit;
	}

	@Override
	public MehrfachPauschalTatbestand clone() {
		return new MehrfachPauschalTatbestand(getVorschrift(), getBezeichnung(), getAnzahl(), getJeweils(),
				getEinheit(), getGebuehrenKlasse());
	}

	/**
	 * Die toString() Methode der Klasse MehrfachPauschalTatbestand liefert einen
	 * String mit der Bezeichnung des Gebührentatbestands und der Felder.
	 * 
	 * @return einen String nach dem Format MehrfachPauschalTatbestand
	 *         [getVorschrift()=%s, getBezeichnung()=%s, getGebuehrenKlasse()=%s,
	 *         getAnzahl()=%s, getEinheitsPreis()=%s, getEinheit()=%s]
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return String.format(
				"MehrfachPauschalTatbestand [getVorschrift()=%s, getBezeichnung()=%s, getGebuehrenKlasse()=%s, getAnzahl()=%s, getEinheitsPreis()=%s, getEinheit()=%s]",
				getVorschrift(), getBezeichnung(), getGebuehrenKlasse(), getAnzahl(), getJeweils(), getEinheit());
	}

	/**
	 * Die Methode berechnet den {@link #getBetrag()} aus {@link #getAnzahl()} und
	 * {@link #getJeweils()}. Sie setzt dann {@link #berechnet} auf
	 * <code>true</code>.
	 * 
	 * @return den {@link #getBetrag()}; solange {@link #getAnzahl()} < 0, also noch
	 *         nicht gesetzt, ist der Betrag 0.0.
	 */
	private double berechne() {
		if (getAnzahl() < 0)
			setBetrag(0.0);
		else
			setBetrag(getAnzahl() * getJeweils());
		berechnet = true;
		return getBetrag();
	}
}
