/**
 * AuslagenTatbestand.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.model.gebuehren;

import javax.xml.bind.JAXB;
import javax.xml.bind.annotation.XmlAttribute;

/**
 * Von GebuehrenTatbestand abgeleitete Klasse, für pauschale Auslagen und
 * streitwertunabhängige Gebühren.
 * 
 * @author Peter Schuster (setrok)
 * @date 06.05.2014
 * 
 */
public class PauschalTatbestand extends GebuehrenTatbestand {

	private static final long serialVersionUID = -7242122091272139243L;
	public static final String ART = "Pauschale";
	private double betrag;

	/**
	 * Dieser Konstruktor (ohne die Höhe der Auslagen) kann benutzt werden, wenn die
	 * Höhe variabel, also nicht immer dieselbe ist (etwa bei Kosten der
	 * Beweisaufnahme). Eine Abfrage der Auslagen liefert so lange -1.0, bis ein
	 * anderer Wert mit {@link #setBetrag(double)} festgesetzt wird.
	 * 
	 * @param vorschrift      Die gesetzliche Vorschrift, nach der sich der
	 *                        GebuehrenTatbestand richtet.
	 * @param bezeichnung     Die Bezeichnung des Gebührentatbestands nach dem
	 *                        jeweiligen Verzeichnis als String, z.B. "Nr. 1100 KV"
	 * @param gebuehrenKlasse Klasse der Gebühr, also den Verweis auf die Klasse
	 *                        eines Ablegers von GebuehrenTabelle, also entweder
	 *                        GerichtsGebuehrenTabelle.class oder
	 *                        AnwaltsGebuehrenTabelle.class
	 * 
	 * @see eu.gronos.kostenrechner.model.gebuehren.calculation.gebuehren.eu.gronos.kostenrechner.controller.GebuehrenSatzTatbestand.GebuehrenTatbestand#Constructor(String,double,Class)
	 */
	public PauschalTatbestand(String vorschrift, String bezeichnung,
			Class<? extends GebuehrenTabelle> gebuehrenKlasse) {
		this(vorschrift, bezeichnung, -1.0, gebuehrenKlasse);
	}

	/**
	 * Dieser Konstruktor (mit Höhe der Auslagen) kann benutzt werden, wenn die Höhe
	 * der Auslage bei Initialisieren bereits feststeht, z.B. weil es eigentlich
	 * eine Gebühr mit fester (streitwertunabhängiger) Höhe ist, z.B. die
	 * Aktenversandpauschale.
	 * 
	 * @param vorschrift      Die gesetzliche Vorschrift, nach der sich der
	 *                        GebuehrenTatbestand richtet.
	 * @param bezeichnung     Die Bezeichnung des Gebührentatbestands nach dem
	 *                        jeweiligen Verzeichnis als String, z.B. "Nr. 1100 KV"
	 * @param betrag          die Höhe der Auslagen oder der streitwertunabhängigen
	 *                        Gebühr/Pauschale, oder ein negativer Wert (-1.0), wenn
	 *                        diese noch nicht feststeht und später gesetzt werden
	 *                        soll.
	 * @param gebuehrenKlasse Klasse der Gebühr, also den Verweis auf die Klasse
	 *                        eines Ablegers von GebuehrenTabelle, also entweder
	 *                        GerichtsGebuehrenTabelle.class oder
	 *                        AnwaltsGebuehrenTabelle.class
	 */
	public PauschalTatbestand(String vorschrift, String bezeichnung, double betrag,
			Class<? extends GebuehrenTabelle> gebuehrenKlasse) {
		// super(bezeichnung, -1.0, gebuehrenKlasse);
		super(vorschrift, bezeichnung, gebuehrenKlasse);
		this.betrag = betrag;
	}

	/**
	 * Konstruktor ohne Parameter für {@link JAXB}
	 * 
	 */
	public PauschalTatbestand() {
		super();
	}

	/**
	 * @return gibt das Feld {@link #betrag} als double zurück.
	 */
	@XmlAttribute(name = "betrag")
	public double getBetrag() {
		return betrag;
	}

	/**
	 * @param betrag d. {@link #betrag}, d. gesetzt werden soll als double
	 */
	public void setBetrag(double betrag) {
		this.betrag = betrag;
	}

	/**
	 * Die toString() Methode der Klasse AuslagenTatbestand liefert einen String mit
	 * der Bezeichnung des Gebührentatbestands und der Höhe der Auslagen in
	 * Klammern.
	 * 
	 * @return einen String nach dem Format PauschalTatbestand [getVorschrift()=%s,
	 *         getBezeichnung()=%s, getGebuehrenKlasse()=%s, betrag=%s]
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return String.format(
				"PauschalTatbestand [getVorschrift()=%s, getBezeichnung()=%s, getGebuehrenKlasse()=%s, betrag=%s]",
				getVorschrift(), getBezeichnung(), getGebuehrenKlasse(), betrag);
	}

	/**
	 * Die Methode dient dazu, einen {@link Object#hashCode()} zu erstellen.
	 * 
	 * @return einen <code>int</code>, der sich aus {@link #getVorschrift()},
	 *         {@link #getBezeichnung()}, {@link #getGebuehrenKlasse()} und
	 *         {@link #getBetrag()} berechnet.
	 * 
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = super.hashCode();
		long temp;
		temp = Double.doubleToLongBits(betrag);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		return result;
	}

	/**
	 * Vergleicht nach den Kriterien {@link #getVorschrift()},
	 * {@link #getBezeichnung()}, {@link #getGebuehrenKlasse()} und
	 * {@link #getBetrag()}
	 * 
	 * @param obj das zu vergleichende {@link Object}
	 * @return <code>true</code>, wenn die Kriterien übereinstimmen, sonst
	 *         <code>false</code>
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (!super.equals(obj)) {
			return false;
		}
		if (!(obj instanceof PauschalTatbestand)) {
			return false;
		}
		PauschalTatbestand other = (PauschalTatbestand) obj;
		if (Double.doubleToLongBits(betrag) != Double.doubleToLongBits(other.betrag)) {
			return false;
		}
		return true;
	}

//	public String tonString() {
//		if (getBetrag() < 0)
//			return String.format("%s (Höhe noch nicht festgelegt)",
//					getBezeichnung());
//		else
//			return String.format("%s (%,.2f EUR)", getBezeichnung(),
//					getBetrag());
//	}
}
