/**
 * Beteiligter.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.model.tenordaten;

import java.io.Serializable;

import javax.xml.bind.annotation.XmlAttribute;

import eu.gronos.kostenrechner.model.baumbach.GerichtsKostenBeteiligter;

/**
 * Die Oberklasse für Beteiligte hält die Methoden für die parteiBezeichner
 * bereit
 * 
 * @author Peter Schuster (setrok)
 * @date 12.06.2014
 * 
 * @todo TODO könnte Xjustiz-Daten speichern...
 */
public class Beteiligter implements Comparable<Beteiligter>, Serializable {

	private static final long serialVersionUID = -6275833726367356470L;
	public static final int SINGULAR = 0;
	public static final int PLURAL = 2;
	public static final int MAENNLICH = 0;
	public static final int WEIBLICH = 1;
	public static final int NOMINATIV = 0;
	public static final int GENITIV = 1;
	public static final int DATIV = 2;
	public static final int AKKUSATIV = 3;
	public static final int KLAEGER = 0;
	public static final int DRITTWIDERBEKLAGTE = 1;
	public static final int BEKLAGTE = 2;
	private static final String[][] BEKLAGTEN_BEUGUNG = new String[][] {
			{ "der Beklagte", "die Beklagte", "die Beklagten", "die Beklagten" },
			{ "des Beklagten", "der Beklagten", "der Beklagten", "der Beklagten" },
			{ "dem Beklagten", "der Beklagten", "den Beklagten", "den Beklagten" },
			{ "den Beklagten", "die Beklagte", "die Beklagten", "die Beklagten" } };
	private static final String[][] DRITTWIDERBEKLAGTEN_BEUGUNG = new String[][] {
			{ "der Drittwiderbeklagte", "die Drittwiderbeklagte", "die Drittwiderbeklagten",
					"die Drittwiderbeklagten" },
			{ "des Drittwiderbeklagten", "der Drittwiderbeklagten", "der Drittwiderbeklagten",
					"der Drittwiderbeklagten" },
			{ "dem Drittwiderbeklagte", "der Drittwiderbeklagten", "den Drittwiderbeklagten",
					"den Drittwiderbeklagten" },
			{ "den Drittwiderbeklagten", "die Drittwiderbeklagte", "die Drittwiderbeklagten",
					"die Drittwiderbeklagten" } };
	private static final String[][] KLAEGER_BEUGUNG = new String[][] {
			{ "der Kläger", "die Klägerin", "die Kläger", "die Klägerinnen" },
			{ "des Klägers", "der Klägerin", "der Kläger", "der Klägerinnen" },
			{ "dem Kläger", "der Klägerin", "den Klägern", "den Klägerinnen" },
			{ "den Kläger", "die Klägerin", "die Kläger", "die Klägerinnen" } };
	public static final int GERICHTSKOSTEN = 8;
	private static final String[][] GERICHTSKOSTEN_BEUGUNG = new String[][] {
			{ "das Gericht", "das Gericht", "die Gerichte", "die Gerichte" },
			{ "des Gerichts", "des Gerichts", "der Gerichte", "der Gerichte" },
			{ "dem Gericht", "dem Gericht", "den Gerichten", "den Gerichten" },
			{ "das Gericht", "das Gericht", "die Gerichte", "die Gerichte" } };
	/**
	 * Der int mit dem Namen genusNumerus speichert das grammatikalische Geschlecht
	 * des Beteiligten sowie, ob es einer oder mehrere sind. Sein Wert entspricht
	 * einer der Konstanten MAENNLICH oder WEIBLICH, gegebenenfalls zuzüglich
	 * PLURAL.
	 */
	private int genusNumerus;
	/**
	 * Der int mit dem Namen typ speichert den Beteiligtentyp des Beteiligten,
	 * entspricht also einer der Konstanten Beteiligter.KLAEGER,
	 * Beteiligter.DRITTWIDERBEKLAGTE, Beteiligter.BEKLAGTE
	 */
	private/* final */int typ;
	/**
	 * Der int mit dem Namen lfdNr cachet die lfdNr, so dass sie für Hash-Vergleiche
	 * benutzt werden kann.
	 */
	private int lfdNr = -1;
	
	/**
	 * Der Konstruktor richtet eine Instanz von Beteiligter mit den übergegebenen
	 * Werten ein.
	 * 
	 * @param typ          eine der Konstanten Beteiligter.KLAEGER,
	 *                     Beteiligter.DRITTWIDERBEKLAGTE, Beteiligter.BEKLAGTE
	 * @param genusNumerus eine der Konstanten MAENNLICH oder WEIBLICH
	 */
	public Beteiligter(int typ, int genusNumerus) {
		super();
		setTyp(typ);
		setGenusNumerus(genusNumerus);
	}

	/**
	 * Dieser Konstruktor richtet eine Instanz von Beteiligter mit denselben Werten
	 * wie der übergegebene Beteiligte ein (soll die Arbeit bei abgeleiteten Klassen
	 * und mit CellEditor erleichtern).
	 * 
	 * @param beteiligter ein Beteiligter.
	 */
	public Beteiligter(Beteiligter beteiligter) {
		super();
		setTyp(beteiligter.getTyp());
		setGenusNumerus(beteiligter.getGenusNumerus());
	}

	/**
	 * Konstruktor für JAXB
	 * 
	 */
	public Beteiligter() {
		super();
	};

	/**
	 * Die statische Methode parteiBezeichner dient dazu, einen passenden Bezeichner
	 * zu den übergebenen Angaben zu einer bestimmten Partei zu erzeugen. Diese kann
	 * dann von den nicht-statischen Methoden aufgerufen werden, aber auch denen zur
	 * Array-Erzeugung.
	 * 
	 * @param beteiligtenTyp    eine der Konstanten KLAEGER, DRITTWIDERBEKLAGTER,
	 *                          BEKLAGTER
	 * @param genusNumerus
	 * @param laufendeNummer    die laufende Nummer der Partei bzgl. ihres Typs
	 * @param casus             einen passenden Bezeichner zu einer bestimmten
	 *                          Partei zu erzeugen.
	 * @param einzigerSeinerArt true, wenn es nur eine Partei dieses Beteiligtentyps
	 *                          gibt.
	 * @return einen String mit einem passenden Bezeichner etwa "der Kläger", "die
	 *         Kläger", "der Beklagte zu 1.)" oder null, wenn es den Beteiligtentyp
	 *         nicht gibt.
	 */
	public static String parteiBezeichner(int beteiligtenTyp, int genusNumerus, int laufendeNummer, int casus,
			boolean einzigerSeinerArt) {
		String[][] zwischen = getBezeichnerBeugungen(beteiligtenTyp);
		if (zwischen == null)
			return null;
		return zwischen[casus][genusNumerus] + (einzigerSeinerArt ? "" : fuegeLaufendeNummerHinzu(laufendeNummer));
	}

	/**
	 * @return typ gibt den typ als int zurück, also eine der Konstanten
	 *         Beteiligter.KLAEGER, Beteiligter.DRITTWIDERBEKLAGTE,
	 *         Beteiligter.BEKLAGTE
	 */
	@XmlAttribute(name = "typ")
	public int getTyp() {
		return typ;
	}

	/**
	 * Die Methode dient dazu, den Typ nachträglich zu verändern.
	 * 
	 * @param typ eine der Konstanten {@link Beteiligter#KLAEGER},
	 *            {@link Beteiligter#DRITTWIDERBEKLAGTE},
	 *            {@link Beteiligter#BEKLAGTE}
	 */
	public void setTyp(int typ) {
		if (typ >= KLAEGER && typ <= BEKLAGTE)
			this.typ = typ;
		if (typ == GERICHTSKOSTEN && this instanceof GerichtsKostenBeteiligter)
			this.typ = typ;
	}

	/**
	 * @return genusNumerus gibt den genusNumerus als int zurück,also eine der
	 *         Konstanten MAENNLICH oder WEIBLICH
	 */
	@XmlAttribute(name = "genusNumerus")
	public int getGenusNumerus() {
		return genusNumerus;
	}

	/**
	 * @param genusNumerus the genusNumerus to set
	 */
	public void setGenusNumerus(int genusNumerus) {
		this.genusNumerus = genusNumerus;
	}

	/**
	 * Die Methode gibt zurück, ob die Instanz ein oder mehrere "echte" Personen
	 * repräsentiert.
	 * 
	 * @return true, wenn genusNumerus eine Pluralform ist.
	 */
	public boolean isPlural() {
		return getGenusNumerus() >= PLURAL;
	}

	/**
	 * @return gibt den lfdNr als int zurück.
	 */
	@XmlAttribute(name = "laufendeNummer")
	public int getLfdNr() {
		return lfdNr;
	}

	/**
	 * Der Setter für lfdNr dient dazu, die lfdNr zu cashen, so dass sie für
	 * Hash-Vergleiche benutzt werden kann.
	 * 
	 * @param lfdNr d. lfdNr, d. gesetzt werden soll
	 */
	public void setLfdNr(int lfdNr) {
		this.lfdNr = lfdNr;
	}

	/**
	 * Die Methode parteiBezeichner dient dazu, einen passenden Bezeichner zu dem
	 * repräsentierten BaumbachBeteiligten zu erzeugen.
	 * 
	 * @param casus             eine der Konstanten NOMINATIV, GENITIV, DATIV,
	 *                          AKKUSATIV
	 * @param laufendeNummer    die laufende Nummer der Partei bzgl. ihres Typs
	 * @param einzigerSeinerArt true, wenn es nur eine Partei dieses Beteiligtentyps
	 *                          gibt.
	 * @return einen String mit einem passenden Bezeichner etwa "der Kläger", "die
	 *         Kläger", "der Beklagte zu 1.)".
	 */
	public String parteiBezeichner(int casus, int laufendeNummer, boolean einzigerSeinerArt) {
		setLfdNr(laufendeNummer);
		return parteiBezeichner(getTyp(), getGenusNumerus(), laufendeNummer, casus, einzigerSeinerArt);
	}

	/**
	 * Die Methode fuegeLaufendeNummerHinzu dient dazu, aus der übergebenen
	 * laufenden Nummer einen String zu machen, der einem "nackten" ParteiBezeichner
	 * hinzugefügt werden kann.
	 * 
	 * @param laufendeNummer die laufende Nummer eines Beteiligten als int
	 * @return einen formatierten String, z.B. " zu 1.)" (führendes Leerzeichen)
	 */
	public static String fuegeLaufendeNummerHinzu(int laufendeNummer) {
		return String.format(" zu %d.)", laufendeNummer);
	}

	/**
	 * Die Methode getBeteiligtenAuswahlListe dient dazu, ein Array von
	 * Beteiligter[] zu haben, um eine JComboBox<Beteiligter> füllen zu können.
	 * 
	 * @param beteiligtenTyp eine der Konstanten KLAEGER, DRITTWIDERBEKLAGTER,
	 *                       BEKLAGTER
	 * @param mitPlural      der Schalter regelt, ob auch die Pluralformen
	 *                       ausgegeben werden sollen; wenn "true", werden insgesamt
	 *                       4 Strings ins String[] gepackt, sonst nur die zwei
	 *                       Singularformen.
	 * @return ein Array Beteiligter[4] mit Beteiligten eines
	 *         <code>beteiligtenTyp</code>s in allen Formen männlich/weiblich,
	 *         ggfls. Singular/Plural.
	 */
	public static Beteiligter[] getBeteiligtenAuswahlListe(int beteiligtenTyp, boolean mitPlural) {
		Beteiligter[] zwischen = new Beteiligter[4];
		if (mitPlural)
			zwischen = new Beteiligter[] { new Beteiligter(beteiligtenTyp, MAENNLICH),
					new Beteiligter(beteiligtenTyp, WEIBLICH), new Beteiligter(beteiligtenTyp, MAENNLICH + PLURAL),
					new Beteiligter(beteiligtenTyp, WEIBLICH + PLURAL) };
		else
			zwischen = new Beteiligter[] { new Beteiligter(beteiligtenTyp, MAENNLICH),
					new Beteiligter(beteiligtenTyp, WEIBLICH) };
		/*
		 * Da compareTo() und equals() auf die laufendeNr schauen, muss ich die
		 * unterschiedlich setzen. Sonst können JComboBox-Instanzen das nicht mehr
		 * auswerten, die über sort() oder so zu gehen scheinen...
		 */
		for (Beteiligter b : zwischen)
			b.setLfdNr(b.getGenusNumerus());
		return zwischen[0] == null ? null : zwischen;
	}

	/**
	 * Die Methode kurzBezeichner dient dazu, einen parteiBezeichner ohne bestimmten
	 * Artikel zurückzugeben.
	 * 
	 * @param casus eine der Konstanten NOMINATIV, GENITIV, DATIV, AKKUSATIV
	 * @return einen String mit dem Bezeichner ohne Artikel und ohne laufende Nummer
	 */
	public String kurzBezeichner(int casus) {
		String zwischen = getBezeichnerBeugungen(getTyp())[casus][getGenusNumerus()];
		return zwischen.substring(4, zwischen.length());
	}

	/**
	 * Die Methode erzeugt eine String-Repräsentation des Beteiligten durch Aufruf
	 * der Methode {@link kurzBezeichner(int)} im Nominativ ohne zusätzliche
	 * laufende Nummer.
	 * 
	 * @return einen Partei-Bezeichner als String ohne laufende Nummer und Artikel
	 * 
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		// damit das Feld lfdNr nicht "kaputt" geht, wird der Getter aufgerufen.
		// return parteiBezeichner(NOMINATIV, getLfdNr(), true);
		return kurzBezeichner(NOMINATIV);
	}

	/**
	 * Die Methode vergleicht zwei Beteiligte auf Gleichheit. Dabei stellt sie
	 * (außer in Fällen von Identität und null) auf Beteiligtentyp und
	 * LaufendeNummer ab.
	 * 
	 * @param andererBeteiligter the reference object with which to compare.
	 * @return true if this object is the same as the obj argument; false otherwise.
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object andererBeteiligter) {
		if (this == andererBeteiligter) {
			return true;
		}
		if (andererBeteiligter == null) {
			return false;
		}
		if (getClass() != andererBeteiligter.getClass()) {
			return false;
		}
		Beteiligter other = (Beteiligter) andererBeteiligter;
		if (typ != other.typ) {
			return false;
		}
		if (lfdNr != other.lfdNr) {
			return false;
		}
		return true;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + typ;
		result = prime * result + lfdNr;
		return result;
	}

	/**
	 * Vergleicht zwei Beteiligter-Objekte, in erster Linie anhand des
	 * Beteiligtentyps, in zweiter Linie nach laufender Nummer. Dient nur für ein
	 * sort.
	 * 
	 * @param andererBeteiligter the object to be compared.
	 * @return a negative integer, zero, or a positive integer as this object is
	 *         less than, equal to, or greater than the specified object.
	 * 
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	@Override
	public int compareTo(Beteiligter andererBeteiligter) {
		// Wenn die Objekte gleich sind, 0 zurückgeben
		if (equals(andererBeteiligter))
			return 0;
		// Nur wenn die Typen identisch sind, auf die Laufende Nummer abstellen
		if (getTyp() == andererBeteiligter.getTyp()) {
			// an Integer.compareTo(o) delegieren
			return (new Integer(getLfdNr())).compareTo(new Integer(andererBeteiligter.getLfdNr()));
		} else {
			// Wenn die Typen ungleich sind, auf diese abstellen
			return (new Integer(getTyp())).compareTo(new Integer(andererBeteiligter.getTyp()));
		}
	}

	/**
	 * Die Methode ist ein Getter für die privaten konstanten zweidimensionalen
	 * String[][]-Arrays mit den Beugeformen für die Beteiligtentypen.
	 * 
	 * @param beteiligtenTyp eine der Konstanten Beteiligter.KLAEGER,
	 *                       Beteiligter.DRITTWIDERBEKLAGTE, Beteiligter.BEKLAGTE
	 * @return ein String[][]-Array mit den Beugeformen des Beteiligtentyps, der
	 *         dann mit String[casus][genusNumerus] benutzt werden kann.
	 */
	private static String[][] getBezeichnerBeugungen(int beteiligtenTyp) {
		String[][] zwischen = null;
		switch (beteiligtenTyp) {
		case GERICHTSKOSTEN:
			zwischen = GERICHTSKOSTEN_BEUGUNG;
			break;
		case BEKLAGTE:
			zwischen = BEKLAGTEN_BEUGUNG;
			break;
		case DRITTWIDERBEKLAGTE:
			zwischen = DRITTWIDERBEKLAGTEN_BEUGUNG;
			break;
		case KLAEGER:
			zwischen = KLAEGER_BEUGUNG;
			break;
		default:
			zwischen = null;
		}
		return zwischen;
	}

}