/**
 * ProzessVerhaeltnis.java
 * eu.gronos.kostenrechner.calculation (Kostenrechner)
 */
package eu.gronos.kostenrechner.model.tenordaten;

import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlElements;

import eu.gronos.kostenrechner.model.baumbach.BaumbachBeteiligtenListe;
import eu.gronos.kostenrechner.model.baumbach.BaumbachBeteiligter;
import eu.gronos.kostenrechner.model.baumbach.BaumbachGesamtschuldnerschaft;
import eu.gronos.kostenrechner.model.baumbach.GerichtsKostenBeteiligter;

/**
 * Klasse zum strukturierten Darstellen von Verurteilungen als Verhaeltnis
 * zwischen Gläubiger und Schuldner
 *
 * @author Peter Schuster (setrok)
 * @date 04.07.2018
 *
 */
public abstract class ProzessVerhaeltnis implements Comparable<ProzessVerhaeltnis> {
	final static int INT_PROZESS = 0;
	final static int INT_HAUPTSACHE = 1;
	final static int INT_KOSTEN = 2;
	final static int INT_VOLLSTRECKUNG = 3;
	private Beteiligter glaeubiger;
	private Beteiligter schuldner;
	private boolean darueberHinaus;

	/**
	 * Konstruktor ohne Parameter für JAXB
	 * 
	 */
	public ProzessVerhaeltnis() {
	}

	/**
	 * Konstruktor mit allen Feldern zum raschen initialisieren
	 * 
	 * @param glaeubiger     {@link #getGlaeubiger()}
	 * @param schuldner      {@link #getSchuldner()}
	 * @param darueberHinaus {@link #isDarueberHinaus()}
	 */
	public ProzessVerhaeltnis(Beteiligter glaeubiger, Beteiligter schuldner, boolean darueberHinaus) {
		this();
		this.glaeubiger = glaeubiger;
		this.schuldner = schuldner;
		this.darueberHinaus = darueberHinaus;
	}

	/**
	 * @return gibt {@link #glaeubiger} als {@link Beteiligter} zurück.
	 */
	@XmlElements({ @XmlElement(name = "glaeubiger", type = Beteiligter.class),
			@XmlElement(name = "staatskasseAlsGlaeubigerin", type = GerichtsKostenBeteiligter.class),
			@XmlElement(name = "glaeubigerMitZusatz", type = BaumbachBeteiligter.class) })
	public Beteiligter getGlaeubiger() {
		return glaeubiger;
	}

	/**
	 * @param glaeubiger d. {@link #glaeubiger}, d. gesetzt werden soll als
	 *                   Beteiligter
	 */
	public void setGlaeubiger(Beteiligter glaeubiger) {
		this.glaeubiger = glaeubiger;
	}

	/**
	 * @return gibt {@link #schuldner} als Beteiligter zurück.
	 */

	@XmlElements({ @XmlElement(name = "schuldner", type = Beteiligter.class),
			@XmlElement(name = "schuldnerMitZusatz", type = BaumbachBeteiligter.class),
			@XmlElement(name = "gesamtschuldner", type = BaumbachGesamtschuldnerschaft.class) })
	public Beteiligter getSchuldner() {
		return schuldner;
	}

	/**
	 * @param schuldner d. {@link #schuldner}, d. gesetzt werden soll als
	 *                  Beteiligter
	 */
	public void setSchuldner(Beteiligter schuldner) {
		this.schuldner = schuldner;
	}

	/**
	 * @return gibt {@link #darueberHinaus} als boolean zurück.
	 * 
	 * @see BaumbachBeteiligtenListe#hatDarueberHinaus(int)
	 */
	@XmlAttribute(name = "darueberHinaus")
	public boolean isDarueberHinaus() {
		return darueberHinaus;
	}

	/**
	 * @param darueberHinaus d. {@link #darueberHinaus}, d. gesetzt werden soll als
	 *                       boolean
	 */
	public void setDarueberHinaus(boolean darueberHinaus) {
		this.darueberHinaus = darueberHinaus;
	}

	/**
	 * Die Methode dient dazu, zwei Prozessverhältnisse zu vergleichen, um sie
	 * sortieren zu können. Stellt zunächst auf die Klasse (Hauptsache vor Kosten
	 * vor Vollstreckbarkeit), dann die Gläubiger, dann auf die Schuldner ab.
	 * 
	 * @param other das zu vergleichende {@link ProzessVerhaeltnis}
	 * @return a negative integer, zero, or a positive integer as this object is
	 *         less than, equal to, or greater than the specified object.
	 * 
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	@Override
	public int compareTo(ProzessVerhaeltnis other) {
		if (equals(other)) {
			return 0;
		}
		if (getClass() != other.getClass()) {
			return new Integer(intForClass(this)).compareTo(new Integer(intForClass(other)));
		}
		if (getGlaeubiger().equals(other.getGlaeubiger())) {
			return getSchuldner().compareTo(other.getSchuldner());
		} else {
			return getGlaeubiger().compareTo(other.getGlaeubiger());
		}
	}

	/**
	 * Die Methode dient zur Prüfung, ob zwei ProzessVerhaeltnisse gleich sind, also
	 * die gleiche Klasse, den gleichen Gläubiger und den gleichen Schuldner haben.
	 * 
	 * @param obj das andere Prozessverhältnis
	 * @return
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (obj == null)
			return false;
		if (obj == this)
			return true;
		if (getClass() != obj.getClass())
			return false;
		ProzessVerhaeltnis anderesPV = (ProzessVerhaeltnis) obj;
		return getGlaeubiger().equals(anderesPV.getGlaeubiger()) && getSchuldner().equals(anderesPV.getSchuldner());
	}

	/**
	 * Die Methode gibt einen Rumpfstring zurück, der nur die Bezeichner der beiden
	 * Parteien enthält
	 * 
	 * @return einen {@link String}
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		String darueber = "";
		if (isDarueberHinaus()) {
			darueber = " darüber hinaus";
		}
		return String.format("%s gegen %s%s",
				getGlaeubiger().parteiBezeichner(Beteiligter.GENITIV, getGlaeubiger().getLfdNr(), false),
				getSchuldner().parteiBezeichner(Beteiligter.AKKUSATIV, getSchuldner().getLfdNr(), false), darueber);
	}

	/**
	 * Die Methode dient dazu, Hauptsache vor Kosten vor Vollstreckbarkeit sortieren
	 * zu können.
	 * 
	 * @param verhaeltnis ein {@link ProzessVerhaeltnis}
	 * @return einen int in absteigender Reihenfolge für
	 *         {@link VollstreckungsVerhaeltnis}, {@link KostenTragungsVerhaeltnis}
	 *         und {@link HauptsacheVerhaeltnis} und sonstige.
	 */
	protected int intForClass(ProzessVerhaeltnis verhaeltnis) {
		if (verhaeltnis instanceof VollstreckungsVerhaeltnis) {
			return INT_VOLLSTRECKUNG;
		}
		if (verhaeltnis instanceof KostenTragungsVerhaeltnis) {
			return INT_KOSTEN;
		}
		if (verhaeltnis instanceof HauptsacheVerhaeltnis) {
			return INT_HAUPTSACHE;
		}
		return INT_PROZESS;
	}

}
