/**
 * VollstreckbarkeitsObjekt.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.model.tenordaten;

import javax.xml.bind.annotation.XmlAttribute;

import eu.gronos.kostenrechner.model.baumbach.BaumbachVollstreckbarkeit;

/**
 * Eine Klasse, die die Vorschriften zur vorläufigen Vollstreckbarkeit
 * speichert.
 * 
 * @author Peter Felix Schuster (SchusterP2)
 * @date 10.11.2014
 */
public class Vollstreckbarkeit implements Comparable<Vollstreckbarkeit> {
	private boolean b708 = false;
	private boolean b709s1 = false;
	private boolean b709s2 = false;
	private boolean b711 = false;
	private boolean b713 = false;

	/**
	 * Initialisiert das {@link Vollstreckbarkeit} mit allen Werten auf
	 * <code>false</code>.
	 * 
	 */
	public Vollstreckbarkeit() {
		super();
	}

	/**
	 * Initialisiert das {@link Vollstreckbarkeit} direkt mit den
	 * übergegebenen Parametern.
	 * 
	 * @param b708   {@link #b708}
	 * @param b709s1 {@link #b709s1}
	 * @param b709s2 {@link #b709s2}
	 * @param b711   {@link #b711}
	 * @param b713   {@link #b713}
	 */
	public Vollstreckbarkeit(boolean b708, boolean b709s1, boolean b709s2, boolean b711, boolean b713)
			throws IllegalArgumentException {
		this();
		this.b708 = b708;
		this.b709s1 = b709s1;
		this.b709s2 = b709s2;
		this.b711 = b711;
		this.b713 = b713;
		if (!checkePlausisbilitaeten()) {
			throw new IllegalArgumentException("Ungültige Kombination von " + toString());
		}
	}

	/**
	 * Der Konstruktor nimmt ein anderes {@link Vollstreckbarkeit} (oder
	 * {@link BaumbachVollstreckbarkeit} und klont alle Felder.
	 * 
	 * @param other ein anderes {@link Vollstreckbarkeit}
	 */
	public Vollstreckbarkeit(Vollstreckbarkeit other) {
		this(other.isB708(), other.isB709s1(), other.isB709s2(), other.isB711(), other.isB713());
	}

	/**
	 * Die Methode prüft, ob gegen Sicherheitsleistung oder mit Abwendungsbefugnis
	 * vollstreckbar ist. Wenn beides mal 0 übergeben wird, liefert sie ein leeres
	 * Objekt. Sozusagen eine Fabrikmethode mit Prüfung.
	 * 
	 * Prüft nicht die Unanfechtbarkeit.
	 * 
	 * @param hauptsache die Vollstreckbarkeit hinsichtlich der Hauptsache im
	 *                   konkreten Vollstreckungsverhältnis.
	 * @param kosten     die Vollstreckbarkeit hinsichtlich der Kosten im konkreten
	 *                   Vollstreckungsverhältnis.
	 * @return ein neues {@link Vollstreckbarkeit}
	 * @throws IllegalArgumentException sollte eigentlich nicht kommen.
	 */
	public static Vollstreckbarkeit pruefeSicherheitsleistung(long hauptsache, long kosten)
			throws IllegalArgumentException {
		if (hauptsache == 0 && kosten == 0) {
			/* Dann alles false, leeres Objekt */
			return new Vollstreckbarkeit();
		} else if (hauptsache > 1250 || (hauptsache == 0 && kosten > 1500)) {
			/* § 709 ZPO */
			return new Vollstreckbarkeit(false, true, true, false, false);
		} else {
			/* § 708, 709 S. 2, 711 ZPO */
			return new Vollstreckbarkeit(true, false, true, true, false);
		}
	}

	/**
	 * @return ob das Urteil nach § 708 ZPO ohne Sicherheitsleistung vorläufig
	 *         vollstreckbar ist, als boolean
	 */
	@XmlAttribute(name = "paragraf708")
	public boolean isB708() {
		return b708;
	}

	/**
	 * @return ob das Urteil nach § 709 S. 1 ZPO nur gegen Sicherheitsleistung
	 *         vorläufig vollstreckbar ist, als boolean
	 */
	@XmlAttribute(name = "paragraf709satz1")
	public boolean isB709s1() {
		return b709s1;
	}

	/**
	 * @return ob die Sicherheitsleistung prozentual gemäß § 709 S. 2 ZPO angegeben
	 *         wird, als boolean
	 */
	@XmlAttribute(name = "paragraf709satz2")
	public boolean isB709s2() {
		return b709s2;
	}

	/**
	 * @return ob der Vollstreckungsschuldner eine Abwendungsbefugnis gemäß § 711
	 *         ZPO hat, als boolean
	 */
	@XmlAttribute(name = "paragraf711")
	public boolean isB711() {
		return b711;
	}

	/**
	 * @return ob wegen Unanfechtbarkeit gemäß § 713 ZPO eine Sicherheitsleistung
	 *         ganz entfällt, als boolean
	 */
	@XmlAttribute(name = "paragraf713")
	public boolean isB713() {
		return b713;
	}

	/**
	 * 
	 * @return a hash code value for the object.
	 * 
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + (b713 ? 1231 : 1237);
		result = prime * result + (b709s1 ? 1231 : 1237);
		result = prime * result + (b711 ? 1231 : 1237);
		result = prime * result + (b708 ? 1231 : 1237);
		result = prime * result + (b709s2 ? 1231 : 1237);
		return result;
	}

	/**
	 * Indicates whether some other object is "equal to" this one.
	 * 
	 * @param obj the reference object with which to compare.
	 * @return <code>true</code> if this object is the same as the obj argument;
	 *         <code>false</code> otherwise.
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		Vollstreckbarkeit other = (Vollstreckbarkeit) obj;
		if (b713 != other.b713)
			return false;
		if (b709s1 != other.b709s1)
			return false;
		if (b711 != other.b711)
			return false;
		if (b708 != other.b708)
			return false;
		if (b709s2 != other.b709s2)
			return false;
		return true;
	}

	/**
	 * @return die ausgewählten Vollstreckbarkeitsvorschriften als Klartext
	 * 
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		StringBuilder builder = new StringBuilder("Vollstreckbarkeitsvorschriften: ");
		if (isB708())
			builder.append("§ 708 ZPO");
		if (isB709s1())
			builder.append("§ 709 Satz 1 ZPO");
		if (isB709s2())
			builder.append("§ 709 Satz 2 ZPO");
		if (isB711())
			builder.append("§ 711 ZPO");
		if (isB713())
			builder.append("§ 713 ZPO");
		// string.append(".");
		return builder.toString().replace("ZPO§", "ZPO, §");
	}

	/**
	 * Die Methode prüft, ob überhaupt ein Flag gesetzt wurde oder ob das Objekt
	 * &quot;leer&quot; ist.
	 * 
	 * @return <code>true</code>, wenn wenigstens ein Flag <code>true</code> ist
	 */
	public boolean hatEintraege() {
		return isB708() || isB709s1() || isB709s2() || isB711() || isB713();
	}

	/**
	 * Die Methode vergleicht das andere {@link Vollstreckbarkeit} mit
	 * diesem. Und zwar nach den Kriterien: § 713 < § 709 < § 708
	 * 
	 * @param other das zu vergleichende Objekt.
	 * @return a negative integer, zero, or a positive integer as this object is
	 *         less than, equal to, or greater than the specified object.
	 * 
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	@Override
	public int compareTo(Vollstreckbarkeit other) {
		if (equals(other))
			return 0;
		else if (isB713() != other.isB713()) {
			return isB713() ? -1 : 1;
		} else if (isB709s1() != other.isB709s1()) {
			return isB709s1() ? -1 : 1;
		} else if (isB711() != other.isB711()) {
			return isB711() ? -1 : 1;
		} else if (isB708() != other.isB708()) {
			return isB708() ? -1 : 1;
		} else if (isB709s2() != other.isB709s2()) {
			return isB709s2() ? -1 : 1;
		} else
			return 0;
	}

	/**
	 * Die Methode dient dazu, fehlerhafte Kombinationen der einzelnen
	 * Vollstreckbarkeitsvorschriften zu erkennen.
	 * 
	 * @return <code>true</code>, wenn die gewählte Kombination gültig ist.
	 */
	boolean checkePlausisbilitaeten() {
		if (isB708()) {
			if (isB711() && isB709s2() && !isB709s1() && !isB713())
				/* §§ 708, 709 S. 2, 711 ZPO */
				return true;
			else if (isB713() && !isB709s1() && !isB709s2() && !isB711())
				/* §§ 708, 713 ZPO */
				return true;
		} else if (isB709s1()) {
			/* § 709 S. 1, S. 2 ZPO */
			if (isB709s2() && !isB708() && !isB713())
				return true;
		} else if (!isB708() && !isB709s1() && !isB709s2() && !isB709s2() && !isB711() && !isB713()) {
			/*
			 * Es darf auch alles false sein, sonst könnte das erste Element der Liste nicht
			 * generiert werden
			 */
			return true;
		}
		return false;
	}

	/**
	 * @param b708 the {@link #isB708} to set
	 */
	void setB708(boolean b708) {
		this.b708 = b708;
	}

	/**
	 * @param b709s1 the {@link #isB709s1} to set
	 */
	void setB709s1(boolean b709s1) {
		this.b709s1 = b709s1;
	}

	/**
	 * @param b709s2 the {@link #isB709s2} to set
	 */
	void setB709s2(boolean b709s2) {
		this.b709s2 = b709s2;
	}

	/**
	 * @param b711 the {@link #isB711} to set
	 */
	void setB711(boolean b711) {
		this.b711 = b711;
	}

	/**
	 * @param b713 the {@link #isB713} to set
	 */
	void setB713(boolean b713) {
		this.b713 = b713;
	}
}
