/**
 * VollstreckungsVerhaeltnis.java
 * eu.gronos.kostenrechner.calculation (Kostenrechner)
 */
package eu.gronos.kostenrechner.model.tenordaten;

import javax.xml.bind.JAXB;
import javax.xml.bind.annotation.XmlElement;

import eu.gronos.kostenrechner.model.baumbach.BaumbachVollstreckbarkeit;

/**
 * Klasse zum strukturierten Darstellen der vorläufigen Vollstreckbarkeit einer
 * Verurteilung
 *
 * @author Peter Schuster (setrok)
 * @date 28.07.2018
 *
 */
public class VollstreckungsVerhaeltnis extends ProzessVerhaeltnis {
	private Vollstreckbarkeit vollstreckbarkeit;

	/**
	 * Standardkonstruktor ohne Parameter für {@link JAXB}
	 * 
	 */
	public VollstreckungsVerhaeltnis() {
		super();
	}

	/**
	 * Der Konstruktor, mit dem man gleich alle Felder setzen kann.
	 * 
	 * @param glaeubiger        {@link #getGlaeubiger()}
	 * @param schuldner         {@link #getSchuldner()}
	 * @param vollstreckbarkeit {@link #getVollstreckbarkeit()}
	 */
	public VollstreckungsVerhaeltnis(Beteiligter glaeubiger, Beteiligter schuldner,
			Vollstreckbarkeit vollstreckbarkeit) {
		super(glaeubiger, schuldner, false);
		this.setVollstreckbarkeit(vollstreckbarkeit);
	}

	/**
	 * Ein Konstruktor, der sich zunutze macht, dass
	 * {@link BaumbachVollstreckbarkeit} fast alle Wete enthält, die für das
	 * Vollstreckungsverhältnis gebraucht werden
	 * 
	 * @param baumbach ein {@link BaumbachVollstreckbarkeit}
	 */
	public VollstreckungsVerhaeltnis(BaumbachVollstreckbarkeit baumbach) {
		this(baumbach.getGlaeubiger(), baumbach.getSchuldner(), new Vollstreckbarkeit(baumbach));
	}

	/**
	 * @return gibt {@link #vollstreckbarkeit} als VollstreckbarkeitsObjekt zurück.
	 */
	@XmlElement(name = "vollstreckbarkeit")
	public Vollstreckbarkeit getVollstreckbarkeit() {
		return vollstreckbarkeit;
	}

	/**
	 * @param vollstreckbarkeit d. {@link #vollstreckbarkeit}, d. gesetzt werden
	 *                          soll als VollstreckbarkeitsObjekt
	 */
	public void setVollstreckbarkeit(Vollstreckbarkeit vollstreckbarkeit) {
		this.vollstreckbarkeit = vollstreckbarkeit;
	}

	/**
	 * Die Methode dient dazu, zwei Prozessverhältnisse zu vergleichen, um sie
	 * sortieren zu können. Stellt zunächst auf die Klasse (Hauptsache vor Kosten
	 * vor Vollstreckbarkeit), ab. Wenn beides {@link VollstreckungsVerhaeltnis}se
	 * sind, dann erst auf die {@link Vollstreckbarkeit}, erst dann die Gläubiger,
	 * dann auf die Schuldner ab.
	 * 
	 * @param other das zu vergleichende {@link ProzessVerhaeltnis}
	 * @return a negative integer, zero, or a positive integer as this object is
	 *         less than, equal to, or greater than the specified object.
	 * 
	 * @see eu.gronos.kostenrechner.model.tenordaten.ProzessVerhaeltnis#compareTo(eu.gronos.kostenrechner.model.tenordaten.ProzessVerhaeltnis)
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	@Override
	public int compareTo(ProzessVerhaeltnis other) {
		if (equals(other)) {
			return 0;
		}
		if (getClass() != other.getClass()) {
			return new Integer(intForClass(this)).compareTo(new Integer(intForClass(other)));
		}
		VollstreckungsVerhaeltnis otherVV = (VollstreckungsVerhaeltnis) other;
		if (getVollstreckbarkeit().equals(otherVV.getVollstreckbarkeit())) {
			return super.compareTo(other);
		} else {
			return getVollstreckbarkeit().compareTo(otherVV.getVollstreckbarkeit());
		}
	}

	@Override
	public String toString() {
		if (getVollstreckbarkeit() == null)
			return super.toString();
		else
			return String.format("Vollstreckungsverhältnis %s gemäß %s", super.toString(),
					getVollstreckbarkeit().toString());
	}

}
