/*
 * KostenJDialog.java
 * eu.gronos.kostenrechner.view (Kostenrechner)
 */
package eu.gronos.kostenrechner.view;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagLayout;
import java.awt.HeadlessException;
import java.awt.Window;
import java.awt.event.FocusAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFormattedTextField;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JRootPane;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.border.EmptyBorder;
import javax.swing.event.HyperlinkListener;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.BeiFocusMarkierenHorcher;
import eu.gronos.kostenrechner.controller.BeschriebeneAktion;
import eu.gronos.kostenrechner.controller.ComponentBeschrifter;
import eu.gronos.kostenrechner.controller.DialogFensterSchliessenHorcher;
import eu.gronos.kostenrechner.controller.HinzufuegenAbbrechenAction;
import eu.gronos.kostenrechner.controller.SprungmarkenHorcher;
import eu.gronos.kostenrechner.interfaces.Vorsorgend;
import eu.gronos.kostenrechner.model.beschriftungen.Beschriftung;
import eu.gronos.kostenrechner.model.beschriftungen.LangBeschriftung;
import eu.gronos.kostenrechner.model.beschriftungen.LangVorsorgeBeschriftung;
import eu.gronos.kostenrechner.model.beschriftungen.NameContainerSammlung;
import eu.gronos.kostenrechner.view.result.FontHelferForWindow;

/**
 * Abstrakte Oberklasse zum {@link HinzufuegenDialog} und für Fenster, die keine
 * Werte zurückgeben sollen.
 *
 * @author Peter Schuster (setrok)
 * @date 24.01.2020
 *
 */
public abstract class KostenJDialog extends JDialog {

	private static final long serialVersionUID = 2500262285732280536L;
	protected static final EmptyBorder BORDER = new EmptyBorder(5, 5, 5, 5);
	/**
	 * Der JPanel mit dem Namen contentPanel ist das JPanel für den eigentlichen
	 * Inhalt
	 */
	protected final JPanel contentPanel = new JPanel();
	private JPanel buttonPane;
	private final BeschriebeneAktion cancelAction = new HinzufuegenAbbrechenAction(this,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(76061));
	private final WindowAdapter fensterSchliesser = new DialogFensterSchliessenHorcher(this);
	public static final FocusAdapter BEI_FOCUS_MARKIEREN = new BeiFocusMarkierenHorcher();
	public static final HyperlinkListener HYPERLINK_LISTENER = new SprungmarkenHorcher();
	private final FontHelferForWindow fufhelfer = new FontHelferForWindow(this);
	private final Beschriftung beschriftung;
	protected final ComponentBeschrifter beschrifter = new ComponentBeschrifter();

	/**
	 * Erstellt einen modalen {@link JDialog} mit einer buttonPane (nur Abbrechen
	 * bzw. Schließen) und einem {@link #contentPanel}.
	 * 
	 * Dieser muss von der abgeleiteten Klasse in der Methode
	 * {@link #fuelleContentPane()} mit {@link JComponent}s gefüllt werden, die von
	 * {@link #showDialogAndAsk()} aufgerufen wird.
	 * 
	 * @param owner        das {@link Window} ({@link JFrame} oder {@link JDialog})
	 *                     von dem aus der {@link KostenJDialog} angezeigt wird
	 * @param beschriftung eine {@link Beschriftung}, mit dem der anzuzeigende Titel
	 *                     {@link JDialog#getTitle()} gesetzt wird
	 * @throws HeadlessException wenn in einer Umgebung aufgerufen, die keinen
	 *                           Bildschirm, Tastatur oder Maus unterstützt
	 */
	public KostenJDialog(Window owner, Beschriftung beschriftung) throws HeadlessException {
		super(owner);
		super.setModal(true);
		//super(owner, true);
		Kostenrechner.getLogger().info("Starte Dialog " + this.getClass().toString());
		this.beschriftung = beschriftung;
		beschrifter.beschrifte(this, beschriftung);
		baueContentPane();
		setButtonPane();
	}
	// super(owner, title, true);beschriftung.getTitle(),
	// super(owner, true);
	// .applyTo
	// super.setAlwaysOnTop(false);
	// beschriftung.applyTo(getContentPane());

	/**
	 * Die Methode dient dazu, den Dialog aufzurufen, den {@link #contentPanel} über
	 * die Methode {@link #fuelleContentPane()} der abgeleiteten Klasse mit
	 * Komponenten zu füllen und mit {@link JDialog#setVisible(boolean)} den Dialog
	 * anzuzeigen.
	 */
	public void showDialog() {
		fuelleContentPane();
		// setBounds();
		pack();
		setVisible(true);
	}

	/**
	 * Die Methode lässt die Position vom Betriebssystem zuweisen und setzt die
	 * Größe auf die bevorzugte Größe. Wird von {@link #showDialogAndAsk()}
	 * aufgerufen.
	 * 
	 * @see java.awt.Window#setBounds(int,int,int,int)
	 * @see java.awt.Window#setLocationByPlatform(boolean)
	 * @see java.awt.Window#setSize(Dimension)
	 * @see java.awt.Window#pack()
	 */
	@Override
	public void pack() {
		setLocationByPlatform(true);
		setSize(fufhelfer.calculatePreferredSizeWith(BORDER.getBorderInsets()));//
		setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);
		addWindowListener(fensterSchliesser);
		super.pack();
	}

	/**
	 * Die Methode schließt den Dialog; kann in einer {@link Action} benutzt werden
	 * 
	 */
	public void fensterSchliessenBeiAbbruch() {
		setVisible(false);
		dispose();
	}

	/**
	 * @return gibt {@link #beschriftung} als {@link Beschriftung} zurück.
	 */
	public Beschriftung getBeschriftung() {
		return beschriftung;
	}

	/**
	 * @return gibt den buttonPane als JPanel zurück.
	 */
	public JPanel getButtonPane() {
		return buttonPane;
	}

	/**
	 * @return gibt {@link #cancelAction} als {@link AbstractAction} zurück.
	 */
	public BeschriebeneAktion getCancelAction() {
		return cancelAction;
	}

	/**
	 * Die Methode füllt den Inhaltsbereich des Dialogs mit JComponents zur Eingabe.
	 * Wird von {@link #showDialogAndAsk()} aufgerufen.
	 * 
	 */
	protected abstract void fuelleContentPane();

	/**
	 * Die Methode initialisiert die allgemeinen JComponents für den Inhaltsbereich
	 * des Dialogs und wird vom Konstruktor selbsttätig aufgerufen.
	 * 
	 */
	protected void baueContentPane() {
		GridBagLayout gbl_contentPanel = new GridBagLayout();
		contentPanel.setBorder(BORDER);
		contentPanel.setLayout(gbl_contentPanel);
		getContentPane().add(contentPanel, BorderLayout.CENTER);
	}

	/**
	 * Die Methode fügt eine Action der ActionMap und der InputMap der JRootPane des
	 * HinzufuegenDialogs hinzu.
	 * 
	 * @param action eine Action, bei der die Values
	 *               {@link Action#ACTION_COMMAND_KEY} und
	 *               {@link Action#ACCELERATOR_KEY} gesetzt sein müssen.
	 */
	protected void addToRootInputMap(Action action) {
		JRootPane rp = getRootPane();
		/* Zunächst die Action der ActionMap hinzufügen unter ihrem Namen */
		rp.getActionMap().put(action.getValue(Action.ACTION_COMMAND_KEY), action);
		/* Dann die Action der InputMap unter ihrer Tastenkombination hinzufügen */
		rp.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put((KeyStroke) action.getValue(Action.ACCELERATOR_KEY),
				action.getValue(Action.ACTION_COMMAND_KEY));
	}

	/**
	 * Die Methode entfernt eine Action aus der ActionMap und der InputMap der
	 * JRootPane des HinzufuegenDialogs.
	 * 
	 * @param action eine Action, bei der die Values
	 *               {@link Action#ACTION_COMMAND_KEY} und
	 *               {@link Action#ACCELERATOR_KEY} gesetzt sein müssen.
	 */
	protected void removeFromInputMap(Action action) {
		JRootPane rp = getRootPane();
		/* Zuerst die Tastenkombination entfernen */
		rp.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).remove((KeyStroke) action.getValue(Action.ACCELERATOR_KEY));
		/* Dann die Action entfernen */
		rp.getActionMap().remove(action.getValue(Action.ACTION_COMMAND_KEY));
	}

	/**
	 * Setzt für ein {@link JFormattedTextField} die {@link AbstractAction} in die
	 * ActionMap und InputMap, die bei Enter ausgeführt werden soll. Zudem fügt die
	 * Methode das Verhalten hinzu, dass bei FocusErlangung der Inhalt markiert
	 * wird.
	 * 
	 * @param field  das {@link JTextField}, dessen Entertaste belegt werden soll
	 * @param action die {@link AbstractAction}, die bei Entertaste ausgeführt
	 *               werden soll
	 */
	protected void setEnterAction(JTextField field, AbstractAction action) {
		if (action != null) {
			field.getInputMap(JComponent.WHEN_FOCUSED).put(KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0),
					action.getValue(Action.ACTION_COMMAND_KEY));
			field.getActionMap().put(action.getValue(Action.ACTION_COMMAND_KEY), action);
		}
		field.setHorizontalAlignment(JTextField.RIGHT);
		field.addFocusListener(BEI_FOCUS_MARKIEREN);
	}

	/**
	 * Die Methode setzt die Abbrechen-Schaltfläche in einen eigenen JPanel des
	 * Dialogs
	 * 
	 */
	protected void setButtonPane() {
		buttonPane = new JPanel();
		buttonPane.setLayout(new FlowLayout(FlowLayout.RIGHT));
		getContentPane().add(buttonPane, BorderLayout.SOUTH);

		JButton cancelButton = ((BeschriebeneAktion) cancelAction).toButton();
		// Im Grundfall muss die Schaltfläche nur "schließen" lauten
		changeCancelBeschriftung(1);

		buttonPane.add(cancelButton);
		addToRootInputMap(cancelAction);
		getRootPane().setDefaultButton(cancelButton);
	}

	/**
	 * Mit dieser Methode können {@link HinzufuegenDialog} und {@link KostenJDialog}
	 * bei der {@link #getCancelAction()} zwischen der Beschriftung Schließen bzw.
	 * Abbrechen wechseln.
	 * 
	 * @param index siehe {@link Vorsorgend#changeTo(ComponentBeschrifter, int)}
	 */
	protected void changeCancelBeschriftung(int index) {
		LangVorsorgeBeschriftung beschriftung = (LangVorsorgeBeschriftung) getCancelAction().getBeschriftung();
		ComponentBeschrifter beschrifter = new ComponentBeschrifter();
		beschrifter.changeTo(beschriftung, index);
		beschriftung.applyTo(getCancelAction(), beschrifter);
	}

}