/*
 * KostenJPanel.java
 * eu.gronos.kostenrechner.view (Kostenrechner)
 */
package eu.gronos.kostenrechner.view;

import java.awt.GridBagLayout;
import java.awt.LayoutManager;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.text.NumberFormat;

import javax.swing.AbstractAction;
import javax.swing.BorderFactory;
import javax.swing.JComponent;
import javax.swing.JFormattedTextField;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;
import javax.swing.JTable;
import javax.swing.KeyStroke;
import javax.swing.ListSelectionModel;
import javax.swing.UIManager;
import javax.swing.border.Border;
import javax.swing.border.TitledBorder;
import javax.swing.text.NumberFormatter;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.ComponentBeschrifter;
import eu.gronos.kostenrechner.controller.EintragEntfernenAktion;
import eu.gronos.kostenrechner.controller.NumberExtractor;
import eu.gronos.kostenrechner.controller.TabulatorBeforeAction;
import eu.gronos.kostenrechner.controller.TabulatorNextAction;
import eu.gronos.kostenrechner.controller.gebuehren.GebuehrenTransferHandler;
import eu.gronos.kostenrechner.model.beschriftungen.Beschriftung;
import eu.gronos.kostenrechner.model.beschriftungen.LangBeschriftung;
import eu.gronos.kostenrechner.model.beschriftungen.NameContainerSammlung;
import eu.gronos.kostenrechner.view.gebuehren.GebuehrenTableRenderer;
import eu.gronos.kostenrechner.view.gebuehren.TooltipHeaderRenderer;

/**
 * Abstrakte Oberklasse, von denen die Klassen für die Registerkarten abgeleitet
 * werden können.
 *
 * @author Peter Schuster (setrok)
 * @date 13.06.2018
 *
 */
public abstract class KostenJPanel extends JPanel {

	private static final long serialVersionUID = -6667909235024117172L;
	public final AbstractAction eintragEntfernenAktion = new EintragEntfernenAktion(
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(63105));
//					"Eintrag entfernen",
//			"Entfernt den markierten Eintrag aus der Tabelle", "remove",

	AbstractAction aaTabNext = new TabulatorNextAction();
	AbstractAction aaTabBefore = new TabulatorBeforeAction();

	public static final NumberFormatter FORMATTER_INT = new NumberFormatter(NumberFormat.getIntegerInstance());

	/**
	 * Dank an JavaBeginners, Mit einem NumberFormatter lässt sich ein TextFeld so
	 * formatieren, dass nur Zahlen als Eingabe akzeptiert werden. Die Grundlage
	 * hierfür bildet ein JFormattedTextField.
	 * 
	 * @url "http://www.javabeginners.de/Swing/Swing-Elemente/Textkomponenten/Textfeld_fuer_Zahlen.php"
	 */
	public static final NumberFormat FORMAT = NumberFormat.getInstance();
	public static final NumberFormatter FORMATTER = new NumberFormatter(FORMAT);
	public static final TooltipHeaderRenderer HEADER_RENDERER = new TooltipHeaderRenderer();
	public static final EuroEditor EURO_EDITOR = new EuroEditor(FORMATTER);
	public static final EuroRenderer EURO_RENDERER = new EuroRenderer();

	private long start = 0;

	private long ende = 0;
	protected final ComponentBeschrifter beschrifter = new ComponentBeschrifter();

	/**
	 * Konstruktor setzt das GridBagLayout als LayoutManager
	 * 
	 */
	protected KostenJPanel() {
		// super(new GridBagLayout());
		super(new GridBagLayout());
		setStart(Kostenrechner.systemDefaultZone.millis());
		FORMAT.setGroupingUsed(false);
		Kostenrechner.getLogger().info("Starte JPanel " + this.getClass().toString());
	}

	/**
	 * Die Methode dient dazu, die Oberflächenelemente für das Panel zu setzen, wie
	 * bisher fuelleXxxPane()
	 * 
	 */
	public abstract void buildPanel();

	/**
	 * Die Methode dient dazu, eine Referenz auf das Hauptfenster zu übermitteln
	 * 
	 * @param kostenrechner
	 */
	public abstract void setKostenrechner(Kostenrechner kostenrechner);

	/**
	 * @return gibt {@link #start} als {@link long} zurück.
	 */
	public long getStart() {
		return start;
	}

	/**
	 * @return gibt {@link #ende} als {@link long} zurück.
	 */
	public long getEnde() {
		return ende;
	}

	/**
	 * Die Methode errechnet die Dauer des Starts
	 * 
	 * @return die Dauer in Millisekunden.
	 */
	public long errechneDauer() {
		if (getEnde() == 0)
			setEnde(Kostenrechner.systemDefaultZone.millis());
		return getEnde() - getStart();
	}

	/**
	 * Verhindert, dass die übergegebene <code>JTable table</code> die
	 * Tabulatortaste und Umschalt+Tabulatortaste "frisst".
	 * 
	 * @param table die JTable
	 * @url "http://www.coderanch.com/t/342895/GUI/java/JTable-focus" aus: Big Moose
	 *      Saloon, A friendly place for programming greenhorns!
	 */
	public void entferneTastaturFalle(JTable table) {
		table.getInputMap(JComponent.WHEN_FOCUSED).put(KeyStroke.getKeyStroke(KeyEvent.VK_TAB, 0), "tabNext");
		table.getActionMap().put("tabNext", aaTabNext);
		table.getInputMap(JComponent.WHEN_FOCUSED)
				.put(KeyStroke.getKeyStroke(KeyEvent.VK_TAB, InputEvent.SHIFT_DOWN_MASK), "tabBefore");
		table.getActionMap().put("tabBefore", aaTabBefore);
	}

	/**
	 * Die Methode dient dazu, bei den JTables für die GebührenTatbestände die
	 * Spaltenbreite einzustellen, den Renderer und den Transferhandler zu setzen.
	 * 
	 * @param table eine JTable für GebührenTatbestände
	 * 
	 */
	public void setzeGebuehrenTabelleAussehen(JTable table) {
		table.getColumnModel().getColumn(0).setMinWidth(180);
		table.getColumnModel().getColumn(0).setPreferredWidth(180);
		table.getColumnModel().getColumn(1).setPreferredWidth(40);
		table.getColumnModel().getColumn(2).setPreferredWidth(60);
		table.setDefaultRenderer(Double.class, new GebuehrenTableRenderer());
		table.getTableHeader().setDefaultRenderer(HEADER_RENDERER);
		table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		table.setTransferHandler(new GebuehrenTransferHandler(table));
	}

	/**
	 * Die Methode fügt das {@link KostenJPanel} in die {@link JTabbedPane} als
	 * neuen Tab ein.
	 * 
	 * @param pane         das {@link JTabbedPane}
	 * @param beschriftung eine {@link Beschriftung}
	 * @throws IllegalArgumentException TODO (kommentieren)
	 */
	public void intoTabbedPane(JTabbedPane pane, Beschriftung beschriftung) throws IllegalArgumentException {
		pane.addTab(beschriftung.getTitle(), null, this, beschriftung.getShortDescription());
		int index = pane.getTabCount() - 1;
		pane.setEnabledAt(index, true);
		pane.setMnemonicAt(index, beschriftung.mnemonicIntValue());
		//.applyTo
		beschrifter.beschrifte(this, beschriftung);
	}

	/**
	 * @param start d. {@link #start}, d. gesetzt werden soll als {@link long}.
	 */
	protected void setStart(long start) {
		this.start = start;
	}

	/**
	 * @param ende d. {@link #ende}, d. gesetzt werden soll als {@link long}.
	 */
	protected void setEnde(long ende) {
		this.ende = ende;
	}

	/**
	 * Die Methode dient dazu, TODO (kommentieren) TODO (kommentieren)
	 */
	protected void logDauer() {
		Kostenrechner.getLogger()
				.info(String.format("%s brauchte: %d millis", this.getClass().toString(), errechneDauer()));
	}

	/**
	 * Die Methode dient dazu, die formatierten Streitwerte wieder zu long zu
	 * machen, indem "," und "." aus den Zeichenketten gestrichen werden.
	 * 
	 * @param fTf          das {@link JFormattedTextField} mit dem Zahlenwert
	 * @param beschriftung die Beschriftung des {@link JLabel}s des Felds
	 * 
	 * @return ein long
	 */
	protected long longAusFormatter(JFormattedTextField fTf, String beschriftung) throws NumberFormatException {
		if (fTf == null)
			return 0L;
		return new NumberExtractor(fTf, beschriftung).extractLong();
		// Long.parseLong(fTF.getText().replace(".",
		// "").replace(",", ""));
	}

	/**
	 * Die Methode baut ein {@link JPanel} mit {@link TitledBorder} in
	 * Seperatorfarbe.
	 * 
	 * @param beschriftung eine {@link Beschriftung} für das {@link JPanel} selbst
	 * @param layout       {@link LayoutManager} This method changes layout-related
	 *                     information, and therefore, invalidates the component
	 *                     hierarchy.
	 * @return das {@link JPanel}
	 */
	protected JPanel createPanel(Beschriftung beschriftung, LayoutManager layout) {
		JPanel panel = new JPanel();
		panel.setLayout(layout);
		Border schwarz = BorderFactory
				.createLineBorder(UIManager.getLookAndFeel().getDefaults().getColor("Button.shadow"));
		TitledBorder brdr = BorderFactory.createTitledBorder(schwarz);
		panel.setBorder(brdr);
		//.applyTo
		beschrifter.beschrifte(panel, beschriftung);
		return panel;
	}
}