/*
 * OsUIChanger.java
 * eu.gronos.kostenrechner.view (Kostenrechner)
 */
package eu.gronos.kostenrechner.view;

import java.awt.event.InputEvent;

import javax.swing.UIManager;
import javax.swing.UnsupportedLookAndFeelException;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.controller.system.MacAboutHandler;
import eu.gronos.kostenrechner.controller.system.MacDockIcon;
import eu.gronos.kostenrechner.controller.system.MacOpenFileHandler;
import eu.gronos.kostenrechner.controller.system.MacQuitHandler;
import eu.gronos.kostenrechner.model.beschriftungen.Beschriftung;
import eu.gronos.kostenrechner.model.beschriftungen.NameContainerSammlung;
import eu.gronos.kostenrechner.view.helpanderror.FehlerDialog;

/**
 * Die Klasse stellt statische Methoden bereit, um zu prüfen, auf welchem OS das
 * Programm läuft und um das Look'n'Feel zu setzen.
 *
 * @author Peter Schuster (setrok)
 * @date 19.07.2019
 *
 */
public class OsLookAndFeelChanger {

	private OsLookAndFeelChanger() {
	}

	/**
	 * Herausfinden, auf welchem OS ich laufe!
	 */
	public static void pruefeOS() {
		if (isWin())
			setWindowsLookAndFeel();
		if (isMac())
			setMacOsLookAndFeel();
	}

	/**
	 * Die Methode setzt das Look And Feel von Mac OS X und sorgt dafür, dass die
	 * native MacOSX-Menüzeile benutzt werden kann
	 * 
	 * @url "https://bugs.openjdk.java.net/browse/JDK-8029440" JDK-Fehler:
	 *      Xdock:name setzt den Dock-Namen nicht in Mac OS X Mavericks
	 * 
	 * @url "http://stackoverflow.com/questions/307024/native-swing-menu-bar-support-for-macos-x-in-java"
	 *      Native Swing Menu Bar Support For MacOS X In Java
	 * @url "https://developer.apple.com/library/mac/documentation/Java/Conceptual/Java14Development/07-NativePlatformIntegration/NativePlatformIntegration.html#//apple_ref/doc/uid/TP40001909-SW1"
	 *      OS X Integration for Java
	 * 
	 * @url "http://stackoverflow.com/questions/6006173/how-do-you-change-the-dock-icon-of-a-java-program"
	 *      Dock-Icon setzen
	 * @url "http://stackoverflow.com/questions/8361947/how-to-get-getclass-getresource-from-a-static-context"
	 *      getResource aus einer static Methode heraus
	 */
	public static void setMacOsLookAndFeel() {
		try {
			System.setProperty("com.apple.mrj.application.apple.menu.about.name",
					((Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(100)).getTitle());
			// Kostenrechner.KOSTENRECHNER_TITLE);
			UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
			System.setProperty("apple.laf.useScreenMenuBar", "true");
			System.setProperty("apple.awt.brushMetalLook", "true");
			MacDockIcon.dockIconSetzen();
			MacAboutHandler.aboutMenuSetzen();
			MacQuitHandler.quitSetzen();
			MacOpenFileHandler.fileHandlerSetzen();
		} catch (ClassNotFoundException | InstantiationException | IllegalAccessException
				| UnsupportedLookAndFeelException | NoClassDefFoundError e1) {
			FehlerHelper.zeigeFehler(
					FehlerDialog.FEHLER_BLANKO + " beim Anpassen an Apple MacOS:\n" + e1.getLocalizedMessage(), e1);
		}
		// Statt CTRL_MASK muss META_MASK genommen werden
		Kostenrechner.ctrlMask = InputEvent.META_DOWN_MASK;// ActionEvent.META_MASK;
	}

	/**
	 * Die Methode setzt das Look&Feel von Windows über den UIManager
	 * 
	 * @see javax.swing.UIManager
	 * @see javax.swing.UIManager#setLookAndFeel(String)
	 */
	public static void setWindowsLookAndFeel() {
		try {
			UIManager.setLookAndFeel("com.sun.java.swing.plaf.windows.WindowsLookAndFeel");
		} catch (ClassNotFoundException | InstantiationException | IllegalAccessException
				| UnsupportedLookAndFeelException e1) {
			FehlerHelper.zeigeFehler(e1.getLocalizedMessage(), e1);
		}
	}

	/**
	 * Prüft, ob ich auf einem Mac laufe.
	 * 
	 * @return dann true, sonst false
	 */
	public static boolean isMac() {
		return System.getProperty("os.name").contains("Mac OS X");
	}

	/**
	 * Die Methode prüft, ob ich auf einem Windows-System laufe.
	 * 
	 * @return dann true, sonst false
	 */
	public static boolean isWin() {
		return System.getProperty("os.name").contains("Windows") && "\\".equals(System.getProperty("file.separator"));
	}

}
