/**
 * BaumbachAssistentDialog.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.view.baumbach;

import java.awt.BorderLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;

import javax.swing.AbstractAction;
import javax.swing.JCheckBox;
import javax.swing.JFormattedTextField;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSpinner;
import javax.swing.JTabbedPane;
import javax.swing.JTable;
import javax.swing.JTextPane;
import javax.swing.ListSelectionModel;
import javax.swing.SwingConstants;
import javax.swing.text.BadLocationException;
import javax.swing.text.Element;
import javax.swing.text.html.HTMLDocument;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.ComponentBeschrifter;
import eu.gronos.kostenrechner.controller.baumbach.AssistentBkStreitwertHorcher;
import eu.gronos.kostenrechner.controller.baumbach.AssistentBkUnterliegenHorcher;
import eu.gronos.kostenrechner.controller.baumbach.AssistentWbkStreitwertHorcher;
import eu.gronos.kostenrechner.controller.baumbach.AssistentWbkUnterliegenHorcher;
import eu.gronos.kostenrechner.controller.baumbach.AssistentWeiterAction;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenAssistentPruefer;
import eu.gronos.kostenrechner.controller.files.HtmlWerkzeugKasten;
import eu.gronos.kostenrechner.controller.files.XmlTransferHandler;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.interfaces.ParsendUndBauend;
import eu.gronos.kostenrechner.interfaces.Vorsorgend;
import eu.gronos.kostenrechner.model.baumbach.ProtoBeteiligtenWerte;
import eu.gronos.kostenrechner.model.baumbach.Streitgenossen;
import eu.gronos.kostenrechner.model.beschriftungen.Beschriftung;
import eu.gronos.kostenrechner.model.beschriftungen.LangBeschriftung;
import eu.gronos.kostenrechner.model.beschriftungen.NameContainerSammlung;
import eu.gronos.kostenrechner.model.beschriftungen.VorsorgendeBeschriftung;
import eu.gronos.kostenrechner.model.tenordaten.Allgemein;
import eu.gronos.kostenrechner.model.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.model.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.view.GitterBeutelBeschraenkungen;
import eu.gronos.kostenrechner.view.HinzufuegenDialog;
import eu.gronos.kostenrechner.view.TabbedPaneFactory;

/**
 * Assistent zum Hinzufügen aller Beteiligten auf einmal. Mehrere Tabs/Schritte.
 * 1.) Zuerst müssen die einfachen (Baumbach?)Beteiligten für die Klage erstellt
 * werden, ohne das Unterliegen und den Streitwert festzulegen (ggf.
 * XJustiz-Import) bzw. mit einheitlichem Streitwert. 2.) Dann können die
 * Beteiligten der Widerklage hinzugefügt werden, wieder einheitlicher
 * Streitwert. 3.) Dann kann angegeben werden, welche Beteiligten
 * gesamtschuldnerisch und zu wieviel verurteilt wurden. Erst jetzt werden die
 * eigentlichen BaumbachBeteiligten initialisiert (oder jetzt bekommen sie ihre
 * Werte).
 * 
 * @author Peter Schuster (setrok)
 * @date 22.06.2014
 * 
 * 
 */
public class BaumbachAssistentDialog extends HinzufuegenDialog<VerfahrensDatenContainer> {

	private static final long serialVersionUID = -170526643502912253L;

	private JTabbedPane tabbedPane;
	private JPanel[] panels;
	private VerfahrensDatenContainer verfahrensDatenContainer;
	private JTextPane textPane;
	private JSpinner spiKlaegerZahl;
	private JSpinner spiBeklagtenZahl;
	private JTable tabKlaegerGenus;
	private JScrollPane scpKlaegerGenus;
	private SimpleBeteiligtenTable tabBeklagtenGenus;
	private JScrollPane scpBeklagteGenus;
	private JCheckBox chbBkSwEinheitlich;
	private JFormattedTextField ftfBkStreitwert;
	private SimpleBeteiligtenTable tabBeklagtenSw;
	private JScrollPane scpBkSw;
	private JCheckBox chBkGesamtschuldner;
	private JList<Beteiligter> liBkGesamtschuldner;
	private JScrollPane scpBeklagtenAuswahlListe;
	private JFormattedTextField ftfBkGesamtUnterliegen;
	private SimpleBeteiligtenTable tbBkUnt;
	private JScrollPane scpBkUnt;
	private JList<Beteiligter> liBkWk;
	private JScrollPane scpBkWk;
	private JSpinner spiDwbkAnz;
	private SimpleBeteiligtenTable tbDwbkGenus;
	private JScrollPane scpDwbkGenus;
	private JCheckBox chbWkEinheitlich;
	private JFormattedTextField ftfWkSw;
	private SimpleBeteiligtenTable tabKlSw;
	private JScrollPane scpKlSw;
	private SimpleBeteiligtenTable tabDwbkSw;
	private JScrollPane scpDwbkSw;
	private JCheckBox chbWkGesamtschuld;
	private JList<Beteiligter> liWkGesamtschuldner;
	private JFormattedTextField ftfWkUnterliegen;
	private SimpleBeteiligtenTable tbKlDarueber;
	private JScrollPane scpKlDarueber;
	private SimpleBeteiligtenTable tbDwbkDarueber;
	private JScrollPane scpDwbkDarueber;
	private JCheckBox chbKlWk;
	private JLabel lblBkStreitwert;
	private JLabel lblBkGesamtUnterliegen;
	private JLabel lblWkSw;
	private JLabel lblWkUnterliegen;

	public final ProtoBeteiligtenWerte primitives;// = new ProtoBeteiligtenWerte();
	private final ParsendUndBauend<Streitgenossen> pruefer = new StreitgenossenAssistentPruefer();
	private AbstractAction weiter = new AssistentWeiterAction(this,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(38011), pruefer);

	/**
	 * Der Konstruktor für den Standardfall ohne Xjustiz-Import
	 * 
	 * @param owner         the JFrame from which the dialog is displayed
	 * @param mitWiderklage sollen auch die Schritte für die Widerklage durchgeführt
	 *                      werden?
	 */
	public BaumbachAssistentDialog(JFrame owner, boolean mitWiderklage) {
		super(owner, (VorsorgendeBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(38000));
		this.primitives = new ProtoBeteiligtenWerte();
		this.primitives.mitWiderklage = mitWiderklage;
		((StreitgenossenAssistentPruefer) pruefer).setZustand(0);
		this.setContainer(null);
		primitives.ohneXjustiz = getVerfahrensDatenContainer() == null;
		setOkButtonToBeFocussed(false);
		Kostenrechner.getLogger().info("BaumbachAssistentDialog - ohne Xjustiz!");
		((StreitgenossenAssistentPruefer) pruefer).setRueckgabewert(new Streitgenossen());
		// setPrimitives darf nur einmal aufgerufen werden! - dann wohl hier
		((StreitgenossenAssistentPruefer) pruefer).setPrimitives(this.primitives);
	}

	/**
	 * Ein Konstruktor, der Verfahrensdaten vom XJustiz-Import annimmt und dann ein
	 * paar Felder schon einmal vorbelegen kann.
	 * 
	 * @param owner     the JFrame from which the dialog is displayed
	 * @param container ein KostenTenorContainer aus einem Xjustiz-Import
	 */
	public BaumbachAssistentDialog(JFrame owner, VerfahrensDatenContainer container) {
		super(owner, (Beschriftung) new ComponentBeschrifter()
				.changeTo(((Vorsorgend<?>) NameContainerSammlung.BESCHRIFTUNGEN.get(38000)), 1));
//		if () {}
		this.primitives = ((StreitgenossenAssistentPruefer) pruefer).pruefeVerfahrensDatenContainer(container,
				new ProtoBeteiligtenWerte());
		((StreitgenossenAssistentPruefer) pruefer).setZustand(0);
		this.setContainer(container);
		setOkButtonToBeFocussed(false);
		Kostenrechner.getLogger().info("BaumbachAssistentDialog - mit Xjustiz!");
		((StreitgenossenAssistentPruefer) pruefer).setRueckgabewert(container.streitgenossen);
		// setPrimitives darf nur einmal aufgerufen werden! - dann wohl hier
		((StreitgenossenAssistentPruefer) pruefer).setPrimitives(this.primitives);
	}

	/**
	 * Die Methode setzt alle Tabs außer den aktuellen und den Start-Tab auf deaktiv
	 * 
	 * @param index Index des zu aktivierenden Tabs
	 */
	public void zustandAnzeigen(int index) {
		for (int i = 0; i < (((StreitgenossenAssistentPruefer) pruefer).getHoechsterZustand())
				&& panels[i] != null; i++) {
			if (i == 0) {
				tabbedPane.setEnabledAt(i, true);
			} else if (i == index) {
				tabbedPane.setEnabledAt(i, true);
				tabbedPane.setSelectedIndex(i);
			} else {
				tabbedPane.setEnabledAt(i, false);
			}
		}
		/* Solange die Eingabe noch nicht abgeschlossen ist, Ok ausblenden */
		if (index < ((StreitgenossenAssistentPruefer) pruefer).getHoechsterZustand() - 1) {
			super.getOkButton().requestFocusInWindow();
		} else {
			restoreOkButton();
			super.getOkButton().requestFocusInWindow();
		}
	}

	/**
	 * Die Methode darf erst ganz zum Schluss true melden, abhängig davon also, ob
	 * eine Widerklage vorliegt oder nicht. Wenn keine Widerklage vorliegt, ist der
	 * Vorgang nach dem Panel 5 fertig, mit Widerklage nach dem Panel 10.
	 * 
	 * @see eu.gronos.kostenrechner.view.HinzufuegenDialog#parseEingabe()
	 */
	@Override
	public boolean parseEingabe() {
		Kostenrechner.getLogger()
				.info(String.format("Zustand: %d", ((StreitgenossenAssistentPruefer) pruefer).getZustand()));
		// return
		boolean boole = false;
		try {
			boole = pruefer.parseEingabe();
		} catch (IllegalArgumentException e) {
			FehlerHelper.zeigeFehler(e.getLocalizedMessage(), e);
			boole = false;
		} catch (NullPointerException e) {
			FehlerHelper.zeigeFehler(e.getLocalizedMessage(), e);
			boole = false;
		}
		return boole;
	}

	/**
	 * @return gibt {@link #verfahrensDatenContainer} als VerfahrensDatenContainer
	 *         zurück.
	 */
	public VerfahrensDatenContainer getVerfahrensDatenContainer() {
		return verfahrensDatenContainer;
	}

	/**
	 * Diese Methode erstellt zum Abschluss BaumbachBeteiligtenListen
	 * 
	 * Erst wenn der letzte Zustand erreicht ist, darf gebaut werden (sollte
	 * eigentlich von parseEingabe() geprüft werden
	 * 
	 * @return soll einen {@link VerfahrensDatenContainer} zurückgeben
	 * 
	 * @see eu.gronos.kostenrechner.view.HinzufuegenDialog#baueRueckgabewert()
	 */
	@Override
	protected VerfahrensDatenContainer baueRueckgabewert() {
		VerfahrensDatenContainer c = new VerfahrensDatenContainer();
		/* Den UnterContainer Streitgenossen erstellt der pruefer */
		c.streitgenossen = pruefer.getRueckgabewert();
		/* Dann die allgemeinen Teile einpacken */
		Allgemein a = c.allgemein;
		// VerfahrensDatenContainerUtil.
		a.speicherdatum = new XmlTransferHandler().speicherDatum();
		if (!primitives.ohneXjustiz) {
			a.aktenzeichen = getVerfahrensDatenContainer().allgemein.aktenzeichen;
		} else {
			a.aktenzeichen = "";
		}
		if (primitives.wkStreitwert > primitives.bkStreitwert)
			a.streitwert = primitives.wkStreitwert;
		else
			a.streitwert = primitives.bkStreitwert;
		a.selectedPanel = StreitgenossenPanel.TAB_STREITGENOSSEN;
		Kostenrechner.getLogger().info(primitives.toString());
		return c;
	}

	/**
	 * Die Methode initialisiert die allgemeinen JComponents für den Inhaltsbereich
	 * des Dialogs und wird vom Konstruktor selbsttätig aufgerufen. In der
	 * abgeleiteten Klasse baut die Methode eine {@link javax.swing.JTabbedPane}
	 * auf, mit so vielen Tabs wie Zuständen, die in einem Array JPanel[] panels
	 * aufbewahrt werden.
	 * 
	 * @see eu.gronos.kostenrechner.view.HinzufuegenDialog#baueContentPane()
	 */
	@Override
	protected void baueContentPane() {
		tabbedPane = new JTabbedPane(JTabbedPane.TOP);
		ComponentBeschrifter beschrifter = new ComponentBeschrifter();
		beschrifter.beschrifte(tabbedPane, NameContainerSammlung.BESCHRIFTUNGEN.get(38001));
		/* BorderLayout.NORTH führt zu Darstellungsfehlern beim Vergrößern */
		getContentPane().add(tabbedPane, BorderLayout.CENTER);
		tabbedPane.setBorder(BORDER);
		panels = new JPanel[StreitgenossenAssistentPruefer.HOECHSTER_ZUSTAND];
		/*
		 * for-Schleife zum Anlegen der Reiter Die tabbedPanes können hier noch nicht
		 * angelegt werden, weil mitWiderklage noch nicht bekannt ist.
		 */
		for (int i = 0; i < panels.length; i++) {
			panels[i] = new JPanel();
			panels[i].setBorder(BORDER);
			panels[i].setLayout(new GridBagLayout());
		}
	}

	/**
	 * Füllt die Tabs mit Oberflächenelemente
	 * 
	 * @see eu.gronos.kostenrechner.view.HinzufuegenDialog#fuelleContentPane()
	 */
	@Override
	protected void fuelleContentPane() {
		baueWeiterButton();
		/* TabbedPanes anlegen */
		final int[] keys = new int[] { 38002, 38020, 38100, 38200, 38300, 38400, 38500, 38600, 38700, 38800, 38900 };
		// ohne Widerklage früher aufhören!
		for (int i = 0; i < panels.length && (primitives.mitWiderklage
				|| i < StreitgenossenAssistentPruefer.HOECHSTER_ZUSTAND_OHNE_WIDERKLAGE); i++) {
			Beschriftung beschriftung = (Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(keys[i]);
			TabbedPaneFactory.intoTabbedPane(tabbedPane, panels[i], beschriftung);
		}

		/*
		 * Jeder Schritt ist ein einzelnes JPanel im tabbedPane.
		 * 
		 * Nur je eine sbtm-Variable für kl, bk, dwbk!
		 */
		// Panel 0:
		final HtmlWerkzeugKasten kit = new HtmlWerkzeugKasten();

		textPane = (JTextPane) beschrifter.beschrifte(new JTextPane(), NameContainerSammlung.BESCHRIFTUNGEN.get(38005));
		textPane.setEditorKit(kit);
		kit.read(willkommenRessource(), textPane);

		if (!primitives.ohneXjustiz) {
			baueAzEin(textPane, getVerfahrensDatenContainer().allgemein.aktenzeichen);
		}

		panels[0].add(textPane, GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 1, GridBagConstraints.BOTH, false));

		// Panel 1:
		// Kann bei Import übersprungen werden
		JLabel lblKlaegerZahl = (JLabel) beschrifter.beschrifte(new JLabel(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38021));
		lblKlaegerZahl.setHorizontalAlignment(SwingConstants.RIGHT);

		spiKlaegerZahl = (JSpinner) beschrifter.beschrifte(new JSpinner(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38025));
		lblKlaegerZahl.setLabelFor(spiKlaegerZahl);
		spiKlaegerZahl.setEnabled(false);

		JLabel lblBeklagtenZahl = (JLabel) beschrifter.beschrifte(new JLabel(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38031));
		lblBeklagtenZahl.setHorizontalAlignment(SwingConstants.RIGHT);

		spiBeklagtenZahl = (JSpinner) beschrifter.beschrifte(new JSpinner(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38035));
		lblBeklagtenZahl.setLabelFor(spiBeklagtenZahl);

		panels[1].add(lblKlaegerZahl,
				GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 1, GridBagConstraints.HORIZONTAL, false));
		panels[1].add(spiKlaegerZahl,
				GitterBeutelBeschraenkungen.getInstance(1, 0, 1, 1, GridBagConstraints.HORIZONTAL, false));
		panels[1].add(lblBeklagtenZahl,
				GitterBeutelBeschraenkungen.getInstance(0, 1, 1, 1, GridBagConstraints.HORIZONTAL, false));
		panels[1].add(spiBeklagtenZahl,
				GitterBeutelBeschraenkungen.getInstance(1, 1, 1, 1, GridBagConstraints.HORIZONTAL, false));

		// Panel 2
		// kann bei Import übersprungen werden
		JLabel lblGenusKlaeger = (JLabel) beschrifter.beschrifte(new JLabel(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38101));

		tabKlaegerGenus = (JTable) beschrifter.beschrifte(new SimpleBeteiligtenTable(Beteiligter.KLAEGER),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38111));
		scpKlaegerGenus = (JScrollPane) beschrifter.beschrifte(new JScrollPane(tabKlaegerGenus),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38110));
		lblGenusKlaeger.setLabelFor(scpKlaegerGenus);

		JLabel lblBeklagtenGenus = (JLabel) beschrifter.beschrifte(new JLabel(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38121));

		tabBeklagtenGenus = (SimpleBeteiligtenTable) beschrifter.beschrifte(
				new SimpleBeteiligtenTable(Beteiligter.BEKLAGTE), NameContainerSammlung.BESCHRIFTUNGEN.get(38131));
		scpBeklagteGenus = (JScrollPane) beschrifter.beschrifte(new JScrollPane(tabBeklagtenGenus),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38130));
		lblBeklagtenGenus.setLabelFor(scpBeklagteGenus);

		panels[2].add(lblGenusKlaeger,
				GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 1, GridBagConstraints.HORIZONTAL, false));
		panels[2].add(scpKlaegerGenus,
				GitterBeutelBeschraenkungen.getInstance(0, 1, 1, 1, GridBagConstraints.BOTH, false));
		panels[2].add(lblBeklagtenGenus,
				GitterBeutelBeschraenkungen.getInstance(1, 0, 1, 1, GridBagConstraints.HORIZONTAL, false));
		panels[2].add(scpBeklagteGenus,
				GitterBeutelBeschraenkungen.getInstance(1, 1, 1, 1, GridBagConstraints.BOTH, false));

		// Panel 3
		chbBkSwEinheitlich = (JCheckBox) beschrifter.beschrifte(new JCheckBox(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38201));

		lblBkStreitwert = (JLabel) beschrifter.beschrifte(new JLabel(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38211));
		lblBkStreitwert.setHorizontalAlignment(SwingConstants.RIGHT);

		ftfBkStreitwert = (JFormattedTextField) beschrifter.beschrifte(new JFormattedTextField(formatter),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38215));
		// Vorbelegen mit dem Wert aus Xjustiz, sofern bekannt
		lblBkStreitwert.setLabelFor(ftfBkStreitwert);
		// "Weiter"-Action der InputMap hinzufügen und mit VK_ENTER verknüpfen
		setEnterAction(ftfBkStreitwert, weiter);

		tabBeklagtenSw = (SimpleBeteiligtenTable) beschrifter.beschrifte(
				new SimpleBeteiligtenTable(Beteiligter.BEKLAGTE), NameContainerSammlung.BESCHRIFTUNGEN.get(38221));
		chbBkSwEinheitlich.addChangeListener(new AssistentBkStreitwertHorcher(ftfBkStreitwert, tabBeklagtenSw));
		scpBkSw = (JScrollPane) beschrifter.beschrifte(new JScrollPane(tabBeklagtenSw),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38220));

		panels[3].add(chbBkSwEinheitlich,
				GitterBeutelBeschraenkungen.getInstance(0, 0, 2, 1, GridBagConstraints.HORIZONTAL, false));
		panels[3].add(lblBkStreitwert,
				GitterBeutelBeschraenkungen.getInstance(0, 1, 1, 1, GridBagConstraints.HORIZONTAL, false));
		panels[3].add(ftfBkStreitwert,
				GitterBeutelBeschraenkungen.getInstance(1, 1, 1, 1, GridBagConstraints.HORIZONTAL, false));
		panels[3].add(scpBkSw, GitterBeutelBeschraenkungen.getInstance(0, 2, 2, 1, GridBagConstraints.BOTH, false));

		// Panel 4
		chBkGesamtschuldner = (JCheckBox) beschrifter.beschrifte(new JCheckBox(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38301));
		chBkGesamtschuldner.setSelected(true);

		JLabel lblBkGesamtschuldner = (JLabel) beschrifter.beschrifte(new JLabel(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38311));

		// Hier muss ich wohl die erste Spalte nehmen...
		liBkGesamtschuldner = new JList<Beteiligter>();
		beschrifter.beschrifte(liBkGesamtschuldner, NameContainerSammlung.BESCHRIFTUNGEN.get(38321));
		liBkGesamtschuldner.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
		liBkGesamtschuldner.setCellRenderer(new BeteiligtenListRenderer(false));
		scpBeklagtenAuswahlListe = (JScrollPane) beschrifter.beschrifte(new JScrollPane(liBkGesamtschuldner),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38320));
		lblBkGesamtschuldner.setLabelFor(scpBeklagtenAuswahlListe);

		lblBkGesamtUnterliegen = (JLabel) beschrifter.beschrifte(new JLabel(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38331));
		lblBkGesamtUnterliegen.setHorizontalAlignment(SwingConstants.RIGHT);

		ftfBkGesamtUnterliegen = (JFormattedTextField) beschrifter.beschrifte(new JFormattedTextField(formatter),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38335));
		lblBkGesamtUnterliegen.setLabelFor(ftfBkGesamtUnterliegen);
		setEnterAction(ftfBkGesamtUnterliegen, weiter);
		/*
		 * Muss nach dem Mnemonic, weil das Zuweisen eines Mnemonic wohl das ChangeEvent
		 * auch auslöst. Und auch nach der Initialisierung der Oberflächenelemente.
		 */
		chBkGesamtschuldner
				.addChangeListener(new AssistentBkUnterliegenHorcher(liBkGesamtschuldner, ftfBkGesamtUnterliegen));

		panels[4].add(chBkGesamtschuldner,
				GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 1, GridBagConstraints.HORIZONTAL, false));
		panels[4].add(lblBkGesamtschuldner,
				GitterBeutelBeschraenkungen.getInstance(0, 1, 1, 1, GridBagConstraints.HORIZONTAL, false));
		panels[4].add(scpBeklagtenAuswahlListe,
				GitterBeutelBeschraenkungen.getInstance(0, 2, 2, 1, GridBagConstraints.BOTH, false));
		panels[4].add(lblBkGesamtUnterliegen,
				GitterBeutelBeschraenkungen.getInstance(0, 3, 1, 1, GridBagConstraints.HORIZONTAL, false));
		panels[4].add(ftfBkGesamtUnterliegen,
				GitterBeutelBeschraenkungen.getInstance(1, 3, 1, 1, GridBagConstraints.HORIZONTAL, false));

		// Panel 5
		// ==> sonst Unterliegen 0
		JLabel lblBkUnt = (JLabel) beschrifter.beschrifte(new JLabel(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38401));
		tbBkUnt = (SimpleBeteiligtenTable) beschrifter.beschrifte(new SimpleBeteiligtenTable(Beteiligter.BEKLAGTE),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38411));
		scpBkUnt = (JScrollPane) beschrifter.beschrifte(new JScrollPane(tbBkUnt),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38410));
		lblBkUnt.setLabelFor(scpBkUnt);

		panels[5].add(lblBkUnt,
				GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 1, GridBagConstraints.HORIZONTAL, false));
		panels[5].add(scpBkUnt, GitterBeutelBeschraenkungen.getInstance(0, 2, 2, 1, GridBagConstraints.BOTH, false));

		// AB HIER NUR WENN (mitWiderklage == true)
		if (primitives.mitWiderklage == false) {
			zustandAnzeigen(((StreitgenossenAssistentPruefer) pruefer).getZustand());
			panelsAusblendenBeiImport();
			return;
		}

		// Panel 6
		// kann bei Import übersprungen werden
		JLabel lblBkWk = (JLabel) beschrifter.beschrifte(new JLabel(), NameContainerSammlung.BESCHRIFTUNGEN.get(38501));
		liBkWk = new JList<Beteiligter>();
		beschrifter.beschrifte(liBkWk, NameContainerSammlung.BESCHRIFTUNGEN.get(38511));
		liBkWk.setCellRenderer(new BeteiligtenListRenderer());
		liBkWk.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		scpBkWk = (JScrollPane) beschrifter.beschrifte(new JScrollPane(liBkWk),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38510));
		lblBkWk.setLabelFor(scpBkWk);

		JLabel lblDwbkAnz = (JLabel) beschrifter.beschrifte(new JLabel(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38521));
		lblDwbkAnz.setHorizontalAlignment(SwingConstants.RIGHT);
		spiDwbkAnz = (JSpinner) beschrifter.beschrifte(new JSpinner(), NameContainerSammlung.BESCHRIFTUNGEN.get(38525));
		lblDwbkAnz.setLabelFor(spiDwbkAnz);
		// Abfragen, ob Wk auch gegen den Kläger geht
		chbKlWk = (JCheckBox) beschrifter.beschrifte(new JCheckBox(), NameContainerSammlung.BESCHRIFTUNGEN.get(38531));

		panels[6].add(lblBkWk,
				GitterBeutelBeschraenkungen.getInstance(0, 0, 3, 1, GridBagConstraints.HORIZONTAL, false));
		panels[6].add(scpBkWk, GitterBeutelBeschraenkungen.getInstance(0, 1, 3, 1, GridBagConstraints.BOTH, false));
		panels[6].add(lblDwbkAnz,
				GitterBeutelBeschraenkungen.getInstance(0, 2, 1, 1, GridBagConstraints.HORIZONTAL, false));
		panels[6].add(spiDwbkAnz,
				GitterBeutelBeschraenkungen.getInstance(1, 2, 1, 1, GridBagConstraints.HORIZONTAL, false));
		panels[6].add(chbKlWk,
				GitterBeutelBeschraenkungen.getInstance(2, 2, 1, 1, GridBagConstraints.HORIZONTAL, false));

		// Panel 7
		// kann bei Import übersprungen werden
		JLabel lblDwbkGenus = (JLabel) beschrifter.beschrifte(new JLabel(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38601));
		tbDwbkGenus = (SimpleBeteiligtenTable) beschrifter.beschrifte(
				new SimpleBeteiligtenTable(Beteiligter.DRITTWIDERBEKLAGTE),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38611));
		scpDwbkGenus = (JScrollPane) beschrifter.beschrifte(new JScrollPane(tbDwbkGenus),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38610));
		lblDwbkGenus.setLabelFor(scpDwbkGenus);
		panels[7].add(lblDwbkGenus,
				GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 1, GridBagConstraints.HORIZONTAL, false));
		panels[7].add(scpDwbkGenus,
				GitterBeutelBeschraenkungen.getInstance(0, 1, 1, 1, GridBagConstraints.BOTH, false));

		// Panel 8
		chbWkEinheitlich = (JCheckBox) beschrifter.beschrifte(new JCheckBox(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38701));

		lblWkSw = (JLabel) beschrifter.beschrifte(new JLabel(), NameContainerSammlung.BESCHRIFTUNGEN.get(38711));
		lblWkSw.setHorizontalAlignment(SwingConstants.RIGHT);
		ftfWkSw = (JFormattedTextField) beschrifter.beschrifte(new JFormattedTextField(formatter),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38715));
		lblWkSw.setLabelFor(ftfWkSw);
		setEnterAction(ftfWkSw, weiter);

		JLabel lblKlSw = (JLabel) beschrifter.beschrifte(new JLabel(), NameContainerSammlung.BESCHRIFTUNGEN.get(38721));
		tabKlSw = (SimpleBeteiligtenTable) beschrifter.beschrifte(new SimpleBeteiligtenTable(Beteiligter.KLAEGER),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38731));
		scpKlSw = (JScrollPane) beschrifter.beschrifte(new JScrollPane(tabKlSw),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38730));
		lblKlSw.setLabelFor(scpKlSw);

		JLabel lblDwbkSw = (JLabel) beschrifter.beschrifte(new JLabel(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38741));
		tabDwbkSw = (SimpleBeteiligtenTable) beschrifter.beschrifte(
				new SimpleBeteiligtenTable(Beteiligter.DRITTWIDERBEKLAGTE),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38751));
		chbWkEinheitlich.addChangeListener(new AssistentWbkStreitwertHorcher(ftfWkSw, tabKlSw, tabDwbkSw, primitives));
		scpDwbkSw = (JScrollPane) beschrifter.beschrifte(new JScrollPane(tabDwbkSw),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38750));
		lblDwbkSw.setLabelFor(scpDwbkSw);

		panels[8].add(chbWkEinheitlich,
				GitterBeutelBeschraenkungen.getInstance(0, 0, 2, 1, GridBagConstraints.HORIZONTAL, false));
		panels[8].add(lblWkSw,
				GitterBeutelBeschraenkungen.getInstance(0, 1, 1, 1, GridBagConstraints.HORIZONTAL, false));
		panels[8].add(ftfWkSw,
				GitterBeutelBeschraenkungen.getInstance(1, 1, 1, 1, GridBagConstraints.HORIZONTAL, false));
		panels[8].add(lblKlSw,
				GitterBeutelBeschraenkungen.getInstance(0, 2, 1, 1, GridBagConstraints.HORIZONTAL, false));
		panels[8].add(scpKlSw, GitterBeutelBeschraenkungen.getInstance(0, 3, 1, 1, GridBagConstraints.BOTH, false));
		panels[8].add(lblDwbkSw,
				GitterBeutelBeschraenkungen.getInstance(1, 2, 1, 1, GridBagConstraints.HORIZONTAL, false));
		panels[8].add(scpDwbkSw, GitterBeutelBeschraenkungen.getInstance(1, 3, 1, 1, GridBagConstraints.BOTH, false));

		// Panel 9
		chbWkGesamtschuld = (JCheckBox) beschrifter.beschrifte(new JCheckBox(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38801));

		JLabel lblWkGesamtschuldner = (JLabel) beschrifter.beschrifte(new JLabel(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38811));
		liWkGesamtschuldner = new JList<Beteiligter>();
		beschrifter.beschrifte(liWkGesamtschuldner, NameContainerSammlung.BESCHRIFTUNGEN.get(38821));
		liWkGesamtschuldner.setCellRenderer(new BeteiligtenListRenderer());
		lblWkGesamtschuldner.setLabelFor(liWkGesamtschuldner);
		liWkGesamtschuldner.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
		JScrollPane scpWkGesamtschuldner = (JScrollPane) beschrifter.beschrifte(new JScrollPane(liWkGesamtschuldner),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38820));

		lblWkUnterliegen = (JLabel) beschrifter.beschrifte(new JLabel(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38831));
		ftfWkUnterliegen = (JFormattedTextField) beschrifter.beschrifte(new JFormattedTextField(formatter),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38835));
		lblWkUnterliegen.setLabelFor(ftfWkUnterliegen);
		chbWkGesamtschuld.addChangeListener(
				new AssistentWbkUnterliegenHorcher(liWkGesamtschuldner, ftfWkUnterliegen, primitives));
		setEnterAction(ftfWkUnterliegen, weiter);

		panels[9].add(chbWkGesamtschuld,
				GitterBeutelBeschraenkungen.getInstance(0, 0, 2, 1, GridBagConstraints.HORIZONTAL, false));
		panels[9].add(lblWkGesamtschuldner,
				GitterBeutelBeschraenkungen.getInstance(0, 1, 2, 1, GridBagConstraints.HORIZONTAL, false));
		panels[9].add(scpWkGesamtschuldner,
				GitterBeutelBeschraenkungen.getInstance(0, 2, 2, 1, GridBagConstraints.BOTH, false));
		panels[9].add(lblWkUnterliegen,
				GitterBeutelBeschraenkungen.getInstance(0, 3, 1, 1, GridBagConstraints.HORIZONTAL, false));
		panels[9].add(ftfWkUnterliegen,
				GitterBeutelBeschraenkungen.getInstance(1, 3, 1, 1, GridBagConstraints.HORIZONTAL, false));

		// Panel 10
		JLabel lblKlDarueber = (JLabel) beschrifter.beschrifte(new JLabel(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38901));
		tbKlDarueber = (SimpleBeteiligtenTable) beschrifter.beschrifte(new SimpleBeteiligtenTable(Beteiligter.KLAEGER),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38911));
		scpKlDarueber = (JScrollPane) beschrifter.beschrifte(new JScrollPane(tbKlDarueber),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38910));
		lblKlDarueber.setLabelFor(scpKlDarueber);

		JLabel lblDwbkDarueber = (JLabel) beschrifter.beschrifte(new JLabel(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38921));
		tbDwbkDarueber = (SimpleBeteiligtenTable) beschrifter.beschrifte(
				new SimpleBeteiligtenTable(Beteiligter.DRITTWIDERBEKLAGTE),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38931));
		scpDwbkDarueber = (JScrollPane) beschrifter.beschrifte(new JScrollPane(tbDwbkDarueber),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38930));
		lblDwbkDarueber.setLabelFor(scpDwbkDarueber);

		panels[10].add(lblKlDarueber,
				GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 1, GridBagConstraints.HORIZONTAL, false));
		panels[10].add(scpKlDarueber,
				GitterBeutelBeschraenkungen.getInstance(0, 1, 1, 1, GridBagConstraints.BOTH, false));
		panels[10].add(lblDwbkDarueber,
				GitterBeutelBeschraenkungen.getInstance(1, 0, 1, 1, GridBagConstraints.HORIZONTAL, false));
		panels[10].add(scpDwbkDarueber,
				GitterBeutelBeschraenkungen.getInstance(1, 1, 1, 1, GridBagConstraints.BOTH, false));

		zustandAnzeigen(((StreitgenossenAssistentPruefer) pruefer).getZustand());
		panelsAusblendenBeiImport();
	}
	// final HTMLEditorKit kit = new HTMLEditorKit();
	// final StyledDocument doc = (StyledDocument) kit.createDefaultDocument();
	// ComponentBeschrifter beschrifter = new ComponentBeschrifter();
	// jta.setEditorKit(kit);
	// jta.setEditable(false);
	// jta.setStyledDocument(doc);
	// final HtmlWerkzeugKasten helper = new HtmlWerkzeugKasten();
	// helper.leseHtml(willkommen, doc, kit);
	// helper.leseCss("resources/kostentenor.css", kit);
	// final String willkommen = willkommenRessource();

	/**
	 * Die Methode entscheidet, welche HTML-Datei als Willkommensseite geladen
	 * werden soll.
	 * 
	 * @return ein {@link String} mit dem Namen der Resource
	 */
	private String willkommenRessource() {
		if (!primitives.ohneXjustiz)
			return "resources/assistent-xjustiz.html";
		else
			return "resources/assistent.html";
	}
	// String willkommen = "resources/assistent.html";
	// return willkommen;

	/**
	 * Die Methode baut das Aktenzeichen aus dem {@link VerfahrensDatenContainer} in
	 * das {@link HTMLDocument} ein.
	 * 
	 * @param jta          das {@link JTextPane}, in dem auch der Platzhalter für
	 *                     das Aktenzeichen steht.
	 * @param aktenzeichen das Aktenzeichen aus der Xjustiz-Datei.
	 */
	private void baueAzEin(JTextPane jta, String aktenzeichen) {
		final HTMLDocument htmlDoc = (HTMLDocument) jta.getDocument();
		Element span = htmlDoc.getElement("az");
		// Der Inhalt des Span-Tags
		if (span != null)
			try {
				System.err.println(htmlDoc.getText(span.getStartOffset(), span.getEndOffset() - span.getStartOffset()));
				htmlDoc.remove(span.getStartOffset(), span.getEndOffset() - span.getStartOffset());
				htmlDoc.insertString(span.getStartOffset(), " dem Verfahren " + aktenzeichen, span.getAttributes());
			} catch (BadLocationException e) {
				FehlerHelper.zeigeFehler(e.getLocalizedMessage(), e);
			}
		else
			System.err.println("Nix kann writen Aktenzeichen!");
	}

	/**
	 * Die Methode rekonstruiert den {@link HinzufuegenDialog#getOkButton()} wieder
	 * zu einer Schaltfläche "Ok". {@link HinzufuegenDialog} enthält dafür
	 * {@link HinzufuegenDialog#getOkAction()}.
	 */
	private void restoreOkButton() {
		super.getOkButton().setAction(super.getOkAction());
		super.setOkActionEnabled(true);
		removeFromInputMap(weiter);
		addToRootInputMap(super.getOkAction());
		beschrifter.beschrifte(super.getOkButton(), NameContainerSammlung.BESCHRIFTUNGEN.get(76051));
	}

	/**
	 * Die Methode ersetzt den {@link HinzufuegenDialog#getOkButton()} durch einen
	 * Weiter-Button
	 */
	private void baueWeiterButton() {
		super.setOkActionEnabled(false);
		removeFromInputMap(getOkAction());
		super.getOkButton().setAction(weiter);
		addToRootInputMap(weiter);
		beschrifter.beschrifte(super.getOkButton(), NameContainerSammlung.BESCHRIFTUNGEN.get(38011));
	}

	/**
	 * Die Methode sorgt dafür, dass bei Xjustiz-Import die nicht benötigten
	 * Panels/Tabs auch nicht stören. Zwar kann man diese Tabs nicht komplett
	 * ausblenden, aber zumindest werden die Titles auf null gesetzt, so dass sie
	 * leer sind. Der Assistent erscheint dadurch nicht mehr so groß.
	 * 
	 */
	private void panelsAusblendenBeiImport() {
		if (primitives.ohneXjustiz)
			return;
		if (primitives.tab1ueberspringen) {
			panels[1].setVisible(false);
			tabbedPane.setTitleAt(1, null);
		}
		if (primitives.tab2ueberspringen) {
			panels[2].setVisible(false);
			tabbedPane.setTitleAt(2, null);
		}
		if (primitives.tab4ueberspringen) {
			panels[4].setVisible(false);
			tabbedPane.setTitleAt(4, null);
		}
		if (primitives.mitWiderklage) {
			if (primitives.tab6ueberspringen) {
				tabbedPane.setTitleAt(6, null);
				panels[6].setVisible(false);
			}
			if (primitives.tab7ueberspringen) {
				tabbedPane.setTitleAt(7, null);
				panels[7].setVisible(false);
			}
			if (primitives.tab9ueberspringen) {
				tabbedPane.setTitleAt(9, null);
				panels[9].setVisible(false);
			}
		}
	}

	/**
	 * @param container d. {@link #verfahrensDatenContainer}, d. gesetzt werden soll
	 *                  als VerfahrensDatenContainer
	 */
	private void setContainer(VerfahrensDatenContainer container) {
		this.verfahrensDatenContainer = container;
	}

}
//private AbstractAction buttonBackup;
// "Weiter", "weiter", "Weiter zum nächsten Reiter",
// getButtonPane().add(new JButton(weiter));
// backupOkButton();
// restoreOkButton();
// addToRootInputMap(weiter);
// tabbedPane.addTab("Start", null, panels[0], "Willkommenstext anzeigen");
// tabbedPane.addTab("Beteiligtenanzahl (1)", null, panels[1], "Anzahl der
// Kläger und Beklagten");
// tabbedPane.addTab("-genus (2)", null, panels[2], "Bezeichnung von Kläger und
// Beklagten");
// tabbedPane.addTab("Klagewert (3)", null, panels[3], "Streitwert der Beklagten
// (für alle einheitlich?)");
// tabbedPane.addTab("Gesamtschuld (4)", null, panels[4],"Wurden die Beklagten
// (ganz oder z.T.) gesamtschuldnerisch verurteilt? Auswahl der Beklagten");
// tabbedPane.addTab("Darüber hinaus (5)", null, panels[5], "Wurden einige
// Beklagte darüber hinaus verurteilt? Welche, zu wie viel?");
// tabbedPane.addTab("Widerklage (6)", null, panels[6], "Welcher Beklagte klagt?
// Wie viele Drittwiderbeklagten?");
// tabbedPane.addTab("Drittwiderklage (7)", null, panels[7], "Bezeichnung der
// Drittwiderbeklagten");
// tabbedPane.addTab("Widerklagewert (8)", null, panels[8],"Wurden Widerbeklagte
// in derselben Höhe verklagt? Welche (jeweils)?");
// tabbedPane.addTab("Gesamtschuld (9)", null, panels[9],"Wurden die
// Widerbeklagten z.T. gesamtschuldnerisch verurteilt? Welche Widerbeklagten, in
// welcher Höhe?");
// tabbedPane.addTab("Darüber hinaus (10)", null, panels[10],"Wurden einzelne
// Widerbeklagten darüber hinaus einzeln verurteilt? Welche, in welcher Höhe?");
// "Willkommen!\n\n" + "Der Assistent führt Sie durch die notwendigen Schritte,
// \n"
// + "um den Baumbach'schen Kostentenor berechnen zu können.\n\n"
// + "Klicken Sie \"Weiter\", um zu beginnen.";
//"Willkommen!\n\n" + "Sie möchten die Beteiligtendaten aus dem Verfahren "
//+ getVerfahrensDatenContainer().allgemein.aktenzeichen + " importieren.\n"
//+ "Der Assistent unternimmt mit Ihnen die noch verbleibenden Schritte, um die noch fehlenden Daten zu erfragen, \n"
//+ "mit denen ein Baumbach'scher Kostentenor berechnet werden kann.\n\n"
//+ "Klicken Sie \"Weiter\", um zu beginnen.";
//JTextArea JTextArea(willkommen)
// "Anzahl der Kläger: ", );
// 0, 100, 1
// setMnemonicLabelFor(, , KeyEvent.VK_K);
// "Anzahl der Beklagten: ", );
// "Bezeichnung der Kläger");
// setMnemonicLabelFor(, , KeyEvent.VK_K);
// "Bezeichnung der Beklagten");
// setMnemonicLabelFor(, , KeyEvent.VK_B);
// "Die Beklagten wurden alle in derselben Höhe in Anspruch genommen", true
// "Streitwert: ", ;
// setMnemonicLabelFor(, , KeyEvent.VK_S);
// setMnemonicLabelFor(, , KeyEvent.VK_B);
// chbBkSwEinheitlich.setMnemonic(KeyEvent.VK_H);
// ftfBkStreitwert.setColumns(40);
// "Die Beklagten wurden ganz oder z.T. gesamtschuldnerisch verurteilt",
// chBkGesamtschuldner.setMnemonic(KeyEvent.VK_G);
// "Gesamtschuldnerisch verurteilte Beklagten auswählen"
// setMnemonicLabelFor(, , KeyEvent.VK_V);
// (new BeteiligtenRenderer());
// "Höhe der Verurteilung",
// ftfBkGesamtUnterliegen.setColumns(40);
// setMnemonicLabelFor(, , KeyEvent.VK_H);
// "Folgende Beklagte wurden darüber hinaus zu weiterer Zahlung verurteilt:"
// setMnemonicLabelFor(, , KeyEvent.VK_D);
// "Bitte wählen Sie den Beklagten aus, der auch Widerkläger ist."
// (new BeteiligtenRenderer());
// setMnemonicLabelFor(, , KeyEvent.VK_B);
// "Wie viele Drittwiderbeklagte gibt es?",
// setMnemonicLabelFor(, , KeyEvent.VK_D);
// "auch der Kläger ist Widerbeklagter", true
// chbKlWk.setMnemonic(KeyEvent.VK_U);
// "Genus der Drittwiderbeklagten"
// setMnemonicLabelFor(, , KeyEvent.VK_G);
// "Alle Widerbeklagten wurden in derselben Höhe in Anspruch genommen", true
// chbWkEinheitlich.setMnemonic(KeyEvent.VK_H);
// "Streitwert der Widerklage:",
// setMnemonicLabelFor(, , KeyEvent.VK_S);
// "Streitwert für den Kläger:"
// setMnemonicLabelFor(, , KeyEvent.VK_K);
// "Streitwert bzgl. der Drittwiderbeklagten:"
// setMnemonicLabelFor(, , KeyEvent.VK_D);
// "Einige Widerbeklagte wurden gesamtschuldnerisch verurteilt.",
// .setMnemonic(KeyEvent.VK_E);
// "Folgende Widerbeklagte wurde gesamtschuldnerisch verurteilt"
// (new BeteiligtenRenderer());
// setMnemonicLabelFor(, , KeyEvent.VK_F);
// "Höhe der Verurteilung", SwingConstants.RIGHT
// setMnemonicLabelFor(, , KeyEvent.VK_H);
// "Folgende Kläger wurden darüber hinaus verurteilt:"
// setMnemonicLabelFor(, , KeyEvent.VK_K);
// "Folgende Drittwiderbeklagte wurden darüber hinaus verurteilt:"
// setMnemonicLabelFor(, , KeyEvent.VK_D);
//.leseComponents(panels[tab].getComponents());
//if ((tab + 1) < HOECHSTER_ZUSTAND_OHNE_WIDERKLAGE || ((tab + 1) < HOECHSTER_ZUSTAND && isMitWiderklage())) {
//System.out.println("Uebertrage: " + tab);
//pruefer.leseComponents(panels[tab + 1].getComponents());
//} else {
//System.err.println("Will nicht: " + tab);
//}
// System.out.println(b);
// /**
// * @param wkGesamtUnterliegen d. {@link #wkGesamtUnterliegen}, d. gesetzt
// werden
// * soll als double
// */
// private void setWkGesamtUnterliegen(double wkGesamtUnterliegen) {
// this.wkGesamtUnterliegen = wkGesamtUnterliegen;
// }

/*
 * @param mitWiderklage d. {@link #mitWiderklage}, d. gesetzt werden soll als
 * boolean private void setMitWiderklage(boolean mitWiderklage) {
 * this.sonstigeWerte.mitWiderklage = mitWiderklage; }
 */

/*
 * @param wkStreitwert d. {@link #wkStreitwert}, d. gesetzt werden soll als
 * double private void setWkStreitwert(double wkStreitwert) {
 * this.sonstigeWerte.wkStreitwert = wkStreitwert; }
 */

/*
 * @param klWk d. {@link #klWk}, d. gesetzt werden soll als boolean private void
 * setKlWk(boolean klWk) { sonstigeWerte.klWk = klWk; }
 */

/*
 * @return gibt {@link #mitWiderklage} als boolean zurück. public boolean
 * isMitWiderklage() { return sonstigeWerte.mitWiderklage; }
 */

/*
 * @return gibt {@link #wkStreitwert} als double zurück. public double
 * getWkStreitwert() { return sonstigeWerte.wkStreitwert; }
 */

/*
 * @return gibt {@link #klWk} als boolean zurück. public boolean isKlWk() {
 * return sonstigeWerte.klWk; }
 */

/*
 * @return gibt {@link #wkGesamtUnterliegen} als double zurück. public double
 * getWkGesamtUnterliegen() { return sonstigeWerte.wkGesamtUnterliegen; }
 */
/*
 * Die Methode prüft, ob in einem Array int[] ein bestimmter int enthalten ist.
 * 
 * @param indices das zu durchsuchende Array int[]
 * 
 * @param index der int-Wert, der enthalten sein könnte
 * 
 * @return true, wenn index in indices enthalten ist. private boolean
 * contains(int[] indices, int index) { boolean contains = false; for (int i :
 * indices) if (i == index) contains = true; return contains; }
 */

// System.out.println(zustand);
//if (primitives.mitWiderklage) {
//	return ((StreitgenossenAssistentPruefer) pruefer).getZustand() == HOECHSTER_ZUSTAND - 1 && parseEingabe(10);
//} else {
//	return ((StreitgenossenAssistentPruefer) pruefer).getZustand() == HOECHSTER_ZUSTAND_OHNE_WIDERKLAGE - 1
//			&& parseEingabe(5);
//}

/*
 * Die Variante von parseEingabe, die nach Import von Xjustiz genutzt werden
 * soll. Überprüft die Eingabe und übergibt sie an das nächste Tab, füllt dieses
 * aber, wenn möglich mit Daten aus Xjustiz.
 * 
 * Der Konstruktor für vdc überprüft schon, ob genug Kläger und Beklagte da sind
 * und setzt die Anzahl dieser Beteiligten.
 * 
 * Um das eigentliche Überspringen muss sich der Weiter-Button kümmern.
 * 
 * Bei den zu überspringenden Tabs werden die zweidimensionalen Object-Arrays
 * gefüllt, die sbtm-Modelle angelegt, setModel bei den JTables ausgeführt und
 * die jeweiligen Streitwert-Textfelder gefüllt.
 * 
 * @param tab die Nummer des Tabulators
 * 
 * @param ueberspringen <code>true</code> = es sollen die bei Xjustiz-Import
 * unnötigen Tabs übersprungen werden
 * 
 * @return true, wenn zum nächsten, nicht zu überspringenden Tab gewechselt
 * werden kann
 */
//public boolean parseEingabe(int tab, boolean ueberspringen) {
//	if (!ueberspringen)
//		return parseEingabe(tab);
//	else {
//		boolean boole = parseEingabe(tab);
//		if (tab == 0) {
//			setzeWerte(getVerfahrensDatenContainer().streitgenossen, false);
//		} else if (tab == 5) {
//			setzeWerte(getVerfahrensDatenContainer().streitgenossen, true);
//		}
//		return boole;
//	}
//}

/*
 * Überprüft die Eingabe und übergibt sie an das nächste Tab
 * 
 * @param tab die Nummer des Tabulators
 * 
 * @return true, wenn das Überprüfen und Übergeben klappt
 */
//private boolean parseEingabe(int tab) {
//	/*
//	 * Mögliche Exceptions und RuntimeExceptions müssen abgefangen werden
//	 */
//	// setPrimitives darf nur einmal aufgerufen werden! - aber wo?
//	primitives.ohneXjustiz = getVerfahrensDatenContainer() == null;
//	((StreitgenossenAssistentPruefer) pruefer).setPrimitives(primitives);
//	pruefer.leseComponents(getComponents());
//	boolean boole = false;
//	try {
//		boole = ((StreitgenossenAssistentPruefer) pruefer).parseEingabe(tab);
//	} catch (IllegalArgumentException e) {
//		FehlerDialog.zeigeFehler(FehlerDialog.FEHLER_BLANKO, e.getLocalizedMessage(), e);
//		boole = false;
//	} catch (NullPointerException e) {
//		FehlerDialog.zeigeFehler(FehlerDialog.FEHLER_BLANKO, e.getLocalizedMessage(), e);
//		boole = false;
//	}
//	return boole;
//}
// BaumbachBeteiligtenListe klaeger =
// ScalableBeteiligtenTableModel.konvertiere(objectsKlaeger);
// BaumbachBeteiligtenListe beklagte =
// ScalableBeteiligtenTableModel.konvertiere(objectsBeklagte);
// BaumbachBeteiligtenListe drittwiderbeklagte;
// String aktenzeichen;
// double streitwert;
// // Höchsten Streitwert ermitteln
// if (primitives.wkStreitwert > primitives.bkStreitwert)
// streitwert = primitives.wkStreitwert;
// else
// streitwert = primitives.bkStreitwert;
// // streitwert = vdc.allgemein.streitwert;
// if (!primitives.ohneXjustiz) {
// aktenzeichen = getVerfahrensDatenContainer().allgemein.aktenzeichen;
// } else {
// aktenzeichen = "";
// }
// if (primitives.mitWiderklage) {
// drittwiderbeklagte = ScalableBeteiligtenTableModel.konvertiere(objectsDwbk);
// } else {
// drittwiderbeklagte = null;
// }
// if (primitives.bkGesamt) {
// /* BaumbachGesamtschuldnerschaft bk bauen */
// // Ist genusNumerus Plural?
// beklagte.add(
// new BaumbachGesamtschuldnerschaft(Beteiligter.BEKLAGTE, Beteiligter.MAENNLICH
// + Beteiligter.PLURAL,
// primitives.bkGesamtUnterliegen, false, primitives.bkGesamtIndices,
// beklagte));
// }
// /*
// * BaumbachGesamtschuldnerschaft kl+dbwk bauen, sofern Voraussetzungen. Zur
// * Sicherheit (auch wenn's eigentlich schon früher abgefragt wird) auch
// * Bedingungen gegen NullPointerExceptions
// */
// if (primitives.wkGesamt && drittwiderbeklagte != null &&
// drittwiderbeklagte.size() > 0) {
// BaumbachBeteiligtenListe zwischen = new BaumbachBeteiligtenListe();
// zwischen.addAll(klaeger);
// zwischen.addAll(drittwiderbeklagte);
// drittwiderbeklagte.add(new
// BaumbachGesamtschuldnerschaft(Beteiligter.DRITTWIDERBEKLAGTE,
// Beteiligter.MAENNLICH + Beteiligter.PLURAL, primitives.wkGesamtUnterliegen,
// true,
// primitives.wkGesamtIndices, zwischen));
// }
// return VerfahrensDatenContainerUtil.erstelleContainer(klaeger,
// drittwiderbeklagte, beklagte, streitwert,
// aktenzeichen);
// private boolean mitWiderklage;
// JTextArea
// private Beteiligter[] beteiligteBeklagte;
// private Beteiligter[] beteiligteWiderbeklagte;
//super.setOkActionEnabled(false);
//super.setOkActionEnabled(true);
//getButtonPane().add(new JButton(weiter));
// buttonBackup);// weiter);
// "Assistent für die Baumbach'sche Kostenformel"
// new SpinnerNumberModel(primitives.klaegerZahl, 1, 1, 1)
// new SpinnerNumberModel(primitives.beklagtenZahl, 0, 100, 1)
// objectsKlaeger = new Object[1][4];
// sbtmKlaeger = new ScalableBeteiligtenTableModel(objectsKlaeger,
// Beteiligter.KLAEGER);
// sbtmKlaeger,
// objectsBeklagte = new Object[2][3];
// sbtmBeklagte = new ScalableBeteiligtenTableModel(objectsBeklagte,
// Beteiligter.BEKLAGTE);
// sbtmBeklagte,
// ftfBkStreitwert.setText(Double.toString(primitives.bkStreitwert));
// sbtmBeklagte = new ScalableBeteiligtenTableModel(objectsBeklagte,
// Beteiligter.BEKLAGTE, 10000.0);
// sbtmBeklagte,
/* Noch einmal auf ausgewählt setzen, damit der ChangeListener ausgelöst wird */
// chbBkSwEinheitlich.setSelected(true);
// new Beteiligter[0]
// ftfBkGesamtUnterliegen.setText("10");
/* Und einmal die Auswahl setzen, um ein ChangeEvent auszulösen */
// chBkGesamtschuldner.setSelected(true);
// sbtmBeklagte = new ScalableBeteiligtenTableModel(objectsBeklagte,
// Beteiligter.BEKLAGTE, 10000.0, 0.0);
// sbtmBeklagte,
// new SpinnerNumberModel(primitives.dwbkZahl, 0, 100, 1)
// objectsDwbk = new Object[primitives.dwbkZahl][4];
// sbtmDrittwiderbeklagte = new ScalableBeteiligtenTableModel(objectsDwbk,
// Beteiligter.DRITTWIDERBEKLAGTE);
// sbtmDrittwiderbeklagte,
// chbWkEinheitlich.setSelected(true);
// ftfWkSw.setText(Double.toString(primitives.wkStreitwert));
// sbtmKlaeger = new ScalableBeteiligtenTableModel(objectsKlaeger,
// Beteiligter.KLAEGER, 0.0);
// sbtmKlaeger,
// sbtmDrittwiderbeklagte = new ScalableBeteiligtenTableModel(objectsDwbk,
// Beteiligter.DRITTWIDERBEKLAGTE, 0.0);
// sbtmDrittwiderbeklagte,
// chbWkGesamtschuld.setSelected(true);
// ftfWkUnterliegen.setText(Double.toString(primitives.wkGesamtUnterliegen));
// sbtmKlaeger = new ScalableBeteiligtenTableModel(objectsKlaeger,
// Beteiligter.KLAEGER, primitives.wkStreitwert,
// primitives.wkUnterliegen);
// sbtmKlaeger,
// sbtmDrittwiderbeklagte = new ScalableBeteiligtenTableModel(objectsDwbk,
// Beteiligter.DRITTWIDERBEKLAGTE,
// primitives.wkStreitwert, primitives.wkUnterliegen);
// sbtmDrittwiderbeklagte,
//private ScalableBeteiligtenTableModel sbtmKlaeger;
//private Object[][] objectsKlaeger;
//private Object[][] objectsBeklagte;
//private ScalableBeteiligtenTableModel sbtmBeklagte;
//private ScalableBeteiligtenTableModel sbtmDrittwiderbeklagte;
//private Object[][] objectsDwbk;
//  // "Assistent für die Baumbach'sche Kostenformel (Xjustiz-Import)");
// if (container == null)
// throw new IllegalArgumentException("Leere Datei oder Datei konnte nicht gelesen werden!");
// this.primitives.mitWiderklage = container.streitgenossen.widerklage;
// if (!VerfahrensDatenContainerUtil.gibtsKlaeger(container) || container.streitgenossen.klaeger.size() != 1)
// throw new IllegalArgumentException("Nur ein Kläger erlaubt!");
// primitives.klaegerZahl = container.streitgenossen.klaeger.size();
// primitives.klWk = (primitives.mitWiderklage
// && container.streitgenossen.klaeger.enthaeltAnWiderklageBeteiligte());
// /* Gibt's überhaupt Beklagte und wenn ja, maximal einen Widerkläger? */
// if (!VerfahrensDatenContainerUtil.gibtsBeklagte(container))
// throw new IllegalArgumentException("Datei enthält keine Beklagten!");
// else if (container.streitgenossen.beklagte.zaehleAnWiderklageBeteiligte() > 1)
// throw new IllegalArgumentException("Nur ein Widerkläger erlaubt!");
// primitives.beklagtenZahl = container.streitgenossen.beklagte.size();
// if (primitives.mitWiderklage) {
// for (int i = 0; i < container.streitgenossen.beklagte.size(); i++)
// if (container.streitgenossen.beklagte.get(i).isAnWiderklageBeteiligt())
// primitives.bkWiderklaegerIndex = i;
// }
// if (VerfahrensDatenContainerUtil.gibtsDrittwiderbeklagte(container.streitgenossen))
// primitives.dwbkZahl = container.streitgenossen.drittwiderbeklagte.size();
// else
// primitives.dwbkZahl = 0;
// /*
// * Gibt es Widerkläger ohne Widerbeklagte oder Widerbeklagte ohne Widerkläger?
// */
// if (this.primitives.mitWiderklage) {
// if (container.streitgenossen.beklagte.zaehleAnWiderklageBeteiligte() > 0
// && container.streitgenossen.klaeger.zaehleAnWiderklageBeteiligte() < 1 && primitives.dwbkZahl < 1)
// throw new IllegalArgumentException("Datei enthält Widerkläger ohne Widerbeklagte!");
// if ((container.streitgenossen.klaeger.zaehleAnWiderklageBeteiligte() > 0 || primitives.dwbkZahl > 0)
// && container.streitgenossen.beklagte.zaehleAnWiderklageBeteiligte() < 1)
// throw new IllegalArgumentException("Datei enthält Widerbeklagte ohne Widerkläger!");
// }
//  // hier einen Default-Wert für den Streitwert nehmen (z.B. Xjustiz)
// primitives.bkStreitwert = container.allgemein.streitwert;
// primitives.wkStreitwert = container.allgemein.streitwert;
// primitives.ohneXjustiz = getVerfahrensDatenContainer() == null;