/*
 * BeteiligtenCellEditor.java
 * eu.gronos.kostenrechner.view.baumbach (Kostenrechner)
 */
package eu.gronos.kostenrechner.view.baumbach;

import java.awt.Component;

import javax.swing.DefaultCellEditor;
import javax.swing.JComboBox;
import javax.swing.JTable;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableModel;

import eu.gronos.kostenrechner.model.tenordaten.Beteiligter;

/**
 * Ein {@link DefaultCellEditor} für das Genus-/Numerus-Feld bei
 * {@link Beteiligter}n
 *
 * @author Peter Schuster (setrok)
 * @date 27.09.2019
 *
 */
public class BeteiligtenCellEditor extends DefaultCellEditor implements TableModelListener {
	private static final long serialVersionUID = -1251100602581421021L;
	public static final BeteiligtenListRenderer BETEILIGTEN_RENDERER_OHNE_LFDNR = new BeteiligtenListRenderer(true);
	private TableModel model;

	/**
	 * Konstruktor.
	 * 
	 * @param typ eine der Konstanten KLAEGER, DRITTWIDERBEKLAGTER, BEKLAGTER
	 */
	public BeteiligtenCellEditor(int typ) {
		super(baueComboBox(typ));
	}

	@Override
	public Object getCellEditorValue() {
		Object cellEditorValue = super.getCellEditorValue();
		remove();
		return cellEditorValue;
	}

	@Override
	public boolean stopCellEditing() {
		boolean stopCellEditing = super.stopCellEditing();
		remove();
		return stopCellEditing;
	}

	@Override
	public void cancelCellEditing() {
		super.cancelCellEditing();
		remove();
	}

	/**
	 * Die Methode wird aufgerufen, wenn sich der genaue Umfang von Zellen, Zeilen
	 * oder Spalten geändert hat. Sie ruft {@link #cancelCellEditing()} auf, wenn
	 * die Tabelle geändert wird
	 * 
	 * @param e ein {@link TableModelEvent}
	 * 
	 * @see javax.swing.event.TableModelListener#tableChanged(javax.swing.event.TableModelEvent)
	 */
	@Override
	public void tableChanged(TableModelEvent e) {
		cancelCellEditing();
		((AbstractTableModel) e.getSource()).removeTableModelListener(this);
	}

	/**
	 * Sets an initial value for the editor. This will cause the editor to
	 * stopEditing and lose any partially edited value if the editor is editing when
	 * this method is called. Returns the component that should be added to the
	 * client's Component hierarchy. Once installed in the client's hierarchy this
	 * component will then be able to draw and receive user input.
	 * 
	 * @param table      table - the JTable that is asking the editor to edit; can
	 *                   be null
	 * @param value      the value of the cell to be edited; it is up to the
	 *                   specific editor to interpret and draw the value. For
	 *                   example, if value is the string "true", it could be
	 *                   rendered as a string or it could be rendered as a check box
	 *                   that is checked. null is a valid value
	 * @param isSelected true if the cell is to be rendered with highlighting
	 * @param row        the row of the cell being edited
	 * @param column     the column of the cell being edited
	 * @return the component for editing
	 * 
	 * @url "http://stackoverflow.com/questions/19030261/jtextfield-selectall-works-strangely"
	 *      JTextField.selectAll() works strangely
	 * 
	 * @see javax.swing.DefaultCellEditor#getTableCellEditorComponent(javax.swing.JTable,
	 *      java.lang.Object, boolean, int, int)
	 */
	@Override
	public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column) {
		Component component = super.getTableCellEditorComponent(table, value, isSelected, row, column);
		model = table.getModel();
		model.addTableModelListener(this);
		return component;
	}

	private static JComboBox<Beteiligter> baueComboBox(int typ) {
		JComboBox<Beteiligter> comboBox = new JComboBox<Beteiligter>(
				Beteiligter.getBeteiligtenAuswahlListe(typ, false));
		comboBox.setRenderer(BETEILIGTEN_RENDERER_OHNE_LFDNR);
		return comboBox;
	}

	/**
	 * Die Methode entfernt sich als Listener wieder.
	 */
	private void remove() {
		if (model != null)
			model.removeTableModelListener(this);
	}

}
